%  Copyright (C) 2003-2004 David Roundy
%
%  This program is free software; you can redistribute it and/or modify
%  it under the terms of the GNU General Public License as published by
%  the Free Software Foundation; either version 2, or (at your option)
%  any later version.
%
%  This program is distributed in the hope that it will be useful,
%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%  GNU General Public License for more details.
%
%  You should have received a copy of the GNU General Public License
%  along with this program; if not, write to the Free Software Foundation,
%  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
\subsection{darcs apply}
\begin{code}
module Apply ( apply ) where
import System ( ExitCode(..), exitWith )
import Prelude hiding ( catch )
import IO ( hClose, stdin, stdout, stderr )
import Control.Exception ( catch, throw, Exception( ExitException ) )
import Monad ( when, unless, liftM )
import List ( nub, sort )
import Maybe ( catMaybes, isNothing )

import SignalHandler ( withSignalsBlocked )
import DarcsCommands ( DarcsCommand(..) )
import DarcsArguments ( DarcsFlag( Reply, Test, NoTest, AnyOrder,
                                   Gui, Interactive, All,
                                   MarkConflicts,
                                   AllowConflicts, Verbose, HappyForwarding
                                 ),
                        get_cc, want_external_merge, working_repo_dir,
                        notest, nocompress, apply_conflict_options,
                        ignoretimes, verbose,
                        reply, verify, list_files,
                        all_gui_interactive,
                        leave_test_dir, happy_forwarding, set_scripts_executable
                      )
import qualified DarcsArguments ( cc )
import Repository ( PatchSet,
                    add_to_inventory, get_unrecorded, slurp_recorded,
                    slurp_pending, read_repo, am_in_repo,
                    write_pending, sync_repo,
                  )
import Pristine ( identifyPristine, write_dirty_Pristine )
import Patch ( Patch, patch2patchinfo, invert, list_touched_files,
               join_patches, unjoin_patches, apply_to_slurpy,
               eq_patches, null_patch,
             )
import PatchInfo ( human_friendly )
import SlurpDirectory ( wait_a_moment, slurp_write_dirty, co_slurp, )
import FastPackedString ( readFilePS, packString, unpackPS, hGetContentsPS,
                          linesPS, takePS, dropPS, PackedString,
                          unlinesPS )
import External ( sendEmail, sendEmailDoc, resendEmail,
                  verifyPS )
import Email ( read_email )
import Lock ( withLock, withStdoutTemp, readBinFile )
import Pull ( merge_with_us_and_pending, save_patches,
              check_unrecorded_conflicts )
import Depends ( get_common_and_uncommon )
import Resolution ( standard_resolution, no_resolution, external_resolution )
import SelectChanges ( with_selected_changes )
import Test ( test_slurpy )
import PatchBundle ( scan_bundle )
import DarcsUtils ( putStrLnError )
import Printer ( packedString, putDocLn, vcat, text, ($$), errorDoc )
#include "impossible.h"
\end{code}
\begin{code}
apply_description :: String
apply_description =
 "Apply patches to a repo."
\end{code}

\options{apply}

\haskell{apply_help}
\begin{code}
apply_help :: String
apply_help =
 "Apply is used to apply a bundle of patches to this repository.\n"++
 "Such a bundle may be created using send.\n"
\end{code}
\begin{code}
stdin_magic :: String
stdin_magic = "magic darcs standard input"
stdindefault :: [String] -> IO [String]
stdindefault [] = return [stdin_magic]
stdindefault x = return x
apply :: DarcsCommand
apply = DarcsCommand {command_name = "apply",
                      command_help = apply_help,
                      command_description = apply_description,
                      command_extra_args = 1,
                      command_extra_arg_help = ["<PATCHFILE>"],
                      command_command = apply_cmd,
                      command_prereq = am_in_repo,
                      command_get_arg_possibilities = list_files,
                      command_argdefaults = stdindefault,
                      command_darcsoptions = [verify, reply,
                                              DarcsArguments.cc,
                                              verbose, ignoretimes,
                                              nocompress,
                                              all_gui_interactive,
                                              apply_conflict_options,
                                              notest, happy_forwarding,
                                              leave_test_dir,
                                              working_repo_dir,
                                              set_scripts_executable]}
\end{code}
\begin{code}
apply_cmd :: [DarcsFlag] -> [String] -> IO ()
apply_cmd opts [patchesfile] = withLock "./_darcs/lock" $
 with_patches_file patchesfile $ \ps -> do
  am_verbose <- return $ Verbose `elem` opts
  let from_whom = get_from ps
  us <- read_repo "."
  either_them <- get_patch_bundle opts ps
  them <- case either_them of
          Right t -> return t
          Left er -> do forwarded <- consider_forwarding opts ps
                        if forwarded
                          then exitWith ExitSuccess
                          else fail er
  (_, us', them') <- return $ get_common_and_uncommon (us, them)
  s <- slurp_recorded "."
  with_selected_changes "apply" fixed_opts s
                            (map fromJustTheirPatch $ reverse $ head them') $
                            \ (_,to_be_applied) -> do
   when (null to_be_applied) $
        do putStrLn "You don't want to apply any patches, so I'm exiting!"
           exitWith ExitSuccess
   redirect_output opts from_whom $ do
    when am_verbose $ putStrLn "We have the following extra patches:"
    when am_verbose $ putDocLn $ vcat $ map (human_friendly.fst) $ head us'
    when am_verbose $ putStrLn "Will apply the following patches:"
    when am_verbose $ putDocLn $ vcat $
         map (human_friendly.fromJust.patch2patchinfo) $ to_be_applied
    (us_patch, work_patch) <- merge_with_us_and_pending opts
                              (map fromJustOurPatch $ reverse $ head us',
                               to_be_applied)
    recorded <- slurp_recorded "."
    recorded_with_pending <- slurp_pending "."
    working <- co_slurp recorded_with_pending "."
    standard_resolved_pw <- standard_resolution work_patch
    announce_merge_conflicts opts standard_resolved_pw
    check_unrecorded_conflicts us_patch
    pw_resolved <-
       if AllowConflicts `elem` opts
       then join_patches `liftM` no_resolution work_patch
       else case want_external_merge opts of
            Nothing -> return $ join_patches standard_resolved_pw
            Just c -> do pend <- get_unrecorded (AnyOrder:opts)
                         join_patches `liftM` external_resolution c working
                              (join_patches $ (++catMaybes [pend]) $
                               map (fromJust.snd) $ reverse $ head us')
                              (join_patches $ map (fromJust.snd) $ reverse $ head them')
                              work_patch
    when am_verbose $ putStrLn "Applying patches to the local directories..."
    case apply_to_slurpy us_patch recorded of
        Nothing -> fail "Error applying patch to recorded!"
        Just rec' ->
            case apply_to_slurpy pw_resolved working of
            Nothing -> fail "Error applying patch to working dir."
            Just work' -> do
                when (not (NoTest `elem` opts) && Test `elem` opts) $
                  do recb <- slurp_recorded "."
                     testproblem <- test_slurpy opts $
                                    fromJust $ apply_to_slurpy us_patch recb
                     when (testproblem /= ExitSuccess) $ do
                          putStrLnError "Error in test..."
                          exitWith $ ExitFailure 1
                save_patches opts $ unjoin_patches us_patch
                mp <- get_unrecorded (AnyOrder:opts)
                withSignalsBlocked $ do
                  identifyPristine >>= write_dirty_Pristine rec'
                  wait_a_moment -- so work will be more recent than rec
                  sequence $ map (add_to_inventory ".".fromJust.patch2patchinfo)
                           to_be_applied
                  slurp_write_dirty opts work'
                  unless (isNothing mp && pw_resolved `eq_patches` us_patch) $
                       write_pending $ join_patches
                           [invert us_patch, fromMaybePatch mp, pw_resolved]
                sync_repo
                putStrLn "Finished applying..."
                exitWith ExitSuccess
     where fixed_opts = if Gui `elem` opts || Interactive `elem` opts
                        then opts
                        else All : opts
           fromJustTheirPatch (pinfo, Nothing)
               = errorDoc $ text "Cannot apply this patch bundle, since we're missing:"
                         $$ human_friendly pinfo
           fromJustTheirPatch (_, Just p) = p
           fromJustOurPatch (pinfo, Nothing)
               = errorDoc $ text ("Cannot apply this patch bundle, "
                               ++ "this is a \"--partial repository")
                         $$ text "We don't have the following patch:"
                         $$ human_friendly pinfo
           fromJustOurPatch (_, Just p) = p
           fromMaybePatch Nothing = null_patch
           fromMaybePatch (Just p) = p
apply_cmd _ _ = impossible
\end{code}

Darcs apply accepts a single argument, which is the name of the patch file
to be applied.  If you omit this argument, the patch is read from standard
input.\footnote{One caveat: don't name your patch file ``magic darcs
standard input'', or darcs will read from standard input instead!}  This
allows you to use apply with a pipe from your email program, for example.

\begin{code}
with_patches_file :: FilePath -> (PackedString -> IO a) -> IO a
with_patches_file fn c
    | fn == stdin_magic = do ps <- hGetContentsPS stdin
                             c ps
    | otherwise = do ps <- readFilePS fn
                     c ps
\end{code}

\begin{options}
--verify
\end{options}

If you specify the \verb!--verify PUBRING! option, darcs will check that
the patch was gpg-signed by a key which is in \verb!PUBRING!, and will
refuse to apply the patch otherwise.

\begin{code}
get_patch_bundle :: [DarcsFlag] -> PackedString
                 -> IO (Either String PatchSet)
get_patch_bundle opts fps = do
    mps <- verifyPS opts $ read_email fps
    mops <- verifyPS opts fps
    case (mps, mops) of
      (Nothing, Nothing) ->
          return $ Left "Patch bundle not properly signed, or gpg failed."
      (Just ps, Nothing) -> return $ scan_bundle ps
      (Nothing, Just ps) -> return $ scan_bundle ps
      (Just ps1, Just ps2) -> case scan_bundle ps1 of
                              Left _ -> return $ scan_bundle ps2
                              Right x -> return $ Right x
\end{code}

\begin{options}
--cc, --reply
\end{options}

If you give the \verb!--reply FROM! option to darcs apply, it will send the
results of the application to the sender of the patch.  This only works if
the patch is in the form of an email with its headers intact, so that darcs
can actually know the origin of the patch.  The reply email will indicate
whether or not the patch was successfully applied.  The \verb!FROM! flag is
the email address that will be used as the ``from'' address when replying.
If the darcs apply is being done automatically, it is important that this
address not be the same as the address at which the patch was received in
order to avoid automatic email loops.

If you want to also send the apply email to another address (for example,
to create something like a ``commits'' mailing list), you can use the
\verb!--cc! option to specify additional recipients.  Note that the
\verb!--cc! option \emph{requires} the \verb!--reply! option, which
provides the ``From'' address.

The \verb!--reply! feature of apply is intended primarily for two uses.
When used by itself, it is handy for when you want to apply patches sent to
you by other developers so that they will know when their patch has been
applied.  For example, in my \verb!.muttrc! (the config file for my mailer)
I have:
\begin{verbatim}
macro pager A "<pipe-entry>darcs apply --verbose \
        --reply droundy@abridgegame.org --repodir ~/darcs
\end{verbatim}
which allows me to apply a patch to darcs directly from my mailer, with the
originator of that patch being sent a confirmation when the patch is
successfully applied.  NOTE: For some reason mutt seems to set the umask
such that patches created with the above macro are not world-readable.  I'm
not sure why this is, but use it with care.

When used in combination with the \verb!--verify! option, the
\verb!--reply! option allows for a nice pushable repository.  When these
two options are used together, any patches that don't pass the verify will
be forwarded to the \verb!FROM! address of the \verb!--reply! option.  This
allows you to set up a repository so that anyone who is authorized can push
to it and have it automatically applied, but if a stranger pushes to it,
the patch will be forwarded to you.  Please (for your own sake!) be certain
that the \verb!--reply FROM! address is different from the one used to send
patches to a pushable repository, since otherwise an unsigned patch will be
forwarded to the repository in an infinite loop.

If you use `\verb!darcs apply --verify PUBRING --reply!' to create a
pushable repo by applying patches automatically as they are received by
email, you will also want to use the \verb!--dont-allow-conflicts! option.

\begin{options}
--dont-allow-conflicts
\end{options}
The \verb!--dont-allow-conflicts! flag causes apply to fail when applying a
patch would cause conflicts.  This flag is recommended on repositories
which will be pushed to or sent to.

\begin{options}
--allow-conflicts
\end{options}

\verb!--allow-conflicts! will allow conflicts, but will keep the local and
recorded versions in sync on the repo.  This means the conflict will exist
in both locations until it is resolved.

\begin{options}
--mark-conflicts
\end{options}

\verb!--mark-conflicts! will add conflict markers to illustrate the the
conflict.

\begin{code}
announce_merge_conflicts :: [DarcsFlag] -> [Patch] -> IO ()
announce_merge_conflicts opts resolved_pw =
    case nub $ sort $ list_touched_files $ join_patches $ tail resolved_pw of
    [] -> return ()
    cfs -> if MarkConflicts `elem` opts || AllowConflicts `elem` opts
              || want_external_merge opts /= Nothing
           then do putStrLn "We have conflicts in the following files:"
                   putStrLn $ unwords cfs
           else do putStrLn "There are conflicts in the following files:"
                   putStrLn $ unwords cfs
                   fail "Refusing to apply patches leading to conflicts."
\end{code}

\begin{options}
--external-merge
\end{options}

You can use an external interactive merge tool to resolve conflicts with the
flag \verb!--external-merge!.  For more details see
subsection~\ref{resolution}.

\begin{options}
--all, --gui, --interactive
\end{options}

If you provide the \verb!--interactive! or \verb!--gui! flag, darcs will
ask you for each change in the patch bundle whether or not you wish to
apply that change.  The opposite is the \verb!--all! flag, which can be
used to override an interactive or gui which might be set in your
``defaults'' file.

\begin{code}
get_from :: PackedString -> String
get_from ps = readFrom $ linesPS ps
    where readFrom [] = ""
          readFrom (x:xs)
           | takePS 5 x == from_start = unpackPS $ dropPS 5 x
           | otherwise = readFrom xs

redirect_output :: [DarcsFlag] -> String -> IO a -> IO a
redirect_output opts to doit = ro opts
    where
  cc = get_cc opts
  ro [] = doit
  ro (Reply f:_) =
    withStdoutTemp $ \tempf-> do {a <- doit;
                                  hClose stdout;
                                  hClose stderr;
                                  return a;
                                 } `catch` (sendit tempf)
        where sendit tempf e@(ExitException ExitSuccess) =
                do body <- sanitizeFile tempf
                   sendEmail f to "Patch applied" cc body
                   throwIO e
              sendit tempf (ExitException _) =
                do body <- sanitizeFile tempf
                   sendEmail f to "Patch failed!" cc body
                   throwIO $ ExitException ExitSuccess
              sendit tempf e =
                do body <- sanitizeFile tempf
                   sendEmail f to "Darcs error applying patch!" cc $
                             body ++ "\n\nCaught exception:\n"++
                             show e++"\n"
                   throwIO $ ExitException ExitSuccess
  ro (_:fs) = ro fs

-- sanitizeFile is used to clean up the stdout/stderr before sticking it in
-- an email.

sanitizeFile :: FilePath -> IO String
sanitizeFile f = sanitize `liftM` readBinFile f
    where sanitize s = wash $ remove_backspaces "" s
          wash ('\000':s) = "\\NUL" ++ wash s
          wash ('\026':s) = "\\EOF" ++ wash s
          wash (c:cs) = c : wash cs
          wash [] = []
          remove_backspaces rev_sofar "" = reverse rev_sofar
          remove_backspaces (_:rs) ('\008':s) = remove_backspaces rs s
          remove_backspaces "" ('\008':s) = remove_backspaces "" s
          remove_backspaces rs (s:ss) = remove_backspaces (s:rs) ss

throwIO :: Exception -> IO a
throwIO e = return $ throw e
\end{code}

\begin{code}
forwarding_message :: PackedString
forwarding_message = packString $
    "The following patch was either unsigned, or signed by a non-allowed\n"++
    "key, or there was a gpg failure.\n"

consider_forwarding :: [DarcsFlag] -> PackedString -> IO Bool
consider_forwarding opts m = cf opts (get_cc opts)
    where cf [] _ = return False
          cf (Reply t:_) cc =
              case break is_from (linesPS m) of
              (m1, f:m2) ->
                  let m_lines = forwarding_message:m1 ++ m2
                      m' = unlinesPS m_lines
                      f' = unpackPS (dropPS 5 f) in do
                      if HappyForwarding `elem` opts
                         then resendEmail t m
                         else sendEmailDoc f' t "A forwarded darcs patch" cc
                                           (packedString m')
                      return True
              _ -> return False -- Don't forward emails lacking headers!
          cf (_:fs) cc = cf fs cc
          is_from l = takePS 5 l == from_start

from_start :: PackedString
from_start = packString "From:"
\end{code}

\begin{options}
--no-test, --test
\end{options}

If you specify the \verb!--test! option, apply will run the test (if a test
exists) prior to applying the patch.  If the test fails, the patch is not
applied.  In this case, if the \verb!--reply! option was used, the results
of the test are send in the reply email.  You can also specify the
\verb!--no-test! option, which will override the \verb!--test! option, and
prevent the test from being run.  This is helpful when setting up a
pushable repository, to keep users from running code.


