/*
 * This file is part of din.
 *
 * din is copyright (c) 2006 - 2012 S Jagannathan <jag@dinisnoise.org>
 * For more information, please visit http://dinisnoise.org
 *
 * din is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * din is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with din.  If not, see <http://www.gnu.org/licenses/>.
 *
*/
#include "key_consts.h"
#include "main.h"
#include "din.h"
#include "console.h"
#include "solver.h"
#include "utils.h"
#include "input.h"
#include "color.h"
#include "random.h"
#include "command.h"
#include "delay.h"
#include "chrono.h"
#include "osc.h"
#include "delay.h"
#include "bot.h"
#include "tcl_interp.h"
#include "font.h"
#include "ansi_color_codes.h"
#include "scale_info.h"
#include "ui_list.h"
#include "keyboard_keyboard.h"
#include <sstream>
#include <algorithm>
using namespace std;

extern string dotdin; // ~/.din directory
extern console cons; // console
extern viewport view; // viewport
extern int mousex, mousey; // mouse pos
extern int lmb, rmb, mmb; // mouse button state
extern int LEFT, BOTTOM, RIGHT, TOP; // din board extents
extern int HEIGHT; // TOP - BOTTOM
extern float DELTA_VOLUME; // delta volume per unit key height
extern int LAST_VOLUME; // last volume in key height units
extern int NUM_VOLUMES; // number of available volumes
extern int NUM_MICROTONES; // default number of microtones in a range
extern int NUM_OCTAVES;
extern map<string, int> NOTE_POS; // interval name -> value, 1 - 1, 1# - 2, 2 - 3  etc
extern int SAMPLE_RATE; // sampling rate
extern map<string, float> INTERVALS; // interval name -> value
extern audio_out aout; // audio output
extern tcl_interp interpreter; // integrated tcl interpreter
extern scale_info scaleinfo; // scale information for microtonal and keyboard keyboard
extern int TRAIL_LENGTH; // drone trail length (== number of trail points)
extern int DRONE_HANDLE_SIZE;

extern chrono clk; // audio clock
extern din din0; // microtonal-keyboard
extern ui_list uis; // list of screens

din::din (cmdlist& cl) :
wave ("waveform1.crv"),
waved ("waveform1.ed"),
wavlis (wave_listener::DIN_BOARD),
win (0, 0, view.xmax, view.ymax),
drone_master_volume (0.01),
drone_wave ("drone.crv"),
droneed ("drone.ed"),
dronelis (wave_listener::DRONE),
fm ("Voice FM", "fm.crv"),
am ("Voice AM", "am.crv"),
gatr ("Gater", "gater.crv"),
gated ("gater.ed"),
gatlib ("gater-patterns.lib"),
gatrlis (&gatr),
moded ("modulation.ed"),
fmlis (&fm),
amlis (&am),
helptext ("din.hlp"),
notation ("numeric")
{

    name = "microtonal-keyboard";

    prev_mousex = 0;
    prev_mousey = 0;

    win_mousex = win_mousey = 0;

    current_range = 0;

    snap_drones = 0;
    do_drone_modulation = 1;
    droneed.add (&drone_wave, &dronelis);

    extern curve_library wavlib;
    droneed.attach_library (&wavlib);

    wavsol (&wave);
    wavplay.set_wave (&wavsol);

    waved.add (&wave, &wavlis);
    waved.attach_library (&wavlib);
    wavlis.edited (&waved, 0);

    gated.attach_library (&gatlib);
    gated.add (&gatr.crv, &gatrlis);
    gated.bv.push_back (&gatr);

    am_depth = 0;
    fm_depth = 0;

    moded.add (&fm.crv, &fmlis);
    moded.add (&am.crv, &amlis);
    moded.bv.push_back (&fm);
    moded.bv.push_back (&am);

    show_cursor_info = 0;

    rising = falling = 0;

}

void din::load_scale () {

  setup_ranges ();
  load_drones ();
  update_drone_tone ();
  update_drone_x (0, last_range);
  find_current_range ();

}

bool din::load_ranges () {

  string fname = dotdin + scaleinfo.name + ".ranges";
  cout << LOAD << "<< loading ranges from: " << fname;
  ifstream file (fname.c_str(), ios::in);
  if (!file) {
    cout << ENDL << FAIL << "!!! couldnt load range pos from " << fname << ", will use defaults +++" << ENDL;
    return false;
  }

  string ignore;

  file >> ignore >> NUM_OCTAVES;
  int n; file >> ignore >> n;
  create_ranges (n);

  int l = 0, r, w;
  for (int i = 0; i < num_ranges; ++i) {
    range& R = ranges[i];
    file >> ignore >> w;
    r = l + w;
    R.extents (l, BOTTOM, r, TOP);
    l = r;
  }

  cout << ", done >>>" << ENDL;
  return true;

}

void din::save_ranges () {

  string fname = dotdin + scaleinfo.name + ".ranges";
  ofstream file (fname.c_str(), ios::out);
  if (file) {
    file << "num_octaves " << NUM_OCTAVES << endl;
    file << "num_ranges " << num_ranges << endl;
    for (int i = 0; i < num_ranges; ++i) {
      range& r = ranges[i];
      file << i << ' ' << r.extents.width << endl;
    }
    cout << PASS << "+++ saved ranges in " << fname << " +++" << ENDL;
  }

}

void din::create_ranges (int n) {

  if (n > 0) {
    num_ranges = n;
    ranges.resize (num_ranges);
    last_range = num_ranges - 1;
    firstr = &ranges [0];
    lastr = &ranges [last_range];
  }

}

void din::setup_ranges (int load) {

  cout << DOING << "*** setting up ranges ***" << ENDL;

  if (load) {
    if (!load_ranges()) {
      create_ranges (NUM_OCTAVES * scaleinfo.num_ranges);
      set_range_size (0, last_range, NUM_MICROTONES);
    }
  } else { // number of octaves has changed
    int last_num_ranges = num_ranges;
    create_ranges (NUM_OCTAVES * scaleinfo.num_ranges);
    set_range_size (last_num_ranges, last_range, NUM_MICROTONES); // new ranges to default size, keep size of existing ranges
  }

  setup_range_notes ();
  update_range_notes ();
  calc_range_label ();
  notate_ranges ();
  find_current_range ();

  cout << PASS "+++ setup ranges +++" << ENDL;

}

void din::setup_range_notes () {

  for (int p = 0, r = 0; p < NUM_OCTAVES; ++p) {
    for (int i = 0, j = 1; i < scaleinfo.num_ranges; ++i, ++j) {
      range& R = ranges[r++];
      R.intervals[0] = scaleinfo.notes[i];
      R.intervals[1] = scaleinfo.notes[j];
    }
  }

}

void din::update_range_notes () {
  float octave_start = scaleinfo.lo_tonic;
  for (int p = 0, r = 0; p < NUM_OCTAVES; ++p) {
    for (int i = 0; i < scaleinfo.num_ranges; ++i) {
      ranges[r++].calc (p, octave_start, scaleinfo.intervals);
    }
    octave_start *= 2;
  }

}

void din::calc_range_label () {

  if (num_ranges) {
    range::char_height = get_line_height ();
    range::ybot1 = firstr->extents.bottom - range::char_height;
    range::ybot2 = range::ybot1 - range::char_height - range::spacer;
    range::ytop1 = firstr->extents.top + range::char_height;
    range::ytop2 = range::ytop1 + range::char_height + range::spacer;
  }

}

void din::set_range_size (int ran, int sz) {

  // set size of range ran and greater
  range& R = ranges[ran];
  int delta = sz - R.extents.width;
  R.extents (R.extents.left, BOTTOM, R.extents.left + sz, TOP);
  for (int i = ran + 1; i < num_ranges; ++i) {
    range& Ri = ranges[i];
    Ri.extents (Ri.extents.left + delta, Ri.extents.bottom, Ri.extents.right + delta, Ri.extents.top);
  }
  update_drone_x (ran, last_range);
  find_visible_ranges ();

}

void din::set_range_size (int s, int t, int sz) {

    int r, l;
    if (s < 1) {
      r = LEFT;
    } else r = ranges[s-1].extents.right;

    for (int i = s; i <= t; ++i) {
      l = r;
      r = l + sz;
      range& R = ranges[i];
      R.extents (l, BOTTOM, r, TOP);
    }

    update_drone_x (s, t);
    find_visible_ranges ();

}

void din::range_left_changed () {

  range& R = ranges [current_range];
  int old_left = R.extents.left;

  if (delta_mousex != 0) {
    R.extents (R.extents.left + delta_mousex, R.extents.bottom, R.extents.right, R.extents.top);
    int delta_left = R.extents.left - old_left;
    for (int i = 0; i < current_range; ++i) {
      range& ir = ranges [i];
      ir.extents (ir.extents.left + delta_left, ir.extents.bottom, ir.extents.right + delta_left, ir.extents.top);
    }
    update_drone_x (0, current_range);
    find_visible_ranges ();
  }

}

void din::range_right_changed () {

  range& R = ranges [current_range];
  int old_right = R.extents.right;

  if (delta_mousex != 0) {
    R.extents (R.extents.left, R.extents.bottom, R.extents.right + delta_mousex, R.extents.top);
    int delta_right = R.extents.right - old_right;
    for (int i = current_range + 1; i < num_ranges; ++i) {
      range& ir = ranges [i];
      ir.extents (ir.extents.left + delta_right, ir.extents.bottom, ir.extents.right + delta_right, ir.extents.top);
    }
    update_drone_x (current_range, last_range);
    find_visible_ranges ();
  }

}

void din::set_key_id (int i) {
  key_id = i;
  notate_ranges ();
}

void din::set_notation (const string& s) {
  if (s == "w" || s == "west" || s == "western") notation = "western";  else notation = "numeric";
  notate_ranges ();
}

void din::notate_ranges () {

  extern const char* WESTERN_FLAT [];
  if (notation == "western") {
    for (int i = 0; i < num_ranges; ++i) {
      range& ri = ranges [i];
      string i0 = ri.intervals[0], i1 = ri.intervals[1];
      int ii0 = NOTE_POS[i0], ii1 = NOTE_POS[i1];
      int kii0 = (key_id + ii0) % 12;
      int kii1 = (key_id + ii1) % 12;
      ri.notes[0].name = WESTERN_FLAT[kii0];
      ri.notes[1].name = WESTERN_FLAT[kii1];
    }
  } else if (notation == "numeric") {
    for (int i = 0; i < num_ranges; ++i) {
      range& ri = ranges [i];
      string i0 = ri.intervals[0], i1 = ri.intervals[1];
      ri.notes[0].name = i0;
      ri.notes[1].name = i1;
    }
  }

  // misc labels
  int m = num_ranges / scaleinfo.num_ranges;
  for (int i = 0; i < num_ranges; ++i) {
    range& ri = ranges [i];
    stringstream ss; ss << 1 + i / scaleinfo.num_ranges << '/' << m; ss >> ri.octave;
    if (ri.intervals[0] == "1") ri.key = range::LEFT; else ri.key = range::NONE; // highlight key note in green
  }
  ranges[last_range].key = range::RIGHT;

}

void din::set_tonic (float s) {
  if (scaleinfo.set_tonic(s)) {
    update_range_notes ();
    update_drone_tone ();
    notate_ranges ();
  }
}

void din::mouse2tonic () {

  // set mouse at tonic
  range& r = ranges[scaleinfo.num_ranges]; // range of middle tonic
  int dx = win_mousex - r.extents.left;
  warp_pointer (-dx, 0);

}

float din::get_tonic () {
  return scaleinfo.tonic;
}

float din::get_note_value (const string& s) {
  return scaleinfo.intervals[s];
}

void din::retune_note (const string& nn, float v) {
  float b4 = scaleinfo.intervals[nn];
  scaleinfo.intervals[nn] = v;
  update_range_notes ();
  update_drone_tone ();
  cons << console::green << "retuned note: " << nn << " from: " << b4 << " to " << v << eol;
}

void din::retune_note () {

  // find nearest note
  range& r = ranges[current_range];
  int left = r.extents.left, right = r.extents.right;
  int delta_left = win_mousex - left, delta_right = right - win_mousex;
  int i = 0; if (delta_left > delta_right) i = 1;
  string label(r.intervals[i]);

  float sas [] = {scaleinfo.lo_tonic, scaleinfo.tonic, scaleinfo.hi_tonic};
  float sa = sas [(int) octave_position];
  float freq = step * SAMPLE_RATE;
  float interval = freq / sa;
  label = r.intervals[i];
  float b4 = scaleinfo.intervals[label];
  scaleinfo.intervals [label] = interval;
  update_range_notes ();
  update_drone_tone ();

  cons << console::green << "retuned " << label << " from " << b4 << " to " << interval << eol;

}

void din::restore_note () {

  // find nearest note
  range& r = ranges[current_range];
  int left = r.extents.left, right = r.extents.right;
  int delta_left = win_mousex - left, delta_right = right - win_mousex;
  int i = 0; if (delta_left > delta_right) i = 1;
  string n (r.intervals[i]);
  if (n != "S") {
    n = r.intervals[i];
    cons << console::green << "restored " << n << " from " << scaleinfo.intervals[n] << " to " << INTERVALS[n] << eol;
    scaleinfo.intervals [n] = INTERVALS [n];
    update_range_notes ();
    update_drone_tone ();
  }
}

void din::restore_all_notes () {
  scaleinfo.intervals = INTERVALS;
  update_range_notes ();
  update_drone_tone ();
}

void din::save_scale () {
  save_ranges ();
  save_drones ();
  scaleinfo.save_custom_tuning ();
}

din::~din () {
  wave.save ("waveform1.crv");
  cout << FAIL << "--- destroyed microtonal-keyboard ---" << ENDL;
}

void din::load_drones () {

  string fdrone = dotdin + scaleinfo.name + ".drone";
  ifstream file (fdrone.c_str(), ios::in);
  drones.clear ();

  rising = falling = 0;

  if (!file) return; else {
    string ignore;
    int num_drones = 0;
    file >> ignore >> num_drones;
    file >> ignore >> drone_master_volume;
    last_drone_master_volume = 0;
    cout << LOAD << "<<< loading " << num_drones << " drones from: " << fdrone;
    for (int i = 0; i < num_drones; ++i) {
      drone di;
      file >> ignore >> di.cx >> di.cy;
      di.set_xy (*this, di.cx, di.cy);
      file >> ignore >> di.player.x;
      file >> ignore >> di.r >> di.g >> di.b;
      file >> ignore >> di.mod.active >> di.mod.am.b >> di.mod.am.db >> di.mod.am.depth >> di.mod.am.bpm >> di.mod.fm.b >> di.mod.fm.db >> di.mod.fm.depth >> di.mod.fm.bpm;
      file >> di.num_trail_points >> di.handle_size;
      di.state = drone::RISING;
      di.fdr.set (0, 1);
      ++rising;
      drones.push_back (di);
    }

    update_drone_players ();
    cout << ", done. >>>" << ENDL;
  }
}

void din::save_drones () {

  drone_wave.save ("drone.crv");
  string drone_file = dotdin + scaleinfo.name + ".drone";
  ofstream file (drone_file.c_str(), ios::out);
  if (file) {
    int num_drones = drones.size (); file << "num_drones " << num_drones << eol;
    file << "master_volume " << drone_master_volume << eol;
    for (int i = 0; i < num_drones; ++i) {
      drone& di = drones[i];
      file << "positon " << di.cx << ' ' << di.cy << eol;
      file << "wave_pos " << di.player.x << eol;
      file << "color " << di.r << ' ' << di.g << ' ' << di.b << eol;
      file << "drone_modulation " << di.mod.active << ' ' << di.mod.am.b << ' ' << di.mod.am.db << ' ' << di.mod.am.depth << ' ' << di.mod.am.bpm << ' ' << di.mod.fm.b << ' ' << di.mod.fm.db << ' ' << di.mod.fm.depth << ' ' << di.mod.fm.bpm << ' ' << di.num_trail_points << ' ' << di.handle_size << eol;
    }
    cout << PASS << "+++ saved " << num_drones << " drones in: " << drone_file << " +++" << ENDL;
  }

}

void din::update_drone_tone () {

  for (int i = 0, num_drones = drones.size (); i < num_drones; ++i) {
    drone& di = drones[i];
    range& r = ranges[di.range];
    di.step = (1 - di.pos) * r.notes[0].step + di.pos * r.notes[1].step;
    di.player.set (di.step, drone_master_volume * di.vol, drone_master_volume * di.vol);
  }

}

void din::update_drone_x (int s, int t) {
  for (int i = 0, num_drones = drones.size (); i < num_drones; ++i) {
    drone& di = drones[i];
    if (di.mod.active == 0) {
      if ((di.range >= s) && (di.range <= t)) {
        range& r = ranges[di.range];
        di.x = di.cx = (int)((1 - di.pos) * r.extents.left + di.pos * r.extents.right);
        di.calc_handle ();
      }
    }
  }
}

void din::update_drone_anchors () {
  for (int i = 0, num_drones = drones.size (); i < num_drones; ++i) drones[i].calc_handle ();
}

void din::update_drone_ranges () {
  for (int i = 0, num_drones = drones.size (); i < num_drones; ++i) {
    drone& di = drones[i];
    if (di.range > last_range) {
      di.range = last_range;
      range& rd = ranges[di.range];
      di.pos = (di.x - rd.extents.left) * 1. / rd.extents.width;
    } else if (di.pos > 1) {
      int r = find_range (di.x, di.range);
      range& rr = ranges[r];
      di.range = r;
      di.pos = (di.x - rr.extents.left) * 1. / rr.extents.width;
    }
  }
}

void din::add_drone () {
  drones.push_back (drone());
  update_drone_players ();
  set_drone (drones.size() - 1);
  clear_selected_drones ();
}

void din::delete_selected_drones () {

  if (selected_drones.size () == 0) pick_drone ();
  for (int i = 0, j = selected_drones.size (); i < j; ++i) {
    int s = selected_drones[i];
    drone& ds = drones[s];
    if (ds.state == drone::ACTIVE) {
      ds.state = drone::FALLING;
      ds.fdr.set (1, 0);
      ++falling;
    }
  }
}

int din::delete_all_drones () {
  if (select_all_drones ()) {
    delete_selected_drones ();
    return 1;
  } else cons << console::red << "Drones are busy rising or falling. Please try later!" << eol;
  return 0;
}

void din::set_drone (int d) {

    if (d == -1) return; // bad drone id

    find_volume ();

    drone& dd = drones[d];

    if (!editing_drone) { // create drone

      // random color
      static const float limit = 0.9;
      dd.r = limit * get_rand_01 ();
      dd.g = limit * get_rand_01 ();
      dd.b = limit * get_rand_01 ();

      // create drone at mouse position
      dd.x = dd.cx = win_mousex;
      dd.y = dd.cy = win_mousey;
      dd.dy = win_mousey - BOTTOM;

      // prep to rise the drones
      ++rising;
      dd.state = drone::RISING;
      dd.fdr.set (0, 1);

    } else { // editing the drone

      // update drone position
      dd.cx += delta_mousex;
      dd.cy -= delta_mousey;

    }

    if (!dd.mod.active) dd.set_xy (*this, dd.cx, dd.cy);


}

void din::clear_selected_drones () {
  for (int i = 0, j = selected_drones.size(); i < j; ++i) drones[selected_drones[i]].sel = 0;
  selected_drones.clear ();
  editing_drone = 0;
}

void din::update_drone_players () {
  for (int i = 0, j = drones.size (); i < j; ++i) {
    drone& di = drones[i];
    di.sol (&drone_wave);
    di.player.set_wave (&di.sol);
  }
}

void din::pick_drone () {
  clear_selected_drones ();
  for (int i = drones.size () - 1; i > -1; --i) {
    drone& di = drones[i];
    if ((di.state == drone::ACTIVE) && inbox (di.handle, win_mousex, win_mousey)) {
      di.sel = 1;
      selected_drones.push_back (i);
      break;
    }
  }
}

void din::draw_drones () {

  glEnable (GL_BLEND);
  glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  int num_drones = drones.size ();

  // draw drone trails
  for (int i = 0; i < num_drones; ++i) {
    drone& di = drones[i];
    if (di.mod.active) {
      glBegin (GL_LINE_STRIP);
      glColor4f (di.r, di.g, di.b, di.fdr.amount);
      list< point<int> >& tq = di.trailq;
      for (list< point<int> >::iterator iter = tq.begin (), jter = tq.end (); iter != jter; ++iter) {
        point<int>& p = *iter;
        glVertex2i (p.x, p.y);
      }
      glEnd ();
    }
  }

  if (dinfo.anchor) {

    // draw drone anchor
    for (int i = 0; i < num_drones; ++i) {
      drone& di = drones[i];
      if (di.range >= visl && di.range <= visr) {
        glColor4f (di.r, di.g, di.b, di.fdr.amount);
        glBegin (GL_LINES);
          glVertex2i (di.x, di.y);
          glVertex2i (di.x, BOTTOM);
        glEnd ();
      }
    }

  }

  // draw drone handles
  for (int i = 0; i < num_drones; ++i) {
    drone& di = drones[i];
    if (di.range >= visl && di.range <= visr) {
      glColor4f (di.r, di.g, di.b, di.fdr.amount);
      glRecti (di.handle.left, di.handle.bottom, di.handle.right, di.handle.top);
      if (di.sel) glColor4f (0, 1, 0, di.fdr.amount); else glColor4f (1, 1, 1, di.fdr.amount);
      glBegin (GL_LINE_LOOP);
        glVertex2i (di.handle.left, di.handle.bottom);
        glVertex2i (di.handle.right, di.handle.bottom);
        glVertex2i (di.handle.right, di.handle.top);
        glVertex2i (di.handle.left, di.handle.top);
      glEnd ();
    }
  }

  // draw drone selection box
  selector.draw ();

}

void din::update_drone_master_volume (float d) {

  int n = drones.size ();
  if (n == 0) return;

  float nd = d / n;
  drone_master_volume += nd;

  for (int i = 0, j = drones.size(); i < j; ++i) {
    drone& di = drones[i];
    di.player.set_volume (drone_master_volume * di.vol, drone_master_volume * di.vol);
  }

  roll_console ();

  cons << "drone master volume = " << drone_master_volume << eol;

}

void din::update_drone_solvers () {
  for (int i = 0, j = drones.size (); i < j; ++i) drones[i].sol.update ();
}

string din::get_selected_drones () {
  stringstream ss;
  for (int i = 0, j = drones.size (); i < j; ++i) if (drones[i].sel) ss << i << ' ';
  return ss.str();
}

void din::set_drone_volume (int i, float v) {
  if (i > -1 && i < (int) drones.size()) {
    drone& di = drones[i];
    di.vol = v;
    di.dy = di.vol / DELTA_VOLUME;
    di.player.set (di.step, drone_master_volume * di.vol, drone_master_volume * di.vol);
    di.calc_handle ();
  }
}

void din::calc_win_mouse () {

  delta_mousex = mousex - prev_mousex;
  delta_mousey = mousey - prev_mousey;

  win_mousex += delta_mousex;
  win_mousey -= delta_mousey;

  tonex = win_mousex;
  toney = win_mousey;

  prev_mousex = mousex;
  prev_mousey = mousey;

}

void din::find_selected_drones () {

  // select drones that lie inside selection box
  //

  clear_selected_drones ();
  for (int i = 0, num_drones = drones.size (); i < num_drones; ++i) {
    drone& di = drones [i];
    if ((di.state == drone::ACTIVE) && inbox (selector.region, di.x, di.y)) {
      selected_drones.push_back (i);
      di.sel = 1;
    }
  }

}

int din::select_all_drones () {
  clear_selected_drones ();
  int result = 1;
  for (int i = 0, j = drones.size (); i < j; ++i) {
    drone& di = drones[i];
    if (di.state == drone::ACTIVE) {
      di.sel = 1;
      selected_drones.push_back (i);
    } else result = 0;
  }
  return result;
}

bool din::handle_input () {

  extern float DELTA_BPM;
  static const float delta_am_depth = 0.01, delta_fm_depth = 1;
  static const float delta_drone_master = 0.001;

  jogged = 0;

  if (osc.handle_input ()) { // oscilloscope
    selector.exists = 0;
    return true;
  }

  if (selector.handle_input (win_mousex, win_mousey)) find_selected_drones (); // box select drones

  if (phrasor0.state == phrasor::recording) { // record mouse pos for playback l8r
    static point<int> pt;
    pt.x = win_mousex; pt.y = win_mousey;
    phrasor0.data.push_back (pt);
    ++phrasor0.sz;
  }

  // movement
  if (keypressedd (k_a, dinfo.scroll.rept, dinfo.scroll.rept)) scroll (-dinfo.scroll.dx, 0); else
  if (keypressedd (k_d, dinfo.scroll.rept, dinfo.scroll.rept)) scroll (+dinfo.scroll.dx, 0); else
  if (keypressedd (k_w, dinfo.scroll.rept, dinfo.scroll.rept)) scroll (0, +dinfo.scroll.dy); else
  if (keypressedd (k_s, dinfo.scroll.rept, dinfo.scroll.rept)) scroll (0, -dinfo.scroll.dy);

  // octave shift
  else if (keypressed (k_z)) modulate_up ();
  else if (keypressed (k_x)) modulate_down ();

  else if (keypressed (k_b)) { // toggle gater
    if (uis.w_gater.enabled()) uis.w_gater.toggle ();
  }

  else if (keypressed (k_f)) { // phrase record
    if (phrasor0.state == phrasor::recording) {
      phrasor0.validate ();
      phrasor0.play ();
      cons << console::green << "phrasor STOPPED recording, started PLAYING." << eol;
    } else {
      phrasor0.rec ();
      cons << console::red << "phrasor is RECORDING." << eol;
    }
  }

  else if (keypressed (k_v)) { // phrase play/pause
    if (phrasor0.state == phrasor::playing) {
      cons << console::yellow << "phrasor has PAUSED." << eol;
      phrasor0.state = phrasor::paused;
      find_current_range ();
    } else {
      cons << console::green << "phrasor is PLAYING." << eol;
      phrasor0.validate ();
      phrasor0.play ();
    }
  }

  else if (keypressed (k_g)) { // phrase clear
    phrasor0.clear ();
    find_current_range ();
    cons << console::red << "phrases deleted." << eol;
  }

  // mute lead
  else if (keypressed (k_space)) {
    uis.w_voice.toggle ();
  }

  // drones
  //
  else if (keypressedd (k_q)) add_drone (); // add drone
  else if (keypressed (k_e)) { // move selected drones
    if (editing_drone) clear_selected_drones (); else {
      if (selected_drones.size () == 0)
        pick_drone ();
      if (selected_drones.size ())
        editing_drone = 1;
    }
  }
  else if (editing_drone) {
      for (int i = 0, j = selected_drones.size(); i < j; ++i) set_drone (selected_drones[i]);
  }
  else if (keypressedd (k_c)) delete_selected_drones ();
  else if (keypressed (k_k)) dinfo.anchor = !dinfo.anchor;
  else if (keypressedd (k_comma, 1./16, 1./128)) update_drone_master_volume (-delta_drone_master); // decrease drone master volume
  else if (keypressedd (k_period, 1./16, 1./128)) update_drone_master_volume (+delta_drone_master); // increase drone master volume
  else if (keypressedd (k_slash)) { // mute/unmute drone master volume
    if (drone_master_volume == 0) update_drone_master_volume (last_drone_master_volume); else {
      last_drone_master_volume = drone_master_volume;
      update_drone_master_volume (-drone_master_volume);
    }
  }
  else if (keypressed (k_j)) {
    do_drone_modulation = !do_drone_modulation;
    if (do_drone_modulation && drones.size ()) cons << console::cyan << "modulating drones" << eol;
    else cons << console::cyan << "modulating voice" << eol;
  }
  // phrase jogging
  //
  else if (keydown (k_left)) { // jog left
    phrasor0.jog (-phrasor::JOG);
    jogged = 1;
  }
  else if (keydown (k_right)) { // jog right
    phrasor0.jog (+phrasor::JOG);
    jogged = 1;
  }
  else if (keypressedd (k_left_bracket)) { // decrease jog amount
    if (--phrasor::JOG < 2) phrasor::JOG = 2;
    cons << console::yellow << "phasor jog: " << phrasor::JOG << eol;
  }
  else if (keypressedd (k_right_bracket)) { // increase jog amount
    ++phrasor::JOG;
    cons << console::yellow << "phasor jog: " << phrasor::JOG << eol;
  }

  else if (keydown (k_down)) phrasor0.set_cue (); // set cue point
  else if (keydown (k_up)) phrasor0.goto_cue (); // goto cue point

  else if (keypressed (k_p)) { // toggle compressor
    uis.w_compress.toggle ();
  }

  else if (keypressed (k_semicolon)) { // set key to pitch under cursor
    set_tonic (step * SAMPLE_RATE);
    mouse2tonic ();
    cons ("key");
  }

  // tuning
  //
  else if (keydown (k_lctrl)) {
    if (keypressed (k_m)) set_range_size (0, last_range, ranges[current_range].extents.width);
    else if (keypressed (k_n)) {
      restore_all_notes ();
      cons << console::green << "restored all notes" << eol;
    }
    else range_left_changed ();
  }

  else if (keydown (k_lshift)) {
    if (keypressed (k_m)) set_range_size (0, last_range, NUM_MICROTONES);
    else if (keypressed (k_n)) restore_note ();
    else range_right_changed ();
  }

  else if (keypressed (k_n)) {
    retune_note ();
  }

  else if (keypressed (k_m)) {
    set_range_size (current_range, NUM_MICROTONES);
  }

  else if (keydown (k_h)) { // adjust board height

      if (delta_mousey != 0) {

        HEIGHT -= delta_mousey;
        TOP = BOTTOM + HEIGHT;
        if (TOP <= BOTTOM) { // set sane values
          TOP = BOTTOM + 2;
          HEIGHT = 2;
        }

        // update volume internals
        calc_volume_vars (HEIGHT);

        // update ranges
        for (int i = 0; i < num_ranges; ++i) {
          range& ri = ranges[i];
          ri.extents.top = TOP;
          ri.extents.height = HEIGHT;
          ri.extents.calc ();
        }
        calc_range_label ();

        // update drones
        for (int i = 0, num_drones = drones.size (); i < num_drones; ++i) {
          drone& di = drones[i];
          if (di.mod.active == 0) {
            di.dy = (int)(di.vol * HEIGHT);
            di.cy = BOTTOM + di.dy;
            di.calc_handle ();
          }
        }
      }
  }
  else if (keypressed (k_i)) { // label mouse cursor, notes, pitches
    show_cursor_info = !show_cursor_info;
  } else if (keypressed (k_f1)) helptext();

  // oscilloscope
  //
  else if (keypressed (k_rctrl)) {
    osc.toggle_visible ();
  }
  else if (keypressed (k_menu)) osc.toggle_pause ();

  // bpms

  extern beat2value octave_shift;

  if (keypressedd (k_f5)) { // decrease gater bpm (lshift - gl, lctrl - gr)
    change_bpm (gatr, -DELTA_BPM);
  } else if (keypressedd (k_f6)) { // increase gater bpm (lshift - gl, lctrl - gr)
    change_bpm (gatr, +DELTA_BPM);
  } else if (keypressedd (k_f7)) { // decrease fm bpm
    if (do_drone_modulation && drones.size ()) {
      change_drone_bpm (drone_modulation::FM, -DELTA_BPM);
    } else {
      change_bpm (fm, -DELTA_BPM);
    }
  } else if (keypressedd (k_f8)) { // increase fm bpm
    if (do_drone_modulation && drones.size ()) {
      change_drone_bpm (drone_modulation::FM, DELTA_BPM);
    } else {
      change_bpm (fm, +DELTA_BPM);
    }
  } else if (keypressedd (k_f9)) { // decrease am bpm
    if (do_drone_modulation && drones.size ()) {
      change_drone_bpm (drone_modulation::AM, -DELTA_BPM);
    } else {
      change_bpm (am, -DELTA_BPM);
    }
  } else if (keypressedd (k_f10)) { // increase am bpm
    if (do_drone_modulation && drones.size ()) {
      change_drone_bpm (drone_modulation::AM, DELTA_BPM);
    } else {
      change_bpm (am, +DELTA_BPM);
    }
  } else if (keypressedd (k_f11)) { // decrease octave shift bpm
    change_bpm (octave_shift, -DELTA_BPM);
  } else if (keypressedd (k_f12)) { // increase octave shift bpm
    change_bpm (octave_shift, +DELTA_BPM);
  }

  // depths
  else if (keypressedd (k_r)) { // decrease am depth
    if (do_drone_modulation && drones.size ()) {
      change_drone_depth (drone_modulation::AM, -1);
    } else {
      am_depth -= delta_am_depth;
      cons << "am depth: " << am_depth << eol;
    }
  } else if (keypressedd (k_t)) { // increase am depth
    if (do_drone_modulation && drones.size ()) {
      change_drone_depth (drone_modulation::AM, +1);
    } else {
      am_depth += delta_am_depth;
      cons << "am depth: " << am_depth << eol;
    }
  } else if (keypressedd (k_y)) { // decrease fm depth
    if (do_drone_modulation && drones.size()) {
      change_drone_depth (drone_modulation::FM, -1);
    } else {
      fm_depth -= delta_fm_depth;
      hz2step (fm_depth, fm_step);
      cons << "fm depth: " << fm_depth << eol;
    }
  } else if (keypressedd (k_u)) { // increase fm depth
    if (do_drone_modulation && drones.size ()) {
      change_drone_depth (drone_modulation::FM, +1);
    } else {
      fm_depth += delta_fm_depth;
      hz2step (fm_depth, fm_step);
      cons << "fm depth: " << fm_depth << eol;
    }
  }

  else if (keypressedd (k_minus)) {
    change_drone_trail_points (-1);
  } else if (keypressedd (k_equals)) {
    change_drone_trail_points (+1);
  } else if (keypressedd (k_9)) {
    change_drone_handle_size (-1);
    update_drone_anchors ();
  } else if (keypressedd (k_0)) {
    change_drone_handle_size (+1);
    update_drone_anchors ();
  }


  return true;

}

void din::change_drone_trail_points (int delta) {

  int n = selected_drones.size ();
  if (n) { // change for selected drones only
    for (int i = 0; i < n; ++i) {
      int s = selected_drones[i];
      drone& ds = drones[s];
      ds.num_trail_points += delta;
      if (ds.num_trail_points < 1) ds.num_trail_points = 1;
    }
  } else { // change for all drones
    n = drones.size ();
    for (int i = 0; i < n; ++i) {
      drone& di = drones[i];
      di.num_trail_points += delta;
      if (di.num_trail_points < 1) di.num_trail_points = 1;
    }
  }

}

void din::change_drone_handle_size (int delta) {

  int n = selected_drones.size ();
  if (n) { // change for selected drones only
    for (int i = 0; i < n; ++i) {
      int s = selected_drones[i];
      drone& ds = drones[s];
      ds.handle_size += delta;
      if (ds.handle_size < 0) ds.handle_size = 0;
    }
  } else { // change for all drones
    n = drones.size ();
    for (int i = 0; i < n; ++i) {
      drone& di = drones[i];
      di.handle_size += delta;
      if (di.handle_size < 0) di.handle_size = 0;
    }
  }

}

void din::change_drone_bpm (int what, float delta) {

  // change FM/AM bpm of drones
  //

  int n = selected_drones.size ();
  if (n) { // change for selected drones only
    for (int i = 0; i < n; ++i) {
      int s = selected_drones[i];
      drone& ds = drones[s];
      ds.change_bpm (s, what, delta);
    }
  } else { // change for all drones
    n = drones.size ();
    for (int i = 0; i < n; ++i) {
      drone& di = drones[i];
      di.change_bpm (i, what, delta);
    }
  }

  cons.rollup (1);

}

void din::change_drone_depth (int what, float delta) {
  int n = selected_drones.size ();
  if (n) { // change for selected drones
    for (int i = 0; i < n; ++i) {
      int s = selected_drones[i];
      drone& di = drones[s];
      di.change_depth (s, what, delta);
    }
  } else { // change for all drones
    n = drones.size ();
    if (!n) cons << console::red << "no drones" << eol;
    for (int i = 0; i < n; ++i) {
      drone& di = drones[i];
      di.change_depth (i, what, delta);
    }
  }

  cons.rollup(1);
}

void warp_pointer (int dx, int dy);

void din::scroll (int dx, int dy) {

  mousex -= dx;
  prev_mousex -= dx;

  mousey += dy;
  prev_mousey += dy;

  win (win.left + dx, win.bottom + dy, win.right + dx, win.top + dy);
  warp_pointer (-dx, dy);

  find_visible_ranges (dx);


}

void din::find_current_range () {

  // find the range where mouse is found

  win_mousex = win.left + mousex;
  win_mousey = win.bottom + view.ymax - mousey;

  if (win_mousex <= ranges[0].extents.left) current_range = 0; else
  if (win_mousex >= ranges[last_range].extents.right) current_range = last_range; else

  for (int i = 0; i < num_ranges; ++i) {
    range& curr = ranges[i];
    box<int>& ext = curr.extents;
    if ( (win_mousex >= ext.left) && (win_mousex <= ext.right)) {
      current_range = i;
      break;
    }
  }

  find_visible_ranges ();

}

void din::find_visible_ranges (int dir) { // bcos we only draw visible ranges

  if (dir > 0) {
    while ((visr < last_range) && (ranges[visr].extents.right < win.right)) ++visr;
    while ((visl < last_range) && (ranges[visl].extents.right < win.left)) ++visl;
  } else if (dir < 0) {
    while ((visl > 0) && (ranges[visl].extents.left > win.left)) --visl;
    while ((visr > 0) && (ranges[visr].extents.left > win.right)) --visr;
  } else {
    visl = current_range;
    visr = current_range;
    while ( (visl > 0) && (win.left < ranges[visl].extents.left) ) --visl;
    while ( (visr < last_range) && (ranges[visr].extents.right < win.right) ) ++visr;
  }

}

int din::find_range (int x, int r) {

  while (1) {
    range& curr = ranges [r];
    int deltax = x - curr.extents.left;
    if (deltax > curr.extents.width) {
      if (++r < num_ranges); else {
        r = num_ranges - 1;
        break;
      }
    }
    else if (deltax < 0) {
      if (--r < 0) {
        r = 0;
        break;
      }
    }
    else
      break;
  }
  return r;
}

bool din::find_tone_and_volume () {

  find_volume ();

  // locate current tone
  range* curr = &ranges [current_range];
  int deltax = tonex - curr->extents.left;
  if (deltax >= curr->extents.width) { // tone in range to the right
    ++current_range;
    if (current_range == num_ranges) { // snap to last range
      current_range = last_range;
      curr = lastr;
    } else {
      curr = &ranges [current_range];
    }
  } else if (deltax < 0) { // tone in range to the left
    --current_range;
    if (current_range < 0) { // snap to first range
      curr = firstr;
      current_range = 0;
    } else {
      curr = &ranges [current_range];
    }
  }

  // located tone so find frequency
  //
  deltax = tonex - curr->extents.left;

  delta = deltax * curr->extents.width_1;
  step = curr->notes[0].step + delta * curr->delta_step; // step determines frequency see note.h

  // octave position of tone among all octaves
  octave_position = curr->notes[0].octave_position + delta * curr->delta_octave_position;

  extern double VOLUME;
  if (show_cursor_info) { // display frequency & volume at mouse cursor
    stringstream ss;
    ss.clear (); ss << (step * SAMPLE_RATE) << " / " << VOLUME; cursor_info = ss.str();
  }

  bool result = false;

  if (dv < 0) { // below keyboard, silence
    wavplay.set (step, 0, 0);
    am_vol = 0;
  } else {
    am_vol = VOLUME;
    extern float WAVE_VOLUME;
    wavplay.set (step, VOLUME * WAVE_VOLUME, VOLUME * WAVE_VOLUME);
    result = true;
  }

  Tcl_UpdateLinkedVar (interpreter.interp, "volume"); // VOLUME is accessible in Tcl interpreter as variable volume

  return result;

}

void din::draw () {

  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glOrtho (0, view.xmax, 0, view.ymax, -1, 1);

  glMatrixMode (GL_MODELVIEW);
  glLoadIdentity ();

  // draw oscilloscope
  if (osc.visible) {
    osc.draw ();
    glLoadIdentity ();
  }

  glMatrixMode (GL_PROJECTION);
  glLoadIdentity ();
  glOrtho (win.left, win.right, win.bottom, win.top, -1, 1);

  // label visible ranges
  for (int i = visl; i < visr; ++i) ranges[i].draw_labels (range::LEFT, show_cursor_info);
  ranges[visr].draw_labels (range::BOTH, show_cursor_info);

  draw_drones (); // draw drones

  // mark range notes
  int rl = ranges[visl].extents.left, rr = ranges[visr].extents.right;
  float tr = 1; glColor3f (tr, tr, tr);
  glBegin (GL_LINES);
    glVertex2i (rl, BOTTOM);
    glVertex2i (rr, BOTTOM);
    glVertex2i (rl, TOP);
    glVertex2i (rr, TOP);
  glEnd ();

  // phrasor markers
  if (phrasor0.state != phrasor::stopped) {
    if (phrasor0.state == phrasor::recording) phrasor0.draw (phrasor0.recl, 1, 0, 0); else {
      phrasor0.draw (phrasor0.cur, 0, 1, 0);
      phrasor0.draw (phrasor0.cue, 0.25, 0.25, 0.25);
    }
  }

  // draw cursor info
  if (show_cursor_info) {
    static const int delta_label = 10;
    glColor3f (1, 0.25, 0.15);
    draw_string (cursor_info, tonex + delta_label, toney + delta_label, 0);
  }

}

void warp_pointer_abs (int x, int y);

void din::enter () {

  if (phrasor0.state != phrasor::playing) {
    warp_pointer_abs (prev_mousex, prev_mousey);
    mousex = prev_mousex;
    mousey = prev_mousey;
    find_current_range ();
  }

}

void din::roll_console () {
  if (cons.rollup() == 0) cons.rollup (1);
}

void din::change_depth (int i, float d) {

  roll_console ();
  if (i == 1) {
    fm_depth += d;
    hz2step (fm_depth, fm_step);
    cons << "fm depth: " << fm_depth << eol;
  } else {
    am_depth += d;
    cons << "am depth: " << am_depth << eol;
  }
}

void din::change_bpm (beat2value& which, float amt) {
  roll_console ();
  float bpm = which.get_bpm () + amt;
  bpm = which.set_bpm (bpm);
  cons << which.name << " bpm: " << bpm << eol;
}

void din::calc_am_fm_gater () {
  am.modulate (aout.ams, aout.result, aout.samples_per_channel, am_depth * am_vol);
  fm.modulate (aout.fms, aout.result, aout.samples_per_channel, fm_step);
  gatr (aout.gatr, aout.samples_per_channel);
}

void din::bg () { // always runs even when din board is not visible

  if (phrasor0.state == phrasor::playing) {
    phrasor0.get (tonex, toney);
    if (!jogged) phrasor0.next ();
  }

}

void din::modulate_drones () {
  for (int i = 0, j = drones.size (); i < j; ++i) {
    drone& di = drones[i];
    int add2_trail = 0;
    drone_modulation& dm = di.mod;
    dm.calc ();
    if (dm.active) {
      int x = di.cx + dm.fm.result, y = di.cy + dm.am.result;
      di.set_xy (*this, x, y);
      add2_trail = 1;
    }

    if (add2_trail) {
      if (di.trailq.size () < di.num_trail_points) {
        di.trailq.push_back (point<int>(di.x, di.y));
      } else di.trailq.pop_front ();
    }

  }
}

void din::render_audio (float* out0, float* out1) {

  calc_am_fm_gater (); // compute AM & FM & gater over bpm

  // find tone/volume from mouse position
  //

  if (find_tone_and_volume ()) {

    wavplay.solve_fm (aout.result, aout.fms, aout.samples_per_channel); // do FM

    float *lout = out0, *rout = out1;
    wavplay (lout, aout.samples_per_channel, aout.result, aout.ams, wavplay.left * uis.fdr_voice.amount); // do AM

    // apply gater
    //

    lout = out0;
    rout = out1;
    if (uis.fdr_gater.on) {
      memcpy (aout.result, lout, aout.samples_channel_size); // voice
      multiply (lout, aout.gatr, aout.samples_per_channel); // voice * gater
      tween (lout, aout.result, aout.samples_per_channel, uis.fdr_gater.amount); // blended
    } else {
      if (dinfo.gater) {
        multiply (lout, aout.gatr, aout.samples_per_channel); // voice * gater
      }
    }
    memcpy (rout, lout, aout.samples_channel_size); // copy left -> right

  }

  // apply drones
  //

  rise_drones ();
    modulate_drones ();
  fall_drones ();

  for (int i = 0, num_drones = drones.size (); i < num_drones; ++i) {
    float* lout = out0, *rout = out1;
    drone& di = drones[i];
    play& dp = di.player;
    dp.solve (aout.result, aout.samples_per_channel);
    dp (lout, aout.samples_per_channel, aout.result, dp.left * di.fdr.amount); // render drone waveform to left
    memcpy (rout, lout, aout.samples_channel_size); // copy left -> right ie drone is mono
  }

}

void din::rise_drones () {
  if (rising) {
    for (int i = 0, j = drones.size (); i < j; ++i) {
      drone& di = drones[i];
      if (di.state == drone::RISING) {
        di.fdr.eval ();
        if (di.fdr.on == 0) {
          di.state = drone::ACTIVE;
          --rising;
        }
      }
    }
  }
}

void din::fall_drones () {
  if (falling) {
    for (int i = 0, j = drones.size (); i < j; ++i) {
      drone& di = drones[i];
      if (di.state == drone::FALLING) {
        di.fdr.eval ();
        if (di.fdr.on == 0) {
          di.state = drone::DEAD;
          dead_drones.push_back (i);
          if (--falling == 0) {
            sort (dead_drones.begin (), dead_drones.end());
            for (int m = dead_drones.size () - 1; m > -1; --m) drones.erase (drones.begin () + dead_drones[m]);
            update_drone_players ();
            dead_drones.clear ();
            clear_selected_drones ();
            if (on_delete != "") {
              cons (on_delete);
              on_delete = "";
            }
            return;
          }
        }
      }
    }
  }
}

din_info::din_info () {

  extern string dotdin;
  ifstream file ((dotdin + "din_info").c_str(), ios::in);
  string ignore;

  if (file) {
    file >> ignore >> height; HEIGHT = height; TOP = BOTTOM + HEIGHT; calc_volume_vars (HEIGHT);
    int rollup; file >> ignore >> rollup; cons.rollup (rollup);
    file >> ignore >> delay;
    file >> ignore >> gater;
    file >> ignore >> compress;
    file >> ignore >> voice;
    file >> ignore >> anchor;
    file >> ignore >> cons.show_tips;
  } else {
    height = 290;
    cons.rollup (0);
    delay = 1;
    compress = 1;
    voice = 1;
    anchor = 1;
    cons.show_tips = 1;
  }

  cons.last ();
}

void din_info::save () {
  extern string dotdin;
  ofstream file ((dotdin + "din_info").c_str(), ios::out);
  if (file) {
    file << "board_height " << HEIGHT << endl;
    file << "rollup " << cons.rollup() << endl;
    file << "delay " << delay << endl;
    file << "gater " << gater << endl;
    file << "compress " << compress << endl;
    file << "voice " << voice << endl;
    file << "anchor " << anchor << endl;
    file << "show_tips " << cons.show_tips << endl;
  } else {
    cout << "cannot save din_info" << eol;
  }
}
