/*
 * Licensed to Elasticsearch under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.elasticsearch.index.fielddata;

import org.elasticsearch.action.admin.indices.create.CreateIndexRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.search.facet.Facets;
import org.elasticsearch.search.facet.terms.TermsFacet;
import org.elasticsearch.test.ElasticsearchIntegrationTest;
import org.hamcrest.Matchers;
import org.junit.Test;

import java.io.IOException;

import static org.elasticsearch.common.settings.ImmutableSettings.settingsBuilder;
import static org.elasticsearch.index.query.QueryBuilders.matchAllQuery;
import static org.elasticsearch.search.facet.FacetBuilders.termsFacet;
import static org.elasticsearch.test.hamcrest.ElasticsearchAssertions.assertAcked;

public class FieldDataFilterIntegrationTests extends ElasticsearchIntegrationTest {

    @Test
    public void testRegexpFilter() throws IOException {
        CreateIndexRequestBuilder builder = prepareCreate("test").setSettings(settingsBuilder()
                .put("index.number_of_shards", between(1,5))
                .put("index.number_of_replicas", 0));
        XContentBuilder mapping = XContentFactory.jsonBuilder().startObject().startObject("type")
                .startObject("properties")
                    .startObject("name")
                        .field("type", "string")
                        .startObject("fielddata")
                            .startObject("filter")
                                .startObject("regex")
                                    .field("pattern", "^bac.*")
                                .endObject()
                            .endObject()
                        .endObject()
                    .endObject()
                    .startObject("not_filtered")
                        .field("type", "string")
                    .endObject()
                .endObject()
                .endObject().endObject();
        assertAcked(builder.addMapping("type", mapping));
        ensureGreen();
        int numDocs = atLeast(5);
        for (int i = 0; i < numDocs; i++) {
            client().prepareIndex("test", "type", "" + 0).setSource("name", "bacon bastards", "not_filtered", "bacon bastards").get();
        }
        refresh();
        SearchResponse searchResponse = client().prepareSearch()
                .setSearchType(SearchType.COUNT)
                .setQuery(matchAllQuery())
                .addFacet(termsFacet("name").field("name"))
                .addFacet(termsFacet("not_filtered").field("not_filtered")).get();
        Facets facets = searchResponse.getFacets();
        TermsFacet nameFacet = facets.facet("name");
        assertThat(nameFacet.getEntries().size(), Matchers.equalTo(1));
        assertThat(nameFacet.getEntries().get(0).getTerm().string(), Matchers.equalTo("bacon"));
        
        TermsFacet notFilteredFacet = facets.facet("not_filtered");
        assertThat(notFilteredFacet.getEntries().size(), Matchers.equalTo(2));
        assertThat(notFilteredFacet.getEntries().get(0).getTerm().string(), Matchers.isOneOf("bacon", "bastards"));
        assertThat(notFilteredFacet.getEntries().get(1).getTerm().string(), Matchers.isOneOf("bacon", "bastards"));
        
    }

}
