/***********************************************************
/        Title: usbserial.c
/       Author: David Corcoran
/      Purpose: Abstracts usb API to serial like calls
/      License: See file COPYING
/         $Id: usbserial_mosx.c,v 1.9 2001/11/30 18:10:16 rousseau Exp $
************************************************************/


#include <stdio.h>
#include <unistd.h>
#include <mach/mach.h>
#include <mach/mach_error.h>

#include <IOKit/IOKitLib.h>
#include <IOKit/usb/IOUSBLib.h>

#include <CoreFoundation/CoreFoundation.h>
#include <CoreFoundation/CFURL.h>
#include <CoreFoundation/CFPlugin.h>

#include <IOKit/IOCFPlugIn.h>

#include <assert.h>
#include "Config.h"
#include "pcscdefines.h"
#include "GemCore.h"
#include "GCdebug.h"
#include "usbserial.h"

#define USBWRITE_PIPE   2
#define USBREAD_PIPE    1
#define USBMAX_READERS  (PCSCLITE_MAX_CHANNELS)

/* Change the following to uniquely match your reader. */
enum {
    kMyVendorID			= 0x08E6,	/* change to match your reader */
    kMyProductID		= 0x0430,	/* change to match your reader */
    kMyDeviceClass		= kIOUSBAnyClass,
    kMyDeviceSubClass		= kIOUSBAnySubClass,
    kMyDeviceProtocol		= kIOUSBAnyProtocol
};
    
static mach_port_t 		masterPort;
static int                      iInitialized = FALSE;
typedef struct _intFace {
  IOUSBInterfaceInterface  **iface;
  UInt32 usbAddr;
} intrFace, *pIntrFace;

static pIntrFace intFace[USBMAX_READERS];

status_t OpenUSB( DWORD lun, DWORD Channel)
{
    CFDictionaryRef 			USBMatch = 0;
    io_iterator_t 			iter = 0;
    io_service_t 			USBDevice = 0;
    io_service_t        		USBIface = 0;
    kern_return_t			kr;
    io_iterator_t       		iterB = 0;
    SInt32				score;
    IOUSBFindInterfaceRequest 		findInterface;
    IOCFPlugInInterface 		**iodev=NULL;
    IOCFPlugInInterface			**iodevB=NULL;
    IOUSBDeviceInterface182 		**dev=NULL;
    HRESULT 				res;
    DWORD				rdrLun;
    UInt16				vendor;
    UInt16				product;
    UInt16				release;
    UInt32   				usbAddr;
    short				iFound;
    int					i;
          
    iFound = FALSE;
    rdrLun = lun >> 16;
    
    if ( iInitialized == FALSE ) {
                
        for (i=0; i < USBMAX_READERS; i++) {
            intFace[i] = (pIntrFace)malloc(sizeof(intrFace));
            (intFace[i])->usbAddr = 0;
        }
    
        /* first create a master_port for my task */
        kr = IOMasterPort(bootstrap_port, &masterPort);
        if (kr || !masterPort)
        {
            DEBUG_CRITICAL2("Couldn't create a master IOKit Port (0x%08X)", kr);
            return STATUS_UNSUCCESSFUL;
        }

        iInitialized = TRUE;
    }

    USBMatch = IOServiceMatching(kIOUSBDeviceClassName);
    if (!USBMatch)
    {
        DEBUG_CRITICAL("Can't create a USB matching dictionary");
        mach_port_deallocate(mach_task_self(), masterPort);
        return STATUS_UNSUCCESSFUL;
    }
    
    /* create an iterator over all matching IOService nubs */
    kr = IOServiceGetMatchingServices(masterPort, USBMatch, &iter);
    if (kr)
    {
        DEBUG_CRITICAL2("Can't create a USB Service iterator (0x%08X)", kr);
        CFRelease(USBMatch);
        mach_port_deallocate(mach_task_self(), masterPort);
        return STATUS_UNSUCCESSFUL;
    }

    while ( (USBDevice = IOIteratorNext(iter)) )
    {
        
        kr = IOCreatePlugInInterfaceForService(USBDevice, kIOUSBDeviceUserClientTypeID,         						kIOCFPlugInInterfaceID, &iodev, &score);
        IOObjectRelease(USBDevice);	/* done with the device object now */
        if (kr || !iodev)
        {
            DEBUG_CRITICAL2("unable to create a plugin (0x%08X)", kr);
            continue;
        }
            
        /* i have the device plugin. I need the device interface */
        res = (*iodev)->QueryInterface(iodev, CFUUIDGetUUIDBytes(kIOUSBDeviceInterfaceID182), 						(LPVOID)&dev);
        (*iodev)->Release(iodev);
        if (res || !dev)
        {
            DEBUG_CRITICAL2("Couldn't create a device interface (0x%08X)", (int)res);
            continue;
        }

        kr = (*dev)->GetDeviceVendor(dev, &vendor);
        kr = (*dev)->GetDeviceProduct(dev, &product);
        kr = (*dev)->GetDeviceReleaseNumber(dev, &release);
        kr = (*dev)->GetLocationID(dev, &usbAddr);

        if ((vendor != kMyVendorID) || (product != kMyProductID))
        {
            /* Continue searching for devices */
            (*dev)->Release(dev);
            continue;
        } else {
            for (i=0; i < USBMAX_READERS; i++) {
                if ( usbAddr == (intFace[i])->usbAddr ) {
                    iFound = TRUE;
                    break;
                }
            }
            
            if ( iFound ) {
                iFound = FALSE;
                continue;
            }
        
            /* We found it now lets break out of the loop */
            break;
        }
   }
 
    kr = (*dev)->USBDeviceOpen(dev);
    if (kr == kIOReturnExclusiveAccess) 
    {
        sleep(1);
        // Try to re-open in case Classic is using the device
        if( kr = (*dev)->USBDeviceOpenSeize(dev) )
        {
            DEBUG_CRITICAL2("unable to open device: 0x%08X", kr);
            (*dev)->Release(dev);
        }
    }
    else
    {
        if ( kr )
        {
            DEBUG_CRITICAL2("unable to open device, not kDeviceExclusive: 0x%08X", kr);
            (*dev)->Release(dev);
        }
    }

    /* Reset in case device is in a bad state    */
    /* Reset is asynchronous and will be removed */
    /* (*dev)->ResetDevice(dev);                 */
        
    /* Time to find the first interface */
    findInterface.bInterfaceClass    = kIOUSBFindInterfaceDontCare;
    findInterface.bInterfaceSubClass = kIOUSBFindInterfaceDontCare;
    findInterface.bInterfaceProtocol = kIOUSBFindInterfaceDontCare;
    findInterface.bAlternateSetting  = kIOUSBFindInterfaceDontCare;

    kr = (*dev)->CreateInterfaceIterator(dev, &findInterface, &iterB );
    USBIface = IOIteratorNext(iterB);
    
    if ( USBIface == 0 ) {
        DEBUG_CRITICAL("No interface found");
        return STATUS_UNSUCCESSFUL;
    }

    score = 0;
    
    /* Create the plugin for the interface service */
    kr  = IOCreatePlugInInterfaceForService(USBIface, kIOUSBInterfaceUserClientTypeID, 			  				    kIOCFPlugInInterfaceID, &iodevB, &score);
    res = (*iodev)->QueryInterface(iodevB, CFUUIDGetUUIDBytes(kIOUSBInterfaceInterfaceID),  
                                  (LPVOID)&(intFace[rdrLun])->iface);
    (intFace[rdrLun])->usbAddr = usbAddr;
    
    (*iodev)->Release(iodev);

    /* Open the interface to open all the pipes */
    kr = (*(intFace[rdrLun])->iface)->USBInterfaceOpen((intFace[rdrLun])->iface);

    if (iter) {
        IOObjectRelease(iter);
        iter = 0;
    }
     

  return STATUS_SUCCESS;  
}

status_t WriteUSB( DWORD lun, DWORD length, unsigned char *buffer )
{
    IOReturn		iorv;
    DWORD		rdrLun;
     
    rdrLun = lun >> 16;
         
    DEBUG_XXD("Attempt to write: ", buffer, length);
        
    /* Make sure the pipe is OK */
    iorv = (*(intFace[rdrLun])->iface)->GetPipeStatus( (intFace[rdrLun])->iface, 
                                                            USBWRITE_PIPE );
    if ( iorv != kIOReturnSuccess ) {
        return STATUS_UNSUCCESSFUL;
    }

    /* Write the data */
    iorv = (*(intFace[rdrLun])->iface)->WritePipe((intFace[rdrLun])->iface, USBWRITE_PIPE, 							buffer, length);
    
    if ( iorv != kIOReturnSuccess ) {
        return STATUS_UNSUCCESSFUL;
    }
            
    return STATUS_SUCCESS;
}

status_t ReadUSB( DWORD lun, DWORD *length, unsigned char *buffer )
{
    IOReturn		iorv;
    UInt32		recvLen;
    DWORD		rdrLun;
    
    rdrLun = lun >> 16;
    
    DEBUG_COMM2("Attempt to read %ld bytes", *length);
    
    /* Make sure the pipe is OK */
    iorv = (*(intFace[rdrLun])->iface)->GetPipeStatus( (intFace[rdrLun])->iface, USBREAD_PIPE );
    if ( iorv != kIOReturnSuccess ) {
        return STATUS_UNSUCCESSFUL;
    }

    /* Try to read up to 256 bytes */
    recvLen = 256;
    iorv = (*(intFace[rdrLun])->iface)->ReadPipe( (intFace[rdrLun])->iface, USBREAD_PIPE, 							buffer, &recvLen );	
    if ( iorv != 0 ) {
        return STATUS_UNSUCCESSFUL;
    }
    
	DEBUG_XXD("received: ", buffer, recvLen);
    
    *length = recvLen;
    return STATUS_SUCCESS;
}

status_t CloseUSB( DWORD lun )
{
    IOReturn iorv;
    DWORD rdrLun;
    
    rdrLun = lun >> 16;

    /* Close the interface and rid the iterator */
    iorv = (*(intFace[rdrLun])->iface)->USBInterfaceClose( (intFace[rdrLun])->iface );
    if ( iorv != kIOReturnSuccess ) {
        return STATUS_UNSUCCESSFUL;
    }

    return STATUS_SUCCESS;
}

