/***************************************************************************
 *   Copyright (C) 2004-2006 by Jim Campbell                               *
 *   ifpgui@gmail.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "fileviewer.h"
#include "progressdlg.h"

#include <qtooltip.h>
#include <qwhatsthis.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qpixmap.h>
#include <qevent.h>
#include <qpoint.h>
#include <qmessagebox.h>
#include <qdragobject.h>
#include <qmime.h>
#include <qstrlist.h>
#include <qstringlist.h>
#include <qapplication.h>
#include <qheader.h>
#include <qpopupmenu.h>
#include <qcursor.h>
#include <sys/stat.h>
#include <unistd.h>
#include "errno.h"
#include "qglobal.h"
#ifdef BSD4
   #include "libgen.h"
#endif

//#include "gui_user.h"
#include "ifpinterface.h"
#include "createdirdlg.h"
#include "iointerface.h"

static const int autoopenTime = 1500;

static const char* folder_closed_xpm[] =
   {
      "16 16 9 1",
      "g c #808080",
      "b c #c0c000",
      "e c #c0c0c0",
      "# c #000000",
      "c c #ffff00",
      ". c None",
      "a c #585858",
      "f c #a0a0a4",
      "d c #ffffff",
      "..###...........",
      ".#abc##.........",
      ".#daabc#####....",
      ".#ddeaabbccc#...",
      ".#dedeeabbbba...",
      ".#edeeeeaaaab#..",
      ".#deeeeeeefe#ba.",
      ".#eeeeeeefef#ba.",
      ".#eeeeeefeff#ba.",
      ".#eeeeefefff#ba.",
      ".##geefeffff#ba.",
      "...##gefffff#ba.",
      ".....##fffff#ba.",
      ".......##fff#b##",
      ".........##f#b##",
      "...........####."
   };

static const char* folder_open_xpm[] =
   {
      "16 16 11 1",
      "# c #000000",
      "g c #c0c0c0",
      "e c #303030",
      "a c #ffa858",
      "b c #808080",
      "d c #a0a0a4",
      "f c #585858",
      "c c #ffdca8",
      "h c #dcdcdc",
      "i c #ffffff",
      ". c None",
      "....###.........",
      "....#ab##.......",
      "....#acab####...",
      "###.#acccccca#..",
      "#ddefaaaccccca#.",
      "#bdddbaaaacccab#",
      ".eddddbbaaaacab#",
      ".#bddggdbbaaaab#",
      "..edgdggggbbaab#",
      "..#bgggghghdaab#",
      "...ebhggghicfab#",
      "....#edhhiiidab#",
      "......#egiiicfb#",
      "........#egiibb#",
      "..........#egib#",
      "............#ee#"
   };

static const char * folder_locked[] =
   {
      "16 16 10 1",
      "h c #808080",
      "b c #ffa858",
      "f c #c0c0c0",
      "e c #c05800",
      "# c #000000",
      "c c #ffdca8",
      ". c None",
      "a c #585858",
      "g c #a0a0a4",
      "d c #ffffff",
      "..#a#...........",
      ".#abc####.......",
      ".#daa#eee#......",
      ".#ddf#e##b#.....",
      ".#dfd#e#bcb##...",
      ".#fdccc#daaab#..",
      ".#dfbbbccgfg#ba.",
      ".#ffb#ebbfgg#ba.",
      ".#ffbbe#bggg#ba.",
      ".#fffbbebggg#ba.",
      ".##hf#ebbggg#ba.",
      "...###e#gggg#ba.",
      ".....#e#gggg#ba.",
      "......###ggg#b##",
      ".........##g#b##",
      "...........####."
   };

static const char * pix_file [] =
   {
      "16 16 7 1",
      "# c #000000",
      "b c #ffffff",
      "e c #000000",
      "d c #404000",
      "c c #c0c000",
      "a c #ffffc0",
      ". c None",
      "................",
      ".........#......",
      "......#.#a##....",
      ".....#b#bbba##..",
      "....#b#bbbabbb#.",
      "...#b#bba##bb#..",
      "..#b#abb#bb##...",
      ".#a#aab#bbbab##.",
      "#a#aaa#bcbbbbbb#",
      "#ccdc#bcbbcbbb#.",
      ".##c#bcbbcabb#..",
      "...#acbacbbbe...",
      "..#aaaacaba#....",
      "...##aaaaa#.....",
      ".....##aa#......",
      ".......##......."
   };

QPixmap *folderLocked = 0;
QPixmap *folderClosed = 0;
QPixmap *folderOpen = 0;
QPixmap *fileNormal = 0;


FileViewToolBar::FileViewToolBar(QWidget *parent, const char *name)
      : QWidget(parent)
   {
   top_layout = new QBoxLayout(this, QBoxLayout::LeftToRight);
   top_layout->setAutoAdd(false);

   panelName       = new QLabel(QString::fromUtf8(name), this);
   top_layout->addWidget(panelName, 0, AlignBottom);

   lblSpace1       = new QLabel("           ", this);
   top_layout->addWidget(lblSpace1);//, 0, AlignBottom);

   UpOneLevel      = new QPushButton("Up 1 Level", this);
   UpOneLevel->setMinimumSize(25, 23);
   UpOneLevel->setFocusPolicy(QWidget::NoFocus);
   UpOneLevel->setPixmap(QPixmap::fromMimeSource("updirectory.png"));
   QToolTip::add(UpOneLevel, tr("Go Up a Directory"));
   QString UpOneLevelText =
      tr("<p>Press this button to go up a directory from the current selected directory.</p>");
   QWhatsThis::add(UpOneLevel, UpOneLevelText);
   top_layout->addWidget(UpOneLevel, 0, AlignBottom);

   Refresh         = new QPushButton("Refresh", this);
   Refresh->setMinimumSize(25, 23);
   Refresh->setFocusPolicy(QWidget::NoFocus);
   Refresh->setPixmap(QPixmap::fromMimeSource("reload.png"));
   QToolTip::add(Refresh, tr("Refresh the current view or selected directory"));
   QString RefreshText = tr("<p>Refresh the current view or selected directory.</p>");
   QWhatsThis::add(Refresh, RefreshText);
   top_layout->addWidget(Refresh, 0, AlignBottom);

   CreateDirectory = new QPushButton("New Directory", this);
   CreateDirectory->setMinimumSize(25, 23);
   CreateDirectory->setFocusPolicy(QWidget::NoFocus);
   CreateDirectory->setPixmap(QPixmap::fromMimeSource("newfolder.png"));
   QToolTip::add(CreateDirectory, tr("Create a new directory"));
   QString CreateDirectoryText = tr("<p>Create a new subdirectory at the current level or below the current selected directory.</p>");
   QWhatsThis::add(CreateDirectory, CreateDirectoryText);
   top_layout->addWidget(CreateDirectory, 0, AlignBottom);

   lblSpace2       = new QLabel("     ", this);
   top_layout->addWidget(lblSpace2);//, 0, AlignBottom);

   Properties      = new QPushButton("Properties", this);
   Properties->setMinimumSize(25, 23);
   Properties->setFocusPolicy(QWidget::NoFocus);
   Properties->setPixmap(QPixmap::fromMimeSource("properties.png"));
   QToolTip::add(Properties, tr("Show the properties for the selected file or directory"));
   QString PropertiesText = tr("<p>Select a file from the list and press this button to show it's properties.</p>");
   QWhatsThis::add(Properties, PropertiesText);
   top_layout->addWidget(Properties, 0, AlignBottom);

   Rename        = new QPushButton("Rename", this);
   Rename->setMinimumSize(25, 23);
   Rename->setFocusPolicy(QWidget::NoFocus);
   Rename->setPixmap(QPixmap::fromMimeSource("rename.png"));
   QToolTip::add(Rename, tr("Rename Selected File or Directory"));
   QString RenameText = tr("<p>Select a file from the list and press this button to rename it.</p>");
   QWhatsThis::add(Rename, RenameText);
   top_layout->addWidget(Rename, 0, AlignBottom);

   lblSpace3       = new QLabel("     ", this);
   top_layout->addWidget(lblSpace3);//, 0, AlignBottom);

   Delete        = new QPushButton("Delete", this);
   Delete->setMinimumSize(25, 23);
   Delete->setFocusPolicy(QWidget::NoFocus);
   Delete->setPixmap(QPixmap::fromMimeSource("delete.png"));
   QToolTip::add(Delete, tr("Delete Selected File or Directory"));
   QString DeleteText = tr("<p>Select a file from the list and press this button to delete it.</p>");
   QWhatsThis::add(Delete, DeleteText);
   top_layout->addWidget(Delete, 0, AlignBottom);

   top_layout->addStretch(10);

   connect(UpOneLevel, SIGNAL(clicked()),
           this, SLOT(sUpOneLevelClicked()));
   connect(Refresh, SIGNAL(clicked()),
           this, SLOT(sRefreshClicked()));
   connect(CreateDirectory, SIGNAL(clicked()),
           this, SLOT(sCreateDirectoryClicked()));
   connect(Properties, SIGNAL(clicked()),
           this, SLOT(sPropertiesClicked()));
   connect(Rename, SIGNAL(clicked()),
           this, SLOT(sRenameClicked()));
   connect(Delete, SIGNAL(clicked()),
           this, SLOT(sDeleteClicked()));
   }

FileViewToolBar::~FileViewToolBar()
   {
   }

void FileViewToolBar::sUpOneLevelClicked(void)
{
   emit UpOneLevelClicked();
}

void FileViewToolBar::sRefreshClicked(void)
{
   emit RefreshClicked();
}

void FileViewToolBar::sCreateDirectoryClicked(void)
{
   emit CreateDirectoryClicked();
}

void FileViewToolBar::sPropertiesClicked(void)
{
   emit PropertiesClicked();
}

void FileViewToolBar::sRenameClicked(void)
{
   emit RenameClicked();
}

void FileViewToolBar::sDeleteClicked(void)
{
   emit DeleteClicked();
}


/*****************************************************************************
 *
 * Class FileItem
 *
 *****************************************************************************/
void FileItem::setPixmap(QPixmap *qpix)
   {
   pix = qpix;
   setup();
   widthChanged(0);
   invalidateHeight();
   repaint();
   }


const QPixmap *FileItem::pixmap(int idx) const
   {
   // only column 0 has a pixmap
   if (idx == 0)
      return pix;
   return(0);
   }

/*****************************************************************************
 *
 * Class Directory
 *
 *****************************************************************************/

Directory::Directory(Directory * parent, const QString& filename, IOInterface *Intf)
      : QListViewItem(parent, filename, "Directory"), file(filename),
      showDirsOnly(parent->showDirsOnly),
      pix(0)
   {
   QString fullFileName;
   parent_dir = parent;
   ioIntf = Intf;

   fullFileName = ioIntf->ProtocolType + fullName();
   readable = ioIntf->get_isReadable(fullFileName);
   setRenameEnabled(0, TRUE);

   if (!readable)
      setPixmap(folderLocked);
   else
      setPixmap(folderClosed);
   }


Directory::Directory(QListView * parent, const QString& filename, IOInterface *Intf)
      : QListViewItem(parent, filename, "Directory"), file(filename),
      showDirsOnly(((DirectoryView*)parent)->showDirsOnly()),
      pix(0)
   {
   QString fullFileName;
   parent_dir = 0;
   ioIntf = Intf;

   fullFileName = ioIntf->ProtocolType + fullName();
   readable = Intf->get_isReadable(fullFileName);
   setRenameEnabled(0, TRUE);
   setPixmap(folderClosed);
   }


void Directory::setPixmap(QPixmap *qpix)
   {
   pix = qpix;
   setup();
   widthChanged(0);
   invalidateHeight();
   repaint();
   }


const QPixmap *Directory::pixmap(int idx) const
   {
   // only column 0 has a pixmap
   if (idx == 0)
      return pix;
   return(0);
   }

void Directory::setOpen(bool opn)
   {
   int ret;
   QString dirName, filenamestr, sizestr, filetype;
   QPtrList<FileInfos> file_info_list;
   FileInfos *fi_temp;

   if (opn)
      setPixmap(folderOpen);
   else
      setPixmap(folderClosed);

   if (opn && !childCount())
      {
      dirName = ioIntf->ProtocolType + fullName();
      //file_info_list = NULL;
      ret = ioIntf->exec_GetDirectoryListing(dirName, &file_info_list);
      if (ret != IO_INTF_OK)
         {
         setPixmap(folderClosed);
         showInfoMsgBox(QObject::tr("Get Directory Listing"), QObject::tr("Could not get the listing for ") + dirName);
         return;
         }

      if (file_info_list.isEmpty())
         {
         readable = FALSE;
         setExpandable(FALSE);
         return;
         }

      listView()->setUpdatesEnabled(FALSE);

      for (fi_temp = file_info_list.first(); fi_temp; fi_temp = file_info_list.next())
         {
         sizestr = " ";
         filenamestr = fi_temp->FileName();
         if (fi_temp->FileType() == IO_INTF_FILE)
            {
            // Add file
            sizestr = QString("%1").arg(fi_temp->FileSize(), 12);
            filetype = "File";
            new FileItem(this, filenamestr, filetype, sizestr);
            }
         else if (fi_temp->FileType() == IO_INTF_DIR)
            {
            // Add directory
            filetype = "Directory";
            new Directory(this, filenamestr, ioIntf);
            }
         else if (fi_temp->FileType() == IO_INTF_SYMLINK)
            {
            // Add directory
            filetype = "Symbolic Link";
            }
         else if (fi_temp->FileType() == IO_INTF_SPECIAL)
            {
            // Add directory
            filetype = "Special";
            new FileItem(this, filenamestr, filetype, sizestr);
            }
         }
      listView()->setUpdatesEnabled(TRUE);
      }
   QListViewItem::setOpen(opn);
   }


void Directory::setup()
   {
   setExpandable(TRUE);
   QListViewItem::setup();
   }


QString Directory::fullName()
   {
   QString fname;

   fname = file.name();
   if (parent_dir)
      {
      fname = parent_dir->fullName() + fname + "/";
      }
   return fname;
   }


void Directory::setDirName(const QString& newName)
   {
   file.setName(newName);
   }


/*****************************************************************************
 *
 * Class DirectoryView
 *
 *****************************************************************************/
DirectoryView::DirectoryView(QWidget *parent, const char *name, bool sdo)
      : QListView(parent, name), dirsOnly(sdo), oldCurrent(0),
      dropItem(0), mousePressed(FALSE)
   {
   autoopen_timer = new QTimer(this);
   if (!folderLocked)
      {
      folderLocked = new QPixmap(folder_locked);
      folderClosed = new QPixmap(folder_closed_xpm);
      folderOpen = new QPixmap(folder_open_xpm);
      fileNormal = new QPixmap(pix_file);
      }

   connect(this, SIGNAL(itemRenamed(QListViewItem *, int, const QString &)),
           this, SLOT(fvItemRenamed(QListViewItem *, int, const QString &)));

   setSelectionMode(QListView::Extended);
   setAcceptDrops(TRUE);
   viewport()->setAcceptDrops(TRUE);

   connect(autoopen_timer, SIGNAL(timeout()),
           this, SLOT(openFolder()));
   msgflag = 0;
   }

void DirectoryView::openFolder()
   {
   autoopen_timer->stop();
   if (dropItem && !dropItem->isOpen())
      {
      dropItem->setOpen(TRUE);
      dropItem->repaint();
      }
   }

void DirectoryView::keyPressEvent(QKeyEvent *kpevent)
   {
   if (kpevent->key() == Qt::Key_Delete)
      {
      fvDeleteFile();
      kpevent->accept();
      }
   else if (kpevent->key() == Qt::Key_F2)
      {
      fvRenameFile();
      kpevent->accept();
      }
   else if (kpevent->key() == Qt::Key_F5)
      {
      fvRefreshFileItem();
      }
   }

void DirectoryView::contentsDragEnterEvent(QDragEnterEvent *dr_event)
   {
   if (!QUriDrag::canDecode(dr_event))
      {
      dr_event->ignore();
      return ;
      }

   oldCurrent = currentItem();

   QListViewItem *itm = itemAt(contentsToViewport(dr_event->pos()));
   if (itm)
      {
      if (itm != dropItem)
         {
         autoopen_timer->stop();
         }
      dropItem = itm;
      autoopen_timer->start(autoopenTime);
      }
   }


void DirectoryView::contentsDragMoveEvent(QDragMoveEvent *dr_event)
   {
   QWidget * srcWidget, *destWidget;

   srcWidget = dr_event->source();
   destWidget = (QWidget*)(this);

   if ((srcWidget == NULL) || (srcWidget == destWidget))
      {
      dr_event->ignore();
      return ;
      }

   if (!QUriDrag::canDecode(dr_event))
      {
      dr_event->ignore();
      return ;
      }

   QPoint vp = contentsToViewport(((QDragMoveEvent*)dr_event)->pos());
   QListViewItem *itm = itemAt(vp);
   if (itm)
      {
      clearSelection();
      setSelected(itm, TRUE);
      setCurrentItem(itm);

      if (itm != dropItem)
         {
         autoopen_timer->stop();
         }
      dropItem = itm;
      autoopen_timer->start(autoopenTime);
      }

   dr_event->accept();
   dr_event->acceptAction();
   }

void DirectoryView::contentsDragLeaveEvent(QDragLeaveEvent *)
   {
   autoopen_timer->stop();
   dropItem = 0;
   }

void DirectoryView::contentsDropEvent(QDropEvent *dr_event)
   {
   autoopen_timer->stop();
   QWidget *srcWidget, *destWidget;
   QString filename, Message, destDir;
   ifp_progress_info pginfo;
   QStrList lst;
   QListViewItem *item;
   QPtrList<FileInfos> list;

   if (!QUriDrag::canDecode(dr_event))
      {
      dr_event->ignore();
      return ;
      }

   srcWidget = dr_event->source();
   destWidget = (QWidget*)(this);

   if ((srcWidget == NULL) || (srcWidget == destWidget))
      {
      dr_event->ignore();
      return ;
      }

   QUriDrag::decode(dr_event, lst);
   if (lst.count() <= 0)
      {
      dr_event->ignore();
      return;
      }
   //-------------------------------------------------------
   // Copy from ifp device to local file system
   //-------------------------------------------------------
   item = itemAt(contentsToViewport(dr_event->pos()));
   destDir = baseDirectory(item);

   dr_event->acceptAction();
   dr_event->accept();

   for (uint cnt = 0; cnt < lst.count(); ++cnt)
      {
      filename = QUriDrag::uriToUnicodeUri(lst.at(cnt));
      backslash2slash(filename);
      list.append(new FileInfos(QString(filename), -1, 0, NULL));
      }
   ioIntf->exec_TransferDownload(destDir, &list);

   setDir(destDir);
   fvRefreshFileItem();
   return;
   }

QString DirectoryView::fullPath(QListViewItem* item)
   {
   QString fullpath = item->text(0);
   while((item = item->parent()))
      {
      if (item->parent())
         fullpath = item->text(0) + "/" + fullpath;
      else
         fullpath = item->text(0) + fullpath;
      }
#ifdef Q_WS_WIN
   if (fullpath.length() > 2 && fullpath[ 1 ] != ':')
      {
      QDir dir(fullpath);
      fullpath = dir.currentDirPath().left(2) + fullpath;
      }
#endif

   return fullpath;
   }

QString DirectoryView::baseDirectory(QListViewItem* item)
   {
   QString baseDir;
   QListViewItem *parentItem;

   baseDir = "/";

   if (item == NULL)
      return(baseDir);
   //how to get the directory?? // are we in a directory??
   if (item->pixmap(0) == fileNormal)
      {
      parentItem = item->parent();
      if (parentItem)
         baseDir = fullPath(parentItem) + "/";
      }
   else
      {
      baseDir = fullPath(item) + "/";
      }

   return baseDir;
   }

QString DirectoryView::currentDirectory(void)
   {
   QListViewItem *item = currentItem();
   if ( !item->isSelected() )
      {
      return("/");
      }

   return(baseDirectory(item));
   }

void DirectoryView::contentsMousePressEvent(QMouseEvent* evnt)
   {
   QListView::contentsMousePressEvent(evnt);
   QPoint pnt(contentsToViewport(evnt->pos()));
   QListViewItem *itm = itemAt(pnt);

   // Ignore all clicks except button 1
   if (evnt->button() != Qt::LeftButton)
      return;

   if (itm)
      {
      // if the user clicked into the root decoration of the item, don't try to start a drag!
      if (pnt.x() > header()->cellPos(header()->mapToActual(0)) +
            treeStepSize() * (itm->depth() + (rootIsDecorated() ? 1 : 0)) + itemMargin() ||
            pnt.x() < header()->cellPos(header()->mapToActual(0)))
         {
         presspos = evnt->pos();
         mousePressed = TRUE;
         }
      }
   }

void DirectoryView::unselectAllChildren(QListViewItem *parentItem)
   {
   QListViewItem *current;

   current = parentItem->firstChild();
   while(current != NULL)
      {
      current->setSelected(false);
      if (current->childCount() > 0)
         unselectAllChildren(current);
      current = current->nextSibling();
      }
   }

int DirectoryView::getListOfSelectedItems(QPtrList<FileInfos> *file_info_list)
   {
   QString filename;
   QListViewItem *item;

   //--------------------------------------------------------
   // make a list of all selected files
   //--------------------------------------------------------
   QListViewItemIterator it(this);
   while(it.current())
      {
      item = it.current();
      if (item->isSelected())
         {
         filename = fullPath(item);
         if (item->pixmap(0) == fileNormal)
            {
            file_info_list->append(new FileInfos(QString(filename), IO_INTF_FILE, 0, (void*)(item)));
            }
         else
            {
            // do not include files or directories below this one
            unselectAllChildren(item);
            file_info_list->append(new FileInfos(QString(filename), IO_INTF_DIR, 0, (void*)(item)));
            }
         }
      ++it;
      }
   return(IO_INTF_OK);
   }

void DirectoryView::contentsMouseMoveEvent(QMouseEvent* evnt)
   {
   if (mousePressed && (presspos - evnt->pos()).manhattanLength() > QApplication::startDragDistance())
      {
      QListViewItem *item;
      QStringList source;
      QString filename;
      QUriDrag * ud = new QUriDrag(this);

      mousePressed = FALSE;

      QListViewItemIterator it(this);
      while(it.current())
         {
         item = it.current();
         filename = fullPath(item);
         if (item->isSelected())
            {
            if (item->pixmap(0) != fileNormal)
               {
               unselectAllChildren(item);
               }
            source << filename;
            }
         ++it;
         }
      ud->setUnicodeUris(source);
      ud->drag();
      }
   }

void DirectoryView::contentsMouseReleaseEvent(QMouseEvent *)
   {
   mousePressed = FALSE;
   }

void DirectoryView::setDir(const QString &s)
   {
   QListViewItem *item;

   QListViewItemIterator it(this);
   ++it;
   for ( ; it.current(); ++it)
      {
      it.current()->setOpen(FALSE);
      }

   QStringList lst(QStringList::split("/", s));
   item = firstChild();
   QStringList::Iterator it2 = lst.begin();
   for ( ; it2 != lst.end(); ++it2)
      {
      while(item)
         {
         if (item->text(0) == *it2)
            {
            item->setOpen(TRUE);
            break;
            }
         item = item->itemBelow();
         }
      }

   if (item)
      {
      clearSelection();
      setCurrentItem(item);
      setSelected(item, TRUE);
      ensureItemVisible(item);
      }
   }

void DirectoryView::contextMenuEvent(QContextMenuEvent *)
{
   QPopupMenu *contextMenu;

   contextMenu = new QPopupMenu(this);

   Q_CHECK_PTR(contextMenu);
   contextMenu->insertSeparator();
   QListViewItem *item = currentItem();
   if (item)
      {
      contextMenu->insertItem(tr("&Rename"),  this, SLOT(fvRenameFile()), Key_F2);
      contextMenu->insertItem(tr("&Delete"),  this, SLOT(fvDeleteFile()), Key_Delete);
      if (item->pixmap(0) != fileNormal)
         {
         contextMenu->insertItem(tr("&Create Directory"), this, SLOT(fvCreateDirectory()));
         }
      contextMenu->insertSeparator();
      contextMenu->insertItem(tr("&Refresh"),  this, SLOT(fvRefreshFileItem()), Key_F5);
      }
   else
      {
      contextMenu->insertItem(tr("&Create Directory"), this, SLOT(fvCreateDirectory()));
      }
   contextMenu->exec(QCursor::pos());
   delete contextMenu;
}

void DirectoryView::fvUpOneLevel(void)
   {
   QString baseDir;
   QListViewItem *parentItem;
   QListViewItem *item = currentItem();

   if (!item)
      {
      setDir("/");
      return;
      }

   if (item->pixmap(0) == fileNormal)
      {
      parentItem = item->parent();
      if (!parentItem)
         {
         baseDir = "/";
         }
      else
         {
         parentItem = item->parent();
         if (!parentItem)
            baseDir = "/";
         else
            baseDir = fullPath(parentItem) + "/";
         }
      }
   else
      {
      parentItem = item->parent();
      if (!parentItem)
         {
         baseDir = "/";
         }
      else
         {
         baseDir = fullPath(parentItem) + "/";
         }
      }
   setDir(baseDir);
   }

void DirectoryView::fvCreateDirectory(void)
{
   QString BaseNewDirPath, newDirPath, err;
   int ret;
   QListViewItem *item;

   item = currentItem();
   BaseNewDirPath = "/";

   if (item)
      {
      BaseNewDirPath = fullPath(item);
      }

   ret = ioIntf->exec_Mkdir(BaseNewDirPath, newDirPath);
   if (ret == IO_INTF_USER_CANCEL)
      {
      // do nothing
      }
   else if (ret != IO_INTF_OK)
      {
      err = ioIntf->get_LastError();
      showInfoMsgBox(tr("Make Directory Status"), err);
      }
   else
      {
      setDir(BaseNewDirPath);
      fvRefreshFileItem();
      backslash2slash(newDirPath);
      setDir(newDirPath);
      }
}

void DirectoryView::fvRenameFile(void)
{
   QListViewItem *item = currentItem();

   if (item)
      {
      OldRenamePath = fullPath(item);
      item->startRename(0);
      }
}

void DirectoryView::fvDeleteFile(void)
{
   int ret;
   FileInfos *fi_temp;
   QListViewItem *item;
   QPtrList<FileInfos> file_info_list;

   //--------------------------------------------------------
   // make a list of all selected files then call exec_Delete
   //--------------------------------------------------------
   ret = getListOfSelectedItems(&file_info_list);

   ret = ioIntf->exec_Delete(&file_info_list);
   for (fi_temp = file_info_list.first(); fi_temp; fi_temp = file_info_list.next())
      {
      if (fi_temp->Processed())
         {
         item = (QListViewItem*)fi_temp->Context();
         delete item;
         }
      }
}

void DirectoryView::clearItem(QListViewItem* item)
   {
   while(item->firstChild())
      delete(item->firstChild());
   }

void DirectoryView::fvItemRenamed(QListViewItem * item, int col, const QString & text)
{
   QString newName, baseName, err, temp;
   int ret;

   if (text)
      {
      // satisfy compilier
      }

   if (col != 0)
      return;

   // rename item
   if (item)
      {
      newName = fullPath(item);
      ret = ioIntf->exec_Rename(OldRenamePath, newName);
      if (ret != 0)
         {
         baseName = basename(OldRenamePath);
         item->setText(0, baseName);

         err = ioIntf->get_LastError();
         showInfoMsgBox(tr("Rename Status"), err);
         }
      if (item->pixmap(0) != fileNormal)
         {
         ((Directory*)item)->setDirName(text);
         }
      }
}

void DirectoryView::fvProperties(void)
   {
   QString fileName;
   QListViewItem *item = currentItem();

   if (item)
      {
      fileName = fullPath(item);
      ioIntf->exec_ShowProperties(fileName);
      }
   }

void DirectoryView::fvRefreshFileItem(void)
   {
   QListViewItem *dirItem;
   QListViewItem *item = currentItem();
   QString setFocusName = fullPath(item);

   if (setFocusName == "/")
      setFocusName = "//";

   if (item)
      {
      if (item->pixmap(0) == fileNormal)
         {
         dirItem = item->parent();
         }
      else
         {
         dirItem = item;
         }
      }
   else
      {
      return;
      }

   clearItem(dirItem);
   ((Directory*)dirItem)->setOpen(TRUE);
   setDir(setFocusName);
   }

void DirectoryView::fvTransferSelectedFiles(QString& destDir)
   {
   int ret;
   QPtrList<FileInfos> list;

   //----------------------------------------------------------------
   // make a list of all selected files then call exec_TransferUpload
   //----------------------------------------------------------------
   ret = getListOfSelectedItems(&list);

   ret = ioIntf->exec_TransferUpload(destDir, &list);
   }

fileManagerView::fileManagerView(QWidget *parent, const char *title, bool sort, int ioIntfType)
      : QWidget(parent)
{
   top_layout = new QBoxLayout(this, QBoxLayout::TopToBottom);
   top_layout->setAutoAdd(false);

   filetoolbar = new FileViewToolBar(this, title);
   filetoolbar->setMinimumSize(100,30);

   top_layout->add(filetoolbar);
   internal_fileview = new DirectoryView(this, "internal_fileview", FALSE);
   top_layout->add(internal_fileview);
   setFocusProxy(internal_fileview);
   switch (ioIntfType)
      {
      case IO_INTF_TYPE_LOCAL :
         internal_fileview->ioIntf = new FileInterface();
         break;
      case IO_INTF_TYPE_IFP :
         internal_fileview->ioIntf = new iFPInterface();
         break;
      default: // IO_INTF_TYPE_LOCAL
         internal_fileview->ioIntf = new FileInterface();
      }
   internal_fileview->addColumn( tr("Name") );
   internal_fileview->setColumnAlignment(0, Qt::AlignLeft);
   internal_fileview->addColumn( tr("Type") );
   internal_fileview->setColumnAlignment(1, Qt::AlignLeft);
   internal_fileview->addColumn( tr("Size") );
   internal_fileview->setColumnAlignment(2, Qt::AlignRight);
   internal_fileview->setTreeStepSize( 20 );

   internal_fileview->setCaption(QString::fromUtf8(title));
   internal_fileview->setAllColumnsShowFocus(TRUE);
   if (sort)
      {
      internal_fileview->setShowSortIndicator(TRUE);
      internal_fileview->setSortColumn(-1);
      internal_fileview->setSorting(1);
      }
   else
      {
      internal_fileview->setSortColumn(-1);
      internal_fileview->setSorting(-1);
      }

   connect(filetoolbar, SIGNAL(UpOneLevelClicked()),
           this, SLOT(sUpOneLevelClicked()));
   connect(filetoolbar, SIGNAL(RefreshClicked()),
           this, SLOT(sRefreshClicked()));
   connect(filetoolbar, SIGNAL(CreateDirectoryClicked()),
           this, SLOT(sCreateDirectoryClicked()));
   connect(filetoolbar, SIGNAL(PropertiesClicked()),
           this, SLOT(sPropertiesClicked()));
   connect(filetoolbar, SIGNAL(RenameClicked()),
           this, SLOT(sRenameClicked()));
   connect(filetoolbar, SIGNAL(DeleteClicked()),
           this, SLOT(sDeleteClicked()));

   // add the root directory and set it to open
   Directory *fileview_root = new Directory(internal_fileview, "/", internal_fileview->ioIntf);
   fileview_root->setOpen(TRUE);
}

fileManagerView::~fileManagerView()
   {
   }

void fileManagerView::setDir(const QString &dirName)
   {
   internal_fileview->setDir(dirName);
   }

void fileManagerView::fmCreateDirectory(void)
   {
   internal_fileview->fvCreateDirectory();
   }

void fileManagerView::fmRenameFile(void)
   {
   internal_fileview->fvRenameFile();
   }

void fileManagerView::fmDeleteFile(void)
   {
   internal_fileview->fvDeleteFile();
   }

void fileManagerView::fmRefreshFileItem(void)
   {
   internal_fileview->fvRefreshFileItem();
   }

void fileManagerView::fmTransferSelectedFiles(QString& destDir)
   {
   internal_fileview->fvTransferSelectedFiles(destDir);
   }

int fileManagerView::fmUploadFileList(QString& destination, QPtrList<FileInfos> *file_info_list)
   {
   int ret;
   ret = internal_fileview->ioIntf->exec_TransferUpload(destination, file_info_list);
   return(ret);
   }

void fileManagerView::sUpOneLevelClicked(void)
   {
   internal_fileview->fvUpOneLevel();
   }

void fileManagerView::sRefreshClicked(void)
   {
   internal_fileview->fvRefreshFileItem();
   }

void fileManagerView::sCreateDirectoryClicked(void)
   {
   internal_fileview->fvCreateDirectory();
   }

void fileManagerView::sPropertiesClicked(void)
   {
   internal_fileview->fvProperties();
   }

void fileManagerView::sRenameClicked(void)
   {
   internal_fileview->fvRenameFile();
   }

void fileManagerView::sDeleteClicked(void)
   {
   internal_fileview->fvDeleteFile();
   }

QString fileManagerView::currentDirectory(void)
   {
   return(internal_fileview->currentDirectory());
   }

bool fileManagerView::isProcessing()
   {
   return(internal_fileview->ioIntf->isProcessing());
   }
