/* -*- mode: c++ -*-

   This file is part of the Life library

   Author(s): Christophe Prud'homme <christophe.prudhomme@ujf-grenoble.fr>
   Date: 2006-03-03

   Copyright (C) 2006 EPFL

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 3.0 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this library; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
/**
   \file fekete.hpp
   \author Goncalo Pena <goncalo.pena@epfl.ch>
   \date 2006-10-02
*/
#ifndef __Fekete_H
#define __Fekete_H 1

#include <life/lifealg/glas.hpp>
#include <life/lifepoly/pointsetinterpolation.hpp>
#include <life/lifepoly/geomap.hpp>
#include <life/lifepoly/warpblend.hpp>
#include <life/lifepoly/gausslobatto.hpp>
#include <life/lifepoly/equispaced.hpp>

#include <boost/assign/list_of.hpp>
#include <boost/assign/std/vector.hpp>

namespace Life
{
template< class Convex,
          uint16_type Order,
          typename T >
class PointSetWarpBlend;

template< class Convex,
          uint16_type Order,
          typename T >
class PointSetGaussLobatto;

template< class Convex,
          uint16_type Order,
          typename T = double >
class PointSetFeketeSimplex : public  PointSetInterpolation<Convex::nDim, Order, T, Simplex>
{

public :

    typedef PointSetWarpBlend<Convex, Order, T> super;

    typedef T value_type;

    static const uint32_type Dim = Convex::nDim;
    static const uint16_type nPoints1D = Order+1;

    static const size_type Shape = Convex::Shape;

    static const bool is_simplex = Convex::is_simplex;
    static const bool is_simplex_product = Convex::is_simplex_product;

    typedef typename super::return_type return_type;

    typedef ublas::vector<value_type> vector_type;

    static const uint32_type topological_dimension = Convex::topological_dimension;
    static const uint32_type numPoints = super::numPoints;

    typedef Reference<Convex, Dim, Convex::nOrder, Dim, value_type> reference_convex_type;

    typedef typename reference_convex_type::points_type points_type;

    static const uint32_type nbPtsPerFace = super::nbPtsPerFace;

    typedef std::vector<uint16_type> orbits_type;

    reference_convex_type RefConv;

    PointSetFeketeSimplex( int interior = 0 )
    {
        PointSetWarpBlend<Convex, Order, T> G(interior);

        points_type final_pts = G.points();

        //Copies information about WarpBlend points
        this->setEid( G.getEid() );
        this->setPtE( G.getPtE() );

        if ( Order > 2 )
            {
                points_type Gt(Dim, nbPtsPerFace);

                Gt = innerFeketePoints();
                Gt = orderInteriorFace( Gt );

                if ( interior )
                    final_pts = Gt;
                else
                    final_pts = putInPointset ( final_pts, Gt, G.interiorRangeById(2, 0) );

            }

        this->setPoints( final_pts );

        this->setName( "fekete", Order );
    }

    ~PointSetFeketeSimplex() {}

private :

    points_type innerFeketePoints()
    {
        points_type inner_pts (Dim, nbPtsPerFace);

        std::vector< std::vector<double> > pts (nbPtsPerFace);

        using namespace boost::assign;

        switch ( Order )
            {
            case 3:
                {
                    pts[0] += -double(1.0)/double(3.0), -double(1.0)/double(3.0);
                    break;
                }

            case 4:
                {
                    pts[0] += -0.566915270681781,  -0.566915270681781;
                    pts[1] += -0.566915270681781,   0.133830541363562;
                    pts[2] +=  0.133830541363562,  -0.566915270681781;

                    break;
                }

            case 5:
                {
                    pts[0] += -0.703961057369349,  -0.703961057370137;
                    pts[1] += -0.683302157170358,  -0.158348921415476;
                    pts[2] += -0.703961057370137,   0.407922114739486;
                    pts[3] += -0.158348921414166,  -0.683302157170358;
                    pts[4] += -0.158348921415476,  -0.158348921414166;
                    pts[5] += 0.407922114739486,  -0.703961057369349;
                    break;
                }

            case 6:
                {
                    pts[0] += -0.333333333333333,  -0.333333333333333;
                    pts[1] += -0.367460408129093,  -0.765638165744237;
                    pts[2] += -0.787329063247951,  -0.787329063247952;
                    pts[3] += -0.765638165744237,  -0.367460408129093;
                    pts[4] += 0.133098573873330,  -0.765638165744237;
                    pts[5] += -0.367460408129093,   0.133098573873330;
                    pts[6] += -0.765638165744237,   0.133098573873330;
                    pts[7] += 0.133098573873330,  -0.367460408129093;
                    pts[8] += -0.787329063247952,   0.574658126495903;
                    pts[9] += 0.574658126495903,  -0.787329063247951;
                    break;
                }

            case 7:
                {

                    pts[0] += -0.473581790085882,  -0.473581790086410;
                    pts[1] += -0.092029311878152,  -0.815941376243689;
                    pts[2] += -0.513270713769908,  -0.822113905089151;
                    pts[3] += -0.840822986239630,  -0.840822986239633;
                    pts[4] += -0.473581790086410,  -0.052836419827708;
                    pts[5] += -0.052836419827708,  -0.473581790085882;
                    pts[6] += -0.815941376243689,  -0.092029311878159;
                    pts[7] += -0.092029311878159,  -0.092029311878152;
                    pts[8] += -0.822113905089151,  -0.513270713769908;
                    pts[9] += 0.335384618859059,  -0.822113905089151;
                    pts[10] += -0.513270713769908,   0.335384618859059;
                    pts[11] += -0.822113905089151,   0.335384618859059;
                    pts[12] += 0.335384618859059,  -0.513270713769908;
                    pts[13] += -0.840822986239633,   0.681645972479263;
                    pts[14] += 0.681645972479263,  -0.840822986239630;
                    break;
                }

            case 8:
                {
                    pts[0] += -0.216365666632575,  -0.567268666734080;
                    pts[1] += -0.579556771293157,  -0.579556771293783;
                    pts[2] += -0.265519472998560,  -0.853908185662976;
                    pts[3] += -0.616451325376556,  -0.861487487652122;
                    pts[4] += -0.876851564079131,  -0.876851564079171;
                    pts[5] += -0.567268666733596,  -0.216365666633543;
                    pts[6] += -0.216365666634537,  -0.216365666631639;
                    pts[7] += -0.579556771294394,   0.159113542587640;
                    pts[8] += 0.159113542586342,  -0.579556771292601;
                    pts[9] += -0.853908185662855,  -0.265519472998911;
                    pts[10] += 0.119427658661863,  -0.853908185662948;
                    pts[11] += -0.265519472998830,   0.119427658661806;
                    pts[12] += -0.853908185662892,   0.119427658661475;
                    pts[13] += 0.119427658661483,  -0.265519472998638;
                    pts[14] += -0.861487487652025,  -0.616451325376726;
                    pts[15] += 0.477938813028659,  -0.861487487651878;
                    pts[16] += -0.616451325376719,   0.477938813028829;
                    pts[17] += -0.861487487652288,   0.477938813028806;
                    pts[18] += 0.477938813028602,  -0.616451325376548;
                    pts[19] += -0.876851564079185,   0.753703128158331;
                    pts[20] += 0.753703128158281,  -0.876851564079116;
                    break;
                }

            case 9:
                {
                    pts[0] += -0.333333333333333,  -0.333333333333333;
                    pts[1] += -0.349513020087187,  -0.643132482384194;
                    pts[2] += -0.659136359813528,  -0.659136359813511;
                    pts[3] += -0.060082471166027,  -0.879835057667481;
                    pts[4] += -0.397951578026397,  -0.882287024211335;
                    pts[5] += -0.691219611179607,  -0.889648384166848;
                    pts[6] += -0.902130860890619,  -0.902130860890625;
                    pts[7] += -0.643132482384194,  -0.349513020087187;
                    pts[8] += -0.007354497528619,  -0.643132482384194;
                    pts[9] += -0.349513020087187,  -0.007354497528619;
                    pts[10] += -0.643132482384194,  -0.007354497528619;
                    pts[11] += -0.007354497528619,  -0.349513020087187;
                    pts[12] += -0.659136359813511,   0.318272719627039;
                    pts[13] +=  0.318272719627039,  -0.659136359813528;
                    pts[14] += -0.879835057667481,  -0.060082471166492;
                    pts[15] += -0.060082471166492,  -0.060082471166027;
                    pts[16] += -0.882287024211335,  -0.397951578026397;
                    pts[17] += 0.280238602237732,  -0.882287024211335;
                    pts[18] += -0.397951578026397,   0.280238602237732;
                    pts[19] += -0.882287024211335,   0.280238602237732;
                    pts[20] += 0.280238602237732,  -0.397951578026397;
                    pts[21] += -0.889648384166848,  -0.691219611179607;
                    pts[22] += 0.580867995346455,  -0.889648384166848;
                    pts[23] += -0.691219611179607,   0.580867995346455;
                    pts[24] += -0.889648384166848,   0.580867995346455;
                    pts[25] += 0.580867995346455,  -0.691219611179607;
                    pts[26] += -0.902130860890625,   0.804261721781244;
                    pts[27] += 0.804261721781244,  -0.902130860890619;
                    break;
                }

            case 10:
                {
                    pts[0] += -0.239750033297594,  -0.520499933404811;
                    pts[1] += -0.516800012629008,  -0.516800012629009;
                    pts[2] += -0.257946265874160,  -0.790148650478844;
                    pts[3] += -0.553131324174815,  -0.752415268984459;
                    pts[4] += -0.781465283043698,  -0.781465283043698;
                    pts[5] += -0.033736396065064,  -0.932527207869870;
                    pts[6] += -0.484262543101076,  -0.925302995602205;
                    pts[7] += -0.818984113526921,  -0.931063288077205;
                    pts[8] += -0.520499933404811,  -0.239750033297595;
                    pts[9] += -0.239750033297595,  -0.239750033297594;
                    pts[10] += -0.516800012629009,   0.033600025258018;
                    pts[11] += 0.033600025258018,  -0.516800012629008;
                    pts[12] += -0.790148650478844,  -0.257946265874160;
                    pts[13] += 0.048094916353004,  -0.790148650478844;
                    pts[14] += -0.257946265874160,   0.048094916353004;
                    pts[15] += -0.790148650478844,   0.048094916353004;
                    pts[16] +=  0.048094916353004,  -0.257946265874160;
                    pts[17] += -0.752415268984459,  -0.553131324174815;
                    pts[18] +=  0.305546593159274,  -0.752415268984459;
                    pts[19] += -0.553131324174815,   0.305546593159274;
                    pts[20] += -0.752415268984459,   0.305546593159274;
                    pts[21] += 0.305546593159274,  -0.553131324174815;
                    pts[22] += -0.781465283043698,   0.562930566087395;
                    pts[23] += 0.562930566087395,  -0.781465283043698;
                    pts[24] += -0.932527207869870,  -0.033736396065066;
                    pts[25] += -0.033736396065066,  -0.033736396065064;
                    pts[26] += -0.925302995602205,  -0.484262543101076;
                    pts[27] += 0.409565538703281,  -0.925302995602205;
                    pts[28] += -0.484262543101076,   0.409565538703281;
                    pts[29] += -0.925302995602205,   0.409565538703281;
                    pts[30] += 0.409565538703281,  -0.484262543101076;
                    pts[31] += -0.931063288077205,  -0.818984113526921;
                    pts[32] += 0.750047401604126,  -0.931063288077205;
                    pts[33] += -0.818984113526921,   0.750047401604126;
                    pts[34] += -0.931063288077205,   0.750047401604126;
                    pts[35] += 0.750047401604126,  -0.818984113526921;
                    break;
                }

            case 11:
                {
                    pts[0] += -0.426288429164565,  -0.426288429164564;
                    pts[1] += -0.184095969280423,  -0.631808061439155;
                    pts[2] += -0.586974557425838,  -0.586974557425838;
                    pts[3] += -0.377399646207547,  -0.775153413812684;
                    pts[4] += -0.776324974302030,  -0.908258463253342;
                    pts[5] += -0.086138932580443,  -0.827722134839114;
                    pts[6] += -0.633345990227640,  -0.798648875634472;
                    pts[7] += -0.908258463253342,  -0.776324974302030;
                    pts[8] += -0.218515810917888,  -0.935838462756195;
                    pts[9] += -0.535040235101238,  -0.937261747926706;
                    pts[10] += -0.928290828736822,  -0.928290828736822;
                    pts[11] += -0.426288429164564,  -0.147423141670871;
                    pts[12] += -0.147423141670871,  -0.426288429164565;
                    pts[13] += -0.631808061439155,  -0.184095969280421;
                    pts[14] += -0.184095969280421,  -0.184095969280423;
                    pts[15] += -0.586974557425838,   0.173949114851675;
                    pts[16] += 0.173949114851675,  -0.586974557425838;
                    pts[17] += -0.775153413812684,  -0.377399646207547;
                    pts[18] += 0.152553060020230,  -0.775153413812684;
                    pts[19] += -0.377399646207547,   0.152553060020230;
                    pts[20] += -0.775153413812684,   0.152553060020230;
                    pts[21] += 0.152553060020230,  -0.377399646207547;
                    pts[22] += -0.908258463253342,   0.684583437555372;
                    pts[23] += 0.684583437555372,  -0.776324974302030;
                    pts[24] += -0.827722134839114,  -0.086138932580443;
                    pts[25] += -0.086138932580443,  -0.086138932580443;
                    pts[26] += -0.798648875634472,  -0.633345990227640;
                    pts[27] += 0.431994865862111,  -0.798648875634472;
                    pts[28] += -0.633345990227640,   0.431994865862111;
                    pts[29] += -0.798648875634472,   0.431994865862111;
                    pts[30] += 0.431994865862111,  -0.633345990227640;
                    pts[31] += -0.776324974302030,   0.684583437555372;
                    pts[32] += 0.684583437555372,  -0.908258463253342;
                    pts[33] += -0.935838462756195,  -0.218515810917888;
                    pts[34] += 0.154354273674083,  -0.935838462756195;
                    pts[35] += -0.218515810917888,   0.154354273674083;
                    pts[36] += -0.935838462756195,   0.154354273674083;
                    pts[37] += 0.154354273674083,  -0.218515810917888;
                    pts[38] += -0.937261747926706,  -0.535040235101238;
                    pts[39] += 0.472301983027943,  -0.937261747926706;
                    pts[40] += -0.535040235101238,   0.472301983027943;
                    pts[41] += -0.937261747926706,   0.472301983027943;
                    pts[42] += 0.472301983027943,  -0.535040235101238;
                    pts[43] += -0.928290828736822,   0.856581657473645;
                    pts[44] += 0.856581657473645,  -0.928290828736822;
                    break;
                }

            case 12:
                {
                    pts[0] += -0.333333333333333,  -0.333333333333333;
                    pts[1] += -0.366118633733524,  -0.559725774975652;
                    pts[2] += -0.074155591290824,  -0.559725774975640;
                    pts[3] += -0.624565774148092,  -0.624565774148093;
                    pts[4] += -0.140340214413457,  -0.719319571173060;
                    pts[5] += -0.431536181329258,  -0.768507319183628;
                    pts[6] += -0.833349444442592,  -0.833349444442587;
                    pts[7] += -0.205647119941729,  -0.865429878796984;
                    pts[8] += -0.644199866452571,  -0.818032093757191;
                    pts[9] += -0.949540550694726,  -0.867065080338688;
                    pts[10] += -0.021888401958102,  -0.956223196083790;
                    pts[11] += -0.394807319693724,  -0.936337673359307;
                    pts[12] += -0.653266898894878,  -0.945296284206355;
                    pts[13] += -0.867065080338677,  -0.949540550694725;
                    pts[14] += -0.559725774975652,  -0.074155591290824;
                    pts[15] += -0.074155591290824,  -0.366118633733524;
                    pts[16] += -0.559725774975640,  -0.366118633733536;
                    pts[17] += -0.366118633733536,  -0.074155591290824;
                    pts[18] += -0.624565774148093,   0.249131548296185;
                    pts[19] += 0.249131548296185,  -0.624565774148092;
                    pts[20] += -0.719319571173060,  -0.140340214413483;
                    pts[21] += -0.140340214413483,  -0.140340214413457;
                    pts[22] += -0.768507319183628,  -0.431536181329258;
                    pts[23] += 0.200043500512886, -0.768507319183628;
                    pts[24] += -0.431536181329258,   0.200043500512886;
                    pts[25] += -0.768507319183628,   0.200043500512886;
                    pts[26] += 0.200043500512886,  -0.431536181329258;
                    pts[27] += -0.833349444442587,   0.666698888885179;
                    pts[28] += 0.666698888885179,  -0.833349444442592;
                    pts[29] += -0.865429878796984,  -0.205647119941729;
                    pts[30] += 0.071076998738713,  -0.865429878796984;
                    pts[31] += -0.205647119941729,   0.071076998738713;
                    pts[32] += -0.865429878796984,   0.071076998738713;
                    pts[33] += 0.071076998738713,  -0.205647119941729;
                    pts[34] += -0.818032093757191,  -0.644199866452571;
                    pts[35] += 0.462231960209762,  -0.818032093757191;
                    pts[36] += -0.644199866452571,   0.462231960209762;
                    pts[37] += -0.818032093757191,   0.462231960209762;
                    pts[38] += 0.462231960209762,  -0.644199866452571;
                    pts[39] += -0.867065080338688,   0.816605631033414;
                    pts[40] += 0.816605631033414,  -0.949540550694726;
                    pts[41] += -0.956223196083790,  -0.021888401958108;
                    pts[42] += -0.021888401958108, -0.021888401958102;
                    pts[43] += -0.936337673359307,  -0.394807319693724;
                    pts[44] += 0.331144993053031,  -0.936337673359307;
                    pts[45] += -0.394807319693724,   0.331144993053031;
                    pts[46] += -0.936337673359307,   0.331144993053031;
                    pts[47] += 0.331144993053031,  -0.394807319693724;
                    pts[48] += -0.945296284206355,  -0.653266898894878;
                    pts[49] += 0.598563183101232,  -0.945296284206355;
                    pts[50] += -0.653266898894878,   0.598563183101232;
                    pts[51] += -0.945296284206355,   0.598563183101232;
                    pts[52] += 0.598563183101232,  -0.653266898894878;
                    pts[53] += -0.949540550694725,   0.816605631033402;
                    pts[54] += 0.816605631033402,  -0.867065080338677;
                    break;
                }


            case 13:
                {
                    pts[0] += -0.411407513886432,  -0.411407513886432;
                    pts[1] += -0.214052183872017,  -0.571895632255967;
                    pts[2] += -0.539573883106313,  -0.539573883106313;
                    pts[3] += -0.287868553838811,  -0.717902048140164;
                    pts[4] += -0.540297218250118,  -0.732989588419528;
                    pts[5] += -0.743226499430370,  -0.743226499430370;
                    pts[6] += -0.071171195753188,  -0.857657608493625;
                    pts[7] += -0.335863001538683,  -0.859363917043220;
                    pts[8] += -0.623056409518609,  -0.862030029517440;
                    pts[9] += -0.852002455492610,  -0.852002455492610;
                    pts[10] +=-0.178334191212880,  -0.954817865487350;
                    pts[11] +=-0.464674822424600,  -0.950458941099439;
                    pts[12] +=-0.701942332364525,  -0.951956806119627;
                    pts[13] +=-0.883949040041051,  -0.955838523904429;
                    pts[14] +=-0.411407513886432,  -0.177184972227137;
                    pts[15] +=-0.177184972227137,  -0.411407513886432;
                    pts[16] +=-0.571895632255967,  -0.214052183872017;
                    pts[17] +=-0.214052183872017,  -0.214052183872017;
                    pts[18] +=-0.539573883106313,   0.079147766212627;
                    pts[19] += 0.079147766212627,  -0.539573883106313;
                    pts[20] +=-0.717902048140164,  -0.287868553838811;
                    pts[21] += 0.005770601978975,  -0.717902048140164;
                    pts[22] +=-0.287868553838811,   0.005770601978975;
                    pts[23] +=-0.717902048140164,   0.005770601978975;
                    pts[24] += 0.005770601978975,  -0.287868553838811;
                    pts[25] +=-0.732989588419528,  -0.540297218250118;
                    pts[26] += 0.273286806669646,  -0.732989588419528;
                    pts[27] +=-0.540297218250118,   0.273286806669646;
                    pts[28] +=-0.732989588419528,   0.273286806669646;
                    pts[29] += 0.273286806669646,  -0.540297218250118;
                    pts[30] +=-0.743226499430370,   0.486452998860741;
                    pts[31] += 0.486452998860741,  -0.743226499430370;
                    pts[32] +=-0.857657608493625,  -0.071171195753188;
                    pts[33] +=-0.071171195753188,  -0.071171195753188;
                    pts[34] +=-0.859363917043220,  -0.335863001538683;
                    pts[35] += 0.195226918581903,  -0.859363917043220;
                    pts[36] +=-0.335863001538683,   0.195226918581903;
                    pts[37] +=-0.859363917043220,   0.195226918581903;
                    pts[38] += 0.195226918581903,  -0.335863001538683;
                    pts[39] +=-0.862030029517440,  -0.623056409518609;
                    pts[40] += 0.485086439036049,  -0.862030029517440;
                    pts[41] +=-0.623056409518609,   0.485086439036049;
                    pts[42] +=-0.862030029517440,   0.485086439036049;
                    pts[43] += 0.485086439036049,  -0.623056409518609;
                    pts[44] +=-0.852002455492610,   0.704004910985221;
                    pts[45] += 0.704004910985221,  -0.852002455492610;
                    pts[46] +=-0.954817865487350,  -0.178334191212880;
                    pts[47] += 0.133152056700230,  -0.954817865487350;
                    pts[48] +=-0.178334191212880,   0.133152056700230;
                    pts[49] +=-0.954817865487350,   0.133152056700230;
                    pts[50] += 0.133152056700230,  -0.178334191212880;
                    pts[51] +=-0.950458941099439,  -0.464674822424600;
                    pts[52] += 0.415133763524038,  -0.950458941099439;
                    pts[53] +=-0.464674822424600,   0.415133763524038;
                    pts[54] +=-0.950458941099439,   0.415133763524038;
                    pts[55] += 0.415133763524038,  -0.464674822424600;
                    pts[56] +=-0.951956806119627,  -0.701942332364525;
                    pts[57] += 0.653899138484151,  -0.951956806119627;
                    pts[58] +=-0.701942332364525,   0.653899138484151;
                    pts[59] +=-0.951956806119627,   0.653899138484151;
                    pts[60] += 0.653899138484151,  -0.701942332364525;
                    pts[61] +=-0.955838523904429,  -0.883949040041051;
                    pts[62] += 0.839787563945479,  -0.955838523904429;
                    pts[63] +=-0.883949040041051,   0.839787563945479;
                    pts[64] +=-0.955838523904429,   0.839787563945479;
                    pts[65] += 0.839787563945479,  -0.883949040041051;
                    break;
                }


            case 14:
                {


                    pts[0] += -0.210489388870677,  -0.579021222258655;
                    pts[1] += -0.183849077476319,  -0.408075461261843;
                    pts[2] += -0.127347936150219,  -0.745304127699574;
                    pts[3] += -0.540572408690582,  -0.540572408690580;
                    pts[4] += -0.364367592015457,  -0.689962524023234;
                    pts[5] += -0.736068596037139,  -0.736068596037137;
                    pts[6] +=  0.038332201767425,  -0.861626855061358;
                    pts[7] += -0.554243348588859,  -0.727059313582804;
                    pts[8] += -0.871281426433905,  -0.769687425283662;
                    pts[9] += -0.176705346706054,  -0.861626855061365;
                    pts[10] +=-0.382367340313839,  -0.872946396511865;
                    pts[11] +=-0.584327224183587,  -0.875713493198755;
                    pts[12] +=-0.769687425283663,  -0.871281426433905;
                    pts[13] +=-0.173252532533942,  -0.957943978206850;
                    pts[14] +=-0.459441974333255,  -0.960810947366647;
                    pts[15] +=-0.697211314708751,  -0.958833352267252;
                    pts[16] +=-0.858673547396615,  -0.945197866913093;
                    pts[17] +=-0.955761107477727,  -0.955761107477727;
                    pts[18] +=-0.579021222258655,  -0.210489388870668;
                    pts[19] +=-0.210489388870668,  -0.210489388870677;
                    pts[20] +=-0.408075461261843,  -0.408075461261838;
                    pts[21] +=-0.408075461261838,  -0.183849077476319;
                    pts[22] +=-0.745304127699574,  -0.127347936150206;
                    pts[23] +=-0.127347936150206,  -0.127347936150219;
                    pts[24] +=-0.540572408690580,  0.081144817381162;
                    pts[25] += 0.081144817381162, -0.540572408690582;
                    pts[26] +=-0.689962524023234,  -0.364367592015457;
                    pts[27] += 0.054330116038691,  -0.689962524023234;
                    pts[28] +=-0.364367592015457,   0.054330116038691;
                    pts[29] +=-0.689962524023234,   0.054330116038691;
                    pts[30] += 0.054330116038691,  -0.364367592015457;
                    pts[31] +=-0.736068596037137,   0.472137192074276;
                    pts[32] += 0.472137192074276,  -0.736068596037139;
                    pts[33] +=-0.861626855061358,  -0.176705346706066;
                    pts[34] +=-0.176705346706066,   0.038332201767425;
                    pts[35] +=-0.727059313582804,  -0.554243348588859;
                    pts[36] += 0.281302662171663,  -0.727059313582804;
                    pts[37] +=-0.554243348588859,   0.281302662171663;
                    pts[38] +=-0.727059313582804,   0.281302662171663;
                    pts[39] += 0.281302662171663,  -0.554243348588859;
                    pts[40] +=-0.769687425283662,   0.640968851717568;
                    pts[41] += 0.640968851717568,  -0.871281426433905;
                    pts[42] +=-0.861626855061365,   0.038332201767419;
                    pts[43] += 0.038332201767419,  -0.176705346706054;
                    pts[44] +=-0.872946396511865,  -0.382367340313839;
                    pts[45] += 0.255313736825704,  -0.872946396511865;
                    pts[46] +=-0.382367340313839,   0.255313736825704;
                    pts[47] +=-0.872946396511865,   0.255313736825704;
                    pts[48] += 0.255313736825704,  -0.382367340313839;
                    pts[49] +=-0.875713493198755,  -0.584327224183587;
                    pts[50] += 0.460040717382342,  -0.875713493198755;
                    pts[51] +=-0.584327224183587,   0.460040717382342;
                    pts[52] +=-0.875713493198755,   0.460040717382342;
                    pts[53] += 0.460040717382342,  -0.584327224183587;
                    pts[54] +=-0.871281426433905,   0.640968851717568;
                    pts[55] += 0.640968851717568,  -0.769687425283663;
                    pts[56] +=-0.957943978206850,  -0.173252532533942;
                    pts[57] += 0.131196510740792,  -0.957943978206850;
                    pts[58] +=-0.173252532533942,   0.131196510740792;
                    pts[59] +=-0.957943978206850,   0.131196510740792;
                    pts[60] += 0.131196510740792,  -0.173252532533942;
                    pts[61] +=-0.960810947366647,  -0.459441974333255;
                    pts[62] += 0.420252921699902,  -0.960810947366647;
                    pts[63] +=-0.459441974333255,   0.420252921699902;
                    pts[64] +=-0.960810947366647,   0.420252921699902;
                    pts[65] += 0.420252921699902,  -0.459441974333255;
                    pts[66] +=-0.958833352267252,  -0.697211314708751;
                    pts[67] += 0.656044666976003,  -0.958833352267252;
                    pts[68] +=-0.697211314708751,   0.656044666976003;
                    pts[69] +=-0.958833352267252,   0.656044666976003;
                    pts[70] += 0.656044666976003,  -0.697211314708751;
                    pts[71] +=-0.945197866913093,  -0.858673547396615;
                    pts[72] += 0.803871414309708,  -0.945197866913093;
                    pts[73] +=-0.858673547396615,   0.803871414309708;
                    pts[74] +=-0.945197866913093,   0.803871414309708;
                    pts[75] += 0.803871414309708,  -0.858673547396615;
                    pts[76] +=-0.955761107477727,   0.911522214955453;
                    pts[77] += 0.911522214955453,  -0.955761107477727;
                    break;
                }

            case 15:
                {
                    pts[0] += -0.333333333333333,  -0.333333333333333;
                    pts[1] += -0.345919243961035,  -0.524125896411661;
                    pts[2] += -0.524125896411661,  -0.345919243961035;
                    pts[3] += -0.158607804822441,  -0.682784390355118;
                    pts[4] += -0.397236169198181,  -0.682930809798947;
                    pts[5] += -0.547891729121480,  -0.547891729121479;
                    pts[6] += -0.229298471311900,  -0.805494870290821;
                    pts[7] += -0.450017853162840,  -0.824969972041269;
                    pts[8] += -0.604881786728350,  -0.732090458434108;
                    pts[9] += -0.762668477812969,  -0.762668477812969;
                    pts[10] +=-0.047709572520027,  -0.904580854959946;
                    pts[11] +=-0.295197559068932,  -0.904875474507678;
                    pts[12] +=-0.604222488756960,  -0.880761064604909;
                    pts[13] +=-0.767507099479299,  -0.893012043623240;
                    pts[14] +=-0.893765292435529,  -0.893765292435529;
                    pts[15] +=-0.164799653568441,  -0.968562022370700;
                    pts[16] +=-0.431133449666801,  -0.960622535118110;
                    pts[17] +=-0.648097761351901,  -0.963860302249784;
                    pts[18] +=-0.836672115891999,  -0.965611696960525;
                    pts[19] +=-0.956100831729230,  -0.956100831729230;
                    pts[20] +=-0.524125896411661,  -0.129954859627304;
                    pts[21] +=-0.129954859627304,  -0.345919243961035;
                    pts[22] +=-0.345919243961035,  -0.129954859627304;
                    pts[23] +=-0.129954859627304,  -0.524125896411661;
                    pts[24] +=-0.682784390355118,  -0.158607804822441;
                    pts[25] +=-0.158607804822441,  -0.158607804822441;
                    pts[26] +=-0.682930809798947,  -0.397236169198181;
                    pts[27] += 0.080166978997128,  -0.682930809798947;
                    pts[28] +=-0.397236169198181,   0.080166978997128;
                    pts[29] +=-0.682930809798947,   0.080166978997128;
                    pts[30] += 0.080166978997128,  -0.397236169198181;
                    pts[31] +=-0.547891729121479,   0.095783458242960;
                    pts[32] += 0.095783458242960,  -0.547891729121480;
                    pts[33] +=-0.805494870290821,  -0.229298471311900;
                    pts[34] += 0.034793341602721,  -0.805494870290821;
                    pts[35] +=-0.229298471311900,   0.034793341602721;
                    pts[36] +=-0.805494870290821,   0.034793341602721;
                    pts[37] += 0.034793341602721,  -0.229298471311900;
                    pts[38] +=-0.824969972041269,  -0.450017853162840;
                    pts[39] += 0.274987825204109,  -0.824969972041269;
                    pts[40] +=-0.450017853162840,   0.274987825204109;
                    pts[41] +=-0.824969972041269,   0.274987825204109;
                    pts[42] += 0.274987825204109,  -0.450017853162840;
                    pts[43] +=-0.732090458434108,  -0.604881786728350;
                    pts[44] += 0.336972245162458,  -0.732090458434108;
                    pts[45] +=-0.604881786728350,   0.336972245162458;
                    pts[46] +=-0.732090458434108,   0.336972245162458;
                    pts[47] += 0.336972245162458,  -0.604881786728350;
                    pts[48] +=-0.762668477812969,   0.525336955625937;
                    pts[49] += 0.525336955625937,  -0.762668477812969;
                    pts[50] +=-0.904580854959946,  -0.047709572520027;
                    pts[51] +=-0.047709572520027,  -0.047709572520027;
                    pts[52] +=-0.904875474507678,  -0.295197559068932;
                    pts[53] += 0.200073033576611,  -0.904875474507678;
                    pts[54] +=-0.295197559068932,   0.200073033576611;
                    pts[55] +=-0.904875474507678,   0.200073033576611;
                    pts[56] += 0.200073033576611,  -0.295197559068932;
                    pts[57] +=-0.880761064604909,  -0.604222488756960;
                    pts[58] += 0.484983553361869,  -0.880761064604909;
                    pts[59] +=-0.604222488756960,   0.484983553361869;
                    pts[60] +=-0.880761064604909,   0.484983553361869;
                    pts[61] += 0.484983553361869,  -0.604222488756960;
                    pts[62] +=-0.893012043623240,  -0.767507099479299;
                    pts[63] += 0.660519143102539,  -0.893012043623240;
                    pts[64] +=-0.767507099479299,   0.660519143102539;
                    pts[65] +=-0.893012043623240,   0.660519143102539;
                    pts[66] += 0.660519143102539,  -0.767507099479299;
                    pts[67] +=-0.893765292435529,   0.787530584871058;
                    pts[68] += 0.787530584871058,  -0.893765292435529;
                    pts[69] +=-0.968562022370700,  -0.164799653568441;
                    pts[70] += 0.133361675939142,  -0.968562022370700;
                    pts[71] +=-0.164799653568441,   0.133361675939142;
                    pts[72] +=-0.968562022370700,   0.133361675939142;
                    pts[73] += 0.133361675939142,  -0.164799653568441;
                    pts[74] +=-0.960622535118110,  -0.431133449666801;
                    pts[75] += 0.391755984784911,  -0.960622535118110;
                    pts[76] +=-0.431133449666801,   0.391755984784911;
                    pts[77] +=-0.960622535118110,   0.391755984784911;
                    pts[78] += 0.391755984784911,  -0.431133449666801;
                    pts[79] +=-0.963860302249784,  -0.648097761351901;
                    pts[80] += 0.611958063601684,  -0.963860302249784;
                    pts[81] +=-0.648097761351901,   0.611958063601684;
                    pts[82] +=-0.963860302249784,   0.611958063601684;
                    pts[83] += 0.611958063601684,  -0.648097761351901;
                    pts[84] +=-0.965611696960525,  -0.836672115891999;
                    pts[85] += 0.802283812852524,  -0.965611696960525;
                    pts[86] +=-0.836672115891999,   0.802283812852524;
                    pts[87] +=-0.965611696960525,   0.802283812852524;
                    pts[88] += 0.802283812852524,  -0.836672115891999;
                    pts[89] +=-0.956100831729230,   0.912201663458459;
                    pts[90] += 0.912201663458459,  -0.956100831729230;
                    break;
                }

            case 16:
                {
                    pts[0]   +=-0.205820142532107,  -0.397089928733963;
                    pts[1]   +=-0.226704334870422,  -0.546591330259216;
                    pts[2]   +=-0.147053788408309,  -0.705892423183383;
                    pts[3]   +=-0.347803355325610,  -0.660885382368244;
                    pts[4]   +=-0.506158324630743,  -0.506158324630741;
                    pts[5]   +=-0.221123161795997,  -0.810980203769963;
                    pts[6]   +=-0.533571481527389,  -0.668243726484405;
                    pts[7]   +=-0.636973186435847,  -0.764734627714721;
                    pts[8]   +=-0.781455865395765,  -0.781455865395762;
                    pts[9]   +=-0.049575040474585,  -0.900849919050827;
                    pts[10]  +=-0.301249843385144,  -0.900753138395183;
                    pts[11]  +=-0.444218443770085,  -0.801617084992224;
                    pts[12]  +=-0.672144704736634,  -0.893074100698409;
                    pts[13]  +=-0.808077031149289,  -0.894419861496723;
                    pts[14]  +=-0.908273312282669,  -0.908273312282669;
                    pts[15]  +=-0.130478222056322,  -0.965904946909586;
                    pts[16]  +=-0.360416071313197,  -0.968507402379088;
                    pts[17]  +=-0.504672216906210,  -0.904515816522186;
                    pts[18]  +=-0.589971542834612,  -0.968078774774457;
                    pts[19]  +=-0.780437138124852,  -0.968331431821818;
                    pts[20]  +=-0.925378007505782,  -0.971660798597053;
                    pts[21]  +=-0.397089928733963,  -0.397089928733930;
                    pts[22]  +=-0.397089928733930,  -0.205820142532107;
                    pts[23]  +=-0.546591330259216,  -0.226704334870362;
                    pts[24]  +=-0.226704334870362,  -0.226704334870422;
                    pts[25]  +=-0.705892423183383,  -0.147053788408308;
                    pts[26]  +=-0.147053788408308,  -0.147053788408309;
                    pts[27]  +=-0.660885382368244,  -0.347803355325610;
                    pts[28]  += 0.008688737693854,  -0.660885382368244;
                    pts[29]  +=-0.347803355325610,   0.008688737693854;
                    pts[30]  +=-0.660885382368244,   0.008688737693854;
                    pts[31]  += 0.008688737693854,  -0.347803355325610;
                    pts[32]  +=-0.506158324630741,   0.012316649261484;
                    pts[33]  += 0.012316649261484,  -0.506158324630743;
                    pts[34]  +=-0.810980203769963,  -0.221123161795997;
                    pts[35]  += 0.032103365565961,  -0.810980203769963;
                    pts[36]  +=-0.221123161795997,   0.032103365565961;
                    pts[37]  +=-0.810980203769963,   0.032103365565961;
                    pts[38]  += 0.032103365565961,  -0.221123161795997;
                    pts[39]  +=-0.668243726484405,  -0.533571481527389;
                    pts[40]  += 0.201815208011793,  -0.668243726484405;
                    pts[41]  +=-0.533571481527389,   0.201815208011793;
                    pts[42]  +=-0.668243726484405,   0.201815208011793;
                    pts[43]  += 0.201815208011793,  -0.533571481527389;
                    pts[44]  +=-0.764734627714721,  -0.636973186435847;
                    pts[45]  += 0.401707814150568,  -0.764734627714721;
                    pts[46]  +=-0.636973186435847,   0.401707814150568;
                    pts[47]  +=-0.764734627714721,   0.401707814150568;
                    pts[48]  += 0.401707814150568,  -0.636973186435847;
                    pts[49]  +=-0.781455865395762,   0.562911730791527;
                    pts[50]  += 0.562911730791527,  -0.781455865395765;
                    pts[51]  +=-0.900849919050827,  -0.049575040474587;
                    pts[52]  +=-0.049575040474587,  -0.049575040474585;
                    pts[53]  +=-0.900753138395183,  -0.301249843385144;
                    pts[54]  += 0.202002981780326,  -0.900753138395183;
                    pts[55]  +=-0.301249843385144,   0.202002981780326;
                    pts[56]  +=-0.900753138395183,   0.202002981780326;
                    pts[57]  += 0.202002981780326,  -0.301249843385144;
                    pts[58]  +=-0.801617084992224,  -0.444218443770085;
                    pts[59]  += 0.245835528762309,  -0.801617084992224;
                    pts[60]  +=-0.444218443770085,   0.245835528762309;
                    pts[61]  +=-0.801617084992224,   0.245835528762309;
                    pts[62]  += 0.245835528762309,  -0.444218443770085;
                    pts[63]  +=-0.893074100698409,  -0.672144704736634;
                    pts[64]  += 0.565218805435043,  -0.893074100698409;
                    pts[65]  +=-0.672144704736634,   0.565218805435043;
                    pts[66]  +=-0.893074100698409,   0.565218805435043;
                    pts[67]  += 0.565218805435043,  -0.672144704736634;
                    pts[68]  +=-0.894419861496723,  -0.808077031149289;
                    pts[69]  += 0.702496892646012,  -0.894419861496723;
                    pts[70]  +=-0.808077031149289,   0.702496892646012;
                    pts[71]  +=-0.894419861496723,   0.702496892646012;
                    pts[72]  += 0.702496892646012,  -0.808077031149289;
                    pts[73]  +=-0.908273312282669,   0.816546624565338;
                    pts[74]  += 0.816546624565338,  -0.908273312282669;
                    pts[75]  +=-0.965904946909586,  -0.130478222056322;
                    pts[76]  += 0.096383168965909,  -0.965904946909586;
                    pts[77]  +=-0.130478222056322,   0.096383168965909;
                    pts[78]  +=-0.965904946909586,   0.096383168965909;
                    pts[79]  += 0.096383168965909,  -0.130478222056322;
                    pts[80]  +=-0.968507402379088,  -0.360416071313197;
                    pts[81]  += 0.328923473692285,  -0.968507402379088;
                    pts[82]  +=-0.360416071313197,   0.328923473692285;
                    pts[83]  +=-0.968507402379088,   0.328923473692285;
                    pts[84]  += 0.328923473692285,  -0.360416071313197;
                    pts[85]  +=-0.904515816522186,  -0.504672216906210;
                    pts[86]  += 0.409188033428395,  -0.904515816522186;
                    pts[87]  +=-0.504672216906210,   0.409188033428395;
                    pts[88]  +=-0.904515816522186,   0.409188033428395;
                    pts[89]  += 0.409188033428395,  -0.504672216906210;
                    pts[90]  +=-0.968078774774457,  -0.589971542834612;
                    pts[91]  += 0.558050317609069,  -0.968078774774457;
                    pts[92]  +=-0.589971542834612,   0.558050317609069;
                    pts[93]  +=-0.968078774774457,   0.558050317609069;
                    pts[94]  += 0.558050317609069,  -0.589971542834612;
                    pts[95]  +=-0.968331431821818,  -0.780437138124852;
                    pts[96]  += 0.748768569946670,  -0.968331431821818;
                    pts[97]  +=-0.780437138124852,   0.748768569946670;
                    pts[98]  +=-0.968331431821818,   0.748768569946670;
                    pts[99]  += 0.748768569946670,  -0.780437138124852;
                    pts[100] +=-0.971660798597053,  -0.925378007505782;
                    pts[101] += 0.897038806102834,  -0.971660798597053;
                    pts[102] +=-0.925378007505782,   0.897038806102834;
                    pts[103] +=-0.971660798597053,   0.897038806102834;
                    pts[104] += 0.897038806102834,  -0.925378007505782;
                    break;
                }

            case 17:
                {
                    pts[0]   +=-0.274418069203627,  -0.451163861592915;
                    pts[1]   +=-0.454542715481369,  -0.454542715481372;
                    pts[2]   +=-0.304432262743245,  -0.603457916280128;
                    pts[3]   +=-0.494494713443969,  -0.621442380901352;
                    pts[4]   +=-0.658725522099670,  -0.658725522099675;
                    pts[5]   +=-0.135472335404057,  -0.729055329191558;
                    pts[6]   +=-0.352087942349477,  -0.733502615179034;
                    pts[7]   +=-0.390991263418732,  -0.850467523022011;
                    pts[8]   +=-0.538875562890167,  -0.776709531025512;
                    pts[9]   +=-0.814453782144371,  -0.814453782144367;
                    pts[10]  +=-0.195103212265680,  -0.834191333597611;
                    pts[11]  +=-0.252286838337513,  -0.919078757215168;
                    pts[12]  +=-0.526146769703608,  -0.899421376980804;
                    pts[13]  +=-0.684707100047466,  -0.799915944717937;
                    pts[14]  +=-0.817800237901515,  -0.916958697330690;
                    pts[15]  +=-0.916944777907565,  -0.916944777907568;
                    pts[16]  +=-0.040450078409046,  -0.919099843181106;
                    pts[17]  +=-0.139681499859092,  -0.973704822173568;
                    pts[18]  +=-0.365272864060711,  -0.966341745722431;
                    pts[19]  +=-0.553244326836927,  -0.969305591113758;
                    pts[20]  +=-0.687104081104119,  -0.907603260148529;
                    pts[21]  +=-0.727794870234229,  -0.972100556591935;
                    pts[22]  +=-0.874388356916094,  -0.973207851954675;
                    pts[23]  +=-0.966114906224907,  -0.966114906224907;
                    pts[24]  +=-0.451163861592915,  -0.274418069203458;
                    pts[25]  +=-0.274418069203458,  -0.274418069203627;
                    pts[26]  +=-0.454542715481372,  -0.090914569037259;
                    pts[27]  +=-0.090914569037259,  -0.454542715481369;
                    pts[28]  +=-0.603457916280128,  -0.304432262743245;
                    pts[29]  +=-0.092109820976628,  -0.603457916280128;
                    pts[30]  +=-0.304432262743245,  -0.092109820976628;
                    pts[31]  +=-0.603457916280128,  -0.092109820976628;
                    pts[32]  +=-0.092109820976628,  -0.304432262743245;
                    pts[33]  +=-0.621442380901352,  -0.494494713443969;
                    pts[34]  += 0.115937094345321,  -0.621442380901352;
                    pts[35]  +=-0.494494713443969,   0.115937094345321;
                    pts[36]  +=-0.621442380901352,   0.115937094345321;
                    pts[37]  += 0.115937094345321,  -0.494494713443969;
                    pts[38]  +=-0.658725522099675,   0.317451044199345;
                    pts[39]  += 0.317451044199345,  -0.658725522099670;
                    pts[40]  +=-0.729055329191558,  -0.135472335404385;
                    pts[41]  +=-0.135472335404385,  -0.135472335404057;
                    pts[42]  +=-0.733502615179034,  -0.352087942349477;
                    pts[43]  += 0.085590557528511,  -0.733502615179034;
                    pts[44]  +=-0.352087942349477,   0.085590557528511;
                    pts[45]  +=-0.733502615179034,   0.085590557528511;
                    pts[46]  += 0.085590557528511,  -0.352087942349477;
                    pts[47]  +=-0.850467523022011,  -0.390991263418732;
                    pts[48]  += 0.241458786440742,  -0.850467523022011;
                    pts[49]  +=-0.390991263418732,   0.241458786440742;
                    pts[50]  +=-0.850467523022011,   0.241458786440742;
                    pts[51]  += 0.241458786440742,  -0.390991263418732;
                    pts[52]  +=-0.776709531025512,  -0.538875562890167;
                    pts[53]  += 0.315585093915678,  -0.776709531025512;
                    pts[54]  +=-0.538875562890167,   0.315585093915678;
                    pts[55]  +=-0.776709531025512,   0.315585093915678;
                    pts[56]  += 0.315585093915678,  -0.538875562890167;
                    pts[57]  +=-0.814453782144367,   0.628907564288738;
                    pts[58]  += 0.628907564288738,  -0.814453782144371;
                    pts[59]  +=-0.834191333597611,  -0.195103212265680;
                    pts[60]  += 0.029294545863291,  -0.834191333597611;
                    pts[61]  +=-0.195103212265680,   0.029294545863291;
                    pts[62]  +=-0.834191333597611,   0.029294545863291;
                    pts[63]  += 0.029294545863291,  -0.195103212265680;
                    pts[64]  +=-0.919078757215168,  -0.252286838337513;
                    pts[65]  += 0.171365595552681,  -0.919078757215168;
                    pts[66]  +=-0.252286838337513,   0.171365595552681;
                    pts[67]  +=-0.919078757215168,   0.171365595552681;
                    pts[68]  += 0.171365595552681,  -0.252286838337513;
                    pts[69]  +=-0.899421376980804,  -0.526146769703608;
                    pts[70]  += 0.425568146684412,  -0.899421376980804;
                    pts[71]  +=-0.526146769703608,   0.425568146684412;
                    pts[72]  +=-0.899421376980804,   0.425568146684412;
                    pts[73]  += 0.425568146684412,  -0.526146769703608;
                    pts[74]  +=-0.799915944717937,  -0.684707100047466;
                    pts[75]  += 0.484623044765403,  -0.799915944717937;
                    pts[76]  +=-0.684707100047466,   0.484623044765403;
                    pts[77]  +=-0.799915944717937,   0.484623044765403;
                    pts[78]  += 0.484623044765403,  -0.684707100047466;
                    pts[79]  +=-0.916958697330690,  -0.817800237901515;
                    pts[80]  += 0.734758935232205,  -0.916958697330690;
                    pts[81]  +=-0.817800237901515,   0.734758935232205;
                    pts[82]  +=-0.916958697330690,   0.734758935232205;
                    pts[83]  += 0.734758935232205,  -0.817800237901515;
                    pts[84]  +=-0.916944777907568,   0.833889555815133;
                    pts[85]  += 0.833889555815133,  -0.916944777907565;
                    pts[86]  +=-0.919099843181106,  -0.040450078409849;
                    pts[87]  +=-0.040450078409849,  -0.040450078409046;
                    pts[88]  +=-0.973704822173568,  -0.139681499859092;
                    pts[89]  += 0.113386322032660,  -0.973704822173568;
                    pts[90]  +=-0.139681499859092,   0.113386322032660;
                    pts[91]  +=-0.973704822173568,   0.113386322032660;
                    pts[92]  += 0.113386322032660,  -0.139681499859092;
                    pts[93]  +=-0.966341745722431,  -0.365272864060711;
                    pts[94]  += 0.331614609783141,  -0.966341745722431;
                    pts[95]  +=-0.365272864060711,   0.331614609783141;
                    pts[96]  +=-0.966341745722431,   0.331614609783141;
                    pts[97]  += 0.331614609783141,  -0.365272864060711;
                    pts[98]  +=-0.969305591113758,  -0.553244326836927;
                    pts[99]  += 0.522549917950685,  -0.969305591113758;
                    pts[100] +=-0.553244326836927,   0.522549917950685;
                    pts[101] +=-0.969305591113758,   0.522549917950685;
                    pts[102] += 0.522549917950685,  -0.553244326836927;
                    pts[103] +=-0.907603260148529,  -0.687104081104119;
                    pts[104] += 0.594707341252649,  -0.907603260148529;
                    pts[105] +=-0.687104081104119,   0.594707341252649;
                    pts[106] +=-0.907603260148529,   0.594707341252649;
                    pts[107] += 0.594707341252649,  -0.687104081104119;
                    pts[108] +=-0.972100556591935,  -0.727794870234229;
                    pts[109] += 0.699895426826163,  -0.972100556591935;
                    pts[110] +=-0.727794870234229,   0.699895426826163;
                    pts[111] +=-0.972100556591935,   0.699895426826163;
                    pts[112] += 0.699895426826163,  -0.727794870234229;
                    pts[113] +=-0.973207851954675,  -0.874388356916094;
                    pts[114] += 0.847596208870770,  -0.973207851954675;
                    pts[115] +=-0.874388356916094,   0.847596208870770;
                    pts[116] +=-0.973207851954675,   0.847596208870770;
                    pts[117] += 0.847596208870770,  -0.874388356916094;
                    pts[118] +=-0.966114906224907,   0.932229812449814;
                    pts[119] += 0.932229812449814,  -0.966114906224907;
                    break;
                }

            case 18:
                {
                    pts[0]   +=-0.333333333333333,  -0.333333333333333;
                    pts[1]   +=-0.341403167501697,  -0.496889379374577;
                    pts[2]   +=-0.496889379374577,  -0.341403167501698;
                    pts[3]   +=-0.180193099610473,  -0.639613800779053;
                    pts[4]   +=-0.380906991836065,  -0.635706815999634;
                    pts[5]   +=-0.512270446598166,  -0.512270446598166;
                    pts[6]   +=-0.242142213764944,  -0.750619748986711;
                    pts[7]   +=-0.426216871527756,  -0.764111722863127;
                    pts[8]   +=-0.559026266119071,  -0.672116309193513;
                    pts[9]   +=-0.697487089299080,  -0.697487089299080;
                    pts[10]  +=-0.081068949345966,  -0.837862101308069;
                    pts[11]  +=-0.293493269298788,  -0.851490067387505;
                    pts[12]  +=-0.561603804138562,  -0.812436645793173;
                    pts[13]  +=-0.710745308672525,  -0.821809722624405;
                    pts[14]  +=-0.833448470288251,  -0.833448470288251;
                    pts[15]  +=-0.127891272883473,  -0.918186951448195;
                    pts[16]  +=-0.440803029204592,  -0.902264822053879;
                    pts[17]  +=-0.593157770542283,  -0.907931574540618;
                    pts[18]  +=-0.728191943973380,  -0.915862562521078;
                    pts[19]  +=-0.926186882686440,  -0.926186882686440;
                    pts[20]  +=-0.132621542700647,  -0.976724412008478;
                    pts[21]  +=-0.282882435206159,  -0.940187562507177;
                    pts[22]  +=-0.406379266660819,  -0.973537374255463;
                    pts[23]  +=-0.589944148624057,  -0.972780306197722;
                    pts[24]  +=-0.753570755371218,  -0.975026063234546;
                    pts[25]  +=-0.838829021310763,  -0.926960440578635;
                    pts[26]  +=-0.889023739684485,  -0.976272446974781;
                    pts[27]  +=-0.970085029900325,  -0.970085029900325;
                    pts[28]  +=-0.496889379374577,  -0.161707453123726;
                    pts[29]  +=-0.161707453123726,  -0.341403167501697;
                    pts[30]  +=-0.341403167501698,  -0.161707453123726;
                    pts[31]  +=-0.161707453123726,  -0.496889379374577;
                    pts[32]  +=-0.639613800779053,  -0.180193099610473;
                    pts[33]  +=-0.180193099610473,  -0.180193099610473;
                    pts[34]  +=-0.635706815999634,  -0.380906991836065;
                    pts[35]  += 0.016613807835700,  -0.635706815999634;
                    pts[36]  +=-0.380906991836065,   0.016613807835700;
                    pts[37]  +=-0.635706815999634,   0.016613807835700;
                    pts[38]  += 0.016613807835700,  -0.380906991836065;
                    pts[39]  +=-0.512270446598166,   0.024540893196332;
                    pts[40]  += 0.024540893196332,  -0.512270446598166;
                    pts[41]  +=-0.750619748986711,  -0.242142213764944;
                    pts[42]  +=-0.007238037248346,  -0.750619748986711;
                    pts[43]  +=-0.242142213764944,  -0.007238037248346;
                    pts[44]  +=-0.750619748986711,  -0.007238037248346;
                    pts[45]  +=-0.007238037248346,  -0.242142213764944;
                    pts[46]  +=-0.764111722863127,  -0.426216871527756;
                    pts[47]  += 0.190328594390883,  -0.764111722863127;
                    pts[48]  +=-0.426216871527756,   0.190328594390883;
                    pts[49]  +=-0.764111722863127,   0.190328594390883;
                    pts[50]  += 0.190328594390883,  -0.426216871527756;
                    pts[51]  +=-0.672116309193513,  -0.559026266119071;
                    pts[52]  += 0.231142575312584,  -0.672116309193513;
                    pts[53]  +=-0.559026266119071,   0.231142575312584;
                    pts[54]  +=-0.672116309193513,   0.231142575312584;
                    pts[55]  += 0.231142575312584,  -0.559026266119071;
                    pts[56]  +=-0.697487089299080,   0.394974178598160;
                    pts[57]  += 0.394974178598160,  -0.697487089299080;
                    pts[58]  +=-0.837862101308069,  -0.081068949345966;
                    pts[59]  +=-0.081068949345966,  -0.081068949345966;
                    pts[60]  +=-0.851490067387505,  -0.293493269298788;
                    pts[61]  += 0.144983336686293,  -0.851490067387505;
                    pts[62]  +=-0.293493269298788,   0.144983336686293;
                    pts[63]  +=-0.851490067387505,   0.144983336686293;
                    pts[64]  += 0.144983336686293,  -0.293493269298788;
                    pts[65]  +=-0.812436645793173,  -0.561603804138562;
                    pts[66]  += 0.374040449931736,  -0.812436645793173;
                    pts[67]  +=-0.561603804138562,   0.374040449931736;
                    pts[68]  +=-0.812436645793173,   0.374040449931736;
                    pts[69]  += 0.374040449931736,  -0.561603804138562;
                    pts[70]  +=-0.821809722624405,  -0.710745308672525;
                    pts[71]  += 0.532555031296929,  -0.821809722624405;
                    pts[72]  +=-0.710745308672525,   0.532555031296929;
                    pts[73]  +=-0.821809722624405,   0.532555031296929;
                    pts[74]  += 0.532555031296929,  -0.710745308672525;
                    pts[75]  +=-0.833448470288251,   0.666896940576502;
                    pts[76]  += 0.666896940576502,  -0.833448470288251;
                    pts[77]  +=-0.918186951448195,  -0.127891272883473;
                    pts[78]  += 0.046078224331668,  -0.918186951448195;
                    pts[79]  +=-0.127891272883473,   0.046078224331668;
                    pts[80]  +=-0.918186951448195,   0.046078224331668;
                    pts[81]  += 0.046078224331668,  -0.127891272883473;
                    pts[82]  +=-0.902264822053879,  -0.440803029204592;
                    pts[83]  += 0.343067851258471,  -0.902264822053879;
                    pts[84]  +=-0.440803029204592,   0.343067851258471;
                    pts[85]  +=-0.902264822053879,   0.343067851258471;
                    pts[86]  += 0.343067851258471,  -0.440803029204592;
                    pts[87]  +=-0.907931574540618,  -0.593157770542283;
                    pts[88]  += 0.501089345082902,  -0.907931574540618;
                    pts[89]  +=-0.593157770542283,   0.501089345082902;
                    pts[90]  +=-0.907931574540618,   0.501089345082902;
                    pts[91]  += 0.501089345082902,  -0.593157770542283;
                    pts[92]  +=-0.915862562521078,  -0.728191943973380;
                    pts[93]  += 0.644054506494458,  -0.915862562521078;
                    pts[94]  +=-0.728191943973380,   0.644054506494458;
                    pts[95]  +=-0.915862562521078,   0.644054506494458;
                    pts[96]  += 0.644054506494458,  -0.728191943973380;
                    pts[97]  +=-0.926186882686440,   0.852373765372881;
                    pts[98]  += 0.852373765372881,  -0.926186882686440;
                    pts[99]  +=-0.976724412008478,  -0.132621542700647;
                    pts[100] += 0.109345954709125,  -0.976724412008478;
                    pts[101] +=-0.132621542700647,   0.109345954709125;
                    pts[102] +=-0.976724412008478,   0.109345954709125;
                    pts[103] += 0.109345954709125,  -0.132621542700647;
                    pts[104] +=-0.940187562507177,  -0.282882435206159;
                    pts[105] += 0.223069997713336,  -0.940187562507177;
                    pts[106] +=-0.282882435206159,   0.223069997713336;
                    pts[107] +=-0.940187562507177,   0.223069997713336;
                    pts[108] += 0.223069997713336,  -0.282882435206159;
                    pts[109] +=-0.973537374255463,  -0.406379266660819;
                    pts[110] += 0.379916640916282,  -0.973537374255463;
                    pts[111] +=-0.406379266660819,   0.379916640916282;
                    pts[112] +=-0.973537374255463,   0.379916640916282;
                    pts[113] += 0.379916640916282,  -0.406379266660819;
                    pts[114] +=-0.972780306197722,  -0.589944148624057;
                    pts[115] += 0.562724454821779,  -0.972780306197722;
                    pts[116] +=-0.589944148624057,   0.562724454821779;
                    pts[117] +=-0.972780306197722,   0.562724454821779;
                    pts[118] += 0.562724454821779,  -0.589944148624057;
                    pts[119] +=-0.975026063234546,  -0.753570755371218;
                    pts[120] += 0.728596818605764,  -0.975026063234546;
                    pts[121] +=-0.753570755371218,   0.728596818605764;
                    pts[122] +=-0.975026063234546,   0.728596818605764;
                    pts[123] += 0.728596818605764,  -0.753570755371218;
                    pts[124] +=-0.926960440578635,  -0.838829021310763;
                    pts[125] += 0.765789461889398,  -0.926960440578635;
                    pts[126] +=-0.838829021310763,   0.765789461889398;
                    pts[127] +=-0.926960440578635,   0.765789461889398;
                    pts[128] += 0.765789461889398,  -0.838829021310763;
                    pts[129] +=-0.976272446974781,  -0.889023739684485;
                    pts[130] += 0.865296186659265,  -0.976272446974781;
                    pts[131] +=-0.889023739684485,   0.865296186659265;
                    pts[132] +=-0.976272446974781,   0.865296186659265;
                    pts[133] += 0.865296186659265,  -0.889023739684485;
                    pts[134] +=-0.970085029900325,   0.940170059800650;
                    pts[135] += 0.940170059800650,  -0.970085029900325;
                    break;
                }

            case 19:
                {
                    pts[0]   +=-0.287000355881118,  -0.425999288237764;
                    pts[1]   +=-0.206758293148238,  -0.586483413703523;
                    pts[2]   +=-0.389224594690171,  -0.516424689183957;
                    pts[3]   +=-0.310764541061254,  -0.657966548675822;
                    pts[4]   +=-0.480302382494788,  -0.613218952389985;
                    pts[5]   +=-0.624981326354916,  -0.624981326354917;
                    pts[6]   +=-0.128321805315155,  -0.743356389369689;
                    pts[7]   +=-0.323351460920740,  -0.766969030527911;
                    pts[8]   +=-0.497244231750085,  -0.746567934356032;
                    pts[9]   +=-0.643816871049108,  -0.758951147831367;
                    pts[10]  +=-0.770126681161834,  -0.770126681161834;
                    pts[11]  +=-0.170097600290176,  -0.844942783021920;
                    pts[12]  +=-0.357925049990872,  -0.857114592267436;
                    pts[13]  +=-0.533042015447951,  -0.851309640748108;
                    pts[14]  +=-0.699217252744167,  -0.856334228586824;
                    pts[15]  +=-0.841136978797186,  -0.841136978797186;
                    pts[16]  +=-0.037328752151830,  -0.925342495696341;
                    pts[17]  +=-0.222195753165427,  -0.925717377197022;
                    pts[18]  +=-0.420154075742092,  -0.925542373085800;
                    pts[19]  +=-0.603685694381808,  -0.923025153620202;
                    pts[20]  +=-0.747149943107145,  -0.927211175398458;
                    pts[21]  +=-0.854489586141069,  -0.927618164018200;
                    pts[22]  +=-0.933895989466742,  -0.933895989466742;
                    pts[23]  +=-0.114781279448575,  -0.976416029994295;
                    pts[24]  +=-0.310533830593028,  -0.974760850589475;
                    pts[25]  +=-0.485753632927623,  -0.974389077853428;
                    pts[26]  +=-0.642843887255066,  -0.975894779040898;
                    pts[27]  +=-0.783926073185151,  -0.977387818113720;
                    pts[28]  +=-0.897917737079264,  -0.978019404606716;
                    pts[29]  +=-0.972658489043534,  -0.972658489043534;
                    pts[30]  +=-0.425999288237764,  -0.287000355881118;
                    pts[31]  +=-0.287000355881118,  -0.287000355881118;
                    pts[32]  +=-0.586483413703523,  -0.206758293148238;
                    pts[33]  +=-0.206758293148238,  -0.206758293148238;
                    pts[34]  +=-0.516424689183957,  -0.389224594690171;
                    pts[35]  +=-0.094350716125871,  -0.516424689183957;
                    pts[36]  +=-0.389224594690171,  -0.094350716125871;
                    pts[37]  +=-0.516424689183957,  -0.094350716125871;
                    pts[38]  +=-0.094350716125871,  -0.389224594690171;
                    pts[39]  +=-0.657966548675822,  -0.310764541061254;
                    pts[40]  +=-0.031268910262925,  -0.657966548675822;
                    pts[41]  +=-0.310764541061254,  -0.031268910262925;
                    pts[42]  +=-0.657966548675822,  -0.031268910262925;
                    pts[43]  +=-0.031268910262925,  -0.310764541061254;
                    pts[44]  +=-0.613218952389985,  -0.480302382494788;
                    pts[45]  += 0.093521334884772,  -0.613218952389985;
                    pts[46]  +=-0.480302382494788,   0.093521334884772;
                    pts[47]  +=-0.613218952389985,   0.093521334884772;
                    pts[48]  += 0.093521334884772,  -0.480302382494788;
                    pts[49]  +=-0.624981326354917,   0.249962652709833;
                    pts[50]  += 0.249962652709833,  -0.624981326354916;
                    pts[51]  +=-0.743356389369689,  -0.128321805315155;
                    pts[52]  +=-0.128321805315155,  -0.128321805315155;
                    pts[53]  +=-0.766969030527911,  -0.323351460920740;
                    pts[54]  += 0.090320491448651,  -0.766969030527911;
                    pts[55]  +=-0.323351460920740,   0.090320491448651;
                    pts[56]  +=-0.766969030527911,   0.090320491448651;
                    pts[57]  += 0.090320491448651,  -0.323351460920740;
                    pts[58]  +=-0.746567934356032,  -0.497244231750085;
                    pts[59]  += 0.243812166106117,  -0.746567934356032;
                    pts[60]  +=-0.497244231750085,   0.243812166106117;
                    pts[61]  +=-0.746567934356032,   0.243812166106117;
                    pts[62]  += 0.243812166106117,  -0.497244231750085;
                    pts[63]  +=-0.758951147831367,  -0.643816871049108;
                    pts[64]  += 0.402768018880476,  -0.758951147831367;
                    pts[65]  +=-0.643816871049108,   0.402768018880476;
                    pts[66]  +=-0.758951147831367,   0.402768018880476;
                    pts[67]  += 0.402768018880476,  -0.643816871049108;
                    pts[68]  +=-0.770126681161834,   0.540253362323669;
                    pts[69]  += 0.540253362323669,  -0.770126681161834;
                    pts[70]  +=-0.844942783021920,  -0.170097600290176;
                    pts[71]  += 0.015040383312096,  -0.844942783021920;
                    pts[72]  +=-0.170097600290176,   0.015040383312096;
                    pts[73]  +=-0.844942783021920,   0.015040383312096;
                    pts[74]  += 0.015040383312096,  -0.170097600290176;
                    pts[75]  +=-0.857114592267436,  -0.357925049990872;
                    pts[76]  += 0.215039642258308,  -0.857114592267436;
                    pts[77]  +=-0.357925049990872,   0.215039642258308;
                    pts[78]  +=-0.857114592267436,   0.215039642258308;
                    pts[79]  += 0.215039642258308,  -0.357925049990872;
                    pts[80]  +=-0.851309640748108,  -0.533042015447951;
                    pts[81]  += 0.384351656196059,  -0.851309640748108;
                    pts[82]  +=-0.533042015447951,   0.384351656196059;
                    pts[83]  +=-0.851309640748108,   0.384351656196059;
                    pts[84]  += 0.384351656196059,  -0.533042015447951;
                    pts[85]  +=-0.856334228586824,  -0.699217252744167;
                    pts[86]  += 0.555551481330991,  -0.856334228586824;
                    pts[87]  +=-0.699217252744167,   0.555551481330991;
                    pts[88]  +=-0.856334228586824,   0.555551481330991;
                    pts[89]  += 0.555551481330991,  -0.699217252744167;
                    pts[90]  +=-0.841136978797186,   0.682273957594371;
                    pts[91]  += 0.682273957594371,  -0.841136978797186;
                    pts[92]  +=-0.925342495696341,  -0.037328752151830;
                    pts[93]  +=-0.037328752151830,  -0.037328752151830;
                    pts[94]  +=-0.925717377197022,  -0.222195753165427;
                    pts[95]  += 0.147913130362450,  -0.925717377197022;
                    pts[96]  +=-0.222195753165427,   0.147913130362450;
                    pts[97]  +=-0.925717377197022,   0.147913130362450;
                    pts[98]  += 0.147913130362450,  -0.222195753165427;
                    pts[99]  +=-0.925542373085800,  -0.420154075742092;
                    pts[100] += 0.345696448827892,  -0.925542373085800;
                    pts[101] +=-0.420154075742092,   0.345696448827892;
                    pts[102] +=-0.925542373085800,   0.345696448827892;
                    pts[103] += 0.345696448827892,  -0.420154075742092;
                    pts[104] +=-0.923025153620202,  -0.603685694381808;
                    pts[105] += 0.526710848002010,  -0.923025153620202;
                    pts[106] +=-0.603685694381808,   0.526710848002010;
                    pts[107] +=-0.923025153620202,   0.526710848002010;
                    pts[108] += 0.526710848002010,  -0.603685694381808;
                    pts[109] +=-0.927211175398458,  -0.747149943107145;
                    pts[110] += 0.674361118505603,  -0.927211175398458;
                    pts[111] +=-0.747149943107145,   0.674361118505603;
                    pts[112] +=-0.927211175398458,   0.674361118505603;
                    pts[113] += 0.674361118505603,  -0.747149943107145;
                    pts[114] +=-0.927618164018200,  -0.854489586141069;
                    pts[115] += 0.782107750159270,  -0.927618164018200;
                    pts[116] +=-0.854489586141069,   0.782107750159270;
                    pts[117] +=-0.927618164018200,   0.782107750159270;
                    pts[118] += 0.782107750159270,  -0.854489586141069;
                    pts[119] +=-0.933895989466742,   0.867791978933485;
                    pts[120] += 0.867791978933485,  -0.933895989466742;
                    pts[121] +=-0.976416029994295,  -0.114781279448575;
                    pts[122] += 0.091197309442870,  -0.976416029994295;
                    pts[123] +=-0.114781279448575,   0.091197309442870;
                    pts[124] +=-0.976416029994295,   0.091197309442870;
                    pts[125] += 0.091197309442870,  -0.114781279448575;
                    pts[126] +=-0.974760850589475,  -0.310533830593028;
                    pts[127] += 0.285294681182503,  -0.974760850589475;
                    pts[128] +=-0.310533830593028,   0.285294681182503;
                    pts[129] +=-0.974760850589475,   0.285294681182503;
                    pts[130] += 0.285294681182503,  -0.310533830593028;
                    pts[131] +=-0.974389077853428,  -0.485753632927623;
                    pts[132] += 0.460142710781051,  -0.974389077853428;
                    pts[133] +=-0.485753632927623,   0.460142710781051;
                    pts[134] +=-0.974389077853428,   0.460142710781051;
                    pts[135] += 0.460142710781051,  -0.485753632927623;
                    pts[136] +=-0.975894779040898,  -0.642843887255066;
                    pts[137] += 0.618738666295964,  -0.975894779040898;
                    pts[138] +=-0.642843887255066,   0.618738666295964;
                    pts[139] +=-0.975894779040898,   0.618738666295964;
                    pts[140] += 0.618738666295964,  -0.642843887255066;
                    pts[141] +=-0.977387818113720,  -0.783926073185151;
                    pts[142] += 0.761313891298871,  -0.977387818113720;
                    pts[143] +=-0.783926073185151,   0.761313891298871;
                    pts[144] +=-0.977387818113720,   0.761313891298871;
                    pts[145] += 0.761313891298871,  -0.783926073185151;
                    pts[146] +=-0.978019404606716,  -0.897917737079264;
                    pts[147] += 0.875937141685979,  -0.978019404606716;
                    pts[148] +=-0.897917737079264,   0.875937141685979;
                    pts[149] +=-0.978019404606716,   0.875937141685979;
                    pts[150] += 0.875937141685979,  -0.897917737079264;
                    pts[151] +=-0.972658489043534,   0.945316978087067;
                    pts[152] += 0.945316978087067,  -0.972658489043534;
                    break;
                }

            default:
                {
                }
            }

        for ( uint32_type i = 0; i<nbPtsPerFace; ++i)
            {
                inner_pts(0,i) = pts[i][0];
                inner_pts(1,i) = pts[i][1];
            }


        return inner_pts;
    }

    points_type putInPointset (points_type final_pts, points_type pts, std::pair<uint16_type, uint16_type> position )
    {
        ublas::subrange(final_pts, 0, 2, position.first, position.second) = pts;

        return final_pts;
    }

    template<int n>
    static bool order(vector_type a, vector_type b) {return a(n) < b(n);}

    points_type orderInteriorFace( points_type pts )
    {
        std::vector<vector_type> aux(pts.size2());

        for (uint16_type i=0; i<pts.size2(); i++)
            aux[i] = ublas::column(pts, i);

        std::sort(aux.begin(), aux.end(), &order<1>);

        for (uint16_type p=0, i=Order-3; i>0; i--)
            {
                std::sort(aux.begin()+p, aux.begin()+p+i+1, &order<0>);
                p += i+1;
            }

        for (uint16_type i=0; i<pts.size2(); i++)
            ublas::column(pts, i) = aux[i];

        return pts;
    }
};

/**
 * \class Fekete point set class for simplices and simplex products
 */
template< class Convex,
          uint16_type Order,
          typename T = double >
class PointSetFekete : public mpl::if_<mpl::or_<mpl::equal_to<mpl::bool_<Convex::is_simplex_product>, mpl::bool_<true> >,
                                                mpl::equal_to<mpl::int_<Convex::nDim>, mpl::int_<1> > >,
                                       mpl::identity<PointSetGaussLobatto<Convex,Order,T> >,
                                       typename mpl::if_<mpl::equal_to<mpl::int_<Convex::nDim>, mpl::int_<3> >,
                                                         mpl::identity<PointSetWarpBlend<Convex,Order,T> >,
                                                         mpl::identity<PointSetFeketeSimplex<Convex,Order,T> > >::type >::type::type
{
    typedef typename mpl::if_<mpl::or_<mpl::equal_to<mpl::bool_<Convex::is_simplex_product>, mpl::bool_<true> >,
                                       mpl::equal_to<mpl::int_<Convex::nDim>, mpl::int_<1> > >,
                              mpl::identity<PointSetGaussLobatto<Convex,Order,T> >,
                              typename mpl::if_<mpl::equal_to<mpl::int_<Convex::nDim>, mpl::int_<3> >,
                                                mpl::identity<PointSetWarpBlend<Convex,Order,T> >,
                                                mpl::identity<PointSetFeketeSimplex<Convex,Order,T> > >::type>::type::type super;
public:
    PointSetFekete( int interior = 0 ) : super( interior ) {}
};

} // Life
#endif /* __Fekete_H */
