/*
 * $Id: ntfsd.h,v 1.10 2001/12/15 05:13:08 antona Exp $
 *
 * ntfsd.h - Exports for writer daemon. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef NTFSD_H
#define NTFSD_H

#include "list.h"
#include "support.h"

/*
 * These keep track of all mounted volumes. Ntfsd needs to know so it can
 * walk the dirty list for each mounted volume.
 */
extern struct list_head ntfs_mounted_volumes;
extern unsigned long ntfs_nr_mounted_volumes;

/**
 * ntfs_flush - commit all dirty buffers to disk
 *
 * ntfs_flush() commits all dirty buffers to disk when called. It should be
 * called as often as possible, as it only does anything when five seconds have
 * elapsed since the last flush to disk and it isn't executing already.
 */
extern __inline__ void ntfs_flush(void);

/**
 * ntfs_timer_start - start the 5 second timer setting ntfs_need_sync to true
 *
 * Start the interval timer and setup a handler for the SIGALRM signal. This
 * function can and should be called whenever volume is mounted, so that we
 * know how many times you need to call ntfs_timer_stop() before we actually
 * stop the timer.
 *
 * Return TRUE on success or FALSE on error. In the latter case, errno contains
 * the error code.
 *
 * Important error codes:
 * 	ECANCELED - The signal handler was set but the timer setting failed.
 * 		    Tried resetting the signal handler but this failed. So,
 * 		    ntfsd will handle SIGALRM should it occur, but it won't do
 * 		    anything as it is not started.
 *	EINTR - One of the system calls was interrupted more than three times
 *		in a row. You could try again.
 *	ENOTSUP - The host system does not support SIGALRM signal. )-:
 *	EFAULT - Internal error. This is BAD.
 *	EMFILE - Times has been started too many times already. Cannot start
 *		 again before at least one ntfs_timer_stop() call is issued.
 *	others - Other error codes returned by one of the system calls.
 */
BOOL ntfs_timer_start(void);

/**
 * ntfs_timer_start - stop the 5 second timer
 *
 * Stop the interval timer and remove the handler for the SIGALRM signal.
 * This function should be called once for every umount performed. The daemon
 * will only be stopped if ntfs_timer_stop() has been called the same number of
 * times as ntfs_timer_start(), i.e. for every call to ntfs_timer_start(), you
 * need to issue a corresponding ntfs_timer_stop().
 *
 * Return TRUE on success or FALSE on error. In the latter case, errno contains
 * the error code.
 *
 * On error, the handler remains active, unless errno is ECANCELED, in which
 * case the timer has been stopped, so no periodic SIGALRM signals will be
 * generated), but the SIGALRM handler is still registered (since
 * uninstallation failed), so the code has to remain loaded in memory. In
 * theory this will never occur but better catch the impossible before practise
 * makes the the theoretically impossible possible.
 *
 * Important error codes:
 * 	ECANCELED - The timer was stopped but the signal handler could not be
 * 		    reset. Tried restarting the timer but this failed. So,
 * 		    we will handle SIGALRM should it occur, but we won't do
 * 		    anything as we are not started any more.
 *	EINTR - One of the system calls was interrupted more than three times
 *		in a row. You could try again.
 *	ENOTSUP - The host system does not support SIGALRM signal. )-:
 *	EFAULT - Internal error. This is BAD.
 *	others - Other error codes returned by one of the system calls.
 */
BOOL ntfs_timer_stop(void);

#endif /* defined NTFSD_H */

