/*
 * $Id: ntfs_rec.c,v 1.9 2001/06/01 02:07:26 antona Exp $
 *
 * ntfs_rec.c - NTFS record handling code. Part of the Linux-NTFS project.
 *
 * Copyright (c) 2000,2001 Anton Altaparmakov.
 *
 * This program/include file is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program/include file is distributed in the hope that it will be 
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty 
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (in the main directory of the Linux-NTFS 
 * distribution in the file COPYING); if not, write to the Free Software
 * Foundation,Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>

#include "types.h"
#include "volume.h"
#include "ntfs_rec.h"
#include "endians.h"

__inline__ void __post_read_mst_fixup(NTFS_RECORD *b, const __u32 size)
{
	__u16 usa_ofs, usa_count;
	__u16 *usa_pos, *data_pos;

	/* Setup the variables. */
	usa_ofs = le16_to_cpu(b->usa_ofs);
	usa_count = le16_to_cpu(b->usa_count) - 1;
	/* Position of usn in update sequence array. */  
	usa_pos = (__u16*)b + usa_ofs/sizeof(__u16);
	/*
	 * Position in protected data of first __u16 that needs fixing up.
	 */
	data_pos = (__u16*)b + NTFS_SECTOR_SIZE/sizeof(__u16) - 1;
        /* Fixup all sectors. */
	while (usa_count--) {
		/*
		 * Increment position in usa and restore original data from
		 * the usa into the data buffer.
		 */
		*data_pos = *(++usa_pos);
                /* Increment position in data as well. */
		data_pos += NTFS_SECTOR_SIZE/sizeof(__u16);
        }
}

int post_read_mst_fixup(NTFS_RECORD *b, const __u32 size)
{
	__u16 usa_ofs, usa_count, usn;
	__u16 *usa_pos, *data_pos;

	/* Setup the variables. */
	usa_ofs = le16_to_cpu(b->usa_ofs);
	/* Decrement usa_count to get number of fixups. */
	usa_count = le16_to_cpu(b->usa_count) - 1;
	/* Size and alignement checks. */
	if ( size & (NTFS_SECTOR_SIZE - 1)	||
	     usa_ofs & 1			||
	     usa_ofs + (usa_count * 2) > size	||
	     (size >> NTFS_SECTOR_SIZE_BITS) != usa_count)
		return 1;
	/* Position of usn in update sequence array. */  
	usa_pos = (__u16*)b + usa_ofs/sizeof(__u16);
	/* 
	 * The update sequence number which has to be equal to each of the
	 * __u16 values before they are fixed up. Note no need to care for
	 * endianness since we are comparing and moving data for on disk
	 * structures which means the data is consistent. - If it is 
	 * consistenty the wrong endianness it doesn't make any difference.
	 */
	usn = *usa_pos;
	/*
	 * Position in protected data of first __u16 that needs fixing up.
	 */
	data_pos = (__u16*)b + NTFS_SECTOR_SIZE/sizeof(__u16) - 1;
        /*
	 * Check for incomplete multi sector transfer(s).
	 */
	while (usa_count--) {
                if (*data_pos != usn) {
			/*
			 * Incomplete multi sector transfer detected! )-:
			 * Set the magic to "BAAD" and return failure.
			 * Note that magic_BAAD is already converted to le32.
			 */
			b->magic = magic_BAAD;
	                return 0;
		}
		data_pos += NTFS_SECTOR_SIZE/sizeof(__u16);
	}
	/* Re-setup the variables. */
	usa_count = le16_to_cpu(b->usa_count) - 1;
	data_pos = (__u16*)b + NTFS_SECTOR_SIZE/sizeof(__u16) - 1;
	/* Fixup all sectors. */
	while (usa_count--) {
		/*
		 * Increment position in usa and restore original data from
		 * the usa into the data buffer.
		 */
		*data_pos = *(++usa_pos);
                /* Increment position in data as well. */
		data_pos += NTFS_SECTOR_SIZE/sizeof(__u16);
        }
	return 1;
}

int pre_write_mst_fixup(NTFS_RECORD *b, const __u32 size)
{
	__u16 usa_ofs, usa_count, usn;
	__u16 *usa_pos, *data_pos;

	/* Sanity check + only fixup if it makes sense. */
	if (!b || is_baad_record(b->magic) || is_hole_record(b->magic))
		return 0;
	/* Setup the variables. */
	usa_ofs = le16_to_cpu(b->usa_ofs);
	/* Decrement usa_count to get number of fixups. */
	usa_count = le16_to_cpu(b->usa_count) - 1;
	/* Size and alignement checks. */
	if ( size & (NTFS_SECTOR_SIZE - 1)	||
	     usa_ofs & 1			||
	     usa_ofs + (usa_count * 2) > size	||
	     (size >> NTFS_SECTOR_SIZE_BITS) != usa_count)
		return 0;
	/* Position of usn in update sequence array. */  
	usa_pos = (__u16*)((__u8*)b + usa_ofs);
	/*
	 * Cyclically increment the update sequence number 
	 * (skipping 0 and -1, i.e. 0xffff).
	 */
	usn = le16_to_cpup(usa_pos) + 1;
	if (usn == 0xffff || !usn)
		usn = 1;
	usn = cpu_to_le16(usn);
	*usa_pos = usn;
	/* Position in data of first __u16 that needs fixing up. */
	data_pos = (__u16*)b + NTFS_SECTOR_SIZE/sizeof(__u16) - 1;
        /* Fixup all sectors. */
        while (usa_count--) {
		/*
		 * Increment the position in the usa and save the 
		 * original data from the data buffer into the usa.
		 */
		*(++usa_pos) = *data_pos;
		/* Apply fixup to data. */
		*data_pos = usn;
		/* Increment position in data as well. */
		data_pos += NTFS_SECTOR_SIZE/sizeof(__u16);
        }
	return 1;
}

