/* Copyright (C) 2000-2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/***************************************************************************
 * CATALOG.C								   *
 *									   *
 * @description: Handling ODBC catalog APIs				   *
 *									   *
 * @author     : MySQL AB (monty@mysql.com, venu@mysql.com)		   *
 * @date       : 2001-Aug-15						   *
 * @product    : myodbc3						   *
 *									   *
 ****************************************************************************/

/***************************************************************************
 * The following ODBC APIs are implemented in this file:		   *
 *									   *
 *   SQLColumnPrivileges (ODBC)						   *
 *   SQLColumns		 (X/Open)					   *
 *   SQLForeignKeys	 (ODBC)						   *
 *   SQLPrimaryKeys	 (ODBC)						   *
 *   SQLProcedureColumns (ODBC)						   *
 *   SQLProcedures	 (ODBC)						   *
 *   SQLSpecialColumns	 (X/Open)					   *
 *   SQLStatistics	 (ISO 92)					   *
 *   SQLTablePrivileges  (ODBC)						   *
 *   SQLTables		 (ODBC)						   *
 *									   *
 ****************************************************************************/

#include "myodbc3.h"

#define valid_input_parameter(A) ((A) && A[0])
#define escape_input_parameter(A,B) if (B && B[0]) myodbc_remove_escape(A,B)

/*
  @type    : internal
  @purpose : returns the next token
*/

static const char *my_next_token(const char *prev_token, 
                                 char **token, 
                                 char *data, 
                                 const char chr)
{
  const char *cur_token;

  if (cur_token= strchr(*token, chr))
  {
    if (prev_token)
    {
      uint len= (uint)(cur_token-prev_token);
      strncpy(data,prev_token, len);
      data[len]= 0;    
    } 
    *token= (char *)cur_token+1;
    prev_token= cur_token;
    return cur_token+1;
  } 
  return 0;
}

/*
  @type    : internal
  @purpose : gets valid input buffer
*/

static char *myodbc_get_valid_buffer(char *to, char *from, int length)
{
  if (!from)
    return "\0";
  if (length == SQL_NTS)
    length= strlen(from);
  strmake(to,from,length);
  return to;
}

/*
  @type    : internal
  @purpose : appends wild card to the query
*/

static void my_append_wild(char *to, 
                           char *end, 
                           const char *wild)
{
  end-= 5;         /* Some extra */
  to= strmov(to," like '");
  
  if (wild)
  {
    while (*wild && to < end)
    {
      if (*wild == '\\' || *wild == '\'')
        *to++= '\\';
      *to++= *wild++;
    }
  }
  *to++= '%';        /* Nicer this way */
  to[0]= '\'';
  to[1]= 0;
}

/*
  @type    : internal
  @purpose : returns tables from a perticular database
*/
static MYSQL_RES *mysql_list_dbtables(DBC FAR    *dbc, 
                                      const char *db,
                                      const char *table) 
{
  MYSQL FAR *mysql= &dbc->mysql;
  char      buff[255];  

  strxmov(buff,"SHOW TABLES FROM `",db,"`",NullS);
  my_append_wild(strmov(buff,buff),buff+sizeof(buff)-1,table);

  MYLOG_DBC_QUERY(dbc, buff);
  if (mysql_query(mysql,buff))
    return 0;
  return mysql_store_result(mysql);  
}


/*
  @type    : internal
  @purpose : returns current qualifier name
*/
my_bool is_default_db(char *def_db, char *user_db)
{
  /* Fix this to return a valid qualifier for all APIs */
  if (valid_input_parameter(user_db) && 
      !strchr(user_db,'%') && 
      cmp_database(def_db, user_db))
    return FALSE;
  return TRUE;
}

/*
  @type    : internal
  @purpose : returns true if supplied table name associated with database name
*/
static my_bool table_has_database_name(const char *db, const char *table_name)
{
  if (valid_input_parameter(db) || strchr(table_name,'.'))
    return TRUE;
  return FALSE;
}

/*
  @type    : internal
  @purpose : validate for give table type from the list
*/
static my_bool check_table_type(const char *TableType, 
                                const char *req_type, 
                                uint       len)
{
  char    req_type_quoted[NAME_LEN+2], req_type_quoted1[NAME_LEN+2];
  char    *type, *table_type= (char *)TableType;
  my_bool found= 0;

  if (!TableType || !TableType[0])
    return found;

  /* 
    Check and return only 'user' tables from current DB and 
    don't return any tables when its called with 
    SYSTEM TABLES.

    Expected Types:
      "TABLE", "VIEW", "SYSTEM TABLE", "GLOBAL TEMPORARY", 
      "LOCAL TEMPORARY", "ALIAS", "SYNONYM",  
  */

  type= strstr(table_type,",");
  sprintf(req_type_quoted,"'%s'",req_type);
  sprintf(req_type_quoted1,"`%s`",req_type);
  while (type)
  {
    while (isspace(*(table_type))) table_type++;
    if (!myodbc_casecmp(table_type,req_type,len) || 
        !myodbc_casecmp(table_type,req_type_quoted,len+2) || 
        !myodbc_casecmp(table_type,req_type_quoted1,len+2))
    {
      found= 1;
      break;
    }
    table_type= ++type;
    type= strstr(table_type,",");
  }
  if (!found)
  {
    while (isspace(*(table_type))) table_type++;
    if (!myodbc_casecmp(table_type,req_type,len) || 
        !myodbc_casecmp(table_type,req_type_quoted,len+2) ||
        !myodbc_casecmp(table_type,req_type_quoted1,len+2))   
      found= 1;
  }
  return found;
}

static MYSQL_ROW fix_fields_copy(STMT FAR *stmt,MYSQL_ROW row)
{
  uint i;
  for (i=0 ; i < stmt->order_count; i++)
    stmt->array[stmt->order[i]]= row[i];
  return stmt->array;
}

/*
****************************************************************************
SQLTables
****************************************************************************
*/

uint SQLTABLES_order[]= {2};
uint SQLTABLES_qualifier_order[]= {0};
char *SQLTABLES_sysvalues[]= {"mysql","",NULL,"SYSTEM TABLE","MySQL System Table"};
char *SQLTABLES_values[]= {"","",NULL,"TABLE","MySQL table"};
char *SQLTABLES_qualifier_values[]= {"",NULL,NULL,NULL,NULL};
char *SQLTABLES_owner_values[]= {NULL,"",NULL,NULL,NULL};
char *SQLTABLES_type_values[2][5]= 
{
  {NULL,NULL,NULL,"TABLE",NULL},
  {NULL,NULL,NULL,"SYSTEM TABLE",NULL}, /* TODO: add temp tables here */
};

MYSQL_FIELD SQLTABLES_fields[]=
{
  {"TABLE_CAT","Catalog", NullS, NullS, NullS, NAME_LEN, 0, 0, 0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_SCHEM","Catalog",NullS, NullS, NullS, NAME_LEN, 0, 0, 0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_NAME","Catalog", NullS, NullS, NullS, NAME_LEN, NAME_LEN, 0, 0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_TYPE","Catalog",NullS,NullS, NullS, NAME_LEN, 5, 0, 0,
   FIELD_TYPE_VAR_STRING},
  {"REMARKS","Catalog",NullS,NullS, NullS, NAME_LEN, 11, 0, 0,
   FIELD_TYPE_VAR_STRING}};

const uint SQLTABLES_FIELDS= array_elements(SQLTABLES_values);

SQLRETURN SQL_API SQLTables(SQLHSTMT    hstmt,
                            SQLCHAR FAR *szTableQualifier,
                            SQLSMALLINT cbTableQualifier,
                            SQLCHAR FAR *szTableOwner,
                            SQLSMALLINT cbTableOwner,
                            SQLCHAR FAR *szTableName,
                            SQLSMALLINT cbTableName,
                            SQLCHAR FAR *szTableType,
                            SQLSMALLINT cbTableType)
{
  char      Qualifier_buff[NAME_LEN+1], 
            Owner_buff[NAME_LEN+1],
            Name_buff[NAME_LEN+1], 
            Type_buff[NAME_LEN+1], 
            *TableQualifier,
            *TableOwner, 
            *TableName, 
            *TableType;
  STMT FAR  *stmt= (STMT FAR*) hstmt;  
  MYSQL_RES *result, *sys_result;
  my_bool   all_dbs= 1, sys_tables, user_tables;

  DBUG_ENTER("SQLTables");

  DBUG_PRINT("enter",("Qualifier: '%s'(%d)  Owner: '%s'(%d)  Table: '%s'(%d)  Type: '%s'(%d)",
          szTableQualifier ? (char*) szTableQualifier : "null", cbTableQualifier,
          szTableOwner ? (char*) szTableOwner : "null", cbTableOwner,
          szTableName ? (char*) szTableName : "null", cbTableName,
          szTableType ? (char*) szTableType : "null", cbTableType));

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  TableQualifier= myodbc_get_valid_buffer((char FAR *) Qualifier_buff,szTableQualifier,
                          cbTableQualifier);
  TableOwner=     myodbc_get_valid_buffer((char FAR*) Owner_buff,szTableOwner,cbTableOwner);
  TableName=      myodbc_get_valid_buffer((char FAR*) Name_buff,szTableName,cbTableName);

  escape_input_parameter(&stmt->dbc->mysql, TableQualifier);
  escape_input_parameter(&stmt->dbc->mysql, TableOwner);
  escape_input_parameter(&stmt->dbc->mysql, TableName);

  if ((!strcmp(TableQualifier,"%") || 
      !(all_dbs= myodbc_casecmp(TableQualifier,"SQL_ALL_CATALOGS",16))) &&
      !TableOwner[0] && !TableName[0])
  {
    /* Return set of allowed qualifiers */
    DBUG_PRINT("info ",("Return set of table qualifiers / Catalog names"));
    
    if (!all_dbs)
      TableQualifier= "%";
    
    pthread_mutex_lock(&stmt->dbc->lock);
    stmt->result= mysql_list_dbs(&stmt->dbc->mysql,TableQualifier);
    pthread_mutex_unlock(&stmt->dbc->lock);

    if (!stmt->result)
    {
      DBUG_PRINT("error",("%d:%s", mysql_errno(&stmt->dbc->mysql),
                         mysql_error(&stmt->dbc->mysql)));
      goto empty_set;
    }
              
    stmt->order         = SQLTABLES_qualifier_order;
    stmt->order_count   = array_elements(SQLTABLES_qualifier_order);
    stmt->fix_fields    = fix_fields_copy;
    stmt->array= (MYSQL_ROW) my_memdup((gptr) SQLTABLES_qualifier_values,
                                       sizeof(SQLTABLES_qualifier_values),
                                       MYF(0));
    mysql_link_fields(stmt,SQLTABLES_fields,SQLTABLES_FIELDS);
    DBUG_RETURN_STATUS(SQL_SUCCESS);
  }

  if (!TableQualifier[0] && (!strcmp(TableOwner,"%") ||
      !myodbc_casecmp(TableOwner,"SQL_ALL_SCHEMAS",15)) && 
      !TableName[0])
  {
    /* Return set of allowed Table owners */
    DBUG_PRINT("info ",("Return set of table owners / Schema names"));
    stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
    stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLTABLES_owner_values,
                                              sizeof(SQLTABLES_owner_values),
                                              MYF(0));
    stmt->result->row_count= 1;
    mysql_link_fields(stmt,SQLTABLES_fields, SQLTABLES_FIELDS);
    DBUG_RETURN_STATUS(SQL_SUCCESS);
  }

  TableType=   myodbc_get_valid_buffer((char FAR*) Type_buff,szTableType,cbTableType);

  if (!TableQualifier[0] && !TableOwner[0] && !TableName[0] &&
      (!strcmp(TableType,"%") ||
       !myodbc_casecmp(TableType,"SQL_ALL_TABLE_TYPES",19)))   
  {
    /* Return set of TableType qualifiers */
    DBUG_PRINT("info ",("Return set of table types"));
    stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
    stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLTABLES_type_values,
                                              sizeof(SQLTABLES_type_values),
                                              MYF(0));
    /* TODO : Fix this to return temporary table types  */
    stmt->result->row_count= sizeof(SQLTABLES_type_values)/
                             sizeof(SQLTABLES_type_values[0]);
    mysql_link_fields(stmt,SQLTABLES_fields,SQLTABLES_FIELDS);
    DBUG_RETURN_STATUS(SQL_SUCCESS);
  }
  sys_tables= 0;
  sys_result= result= 0;

  escape_input_parameter(&stmt->dbc->mysql, TableType);
  user_tables= check_table_type(TableType,"TABLE",5);

  if (check_table_type(TableType,"SYSTEM TABLE",12) || 
      check_table_type(TableType,"SYSTEM",6))
    sys_tables= 1;

  if (!user_tables && !sys_tables)
  {
    /* Return current db tables if no table type specified (hack for MS VC) */
    if (!szTableType || !cbTableType)
      user_tables= 1;
  }
  
  if ((TableType[0] && !user_tables && !sys_tables) ||
      TableQualifier[0] && strcmp(TableQualifier,"%") &&
      TableOwner[0] && strcmp(TableOwner,"%") &&
      strcmp(TableOwner,stmt->dbc->database))
  {
    /* 
      Return empty set if unknown TableType or if 
      Owner is used 
    */
    goto empty_set;
  }

  /* User Tables with type as 'TABLE' */
  if (user_tables)
  {
    if (szTableQualifier && cmp_database(stmt->dbc->mysql.db, TableQualifier))
    {
      /* Return tables from a perticular database */
      DBUG_PRINT("info ",("Return set of tables '%s' from '%s'",TableName, TableQualifier));
      pthread_mutex_lock(&stmt->dbc->lock);                         
      result= mysql_list_dbtables(stmt->dbc,TableQualifier,TableName);
      pthread_mutex_unlock(&stmt->dbc->lock);
    }
    else
    {
      /* Return tables from default/current database */    
      DBUG_PRINT("info ",("Returning set of current database tables '%s'",TableName));
      pthread_mutex_lock(&stmt->dbc->lock);
      result= mysql_list_tables(&stmt->dbc->mysql,TableName);
      pthread_mutex_unlock(&stmt->dbc->lock);
    }
    if (!result)
    {
      DBUG_PRINT("error",("%d:%s", 
                  mysql_errno(&stmt->dbc->mysql),
                  mysql_error(&stmt->dbc->mysql)));
    }
  }
  /* System tables with type as 'SYSTEM' or 'SYSTEM TABLE' */
  if (sys_tables)
  { 
    DBUG_PRINT("info ",("Return set of system tables"));
    pthread_mutex_lock(&stmt->dbc->lock);                         
    stmt->result= mysql_list_dbtables(stmt->dbc,"mysql",TableName);
    pthread_mutex_unlock(&stmt->dbc->lock);
    
    if (!(sys_result= stmt->result))
    {
      DBUG_PRINT("error",("%d:%s", 
                  mysql_errno(&stmt->dbc->mysql),
                  mysql_error(&stmt->dbc->mysql)));
    }
  }
  {
    MYSQL_ROW    data, row;
    MEM_ROOT     *alloc;
    char         *db, *owner= "";
    my_ulonglong row_count;

    /*
      Append tables together if system and user tables
      are used
    */
    if (!sys_result)
      row_count= 0;
    else
    {
      row_count= sys_result->row_count;
      alloc= &sys_result->field_alloc;
    }
    if (result)
    {
      row_count+= result->row_count;
      if (!sys_result)
        alloc= &result->field_alloc;
    }
    if (!row_count)
      goto empty_set;

    if (sys_result)
    {
      char buff[NAME_LEN+7];

      if (option_flag(stmt, FLAG_NO_CATALOG))
        db= owner= "";
      else
        db= "mysql";      

      if (!(stmt->result_array= (char**) my_malloc(
         (uint)(sizeof(char *)*SQLTABLES_FIELDS*row_count), 
         MYF(MY_FAE | MY_ZEROFILL))))      
        DBUG_RETURN_STATUS(set_error(stmt,MYERR_S1001,NULL,4001));
      
      data= stmt->result_array;

      /* 
        Prefix all system tables with 'mysql.', so that they can 
        be used directly in describing columns related information

        Must needed like this inorder to make system tables editable 
        by ODBC tools
      */
      while ((row = mysql_fetch_row(sys_result)))
      {    
        data[0]= db;
        data[1]= owner;      
        sprintf(buff,"mysql.%s",row[0]);
        data[2]= strdup_root(alloc,buff);
        data[3]= "SYSTEM TABLE";      
        data[4]= "MySQL System Table";
        data+= SQLTABLES_FIELDS;
      }
    }
    if (result)
    {
      if (option_flag(stmt, FLAG_NO_CATALOG))
        db= owner= "";
      else
        db= is_default_db(stmt->dbc->mysql.db,TableQualifier) ?
                          stmt->dbc->mysql.db : 
                          strdup_root(alloc,TableQualifier);

      if (sys_result)
      {    
        while ((row = mysql_fetch_row(result)))
        {    
          data[0]= db;
          data[1]= owner;      
          data[2]= strdup_root(alloc,row[0]);
          data[3]= "TABLE";      
          data[4]= "MySQL Table";
          data  += SQLTABLES_FIELDS;
        }
        mysql_free_result(result);
      }
      else
      {
        stmt->result     = result;
        stmt->order      =  SQLTABLES_order;
        stmt->order_count= array_elements(SQLTABLES_order);
        stmt->fix_fields = fix_fields_copy;
        stmt->array      = (MYSQL_ROW) my_memdup(
                           (gptr)SQLTABLES_values,
                           sizeof(SQLTABLES_values),MYF(0));
        stmt->array[0]   = db;
        stmt->array[1]   = owner;
      }
    }
    stmt->result->row_count= row_count;
  }
  mysql_link_fields(stmt,SQLTABLES_fields,SQLTABLES_FIELDS);
  DBUG_PRINT("info ",("total tables: %ld", stmt->result->row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS);
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLTABLES_values,
                                            sizeof(SQLTABLES_values), 
                                            MYF(0));
  mysql_link_fields(stmt,SQLTABLES_fields, SQLTABLES_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}


/*
****************************************************************************
SQLColumns
****************************************************************************
*/

char SC_type[10],SC_typename[20],SC_precision[10],SC_length[10],SC_scale[10],
  SC_nullable[10], SC_coldef[10], SC_sqltype[10],SC_octlen[10],
  SC_pos[10],SC_isnull[10];

char *SQLCOLUMNS_values[]= {
  "","",NullS,NullS,SC_type,SC_typename,
  SC_precision,
  SC_length,SC_scale,"10",SC_nullable,"MySQL column",
  SC_coldef,SC_sqltype,NullS,SC_octlen,NullS,SC_isnull
};

MYSQL_FIELD SQLCOLUMNS_fields[]=
{
  {"TABLE_CAT","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_SCHEM","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_NAME","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"COLUMN_NAME","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"DATA_TYPE","MySQL_Catalog",NullS,NullS,NullS,5,5,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"TYPE_NAME","MySQL_Catalog",NullS,NullS,NullS,20,20,NOT_NULL_FLAG,0,
   FIELD_TYPE_VAR_STRING},
  {"COLUMN_SIZE","MySQL_Catalog",NullS,NullS,NullS,11,11,0,0,FIELD_TYPE_LONG},
  {"BUFFER_LENGTH","MySQL_Catalog",NullS,NullS,NullS,11,11,0,0,
   FIELD_TYPE_LONG},
  {"DECIMAL_DIGITS","MySQL_Catalog",NullS,NullS,NullS,2,2,0,0,FIELD_TYPE_SHORT},
  {"NUM_PREC_RADIX","MySQL_Catalog",NullS,NullS,NullS,2,2,0,0,FIELD_TYPE_SHORT},
  {"NULLABLE","MySQL_Catalog",NullS,NullS,NullS,5,5,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"REMARKS","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,0,0,
   FIELD_TYPE_VAR_STRING},
  {"COLUMN_DEF","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,0,0,
   FIELD_TYPE_VAR_STRING},
  {"SQL_DATA_TYPE","MySQL_Catalog",NullS,NullS,NullS,5,5,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"SQL_DATETIME_SUB","MySQL_Catalog",NullS,NullS,NullS,2,2,0,0,FIELD_TYPE_SHORT},
  {"CHAR_OCTET_LENGTH","MySQL_Catalog",NullS,NullS,NullS,11,11,0,0,
   FIELD_TYPE_LONG},
  {"ORDINAL_POSITION","MySQL_Catalog",NullS,NullS,NullS,11,11,NOT_NULL_FLAG,0,
   FIELD_TYPE_LONG},
  {"IS_NULLABLE","MySQL_Catalog",NullS,NullS,NullS,3,3,0,0,
   FIELD_TYPE_VAR_STRING}};

const uint SQLCOLUMNS_FIELDS= array_elements(SQLCOLUMNS_values);


/*
  @type    : internal
  @purpose : returns columns from a perticular table from a 
             specified database
*/

static MYSQL_RES* mysql_list_dbcolumns(STMT FAR   *stmt, 
                                       const char *TableQualifier,
                                       const char *TableName,
                                       const char *ColumnName)
{
  DBC FAR   *dbc= stmt->dbc;
  MYSQL FAR *mysql= &dbc->mysql;
  MYSQL_RES *result;
  MYSQL_ROW row;

  if (table_has_database_name(TableQualifier,TableName))
  {
    char buff[255], tab_buff[NAME_LEN*2+7];
    char *select, *to;
    
    if (valid_input_parameter(TableQualifier))   
      strxmov(tab_buff,TableQualifier,".`",TableName,"`",NullS);
    else
      strxmov(tab_buff,TableName,NullS);

    strxmov(buff,"SHOW FIELDS FROM ",tab_buff,NullS);
    my_append_wild(strmov(buff,buff),buff+sizeof(buff)-1,ColumnName);

    MYLOG_QUERY(stmt, buff);
    
    pthread_mutex_lock(&dbc->lock);
    if (mysql_query(mysql,buff) || 
        !(result= mysql_store_result(mysql)))
    {      
      pthread_mutex_lock(&dbc->lock);
      return 0;
    }    
    pthread_mutex_unlock(&dbc->lock);
    
    if (!(select= (char *) my_malloc(sizeof(char *)*(ulong)result->row_count*
                                     (NAME_LEN+1)+NAME_LEN *2, MYF(MY_FAE))))
    {  
      DBUG_PRINT("error",("Memory allocation failed"));
      return 0;
    }  
    to= strxmov(select,"SELECT ",NullS);
    while (row= mysql_fetch_row(result))
      to= strxmov(to, to, row[0], ",",NullS);
    *(to-1)= '\0';
   
    if (valid_input_parameter(TableQualifier))
      strxmov(select,select," FROM ",TableQualifier,".`",TableName,"`",NullS);  
    else      
      strxmov(select,select," FROM ",TableName,NullS);  
    
    mysql_free_result(result);

    MYLOG_QUERY(stmt, select);
    
    pthread_mutex_lock(&dbc->lock);
    if (mysql_query(mysql,select))
    {      
      pthread_mutex_unlock(&dbc->lock);
      return 0;
    }
    result= mysql_store_result(&dbc->mysql);
    pthread_mutex_unlock(&dbc->lock);
    return result;
  }
  pthread_mutex_lock(&dbc->lock);
  result= mysql_list_fields(mysql,TableName,ColumnName);
  pthread_mutex_unlock(&dbc->lock);
  return result;
}

/*
  @type    : ODBC 1.0 API
  @purpose : returns the list of column names in specified tables.
       The driver returns this information as a result set on the
       specified StatementHandle
*/

SQLRETURN SQL_API SQLColumns(SQLHSTMT hstmt,
                             SQLCHAR FAR *szTableQualifier,
                             SQLSMALLINT cbTableQualifier,
                             SQLCHAR FAR *szTableOwner,
                             SQLSMALLINT cbTableOwner,
                             SQLCHAR FAR *szTableName, 
                             SQLSMALLINT cbTableName,
                             SQLCHAR FAR *szColumnName,
                             SQLSMALLINT cbColumnName)
{
  STMT FAR    *stmt= (STMT FAR*) hstmt;
  char        buff[80];
  char        Qualifier_buff[NAME_LEN+1],
              Table_buff[NAME_LEN+1], 
              Column_buff[NAME_LEN+1], 
              *TableQualifier, *TableName,*ColumnName;
  char        *db= "";
  MYSQL_RES   *result;
  MYSQL_FIELD *curField;
  char        **row;
  MEM_ROOT    *alloc;
  ulong       transfer_length,precision,display_size;
    
  DBUG_ENTER("SQLColumns");  

  DBUG_PRINT("enter",("Qualifier: '%s'(%d)  Owner: '%s'(%d)  Table: '%s'(%d)  Column: '%s'(%d)",
          szTableQualifier ? (char*) szTableQualifier : "null", cbTableQualifier,
          szTableOwner ? (char*) szTableOwner : "null", cbTableOwner,
          szTableName ? (char*) szTableName : "null", cbTableName,
          szColumnName ? (char*) szColumnName : "null", cbColumnName));

  TableQualifier= myodbc_get_valid_buffer((char *) Qualifier_buff, szTableQualifier, cbTableQualifier);
  TableName=      myodbc_get_valid_buffer((char *) Table_buff, szTableName, cbTableName);
  ColumnName=     myodbc_get_valid_buffer((char *) Column_buff, szColumnName, cbColumnName);

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  if (!valid_input_parameter(TableName))
    goto empty_set;
  
  escape_input_parameter(&stmt->dbc->mysql, TableQualifier);
  escape_input_parameter(&stmt->dbc->mysql, TableName);
  escape_input_parameter(&stmt->dbc->mysql, ColumnName);

  stmt->result= mysql_list_dbcolumns(stmt,TableQualifier,TableName,ColumnName);
  if (!(result= stmt->result))
  {  
    DBUG_PRINT("error",("%d:%s", mysql_errno(&stmt->dbc->mysql),
                         mysql_error(&stmt->dbc->mysql)));
    goto empty_set;
  }
  stmt->result_array= (char**) my_malloc(sizeof(char*)*SQLCOLUMNS_FIELDS*
           result->field_count,
           MYF(MY_FAE | MY_ZEROFILL));
  alloc= &result->field_alloc;

  if (!option_flag(stmt, FLAG_NO_CATALOG))
    db= is_default_db(stmt->dbc->mysql.db,TableQualifier) ?
                      stmt->dbc->mysql.db : 
                      strdup_root(alloc,TableQualifier);
  
  for (row= stmt->result_array;
       (curField= mysql_fetch_field(result)) ; )
  {
    int type;
      
    row[0]= db;
    row[2]= curField->table;
    row[3]= curField->name;
    row[1]= "";         /* No owner */
    curField->max_length= curField->length;
    type= unireg_to_sql_datatype(stmt,curField,buff,
                                 &transfer_length,&precision,
                                 &display_size);
    row[5]= strdup_root(alloc,buff);
    sprintf(buff,"%d",type);
    row[13]= row[4]= strdup_root(alloc,buff);
    sprintf(buff,"%ld",precision);
    row[6]= strdup_root(alloc,buff);
    sprintf(buff,"%ld",transfer_length);
    row[7]= strdup_root(alloc,buff);
    if (IS_NUM(curField->type))
    {   
      sprintf(buff,"%d",curField->decimals);
      row[8]= strdup_root(alloc,buff);  /* Scale */
      row[9]= "10";   
    }
    else
    {
      row[8]= row[9]= NullS;
      row[15]= strdup_root(alloc,buff);
    }
    if ((curField->flags & (NOT_NULL_FLAG | AUTO_INCREMENT_FLAG)) == NOT_NULL_FLAG)
    { 
      sprintf(buff,"%d",SQL_NO_NULLS);
      row[10]= strdup_root(alloc,buff);
      row[17]= strdup_root(alloc,"NO");
    }
    else 
    {   
      sprintf(buff,"%d",SQL_NULLABLE);
      row[10]= strdup_root(alloc,buff);
      row[17]= strdup_root(alloc,"YES");
    }
    row[11]= ""; 

    /* 
       TODO: When 4.1 supports correct DEFAULT valued fix this
       Default value: 

       The default value of the column. The value in this column should be 
       interpreted as a string if it is enclosed in quotation marks. 

       if NULL was specified as the default value, then this column is the 
       word NULL, not enclosed in quotation marks. If the default value 
       cannot be represented without truncation, then this column contains 
       TRUNCATED, with no enclosing single quotation marks. If no default 
       value was specified, then this column is NULL.
       
       The value of COLUMN_DEF can be used in generating a new column 
       definition, except when it contains the value TRUNCATED

    */
    if (!curField->def)
      row[12]= NullS;
    else
    {
      if (curField->type == FIELD_TYPE_TIMESTAMP &&
          !strcmp(curField->def,"0000-00-00 00:00:00"))
       row[12]= NullS;      
      else
      {
        char *def= alloc_root(alloc, strlen(curField->def)+3);       
        if (IS_NUM(curField->type))
          sprintf(def,"%s",curField->def);
        else
          sprintf(def,"'%s'",curField->def);
        row[12]= def;      
      }
    }
    row+= SQLCOLUMNS_FIELDS;
  }
  result->row_count= result->field_count;
  mysql_link_fields(stmt,SQLCOLUMNS_fields,SQLCOLUMNS_FIELDS);
  DBUG_PRINT("info ",("total columns: %ld", result->row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS);
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLCOLUMNS_values,
                                            sizeof(SQLCOLUMNS_values), 
                                            MYF(0));
  mysql_link_fields(stmt,SQLCOLUMNS_fields, SQLCOLUMNS_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}

/*
****************************************************************************
SQLStatistics
****************************************************************************
*/

char SS_type[10];
uint SQLSTAT_order[]={2,3,5,7,8,9,10};
char *SQLSTAT_values[]={NullS,NullS,"","",NullS,"",SS_type,"","","","",NullS,NullS};

MYSQL_FIELD SQLSTAT_fields[]=
{
  {"TABLE_CAT","MySQL_Stat",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_SCHEM","MySQL_Stat",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_NAME","MySQL_Stat",NullS,NullS,NullS,NAME_LEN,NAME_LEN,NOT_NULL_FLAG,
   0, FIELD_TYPE_VAR_STRING},
  {"NON_UNIQUE","MySQL_Stat",NullS,NullS,NullS,1,1,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"INDEX_QUALIFIER","MySQL_Stat",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"INDEX_NAME","MySQL_Stat",NullS,NullS,NullS,NAME_LEN,NAME_LEN,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TYPE","MySQL_Stat",NullS,NullS,NullS,1,1,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"ORDINAL_POSITION","MySQL_Stat",NullS,NullS,NullS,1,2,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"COLUMN_NAME","MySQL_Stat",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"ASC_OR_DESC","MySQL_Stat",NullS,NullS,NullS,1,1,0,0,FIELD_TYPE_VAR_STRING},
  {"CARDINALITY","MySQL_Stat",NullS,NullS,NullS,11,11,0,0,FIELD_TYPE_LONG},
  {"PAGES","MySQL_Stat",NullS,NullS,NullS,9,9,0,0,FIELD_TYPE_LONG},
  {"FILTER_CONDITION","MySQL_Stat",NullS,NullS,NullS,10,10,0,0,
   FIELD_TYPE_VAR_STRING},
};

const uint SQLSTAT_FIELDS= array_elements(SQLSTAT_fields);

/*
  @type    : internal
  @purpose : returns columns from a perticular table
*/
static MYSQL_RES *mysql_list_dbkeys(DBC FAR    *dbc, 
                                    const char *db,
                                    const char *table) 
{
  MYSQL FAR *mysql= &dbc->mysql;
  char      buff[255];
     
  if (valid_input_parameter(db))
    strxmov(buff,"SHOW KEYS FROM ",db,".`",table,"`",NullS);  
  else      
    strxmov(buff,"SHOW KEYS FROM `",table,"`",NullS);

  MYLOG_DBC_QUERY(dbc, buff);
  if (mysql_query(mysql,buff))
    return 0;
  return mysql_store_result(mysql);
}


/*
  @type    : ODBC 1.0 API
  @purpose : retrieves a list of statistics about a single table and the
       indexes associated with the table. The driver returns the
       information as a result set.
*/

SQLRETURN SQL_API SQLStatistics(SQLHSTMT hstmt,
                                SQLCHAR FAR *szTableQualifier,
                                SQLSMALLINT cbTableQualifier,
                                SQLCHAR FAR *szTableOwner,
                                SQLSMALLINT cbTableOwner,
                                SQLCHAR FAR *szTableName,
                                SQLSMALLINT cbTableName,
                                SQLUSMALLINT fUnique,
                                SQLUSMALLINT fAccuracy)
{
  STMT FAR  *stmt= (STMT FAR*) hstmt;
  MYSQL FAR *mysql= &stmt->dbc->mysql;
  DBC FAR   *dbc= stmt->dbc;
  char      Qualifier_buff[NAME_LEN+1],
            Table_buff[NAME_LEN+1], 
            *TableQualifier, *TableName;
  
  DBUG_ENTER("SQLStatistics");

  DBUG_PRINT("enter",("Qualifier: '%s'(%d)  Owner: '%s'(%d)  Table: '%s'(%d)  fUnique: %d, fAccuracy: %d)",
          szTableQualifier ? (char*) szTableQualifier : "null", cbTableQualifier,
          szTableOwner ? (char*) szTableOwner : "null", cbTableOwner,
          szTableName ? (char*) szTableName : "null", cbTableName,
          fUnique, fAccuracy));

  TableQualifier= myodbc_get_valid_buffer((char *) Qualifier_buff, szTableQualifier, cbTableQualifier);
  TableName=      myodbc_get_valid_buffer((char *) Table_buff, szTableName, cbTableName);

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  if (!valid_input_parameter(TableName))
    goto empty_set;
  
  escape_input_parameter(mysql, TableQualifier);
  escape_input_parameter(mysql, TableName);

  pthread_mutex_lock(&dbc->lock);
  if (!(stmt->result= mysql_list_dbkeys(stmt->dbc,TableQualifier,TableName)))
  {    
    DBUG_PRINT("error",("%d:%s", mysql_errno(mysql),mysql_error(mysql)));
    pthread_mutex_unlock(&dbc->lock);
    goto empty_set;
  }
  pthread_mutex_unlock(&dbc->lock);
  int2str(SQL_INDEX_OTHER,SS_type,10);
  stmt->order=       SQLSTAT_order;
  stmt->order_count= array_elements(SQLSTAT_order);
  stmt->fix_fields=  fix_fields_copy;
  stmt->array= (MYSQL_ROW) my_memdup((gptr) SQLSTAT_values,
                           sizeof(SQLSTAT_values),MYF(0));

  if (option_flag(stmt, FLAG_NO_CATALOG))
    stmt->array[0]= "";
   else
    stmt->array[0]= is_default_db(mysql->db,TableQualifier) ?
                                  mysql->db : 
                                  strdup_root(&stmt->result->field_alloc,TableQualifier);

  if (fUnique == SQL_INDEX_UNIQUE)
  {    
    /* This is too low level... */
    MYSQL_ROWS **prev,*pos;
    prev= &stmt->result->data->data;
    for (pos= *prev; pos; pos= pos->next)
    {
      if (pos->data[1][0] == '0') /* Unlink nonunique index */
      {
        (*prev)=pos;
        prev= &pos->next;
      }
      else
        stmt->result->row_count--;
    }
    (*prev)= 0;
    mysql_data_seek(stmt->result,0);  /* Restore pointer */
  }
  mysql_link_fields(stmt,SQLSTAT_fields,SQLSTAT_FIELDS);
  DBUG_PRINT("info ",("total stats count: %ld", stmt->result->row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS);
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLSTAT_values,
                                            sizeof(SQLSTAT_values), 
                                            MYF(0));
  mysql_link_fields(stmt,SQLSTAT_fields, SQLSTAT_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}

/*
****************************************************************************
SQLTablePrivileges
****************************************************************************
*/

/*
  @type    : internal
  @purpose : checks for the grantability 
*/

static my_bool is_grantable(char *grant_list)
{
  char *grant=dupp_str(grant_list,SQL_NTS);;
  if (grant_list && grant_list[0])
  {
    char seps[]   = ",";
    char *token;
    token = strtok( grant, seps );
    while( token != NULL )
    {
      if (!strcmp(token,"Grant")) 
      {
        x_free(grant);
        return(1);
      }
      token = strtok( NULL, seps );
    }
  }
  x_free(grant);
  return(0);
}

/*
  @type    : internal
  @purpose : returns a table privileges result 
*/
static MYSQL_RES *mysql_list_table_priv(DBC FAR *dbc, 
                                        const char *qualifier, 
                                        const char *table)
{
  MYSQL FAR *mysql= &dbc->mysql;
  char      buff[255+2*NAME_LEN+1];
 
  my_append_wild(strmov(buff,
    "SELECT Db,User,Table_name,Grantor,Table_priv\
    FROM mysql.tables_priv WHERE Table_name"),
    buff+sizeof(buff),table);
  strxmov(buff,buff," AND Db",NullS);
  my_append_wild(strmov(buff,buff),buff+sizeof(buff),qualifier);
  strxmov(buff,buff," ORDER BY Db,Table_name,Table_priv,User",NullS);

  MYLOG_DBC_QUERY(dbc, buff);
  if (mysql_query(mysql,buff))
    return 0;
  return mysql_store_result(mysql);
}

#define MY_MAX_TABPRIV_COUNT 21
#define MY_MAX_COLPRIV_COUNT 3

char *SQLTABLES_priv_values[]= 
{
  NULL,"",NULL,NULL,NULL,NULL,NULL
};

MYSQL_FIELD SQLTABLES_priv_fields[]=
{
  {"TABLE_CAT","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_SCHEM","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_NAME","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"GRANTOR","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"GRANTEE","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,NOT_NULL_FLAG,
   0,FIELD_TYPE_VAR_STRING},
  {"PRIVILEGE","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"IS_GRANTABLE","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
};

const uint SQLTABLES_PRIV_FIELDS= array_elements(SQLTABLES_priv_values);

/*
  @type    : ODBC 1.0 API
  @purpose : returns a list of tables and the privileges associated with
             each table. The driver returns the information as a result
             set on the specified statement.
*/

SQLRETURN SQL_API SQLTablePrivileges(SQLHSTMT hstmt,
                                     SQLCHAR FAR *szTableQualifier,
                                     SQLSMALLINT cbTableQualifier,
                                     SQLCHAR FAR *szTableOwner,
                                     SQLSMALLINT cbTableOwner,
                                     SQLCHAR FAR *szTableName,
                                     SQLSMALLINT cbTableName)
{
  char     Qualifier_buff[NAME_LEN+1],Name_buff[NAME_LEN+1],
           *TableQualifier,*TableName;
  char     **data, **row;
  MEM_ROOT *alloc;
  STMT FAR *stmt= (STMT FAR*) hstmt;
  uint     row_count;
  
  DBUG_ENTER("SQLTablePrivileges");
  
  DBUG_PRINT("enter",("Qualifier: '%s'  Owner: '%s'  Table: '%s'",
          szTableQualifier ? (char*) szTableQualifier : "null",
          szTableOwner ? (char*) szTableOwner : "null",
          szTableName ? (char*) szTableName : "null"));

  TableQualifier= myodbc_get_valid_buffer((char FAR *) Qualifier_buff,szTableQualifier,
         cbTableQualifier);
  TableName= myodbc_get_valid_buffer((char FAR*) Name_buff,szTableName,cbTableName);

  escape_input_parameter(&stmt->dbc->mysql, TableQualifier);
  escape_input_parameter(&stmt->dbc->mysql, TableName);

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  pthread_mutex_lock(&stmt->dbc->lock);
  if (!(stmt->result= mysql_list_table_priv(stmt->dbc,
                                            TableQualifier,TableName)))
  {
    DBUG_PRINT("error",("%d:%s", mysql_errno(&stmt->dbc->mysql),
                           mysql_error(&stmt->dbc->mysql)));
    pthread_mutex_unlock(&stmt->dbc->lock);
    goto empty_set;
  }
  pthread_mutex_unlock(&stmt->dbc->lock);

  /* Allocate max buffers, to avoid reallocation */
  stmt->result_array= (char**) my_malloc(sizeof(char*)* SQLTABLES_PRIV_FIELDS *
                                        (ulong)stmt->result->row_count * 
                                        MY_MAX_TABPRIV_COUNT, 
                                        MYF(MY_FAE | MY_ZEROFILL));
  alloc= &stmt->result->field_alloc;
  data= stmt->result_array;
  row_count= 0;
  while ((row= mysql_fetch_row(stmt->result)))
  {     
    char  *grants= row[4];
    char  token[NAME_LEN+1];
    const char *grant= (const char *)grants;
        
    for (;;)
    {
      data[0]= row[0];         
      data[1]= "";         
      data[2]= row[2];
      data[3]= row[3];   
      data[4]= row[1];
      data[6]= is_grantable(row[4]) ? "YES":"NO";    
      row_count++;
    
      if (!(grant= my_next_token(grant,&grants,token,',')))
      {
        /* End of grants .. */
        data[5]= strdup_root(alloc,grants);
        data+= SQLTABLES_PRIV_FIELDS; 
        break;
      }      
      data[5]= strdup_root(alloc,token);    
      data+= SQLTABLES_PRIV_FIELDS;      
    } 
  }  
  stmt->result->row_count= row_count;
  mysql_link_fields(stmt,SQLTABLES_priv_fields,SQLTABLES_PRIV_FIELDS);
  DBUG_PRINT("info ",("total table priv count: %ld", row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS);
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLTABLES_priv_values,
                                            sizeof(SQLTABLES_priv_values), 
                                            MYF(0));
  mysql_link_fields(stmt,SQLTABLES_priv_fields,SQLTABLES_PRIV_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}



/*
****************************************************************************
SQLColumnPrivileges
****************************************************************************
*/

/*
  @type    : internal
  @purpose : returns a column privileges result 
*/
static MYSQL_RES *mysql_list_column_priv(MYSQL *mysql, 
                                         const char *qualifier,
                                         const char *table, 
                                         const char *column)
{
  char buff[255+3*NAME_LEN+1];
  
  my_append_wild(strmov(buff,
    "SELECT c.Db, c.User,c.Table_name,c.Column_name,\
    t.Grantor,c.Column_priv,t.Table_priv FROM mysql.columns_priv as c,\
    mysql.tables_priv as t WHERE c.Table_name"),
    buff+sizeof(buff),table);
  strxmov(buff,buff," AND c.Db",NullS);
  my_append_wild(strmov(buff,buff),buff+sizeof(buff),qualifier);
  strxmov(buff,buff," AND c.Column_name",NullS);
  my_append_wild(strmov(buff,buff),buff+sizeof(buff),column);
  strxmov(buff,buff," AND c.Table_name=t.Table_name",
          " ORDER BY c.Db,c.Table_name,c.Column_name,c.Column_priv", NullS);
  
  if (mysql_query(mysql,buff))
    return 0;

  return mysql_store_result(mysql);
}

char *SQLCOLUMNS_priv_values[]= 
{
  NULL,"",NULL,NULL,NULL,NULL,NULL,NULL
};

MYSQL_FIELD SQLCOLUMNS_priv_fields[]=
{
  {"TABLE_CAT","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_SCHEM","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_NAME","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"COLUMN_NAME","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"GRANTOR","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"GRANTEE","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,NOT_NULL_FLAG,
   0,FIELD_TYPE_VAR_STRING},
  {"PRIVILEGE","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,NAME_LEN,
   NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"IS_GRANTABLE","MySQL_Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
};

const uint SQLCOLUMNS_PRIV_FIELDS= array_elements(SQLCOLUMNS_priv_values);

/*
  @type    : ODBC 1.0 API
  @purpose : returns a list of columns and associated privileges for the
             specified table. The driver returns the information as a result
             set on the specified StatementHandle.
*/

SQLRETURN SQL_API SQLColumnPrivileges(SQLHSTMT hstmt,
                                      SQLCHAR FAR *szTableQualifier,
                                      SQLSMALLINT cbTableQualifier,
                                      SQLCHAR FAR *szTableOwner,
                                      SQLSMALLINT cbTableOwner,
                                      SQLCHAR FAR *szTableName,
                                      SQLSMALLINT cbTableName,
                                      SQLCHAR FAR *szColumnName,
                                      SQLSMALLINT cbColumnName)
{ 
  STMT FAR *stmt=(STMT FAR*) hstmt;
  char     Qualifier_buff[NAME_LEN+1],Table_buff[NAME_LEN+1],
           Column_buff[NAME_LEN+1],
           *TableQualifier,*TableName, *ColumnName;
  char     **row, **data;
  MEM_ROOT *alloc;
  uint     row_count;
  
  DBUG_ENTER("SQLColumnPrivileges");
  
  DBUG_PRINT("enter",("Qualifier: '%s'  Owner: '%s'  Table: '%s'  column: '%s'",
          szTableQualifier ? (char*) szTableQualifier : "null",
          szTableOwner ? (char*) szTableOwner : "null",
          szTableName ? (char*) szTableName : "null",
          szColumnName ? (char*) szColumnName : "null"));

  TableQualifier=myodbc_get_valid_buffer((char FAR *) Qualifier_buff,szTableQualifier,
         cbTableQualifier);
  TableName=   myodbc_get_valid_buffer((char FAR*) Table_buff,szTableName,cbTableName);
  ColumnName=  myodbc_get_valid_buffer((char FAR*) Column_buff,szColumnName,cbColumnName);
  
  escape_input_parameter(&stmt->dbc->mysql, TableQualifier);
  escape_input_parameter(&stmt->dbc->mysql, TableName);
  escape_input_parameter(&stmt->dbc->mysql, ColumnName);

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  pthread_mutex_lock(&stmt->dbc->lock);
  if (!(stmt->result= mysql_list_column_priv(&stmt->dbc->mysql,
                                             TableQualifier,
                                             TableName,ColumnName)))
  {
    DBUG_PRINT("error",("%d:%s", mysql_errno(&stmt->dbc->mysql),
                           mysql_error(&stmt->dbc->mysql)));
    pthread_mutex_unlock(&stmt->dbc->lock);
    goto empty_set;
  }
  pthread_mutex_unlock(&stmt->dbc->lock);
  stmt->result_array= (char**) my_malloc(sizeof(char*)*SQLCOLUMNS_PRIV_FIELDS*
         (ulong) stmt->result->row_count *MY_MAX_COLPRIV_COUNT, MYF(MY_FAE | MY_ZEROFILL));
  alloc= &stmt->result->field_alloc;
  data= stmt->result_array;
  row_count= 0;
  while ((row= mysql_fetch_row(stmt->result)))
  {
    char  *grants= row[5];
    char  token[NAME_LEN+1];
    const char *grant= (const char *)grants;

    for (;;)
    {
      data[0]= row[0];
      data[1]= "";
      data[2]= row[2];
      data[3]= row[3];
      data[4]= row[4];
      data[5]= row[1];
      data[7]= is_grantable(row[6]) ? "YES":"NO";
      row_count++;

      if (!(grant= my_next_token(grant,&grants,token,',')))
      {
        /* End of grants .. */
        data[6]= strdup_root(alloc,grants);
        data+= SQLCOLUMNS_PRIV_FIELDS;
        break;
      }
      data[6]= strdup_root(alloc,token);
      data+= SQLCOLUMNS_PRIV_FIELDS;
    }
  }
  stmt->result->row_count= row_count;  
  mysql_link_fields(stmt,SQLCOLUMNS_priv_fields,SQLCOLUMNS_PRIV_FIELDS);
  DBUG_PRINT("info ",("total columns priv count: %ld", row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS); 
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLCOLUMNS_priv_values,
                                            sizeof(SQLCOLUMNS_priv_values), 
                                            MYF(0)); 
  mysql_link_fields(stmt,SQLCOLUMNS_priv_fields,SQLCOLUMNS_PRIV_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}

/*
****************************************************************************
SQLSpecialColumns
****************************************************************************
*/

MYSQL_FIELD SQLSPECIALCOLUMNS_fields[]=
{
  {"SCOPE","MySQL_SpecialColumns",NullS,NullS,NullS,5,5,
   NOT_NULL_FLAG,0,FIELD_TYPE_SHORT},
  {"COLUMN_NAME","MySQL_SpecialColumns",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN,NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"DATA_TYPE","MySQL_SpecialColumns",NullS,NullS,NullS,5,5,
   NOT_NULL_FLAG,0,FIELD_TYPE_SHORT},
  {"TYPE_NAME","MySQL_SpecialColumns",NullS,NullS,NullS,20,20,
   NOT_NULL_FLAG,0,FIELD_TYPE_VAR_STRING},
  {"COLUMN_SIZE","MySQL_SpecialColumns",NullS,NullS,NullS,7,7,0,0,
   FIELD_TYPE_LONG},
  {"BUFFER_LENGTH","MySQL_SpecialColumns",NullS,NullS,NullS,7,7,0,0,
   FIELD_TYPE_LONG},
  {"DECIMAL_DIGITS","MySQL_SpecialColumns",NullS,NullS,NullS,3,3,0,0,
   FIELD_TYPE_SHORT},
  {"PSEUDO_COLUMN","MySQL_SpecialColumns",NullS,NullS,NullS,3,3,0,0,
   FIELD_TYPE_SHORT}
};

char *SQLSPECIALCOLUMNS_values[]= {
  0,NULL,0,NULL,0,0,0,0
};

const uint SQLSPECIALCOLUMNS_FIELDS= array_elements(SQLSPECIALCOLUMNS_fields);


/*
  @type    : ODBC 1.0 API
  @purpose : retrieves the following information about columns within a
       specified table:
       - The optimal set of columns that uniquely identifies a row
         in the table.
       - Columns that are automatically updated when any value in the
         row is updated by a transaction
*/

SQLRETURN SQL_API SQLSpecialColumns(SQLHSTMT hstmt,
                                    SQLUSMALLINT fColType,
                                    SQLCHAR FAR *szTableQualifier,
                                    SQLSMALLINT cbTableQualifier,
                                    SQLCHAR FAR *szTableOwner,
                                    SQLSMALLINT cbTableOwner,
                                    SQLCHAR FAR *szTableName,
                                    SQLSMALLINT cbTableName,
                                    SQLUSMALLINT fScope,
                                    SQLUSMALLINT fNullable)
{  
  STMT FAR    *stmt=(STMT FAR*) hstmt;
  char        buff[80];
  char        **row;
  MYSQL_RES   *result;
  MYSQL_FIELD *field;
  MEM_ROOT    *alloc;
  char        Qualifier_buff[NAME_LEN+1],Table_buff[NAME_LEN+1],
              *TableQualifier,*TableName;
  ulong       transfer_length,precision,display_size;
  uint        field_count;
  my_bool     primary_key;

  DBUG_ENTER("SQLSpecialColumns");
  
  DBUG_PRINT("enter",("ColType: %d  Qualifier: '%s'  Owner: '%s' Table: '%s'  Scope: '%d'  Nullable: %d",
          fColType,
          szTableQualifier ? (char*) szTableQualifier : "null",
          szTableOwner ? (char*) szTableOwner : "null",
          szTableName ? (char*) szTableName : "null",
          fScope, fNullable));

  TableQualifier=myodbc_get_valid_buffer((char FAR *) Qualifier_buff,szTableQualifier,
         cbTableQualifier);
  TableName=   myodbc_get_valid_buffer((char FAR*) Table_buff,szTableName,cbTableName);
  
  escape_input_parameter(&stmt->dbc->mysql, TableQualifier);
  escape_input_parameter(&stmt->dbc->mysql, TableName);

  CLEAR_STMT_ERROR(hstmt);
  
  stmt->result= mysql_list_dbcolumns(stmt,TableQualifier,TableName,0);
  if (!(result= stmt->result))
  {    
    DBUG_PRINT("error",("%d:%s", mysql_errno(&stmt->dbc->mysql),
                           mysql_error(&stmt->dbc->mysql)));
    goto empty_set;
  }

  if (fColType == SQL_ROWVER)
  {      
    /* Find possible timestamp */
    if (!(stmt->result_array= (char**) my_malloc(sizeof(char*)*SQLSPECIALCOLUMNS_FIELDS*
                                       result->field_count, MYF(MY_FAE | MY_ZEROFILL))))
    { 
      DBUG_PRINT("error",("Memory allocation failed"));
      goto empty_set;
    }
    
    /* Convert mysql fields to data that odbc wants */
    alloc= &result->field_alloc;
    field_count= 0;
    mysql_field_seek(result,0);
    for (row= stmt->result_array;
        (field = mysql_fetch_field(result));)
    {
      int type;
      if ((field->type != FIELD_TYPE_TIMESTAMP))
        continue;
      field_count++;
      sprintf(buff,"%d",SQL_SCOPE_SESSION);
      row[0]= strdup_root(alloc,buff);
      row[1]= field->name;
      type= unireg_to_sql_datatype(stmt,field,buff,&transfer_length,
                                   &precision,&display_size);
      row[3]= strdup_root(alloc,buff);
      sprintf(buff,"%d",type);
      row[2]= strdup_root(alloc,buff);
      sprintf(buff,"%d",precision);
      row[4]= strdup_root(alloc,buff);
      sprintf(buff,"%d",transfer_length);
      row[5]= strdup_root(alloc,buff);
      sprintf(buff,"%d",field->decimals);
      row[6]= strdup_root(alloc,buff);
      sprintf(buff,"%d",SQL_PC_NOT_PSEUDO);
      row[7]= strdup_root(alloc,buff);
      row+= SQLSPECIALCOLUMNS_FIELDS;
    }
    result->row_count= field_count;
    mysql_link_fields(stmt,SQLSPECIALCOLUMNS_fields,SQLSPECIALCOLUMNS_FIELDS);
    DBUG_PRINT("info ",("total columns count: %ld", field_count));
    DBUG_RETURN_STATUS(SQL_SUCCESS);
  }

  if (fColType != SQL_BEST_ROWID)
  {
    DBUG_RETURN(set_error(stmt,MYERR_S1000,
        "Unsupported argument to SQLSpecialColumns",
        4000));
  }

  /*
   * The optimal set of columns for identifing a row is either
   * the primary key, or if there is no primary key, then
   * all the fields.
   */

  /* Check if there is a primary (unique) key */
  primary_key= 0;
  while ((field= mysql_fetch_field(result)))
  {
    if (field->flags & PRI_KEY_FLAG)
    {
      primary_key=1;
      break;
    }
  }
  if (!(stmt->result_array= (char**) my_malloc(sizeof(char*)*SQLSPECIALCOLUMNS_FIELDS*
                                     result->field_count, MYF(MY_FAE | MY_ZEROFILL))))
  { 
    DBUG_PRINT("error",("Memory allocation failed"));
    goto empty_set;
  }

  /* Convert MySQL fields to data that odbc wants */
  alloc= &result->field_alloc;
  field_count= 0;
  mysql_field_seek(result,0);
  for (row= stmt->result_array ;
       (field= mysql_fetch_field(result)); )
  {
    int type;
    if (primary_key && !(field->flags & PRI_KEY_FLAG))
      continue;
#ifndef SQLSPECIALCOLUMNS_RETURN_ALL_COLUMNS
    /* The ODBC 'standard' doesn't want us to return all columns if there is
       no primary or unique key */
    if (!primary_key)
      continue;
#endif
    field_count++;
    sprintf(buff,"%d",SQL_SCOPE_SESSION);
    row[0]= strdup_root(alloc,buff);
    row[1]= field->name;
    type= unireg_to_sql_datatype(stmt,field,buff,&transfer_length,
                                 &precision,&display_size);
    row[3]= strdup_root(alloc,buff);
    sprintf(buff,"%d",type);
    row[2]= strdup_root(alloc,buff);
    sprintf(buff,"%d",precision);
    row[4]= strdup_root(alloc,buff);
    sprintf(buff,"%d",transfer_length);
    row[5]= strdup_root(alloc,buff);
    sprintf(buff,"%d",field->decimals);
    row[6]= strdup_root(alloc,buff);
    sprintf(buff,"%d",SQL_PC_NOT_PSEUDO);
    row[7]= strdup_root(alloc,buff);
    row+= SQLSPECIALCOLUMNS_FIELDS;
  }
  result->row_count= field_count;
  mysql_link_fields(stmt,SQLSPECIALCOLUMNS_fields,SQLSPECIALCOLUMNS_FIELDS);
  DBUG_PRINT("info ",("total columns count: %ld", field_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS); 
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLSPECIALCOLUMNS_values,
                                            sizeof(SQLSPECIALCOLUMNS_values), 
                                            MYF(0)); 
  mysql_link_fields(stmt,SQLSPECIALCOLUMNS_fields,SQLSPECIALCOLUMNS_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}

/*
****************************************************************************
SQLPrimaryKeys
****************************************************************************
*/

MYSQL_FIELD SQLPRIM_KEYS_fields[]=
{
  {"TABLE_CAT","MySQL_Primary_keys",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_SCHEM","MySQL_Primary_keys",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"TABLE_NAME","MySQL_Primary_keys",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN, NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"COLUMN_NAME","MySQL_Primary_keys",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN, NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"KEY_SEQ","MySQL_Primary_keys",NullS,NullS,NullS,2,2,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"PK_NAME","MySQL_Primary_keys",NullS,NullS,NullS,128,0,0,0,
   FIELD_TYPE_VAR_STRING},
};

const uint SQLPRIM_KEYS_FIELDS= array_elements(SQLPRIM_KEYS_fields);

char *SQLPRIM_KEYS_values[]= {
  NULL,"",NULL,NULL,0,NULL
};

/*
  @type    : ODBC 1.0 API
  @purpose : returns the column names that make up the primary key for a table.
       The driver returns the information as a result set. This function
       does not support returning primary keys from multiple tables in
       a single call
*/

SQLRETURN SQL_API SQLPrimaryKeys(SQLHSTMT hstmt,
                                 SQLCHAR FAR *szTableQualifier,
                                 SQLSMALLINT cbTableQualifier,
                                 SQLCHAR FAR *szTableOwner,
                                 SQLSMALLINT cbTableOwner,
                                 SQLCHAR FAR *szTableName,
                                 SQLSMALLINT cbTableName)
{
  char      Qualifier_buff[NAME_LEN+1],Table_buff[NAME_LEN+1],
            *TableQualifier,*TableName;
  STMT FAR  *stmt= (STMT FAR*) hstmt;
  MYSQL_ROW row;
  char      **data;  
  uint      row_count;

  DBUG_ENTER("SQLPrimaryKeys"); 

  DBUG_PRINT("enter",("Qualifier: '%s'(%d)  Owner: '%s'(%d)  Table: '%s'(%d)",
          szTableQualifier ? (char*) szTableQualifier : "null", cbTableQualifier,
          szTableOwner ? (char*) szTableOwner : "null", cbTableOwner,
          szTableName ? (char*) szTableName : "null", cbTableName));

  TableQualifier= myodbc_get_valid_buffer((char FAR *) Qualifier_buff,szTableQualifier, cbTableQualifier);
  TableName=      myodbc_get_valid_buffer((char FAR*) Table_buff,szTableName,cbTableName);
  
  escape_input_parameter(&stmt->dbc->mysql, TableQualifier);
  escape_input_parameter(&stmt->dbc->mysql, TableName);

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);
  
  pthread_mutex_lock(&stmt->dbc->lock);
  if (!(stmt->result= mysql_list_dbkeys(stmt->dbc,TableQualifier,TableName)))
  {    
    DBUG_PRINT("error",("%d:%s", mysql_errno(&stmt->dbc->mysql),
                           mysql_error(&stmt->dbc->mysql)));
    pthread_mutex_unlock(&stmt->dbc->lock);
    goto empty_set;
  }
  pthread_mutex_unlock(&stmt->dbc->lock);
  stmt->result_array= (char**) my_malloc(sizeof(char*)*SQLPRIM_KEYS_FIELDS*
               (ulong) stmt->result->row_count,
               MYF(MY_FAE | MY_ZEROFILL));
  row_count= 0;
  data= stmt->result_array;
  while ((row= mysql_fetch_row(stmt->result)))
  {
    if (row[1][0] == '0')     /* If unique index */
    {
      if (row_count && !strcmp(row[3],"1"))
        break;    /* Allready found unique key */
      row_count++;
      data[0]= data[1]=0;
      data[2]= row[0];
      data[3]= row[4];
      data[4]= row[3];
      data[5]= "PRIMARY";
      data+= SQLPRIM_KEYS_FIELDS;
    }
  }
  stmt->result->row_count= row_count;

  mysql_link_fields(stmt,SQLPRIM_KEYS_fields,SQLPRIM_KEYS_FIELDS);
  DBUG_PRINT("info ",("total keys count: %ld", row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS);
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLPRIM_KEYS_values,
                                            sizeof(SQLPRIM_KEYS_values), 
                                            MYF(0)); 
  mysql_link_fields(stmt,SQLPRIM_KEYS_fields,SQLPRIM_KEYS_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}

/*
****************************************************************************
SQLForeignJeys
****************************************************************************
*/

MYSQL_FIELD SQLFORE_KEYS_fields[]=
{
  {"PKTABLE_CAT","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"PKTABLE_SCHEM","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"PKTABLE_NAME","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN, NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"PKCOLUMN_NAME","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN, NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"FKTABLE_CAT","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"FKTABLE_SCHEM","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN, NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"FKTABLE_NAME","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN, NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"FKCOLUMN_NAME","MySQL_Foreign_keys",NullS,NullS,NullS,NAME_LEN,
   NAME_LEN, NOT_NULL_FLAG, 0, FIELD_TYPE_VAR_STRING},
  {"KEY_SEQ","MySQL_Foreign_keys",NullS,NullS,NullS,2,2,NOT_NULL_FLAG,0,
   FIELD_TYPE_SHORT},
  {"UPDATE_RULE","MySQL_Foreign_keys",NullS,NullS,NullS,2,2,0,0,
   FIELD_TYPE_SHORT},
  {"DELETE_RULE","MySQL_Foreign_keys",NullS,NullS,NullS,2,2,0,0,
   FIELD_TYPE_SHORT},
  {"FK_NAME","MySQL_Foreign_keys",NullS,NullS,NullS,128,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"PK_NAME","MySQL_Foreign_keys",NullS,NullS,NullS,128,0,0,0,
   FIELD_TYPE_VAR_STRING},
  {"DEFERRABILITY","MySQL_Foreign_keys",NullS,NullS,NullS,2,2,0,0,
   FIELD_TYPE_SHORT},
};

const uint SQLFORE_KEYS_FIELDS= array_elements(SQLFORE_KEYS_fields);

char *SQLFORE_KEYS_values[]= {
  NULL,"",NULL,NULL,
  NULL,"",NULL,NULL,
  0,0,0,NULL,NULL,0
};

/*
  @type    : internal
  @purpose : returns table status in the form of resultset
*/
static MYSQL_RES *mysql_table_status(STMT *stmt, 
                                     const char *qualifier, 
                                     const char *table)
{
  MYSQL *mysql= &stmt->dbc->mysql;
  char  buff[255];  

  strxmov(buff,"show table status from `",qualifier,"`", NullS);
  my_append_wild(strmov(buff,buff),buff+sizeof(buff),table);
  
  MYLOG_QUERY(stmt, buff);
  if (mysql_query(mysql,buff))
    return 0;
  return mysql_store_result(mysql);
}

/*
  @type    : ODBC 1.0 API
  @purpose : returns
       - A list of foreign keys in the specified table (columns
         in the specified table that refer to primary keys in
         other tables).
       - A list of foreign keys in other tables that refer to the primary
         key in the specified table
*/

SQLRETURN SQL_API SQLForeignKeys(SQLHSTMT hstmt,
                                 SQLCHAR FAR *szPkTableQualifier,
                                 SQLSMALLINT cbPkTableQualifier,
                                 SQLCHAR FAR *szPkTableOwner,
                                 SQLSMALLINT cbPkTableOwner,
                                 SQLCHAR FAR *szPkTableName,
                                 SQLSMALLINT cbPkTableName,
                                 SQLCHAR FAR *szFkTableQualifier,
                                 SQLSMALLINT cbFkTableQualifier,
                                 SQLCHAR FAR *szFkTableOwner,
                                 SQLSMALLINT cbFkTableOwner,
                                 SQLCHAR FAR *szFkTableName,
                                 SQLSMALLINT cbFkTableName)
{
  STMT FAR *stmt=(STMT FAR*) hstmt;
  uint row_count= 0;
  
  DBUG_ENTER("SQLForeignKeys");

  DBUG_PRINT("enter",("PKQualifier: '%s'(%d)  PKOwner: '%s'(%d)  PKTable: '%s'(%d) \
          FKQualifier: '%s'(%d)  FKOwner: '%s'(%d)  FKTable: '%s'(%d)",
          szPkTableQualifier ? (char*) szPkTableQualifier : "null", cbPkTableQualifier,
          szPkTableOwner ? (char*) szPkTableOwner : "null", cbPkTableOwner,
          szPkTableName ? (char*) szPkTableName : "null", cbPkTableName,
          szFkTableQualifier ? (char*) szFkTableQualifier : "null", cbFkTableQualifier,
          szFkTableOwner ? (char*) szFkTableOwner : "null", cbFkTableOwner,
          szFkTableName ? (char*) szFkTableName : "null", cbFkTableName));

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  if (is_minimum_version(stmt->dbc->mysql.server_version,"3.23",4))
  {
    STMT FAR  *stmt=(STMT FAR*) hstmt;
    MEM_ROOT  *alloc;
    MYSQL_ROW row;
    char      **data;
    char      PkQualifier_buff[NAME_LEN+1],PkName_buff[NAME_LEN+1],
              *PkTableQualifier,*PkTableName;
    char      FkQualifier_buff[NAME_LEN+1],FkName_buff[NAME_LEN+1],
              *FkTableQualifier,*FkTableName;  
    uint       comment_id;

    PkTableQualifier= myodbc_get_valid_buffer((char FAR *)PkQualifier_buff,szPkTableQualifier,
             cbPkTableQualifier);
    PkTableName= myodbc_get_valid_buffer((char FAR*) PkName_buff,szPkTableName,cbPkTableName);
    FkTableQualifier= myodbc_get_valid_buffer((char FAR *)FkQualifier_buff,szFkTableQualifier,
             cbFkTableQualifier);
    FkTableName= myodbc_get_valid_buffer((char FAR*)FkName_buff,szFkTableName,cbFkTableName);
    
    if (FkTableQualifier && !FkTableQualifier[0])
      FkTableQualifier= stmt->dbc->database;

    CLEAR_STMT_ERROR(hstmt);

    pthread_mutex_lock(&stmt->dbc->lock);
    if (!(stmt->result= mysql_table_status(stmt,FkTableQualifier,FkTableName)))
    {    
      DBUG_PRINT("error",("%d:%s", mysql_errno(&stmt->dbc->mysql),
                           mysql_error(&stmt->dbc->mysql)));
      pthread_mutex_unlock(&stmt->dbc->lock);
      goto empty_set;
    }    
    pthread_mutex_unlock(&stmt->dbc->lock);
    stmt->result_array= (char**) my_malloc(sizeof(char*)*SQLFORE_KEYS_FIELDS*
             (ulong)stmt->result->field_count,
             MYF(MY_FAE | MY_ZEROFILL));

    /* Convert mysql fields to data that odbc wants */
    alloc= &stmt->result->field_alloc;
    data= stmt->result_array;    
    comment_id= stmt->result->field_count-1;

    while ((row= mysql_fetch_row(stmt->result)))
    {
      if ((row[1] && strcmp(row[1],"InnoDB")==0))
      {
        const char *token,*pktoken,*fk_cols_start,*pk_cols_start;
        char       *comment_token, ref_token[NAME_LEN+1];
        char       *pkcomment,*fkcomment;
        uint       key_seq,pk_length,fk_length;
        
        if (!(comment_token= strchr(row[comment_id],';')))
          continue; /* MySQL 4.1 and above, the comment field is '15' */

        do 
        {
          /*       
            Found reference information in comment field from InnoDB type, 
            and parse the same to get the FK information .. 
          */
          key_seq= 1;

          if (!(token= my_next_token(NULL,&comment_token,NULL,'(')))
            break;
          fk_cols_start = token;

          if (!(token= my_next_token(token,&comment_token,ref_token,')')))
            continue;        
          fk_length= (uint)((token-1)-fk_cols_start);

          if (!(token= my_next_token(token+7,&comment_token,ref_token,'/')))
            continue;
          data[0]= strdup_root(alloc,ref_token); /* PKTABLE_CAT */

          if (!(token= my_next_token(token,&comment_token,ref_token,'(')) ||
              myodbc_casecmp(PkTableName,ref_token,strlen(PkTableName)))
            continue;

          data[2]= strdup_root(alloc,ref_token); /* PKTABLE_TABLE */        
          pk_cols_start = token;
        
          if (!(token= my_next_token(token,&comment_token,ref_token,')')))
            continue;
          pk_length= (uint)((token-1)-pk_cols_start);
             
          data[1]= "";                           /* PKTABLE_SCHEM */
          data[4]= strdup_root(alloc,FkTableQualifier); /* FKTABLE_CAT */
          data[5]= "";                           /* FKTABLE_SCHEM */
          data[6]= row[0];                       /* FKTABLE_TABLE */   
 
          /* 
             TODO : FIX both UPDATE_RULE and DELETE_RULE after 
             3.23.52 is released, which supports this feature in 
             server by updating the 'comment' field as well as 
             from SHOW CREATE TABLE defination..

             right now return only SQL_CASCADE as the DELETE/UPDATE 
             rule
          */ 

          data[9]=  "1"; /*SQL_CASCADE*/        /* UPDATE_RULE */ 
          data[10]= "1"; /*SQL_CASCADE*/        /* DELETE_RULE */ 
          data[11]= "NULL";                     /* FK_NAME */
          data[12]= "NULL";                     /* PK_NAME */
          data[13]= "7"; /*SQL_NOT_DEFERRABLE*/ /* DEFERRABILITY */

          token = fkcomment = (char *)fk_cols_start; 
          pktoken = pkcomment = (char *)pk_cols_start;
          fkcomment[fk_length]= '\0';
          pkcomment[pk_length]= '\0';
    
          while (token= my_next_token(token,&fkcomment,ref_token,' '))
          {
            /* Multiple columns exists .. parse them to individual rows */
            char **prev_data= data;
            data[7]= strdup_root(alloc,ref_token);    /* FKTABLE_COLUMN */
            pktoken= my_next_token(pktoken,&pkcomment,ref_token,' ');
            data[3]= strdup_root(alloc,ref_token);    /* PKTABLE_COLUMN */
            sprintf(ref_token,"%d",key_seq++);
            data[8]= strdup_root(alloc,ref_token);    /* KEY_SEQ */
            data+= SQLFORE_KEYS_FIELDS;
            row_count++;
            for (fk_length= SQLFORE_KEYS_FIELDS; fk_length--;)
              data[fk_length]= prev_data[fk_length];
          }                
          data[7]= strdup_root(alloc,fkcomment);      /* FKTABLE_COLUMN */ 
          data[3]= strdup_root(alloc,pkcomment);      /* PKTABLE_COLUMN */
          sprintf(ref_token,"%d",key_seq);
          data[8]= strdup_root(alloc,ref_token);      /* KEY_SEQ */

          data+= SQLFORE_KEYS_FIELDS;
          row_count++;

        } while (comment_token = strchr(comment_token,';'));/* multi table ref */
      }
    } 
  }
  else /* NO FOREIGN KEY support from SERVER */
  {     
    stmt->result=(MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
    stmt->result->eof=1;
  }  
  stmt->result->row_count= row_count;
  mysql_link_fields(stmt,SQLFORE_KEYS_fields,SQLFORE_KEYS_FIELDS);
  DBUG_PRINT("info ",("total keys count: %ld", row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS);
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLFORE_KEYS_values,
                                            sizeof(SQLFORE_KEYS_values), 
                                            MYF(0)); 
  mysql_link_fields(stmt,SQLFORE_KEYS_fields,SQLFORE_KEYS_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}

/*
****************************************************************************
SQLProcesures and SQLProcedureColumns
****************************************************************************
*/

/*
  @type    : internal
  @purpose : returns tables from a perticular database
*/
static MYSQL_RES *mysql_list_sysprocs(DBC FAR *dbc, 
                                      const char *wild) 
{
  MYSQL FAR *mysql= &dbc->mysql;
  char buff[NAME_LEN+50];  

  strxmov(buff,"SELECT name FROM mysql.proc WHERE type='Procedure' and name",NullS);
  my_append_wild(strmov(buff,buff),buff+sizeof(buff)-1,wild);

  MYLOG_DBC_QUERY(dbc, buff);
  if (mysql_query(mysql,buff))
    return 0;
  return mysql_store_result(mysql);
}

char *SQLPROCEDURES_values[]= {"","",NULL,0,0,0,"","procedure"};

MYSQL_FIELD SQLPROCEDURES_fields[]= 
{
  {"PROCEDURE_CAT",    "Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,FIELD_TYPE_VAR_STRING},
  {"PROCEDURE_SCHEM",  "Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,FIELD_TYPE_VAR_STRING},
  {"PROCEDURE_NAME",   "Catalog",NullS,NullS,NullS,NAME_LEN,0,0,0,FIELD_TYPE_VAR_STRING},
  {"NUM_INPUT_PARAMS", "Catalog",NullS,NullS,NullS,11,11,0,0,FIELD_TYPE_LONG},
  {"NUM_OUTPUT_PARAMS","Catalog",NullS,NullS,NullS,11,11,0,0,FIELD_TYPE_LONG},
  {"NUM_RESULT_SETS",  "Catalog",NullS,NullS,NullS,11,11,0,0,FIELD_TYPE_LONG},
  {"REMARKS",          "Catalog",NullS,NullS, NullS, NAME_LEN, 11, 0, 0, FIELD_TYPE_VAR_STRING},
  {"PROCEDURE_TYPE",   "Catalog",NullS,NullS,NullS,2,2,0,0,FIELD_TYPE_SHORT}
};

const uint SQLPROCEDURES_FIELDS= array_elements(SQLPROCEDURES_values);
uint SQLPROCEDURES_order[]= {2};

/*
  @type    : ODBC 1.0 API
  @purpose : returns the list of procedure names stored in a specific data
  source. Procedure is a generic term used to describe an
  executable object, or a named entity that can be invoked
  using input and output parameters
*/

SQLRETURN SQL_API SQLProcedures(SQLHSTMT    hstmt,
                                SQLCHAR FAR *szProcQualifier,
                                SQLSMALLINT cbProcQualifier,
                                SQLCHAR FAR *szProcOwner,
                                SQLSMALLINT cbProcOwner,
                                SQLCHAR FAR *szProcName,
                                SQLSMALLINT cbProcName)
{
  char      Qualifier_buff[NAME_LEN+1],
            Name_buff[NAME_LEN+1], 
            *ProcQualifier,
            *ProcName;
  STMT FAR  *stmt= (STMT FAR*) hstmt; 

  DBUG_ENTER("SQLProcedures");

  DBUG_PRINT("enter",("Qualifier: '%s'(%d)  Owner: '%s'(%d)  Procedure: '%s'(%d)",
          szProcQualifier ? (char*) szProcQualifier : "null", cbProcQualifier,
          szProcOwner ? (char*) szProcOwner : "null", cbProcOwner,
          szProcName ? (char*) szProcName : "null", cbProcName));

  CLEAR_STMT_ERROR(hstmt);
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  if (!is_minimum_version(stmt->dbc->mysql.server_version,"5.0",3))
  {
    DBUG_PRINT("error",("%s", "Driver doesn't support for this MySQL server version, upgrade to >= 5.0"));
    goto empty_set;
  }
  my_SQLFreeStmt(hstmt,MYSQL_RESET);

  ProcQualifier= myodbc_get_valid_buffer((char FAR *)Qualifier_buff,szProcQualifier,cbProcQualifier);
  ProcName= myodbc_get_valid_buffer((char FAR*)Name_buff,szProcName,cbProcName);

  escape_input_parameter(&stmt->dbc->mysql, ProcQualifier);
  escape_input_parameter(&stmt->dbc->mysql, ProcName); 

  pthread_mutex_lock(&stmt->dbc->lock);
  stmt->result= mysql_list_sysprocs(stmt->dbc,ProcName);
  pthread_mutex_unlock(&stmt->dbc->lock);
  
  if (!stmt->result)
  {
    DBUG_PRINT("error",("%d:%s",mysql_errno(&stmt->dbc->mysql),
                                mysql_error(&stmt->dbc->mysql)));
    goto empty_set;
  }
  stmt->order      = SQLPROCEDURES_order;
  stmt->order_count= array_elements(SQLPROCEDURES_order);
  stmt->fix_fields = fix_fields_copy;
  stmt->array      = (MYSQL_ROW) my_memdup(
                       (gptr)SQLPROCEDURES_values,
                       sizeof(SQLPROCEDURES_values),MYF(0));

  mysql_link_fields(stmt,SQLPROCEDURES_fields,SQLPROCEDURES_FIELDS);  
  DBUG_PRINT("info ",("total procedures count: %ld", stmt->result->row_count));
  DBUG_RETURN_STATUS(SQL_SUCCESS);
  
empty_set:
  DBUG_PRINT("info ",("Can't match anything; Returning empty set"));
  stmt->result= (MYSQL_RES*) my_malloc(sizeof(MYSQL_RES),MYF(MY_ZEROFILL));
  stmt->result->row_count= 0;
  stmt->result_array= (MYSQL_ROW) my_memdup((gptr) SQLPROCEDURES_values,
                                            sizeof(SQLPROCEDURES_values), 
                                            MYF(0)); 
  mysql_link_fields(stmt,SQLPROCEDURES_fields,SQLPROCEDURES_FIELDS);
  DBUG_RETURN_STATUS(SQL_SUCCESS);
}

/*
  @type    : ODBC 1.0 API
  @purpose : returns the list of input and output parameters, as well as
  the columns that make up the result set for the specified
  procedures. The driver returns the information as a result
  set on the specified statement
*/

SQLRETURN SQL_API
SQLProcedureColumns(SQLHSTMT hstmt,
		    SQLCHAR FAR *szProcQualifier __attribute__((unused)),
		    SQLSMALLINT cbProcQualifier __attribute__((unused)),
		    SQLCHAR FAR *szProcOwner __attribute__((unused)),
		    SQLSMALLINT cbProcOwner __attribute__((unused)),
		    SQLCHAR FAR *szProcName __attribute__((unused)),
		    SQLSMALLINT cbProcName __attribute__((unused)),
		    SQLCHAR FAR *szColumnName __attribute__((unused)),
		    SQLSMALLINT cbColumnName __attribute__((unused)))
{
  return set_error(hstmt,MYERR_S1000,
		   "Driver doesn't support this yet",4000);
}
