/* Copyright (C) 2000-2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MYODBCSetup.h"

/*!
    \internal
    \brief      Parse state.

                These are the different states we can be in while parsing 
                an attributes string.
*/  
typedef enum tMYODBC_ATTR_PARSE_STATE
{
  MYODBC_ATTR_PARSE_STATE_NAME_START,   /* looking for start of name  */
  MYODBC_ATTR_PARSE_STATE_NAME,         /* looking for end of name    */
  MYODBC_ATTR_PARSE_STATE_EQUAL,        /* looking for equal sign     */
  MYODBC_ATTR_PARSE_STATE_VALUE_START,  /* looking for start of value */
  MYODBC_ATTR_PARSE_STATE_VALUE         /* looking for end of value   */
  
} MYODBC_ATTR_PARSE_STATE;

#ifndef Q_WS_X11
char *strndup( const char *s, size_t n )
{
    size_t nAvail;
    char *p;

    if ( !s )
        return 0;

    nAvail = min( strlen(s) + 1, n + 1 );
    p      = malloc( nAvail );
    memcpy( p, s, nAvail );
    p[nAvail - 1] = '\0';

/*
printf( "[PAH][%s][%d] nAvail=%d p=(%s)\n", __FILE__, __LINE__, nAvail, p );
*/
    return p; 
}
#endif

/*!
    \internal
    \brief      Parse a string of "name=value" pairs.

                ConfigDSN receives connection information from the installer DLL 
                as a list of attributes in the form of keyword-value pairs. Each 
                pair is terminated with a null byte, and the entire list is 
                terminated with a null byte. (That is, two null bytes mark the 
                end of the list.) Spaces are not allowed around the equal sign in 
                the keyword-value pair.  
*/  
MYODBC_NAMEVALUE **MYODBCSetupGetAttributesParsed( LPCSTR pszAttributes )
{
  MYODBC_NAMEVALUE **     aNameValues    = 0;
  MYODBC_ATTR_PARSE_STATE nState         = MYODBC_ATTR_PARSE_STATE_NAME_START;
  char *                  pAnchorChar    = (char *)pszAttributes;
  char *                  pScanChar      = (char *)pszAttributes;
  int                     nLastAttr      = 0;

  /* we always have a null terminator as last pointer in array */
  aNameValues = (MYODBC_NAMEVALUE **)malloc( sizeof(MYODBC_NAMEVALUE *) );
  aNameValues[nLastAttr] = 0;

  /* short circuit if we have not been given stuff to parse */
  if ( !pszAttributes )
    return aNameValues;

  /* scan the input */
  while ( 1 )
  {
    switch ( nState )
    {
      case MYODBC_ATTR_PARSE_STATE_NAME_START:
        {
          if ( isalpha( *pScanChar ) )
          {
            pAnchorChar = pScanChar;
            nState = MYODBC_ATTR_PARSE_STATE_NAME; 
          }
        }
        break;
      case MYODBC_ATTR_PARSE_STATE_NAME:
        {
          if ( (!isalpha( *pScanChar ) && !isdigit( *pScanChar )) || *pScanChar == '=' )
          {
            /* use last attr for current parse data */
            aNameValues[nLastAttr] = (MYODBC_NAMEVALUE*)malloc( sizeof(MYODBC_NAMEVALUE) );
            aNameValues[nLastAttr]->pszName = (char *)strndup( pAnchorChar, pScanChar - pAnchorChar );
            aNameValues[nLastAttr]->pszValue= 0; 

            /* add a new null attr to array for termination */
            nLastAttr++;
            aNameValues = (MYODBC_NAMEVALUE**)realloc( aNameValues, sizeof(MYODBC_NAMEVALUE*) * (nLastAttr + 1) );
            aNameValues[nLastAttr] = 0;

            if ( *pScanChar == '=' )
              nState = MYODBC_ATTR_PARSE_STATE_VALUE_START;
            else
              nState = MYODBC_ATTR_PARSE_STATE_EQUAL;       
          }
        }
        break;
      case MYODBC_ATTR_PARSE_STATE_EQUAL:
        {
          if ( *pScanChar == '=' )
            nState = MYODBC_ATTR_PARSE_STATE_VALUE_START;
        }
        break;
      case MYODBC_ATTR_PARSE_STATE_VALUE_START:
        {
          if ( !isspace( *pScanChar ) )
          {
            pAnchorChar = pScanChar;
            nState = MYODBC_ATTR_PARSE_STATE_VALUE;
          }
        }
        break;
      case MYODBC_ATTR_PARSE_STATE_VALUE:
        {
          if ( *pScanChar == '\0' || *pScanChar == ';' )
          {
            aNameValues[nLastAttr - 1]->pszValue = (char *)strndup( pAnchorChar, pScanChar - pAnchorChar );
          }
          /*!
            \note   Linux

                    In some cases on unixODBC a semi-colon will be used
                    to indicate the end of name/value pair. This is like
                    in SQLDriverConnect(). This is incorrect but we try
                    to simply ignore semi-colon and hope rest of format
                    is ok.    
          */
          if ( *pScanChar == ';' )
              pScanChar++;
        }
        break;
      default:
        fprintf( stderr, "[%s][%d][ERROR] Unhandled state.\n", __FILE__, __LINE__ );
        return aNameValues;
    }

    /* '\0' is used to terminate a name/value pair */
    if ( *pScanChar == '\0' )
      nState = MYODBC_ATTR_PARSE_STATE_NAME_START;

    /* two successive '\0' indicate end of data */
    if ( pScanChar[0] == '\0' && pScanChar[1] == '\0' )
      break;

    pScanChar++;

  } /* while scan */

  return aNameValues;
}


