# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2010 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import os
import re
import sys

import ninix.config
import ninix.alias
import ninix.dll

NINIX_HOME = '~/.ninix'
NINIX_USER = None

def print_error(message):
    sys.stderr.write(''.join((message, '\n')))

def get_ninix_home():
    return os.path.expanduser(NINIX_HOME)

def get_ninix_user():
    if NINIX_USER is None:
        return get_ninix_home()
    return os.path.expanduser(NINIX_USER)

def get_pango_fontrc():
    return os.path.join(get_ninix_user(), 'pango_fontrc')

def get_preferences():
    return os.path.join(get_ninix_user(), 'preferences')

def load_config():
    home_dir = get_ninix_home()
    if not os.path.exists(home_dir):
        return None
    ghosts, shells = search_ghosts(home_dir)
    balloons = search_balloons(home_dir)
    plugins = search_plugins(home_dir)
    nekoninni = search_nekoninni(home_dir)
    katochan = search_katochan(home_dir)
    kinoko = search_kinoko(home_dir)
    return ghosts, shells, balloons, plugins, nekoninni, katochan, kinoko

def get_shiori(path):
    table = {}
    shiori_lib = ninix.dll.Library('shiori', path, saori_lib=None)
    for filename in os.listdir(path):
        if is_readable(os.path.join(path, filename)):
            name = None
            if filename.endswith('.py'):
                name =  filename[:-3]
            elif filename.endswith('.pyc'):
                name =  filename[:-4]
            if name and name not in table:
                shiori = shiori_lib.request(('', name))
                if shiori:
                    table[name] = shiori
    return table

def search_ghosts(home_dir, target=None):
    ghosts = []
    shells = []
    if target:
        dirlist = []
        dirlist.extend(target)
    else:
        try:
            dirlist = os.listdir(os.path.join(home_dir, 'ghost'))
        except OSError:
            dirlist = []
    shiori_table = get_shiori(
        os.path.join(os.environ['PYTHONPATH'], 'ninix/dll'))
    for subdir in dirlist:
        prefix = os.path.join(home_dir, 'ghost', subdir)
        ghost_dir = os.path.join(prefix, 'ghost', 'master')
        desc = read_descript_txt(ghost_dir)
        if desc is None:
            desc = ninix.config.null_config()
        shiori_dll = desc.get('shiori')
        # find a pseudo AI, shells, and a built-in balloon
        candidate = {'name': '', 'score': 0}
        # SHIORI compatible modules
        for name, shiori in shiori_table.iteritems():
            score = int(shiori.find(ghost_dir, shiori_dll))
            if score > candidate['score']:
                candidate['name'] = name
                candidate['score'] = score
        shell_name, surface_set = find_surface_set(prefix)
        balloon = find_balloon(prefix)
        if candidate['score'] == 0:
            if surface_set:
                # use as a shell
                shells.append((shell_name, surface_set, balloon))
            continue
        shiori_name = candidate['name']
        if desc.get('name') == 'default':
            pos = 0
        else:
            pos = len(ghosts)
        use_makoto = find_makoto_dll(ghost_dir)
        ghosts.insert(pos, (
            desc, ghost_dir, use_makoto, surface_set, balloon, prefix,
            shiori_dll, shiori_name))
    return ghosts, shells

def search_balloons(home_dir):
    balloon_dir = os.path.join(home_dir, 'balloon')
    try:
        dirlist = os.listdir(balloon_dir)
    except OSError:
        dirlist = []
    buf = []
    for subdir in dirlist:
        path = os.path.join(balloon_dir, subdir)
        if not os.path.isdir(path):
            continue
        desc = read_descript_txt(path) # REQUIRED
        if not desc:
            continue
        balloon_info = read_balloon_info(path) # REQUIRED
        if not balloon_info:
            continue
        if 'balloon_dir' in balloon_info: # XXX
            print 'Oops: key confliction'
            continue
        else:
            balloon_info['balloon_dir'] = (subdir, ninix.config.null_config())
        if desc.get('name') == 'default':
            pos = 0
        else:
            pos = len(buf)
        buf.insert(pos, (desc, balloon_info))
    return buf

def search_plugins(home_dir):
    buf = []
    plugin_dir = os.path.join(home_dir, 'plugin')
    try:
        dirlist = os.listdir(plugin_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        plugin = read_plugin_txt(os.path.join(plugin_dir, subdir))
        if plugin is None:
            continue
        buf.append(plugin)
    return buf

def search_nekoninni(home_dir): ## FIXME
    buf = []
    skin_dir = os.path.join(home_dir, 'nekodorif/skin')
    try:
        dirlist = os.listdir(skin_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        nekoninni = read_profile_txt(os.path.join(skin_dir, subdir))
        if nekoninni is None:
            continue
        buf.append(nekoninni)
    return buf

def search_katochan(home_dir): ## FIXME
    buf = []
    katochan_dir = os.path.join(home_dir, 'nekodorif/katochan')
    try:
        dirlist = os.listdir(katochan_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        katochan = read_katochan_txt(os.path.join(katochan_dir, subdir))
        if katochan is None:
            continue
        buf.append(katochan)
    return buf

def search_kinoko(home_dir):
    buf = []
    kinoko_dir = os.path.join(home_dir, 'kinoko')
    try:
        dirlist = os.listdir(kinoko_dir)
    except OSError:
        dirlist = []
    for subdir in dirlist:
        kinoko = read_kinoko_ini(os.path.join(kinoko_dir, subdir))
        if kinoko is None:
            continue
        buf.append(kinoko)
    return buf

def read_kinoko_ini(top_dir):
    path = os.path.join(top_dir, 'kinoko.ini')
    kinoko = {}
    kinoko['base'] = 'surface0.png'
    kinoko['animation'] = None
    kinoko['category'] = None
    kinoko['title'] = None
    kinoko['ghost'] = None
    kinoko['dir'] = top_dir
    kinoko['offsetx'] = 0
    kinoko['offsety'] = 0
    kinoko['ontop'] = 0
    kinoko['baseposition'] = 0
    kinoko['baseadjust'] = 0
    kinoko['extractpath'] = None
    kinoko['nayuki'] = None
    if is_readable(path):
        f = open(path)
        line = f.readline()
        if not line.strip() or line.strip() != '[KINOKO]':
            f.close()
            return None
        lineno = 0
        error = None
        for line in f:
            lineno += 1
            if line.endswith(chr(0)): # XXX
                line = line[:-1]
            if not line.strip():
                continue
            if '=' not in line:
                error = 'line %d: syntax error' % lineno
                break
            name, value = [x.strip() for x in line.split('=', 1)]
            if name in ['title', 'ghost', 'category']:
                kinoko[name] = unicode(value, 'Shift_JIS', 'ignore')
            elif name in ['offsetx', 'offsety']:
                kinoko[name] = int(value)
            elif name in ['base', 'animation', 'extractpath']:
                kinoko[name] = value
            elif name in ['ontop', 'baseposition', 'baseadjust']:
                kinoko[name] = int(value)
        f.close()
        if error:
            sys.stderr.write('Error: %s\n%s (skipped)\n' % (error, path))
            return None
    if kinoko['title']:
        return kinoko
    else:
        return None

def read_profile_txt(top_dir): ## FIXME
    path = os.path.join(top_dir, 'profile.txt')
    name = None
    if is_readable(path):
        f = open(path)
        line = f.readline()
        if line:
            name = unicode(line.strip(), 'Shift_JIS', 'ignore')
        f.close()
    if name:
        return (name, top_dir) ## FIXME
    else:
        return None

def read_katochan_txt(top_dir): ## FIXME
    path = os.path.join(top_dir, 'katochan.txt')
    katochan = {}
    katochan['dir'] = top_dir
    if is_readable(path):
        f = open(path)
        name = None
        lineno = 0
        error = None
        for line in f:
            lineno += 1
            if not line.strip():
                continue
            if line.startswith('#'):
                name = line[1:].strip()
                continue
            elif not name:
                error = 'line %d: syntax error' % lineno
                break
            else:
                value = line.strip()
                if name in ['name', 'category']:
                    katochan[name] = unicode(value, 'Shift_JIS', 'ignore')
                if name.startswith('before.script') or \
                        name.startswith('hit.script') or \
                        name.startswith('after.script') or \
                        name.startswith('end.script') or \
                        name.startswith('dodge.script'):
                    ## FIXME: should be array
                    katochan[name] = unicode(value, 'Shift_JIS', 'ignore')
                elif name in ['before.fall.speed', 'before.slide.magnitude',
                              'before.slide.sinwave.degspeed',
                              'before.appear.ofset.x',
                              'before.appear.ofset.y',
                              'hit.waittime', 'hit.ofset.x', 'hit.ofset.y',
                              'after.fall.speed', 'after.slide.magnitude',
                              'after.slide.sinwave.degspeed']:
                    katochan[name] = int(value)
                elif name in ['target',
                              'before.fall.type', 'before.slide.type',
                              'before.wave', 'before.wave.loop',
                              'before.appear.direction',
                              'hit.wave', 'hit.wave.loop',
                              'after.fall.type', 'after.slide.type',
                              'after.wave', 'after.wave.loop',
                              'end.wave', 'end.wave.loop',
                              'end.leave.direction',
                              'dodge.wave', 'dodge.wave.loop']:
                    katochan[name] = value
                else:
                    name = None
        f.close()
        if error:
            sys.stderr.write('Error: %s\n%s (skipped)\n' % (error, path))
            return None
    if katochan['name']:
        return katochan
    else:
        return None

def read_descript_txt(top_dir):
    path = os.path.join(top_dir, 'descript.txt')
    if is_readable(path):
        return ninix.config.create_from_file(path)
    return None

def read_install_txt(top_dir):
    path = os.path.join(top_dir, 'install.txt')
    if is_readable(path):
        return ninix.config.create_from_file(path)
    return None

def read_alias_txt(top_dir):
    path = os.path.join(top_dir, 'alias.txt')
    if is_readable(path):
        return ninix.alias.create_from_file(path)
    return None

def find_makoto_dll(top_dir):
    if is_readable(os.path.join(top_dir, 'makoto.dll')):
        return 1
    return 0

def find_surface_set(top_dir):
    desc = read_descript_txt(os.path.join(top_dir, 'ghost', 'master'))
    if desc:
        shell_name = desc.get('name')
    else:
        shell_name = None
    if not shell_name:
        inst = read_install_txt(top_dir)
        if inst:
            shell_name = inst.get('name')
    surface_set = []
    shell_dir = os.path.join(top_dir, 'shell')
    for name, desc, subdir in find_surface_dir(shell_dir):
        surface_dir = os.path.join(shell_dir, subdir)
        surface_info, alias, tooltips = read_surface_info(surface_dir)
        if surface_info and \
           'surface0' in surface_info and 'surface10' in surface_info:
            if alias is None:
                alias = read_alias_txt(surface_dir)
            surface_set.append((name, surface_dir, desc, alias, surface_info,
                                tooltips))
    return shell_name, surface_set

def find_surface_dir(top_dir):
    buf = []
    path = os.path.join(top_dir, 'surface.txt')
    if os.path.exists(path):
        config = ninix.config.create_from_file(path)
        for name, subdir in config.items():
            desc = read_descript_txt(os.path.join(top_dir, subdir.lower()))
            if desc is None:
                desc = ninix.config.null_config()
            buf.append((name, desc, subdir.lower()))
    else:
        try:
            dirlist = os.listdir(top_dir)
        except OSError:
            dirlist = []
        for subdir in dirlist:
            desc = read_descript_txt(os.path.join(top_dir, subdir))
            if desc is None:
                desc = ninix.config.null_config()
            name = desc.get('name', subdir)
            buf.append((name, desc, subdir))
    return buf

re_surface = re.compile('surface([0-9]+)\.(png|dgp|ddp)')

def read_surface_info(surface_dir):
    surface = {}
    try:
        filelist = os.listdir(surface_dir)
    except OSError:
        filelist = []
    filename_alias = {}
    path = os.path.join(surface_dir, 'alias.txt')
    if os.path.exists(path):
        dic = ninix.alias.create_from_file(path)
        for basename, alias in dic.iteritems():
            if basename.startswith('surface'):
                filename_alias[alias] = basename
    # find png image and associated configuration file
    for filename in filelist:
        basename, suffix = os.path.splitext(filename)
        if basename in filename_alias:
            match = re_surface.match(
                ''.join((filename_alias[basename], suffix)))
        else:
            match = re_surface.match(filename)
        if not match:
            continue
        img = os.path.join(surface_dir, filename)
        if not is_readable(img):
            continue
        key = ''.join(('surface', str(int(match.group(1)))))
        txt = os.path.join(surface_dir, ''.join((basename, 's.txt')))
        if is_readable(txt):
            config = ninix.config.create_from_file(txt)
        else:
            config = ninix.config.null_config()
        txt = os.path.join(surface_dir, ''.join((basename, 'a.txt')))
        if is_readable(txt):
            config.update(ninix.config.create_from_file(txt))
        surface[key] = (img, config)
    # find surfaces.txt
    alias = None
    tooltips = {}
    for key, config in read_surfaces_txt(surface_dir):
        if key == '__alias__':
            alias = config
        elif key == '__tooltips__':
            tooltips = config
        elif key.startswith('surface'):
            try:
                img, null_config = surface[key]
            except KeyError:
                img = None
            surface[key] = (img, config)
    # find surface elements
    for key, (img, config) in surface.items():
        for key, method, filename, x, y in list_surface_elements(config):
            filename = filename.lower()
            basename, suffix = os.path.splitext(filename)
            if basename not in surface:
                surface[basename] = (os.path.join(surface_dir, filename),
                                     ninix.config.null_config())
    return surface, alias, tooltips

def read_surfaces_txt(surface_dir):
    config_list = []
    path = os.path.join(surface_dir, 'surfaces.txt')
    try:
        f = open(path)
    except IOError:
        return config_list
    alias_buffer = []
    tooltips = {}
    charset = 'Shift_JIS'
    while 1:
        line = f.readline()
        if not line:
            break
        if line.startswith('#') or line.startswith('//'):
            continue
        key = line.strip()
        if key.startswith('charset'):
            try:
                charset = line.split(',', 1)[1].strip()
            except:
                pass
            continue
        if not key:
            continue
        try:
            while 1:
                line = f.readline()
                if not line:
                    raise ValueError, 'unexpected end of file'
                line = line.replace('\x81\x40', '').strip()
                if not line:
                    continue
                elif line == '{':
                    break
                key = line # ignore the preceding key
            buf = []
            while 1:
                line = f.readline()
                if not line:
                    raise ValueError, 'unexpected end of file'
                line = line.replace('\x81\x40', '').strip()
                if not line:
                    continue
                elif line == '}':
                    break
                buf.append(line)
            if key in ['sakura.surface.alias', 'kero.surface.alias']:
                alias_buffer.append(key)
                alias_buffer.append('{')
                alias_buffer.extend(buf)
                alias_buffer.append('}')
            elif key.endswith('.tooltips'):
                try:
                    key = key[:-9]
                except:
                    pass
                value = {}
                for line in buf:
                    line = line.split(',', 1)
                    region, text = [unicode(s.strip(), charset, 'ignore') for s in line]
                    value[region] = text
                tooltips[key] = value
            elif key.startswith('surface'):
                try:
                    key = ''.join((key[:7], str(int(key[7:]))))
                except ValueError:
                    pass
                config_list.append((key, ninix.config.create_from_buffer(buf)))
        except ValueError, error:
            print_error('%s: %s (parsing not completed)' % (path, error))
            break
    if alias_buffer:
        config_list.append(('__alias__', ninix.alias.create_from_buffer(alias_buffer)))
    config_list.append(('__tooltips__', tooltips))
    return config_list

def list_surface_elements(config):
    buf = []
    for n in range(256):
        key = ''.join(('element', str(n)))
        if key not in config:
            break
        spec = [value.strip() for value in config[key].split(',')]
        try:
            method, filename, x, y = spec
            x = int(x)
            y = int(y)
        except ValueError:
            print_error('invalid element spec for %s: %s' % (key, config[key]))
            continue
        buf.append((key, method, filename, x, y))
    return buf

def find_balloon(top_dir):
    inst = read_install_txt(top_dir)
    if inst:
        balloon_dir = inst.get('balloon.directory')
        if balloon_dir:
            path = os.path.join(top_dir, 'ghost', 'master', balloon_dir.lower())
            desc = read_descript_txt(path)
            info = read_balloon_info(path)
            if desc and info:
                return (desc, info)
    return None

re_balloon = re.compile('balloon([skc][0-9]+)\.(png)')
re_annex   = re.compile('(arrow[01]|sstp)\.(png)')

def read_balloon_info(balloon_dir):
    balloon = {}
    try:
        filelist = os.listdir(balloon_dir)
    except OSError:
        filelist = []
    for filename in filelist:
        match = re_balloon.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if match.group(2) != 'png' and \
           is_readable(''.join((img[-3:], 'png'))):
                continue
        if not is_readable(img):
            continue
        key = match.group(1)
        txt = os.path.join(balloon_dir, 'balloon%ss.txt' % key)
        if is_readable(txt):
            config = ninix.config.create_from_file(txt)
        else:
            config = ninix.config.null_config()
        balloon[key] = (img, config)
    for filename in filelist:
        match = re_annex.match(filename)
        if not match:
            continue
        img = os.path.join(balloon_dir, filename)
        if not is_readable(img):
            continue
        key = match.group(1)
        config = ninix.config.null_config()
        balloon[key] = (img, config)
    return balloon

def read_plugin_txt(plugin_dir):
    path = os.path.join(plugin_dir, 'plugin.txt')
    try:
        f = open(path)
    except IOError:
        return None
    charset = 'EUC-JP' # default
    plugin_name = startup = None
    menu_items = []
    error = None
    lineno = 0
    for line in f:
        lineno += 1
        if not line.strip() or line.startswith('#'):
            continue
        if ':' not in line:
            error = 'line %d: syntax error' % lineno
            break
        name, value = [x.strip() for x in line.split(':', 1)]
        if name == 'charset':
            charset = value
        elif name == 'name':
            plugin_name = unicode(value, charset, 'ignore')
        elif name == 'startup':
            startup_list = value.split(',')
            startup_list[0] = os.path.join(plugin_dir, startup_list[0])
            if not os.path.exists(startup_list[0]):
                error = 'line %d: invalid program name' % lineno
                break
            startup = startup_list
        elif name == 'menuitem':
            menuitem_list = unicode(value, charset, 'ignore').split(',')
            if len(menuitem_list) < 2:
                error = 'line %d: syntax error' % lineno
                break
            menuitem_list[1] = os.path.join(plugin_dir, menuitem_list[1])
            if not os.path.exists(menuitem_list[1]):
                error = 'line %d: invalid program name' % lineno
                break
            menu_items.append((menuitem_list[0], menuitem_list[1:]))
        else:
            error = 'line %d: syntax error' % lineno
            break
    else:
        if plugin_name is None:
            error = "the 'name' header field is required"
        elif not startup and not menu_items:
            error = "either 'startup' or 'menuitem' header field is required"
    f.close()
    if error:
        sys.stderr.write('Error: %s\n%s (skipped)\n' % (error, path))
        return None
    return plugin_name, plugin_dir, startup, menu_items

def is_readable(path):
    try:
        open(path).read(64)
    except IOError:
        return 0
    return 1
    
###   TEST   ###

def test():
    import locale
    locale.setlocale(locale.LC_ALL, '')
    global NINIX_HOME
    try:
        NINIX_HOME = os.environ['NINIX_HOME']
    except KeyError:
        pass
    config = load_config()
    if config is None:
        raise SystemExit, 'Home directory not found.\n'
    ghosts, shells, balloons, plugins, nekoninni, katochan, kinoko = config ## FIXME
    # ghosts
    for desc, shiori_dir, use_makoto, surface_set, balloon, prefix, shiori_dll, shiori_name in ghosts: ## FIXME
        print 'GHOST', '=' * 50
        print prefix
        print str(desc).encode('utf-8', 'ignore')
        print shiori_dir
        print shiori_dll
        print shiori_name
        print 'use_makoto =', use_makoto
        if surface_set:
            for name, surface_dir, desc, alias, surface in surface_set:
                print '-' * 50
                print 'surface:', name.encode('utf-8', 'ignore')
                print str(desc).encode('utf-8', 'ignore')
                for k, v in surface.iteritems():
                    print k, '=', v[0]
                    print str(v[1]).encode('utf-8', 'ignore')
                if alias:
                    buf = []
                    for k, v in alias.iteritems():
                        if k in ['sakura.surface.alias', 'kero.surface.alias']:
                            print ''.join((k, ':'))
                            for alias_id, alias_list in v.iteritems():
                                print alias_id, \
                                      ''.join(('= [', ', '.join(alias_list), ']'))
                            print
                        else:
                            buf.append((k, v))
                    if buf:
                        print 'filename alias:'
                        for k, v in buf:
                            print k, '=', v
                        print
        if balloon:
            print '-' * 50
            desc, balloon = balloon
            print str(desc).encode('utf-8', 'ignore')
            for k, v in balloon.iteritems():
                print k, '=', v[0]
                print str(v[1]).encode('utf-8', 'ignore')
    # shells
    for shell_name, surface_set, balloon in shells:
        print 'SHELL', '=' * 50
        print shell_name.encode('utf-8', 'ignore')
        for name, surface_dir, desc, alias, surface in surface_set:
            print '-' * 50
            print 'surface:', name.encode('utf-8', 'ignore')
            print str(desc).encode('utf-8', 'ignore')
            for k, v in surface.iteritems():
                print k, '=', v[0]
                print str(v[1]).encode('utf-8', 'ignore')
            if alias:
                buf = []
                for k, v in alias.iteritems():
                    if k in ['sakura.surface.alias', 'kero.surface.alias']:
                        print ''.join((k, ':'))
                        for alias_id, alias_list in v.iteritems():
                            print alias_id, ''.join(('= [', ', '.join(alias_list), ']'))
                        print
                    else:
                        buf.append((k, v))
                if buf:
                    print 'filename alias:'
                    for k, v in buf:
                        print k, '=', v
                    print
        if balloon:
            print '-' * 50
            desc, balloon = balloon
            print str(desc).encode('utf-8', 'ignore')
            for k, v in balloon.iteritems():
                print k, '=', v[0]
                print str(v[1]).encode('utf-8', 'ignore')
    # balloons
    for desc, balloon in balloons:
        print 'BALLOON', '=' * 50
        print str(desc).encode('utf-8', 'ignore')
        for k, v in balloon.iteritems():
            print k, '=', v[0]
            print str(v[1]).encode('utf-8', 'ignore')
    # plugins
    for plugin_name, plugin_dir, startup, menu_items in plugins:
        print 'PLUGIN', '=' * 50
        print 'name =', plugin_name.encode('utf-8', 'ignore')
        if startup:
            print 'startup =', ''.join(('["', '", "'.join(startup), '"]'))
        for label, argv in menu_items:
            print "menuitem '%s' =" % label.encode('utf-8', 'ignore'),
            print ''.join(('["', '", "'.join(argv), '"]'))
    ## FIXME
    # kinoko
    # nekoninni
    # katochan

if __name__ == '__main__':
    test()
