/*
  uniontable: concatenate multiple NoSQL tables and print them to stdout
  in the form of one larger table. The input tables must all have the
  same No. of columns, i.e. they must be ``union compatible''.

  Copyright (c) 2003,2006 Carlo Strozzi

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 dated June, 1991.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  $Id: uniontable.c,v 1.3 2006/03/10 11:26:13 carlo Exp $

*/

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <sys/file.h>
#include <string.h>
#include "util.h"

#ifndef HELPDIR
#define HELPDIR "/usr/local/nosql/help"
#endif
#ifndef HELPFILE
#define HELPFILE (HELPDIR "/uniontable.txt")
#endif

#ifndef DOCDIR 
#define DOCDIR "/usr/local/nosql/doc"
#endif
#ifndef COPYING
#define COPYING (DOCDIR "/COPYING")
#endif
#ifndef WARRANTY
#define WARRANTY (DOCDIR "/WARRANTY")
#endif

#define MAXFNAME 256

/* global variables */

struct {
  unsigned int header : 1;
  unsigned int valid : 1;
} flags;

static const char *progname = "uniontable";  /* global pointer to argv[0] */


/* Unbuffered input routine */

static int getch(int fd) {
    char c;
    return (read(fd, &c, 1) == 1) ? (unsigned char) c : EOF;
}

static int eusage(void) {
    fprintf(stderr, "Usage: %s [-N] [-f table] table [table ...]\n", progname);
    return 2;
}

int main(int argc, char *argv[]) {

    int i=0, j, ifd, min_args=2;
    char c;
    char h_file[MAXFNAME] = "";
    Strlist *args;

    FILE *ofile = NULL;

    flags.header = 1;
    flags.valid = 0;

    args = strlist();
    slappend(args, "tail");
    slappend(args, "-q");
    slappend(args, "-n");
    slappend(args, "+2");

    while (++i < argc && *argv[i] == '-') {

      min_args++;

      if (!strcmp(argv[i], "-N") ||
                     !strcmp(argv[i], "--no-header")) flags.header = 0;

      else if (!strcmp(argv[i], "-f") ||
                     !strcmp(argv[i], "--first-table")) {

         if (++i >= argc || *argv[i] == '-') exit(eusage());

         if (strlen(argv[i]) >= (MAXFNAME-1)) {
            fprintf(stderr, "%s: header file name too long\n",progname);
            exit(2);
         }

         strcpy(h_file, argv[i]);

         min_args++;
      }

      else if (!strcmp(argv[i], "-o") ||
                     !strcmp(argv[i], "--output")) {

         if (++i >= argc || *argv[i] == '-') exit(eusage());

         min_args++;

         if ((ofile = freopen(argv[i], "w", stdout)) < 0) {
            perror(argv[0]);
            exit(2);
         }
      }

      else if (!strcmp(argv[i], "-h") ||
                     !strcmp(argv[i], "--help")) {

         execlp("grep","grep","-v","^#",HELPFILE,(char *) 0);
         perror("grep");
         exit(1);
      }

      else if (!strcmp(argv[i], "--show-copying")) {
        execlp("cat","cat",COPYING,(char *) 0);
        perror("cat");
        exit(1);
      }
      else if (!strcmp(argv[i], "--show-warranty")) {
        execlp("cat","cat",WARRANTY,(char *) 0);
        perror("cat");
        exit(1);
      }
    }

    if (argc < min_args) exit(eusage());

    if (!*h_file) {
       if (strlen(argv[i]) >= (MAXFNAME-1)) {
          fprintf(stderr, "%s: header file name too long\n",progname);
          exit(2);
       }
       strcpy(h_file, argv[i]);
    }

    if ((ifd = open(h_file, O_RDONLY, 0)) < 0) {
       fprintf(stderr, "%s: unable to open header file %s\n",\
                                        progname, h_file);
       exit(2);
    }

    j=i;
    while (argv[j]) slappend(args, argv[j++]);

    while ((c = getch(ifd)) != EOF) {

          if (c == '\001') flags.valid = 1;
          if (flags.header && !strcmp(argv[i],h_file)) putchar(c);

          if (c == '\n') {
             if (!flags.valid) break;

             fflush(NULL);       /* Make sure the header is output */
             close(ifd);            /* close when no longer needed */

             execvp("tail", sl2vec(args));

             perror("tail");
             exit(2);
          }
    }

    if (!flags.valid) {
       fprintf(stderr,"%s: missing table header in input\n",progname);
       exit(2);
    }

    exit(0);                        /* Not reached */
}

/* EOF */
