//                                               -*- C++ -*-
/**
 *  @file  LowDiscrepancySequenceImplementation.cxx
 *  @brief This class is a top-level class for low discrepancy sequences
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy$
 *  @date:   $LastChangedDate$
 *  Id:      $Id$
 */
#include "LowDiscrepancySequenceImplementation.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Stat
    {
      /**
       * @class LowDiscrepancySequenceImplementation
       */

      typedef Common::NotYetImplementedException NotYetImplementedException;
      typedef Common::InvalidRangeException	 InvalidRangeException;
      typedef Common::InvalidArgumentException	 InvalidArgumentException;

      CLASSNAMEINIT(LowDiscrepancySequenceImplementation);

      /* Constructor with parameters */
      LowDiscrepancySequenceImplementation::LowDiscrepancySequenceImplementation(const UnsignedLong dimension) :
	Base::Common::PersistentObject(),
	dimension_(dimension)
      {
	// Nothing to do
      }


      /* Virtual constructor */
      LowDiscrepancySequenceImplementation * LowDiscrepancySequenceImplementation::clone() const
      {
	return new LowDiscrepancySequenceImplementation(*this);
      }


      /* initialize the sequence */
      void LowDiscrepancySequenceImplementation::initialize(const UnsignedLong dimension)
      {
	throw NotYetImplementedException(HERE); // this function is supposed to be pure virtual
      }


      /* Dimension accessor*/
      UnsignedLong LowDiscrepancySequenceImplementation::getDimension() const
      {
	return dimension_;
      }


      /* Generate a low discrepancy vector of numbers uniformly distributed over [0, 1) */
      LowDiscrepancySequenceImplementation::NumericalPoint LowDiscrepancySequenceImplementation::generate()
      {
	throw NotYetImplementedException(HERE); // this function is supposed to be pure virtual
      }


      /* Generate a sample of pseudo-random vectors of numbers uniformly distributed over [0, 1) */
      NumericalSample LowDiscrepancySequenceImplementation::generate(const UnsignedLong size)
      {
	NumericalSample sequenceSample(size, dimension_);
	for(UnsignedLong i = 0; i < size ; ++i)
	  {
	    const NumericalPoint realization(generate());
	    for (UnsignedLong j = 0; j < dimension_; ++j)
		sequenceSample[i][j] = realization[j];
	  }
	return sequenceSample;
      }


      /* Compute the star discrepancy of a sample uniformly distributed over [0, 1) */
      NumericalScalar LowDiscrepancySequenceImplementation::ComputeStarDiscrepancy(const NumericalSample & sample)
      {
	// computationnaly heavy function : O(N²), let N the size of the sample
	const UnsignedLong size(sample.getSize());

	// discrepancy is the maximum of the local discrepancy
	const NumericalPoint lowerPoint(sample.getDimension());
	NumericalScalar discrepancy(0.0);
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    const NumericalScalar local(ComputeLocalDiscrepancy(sample, Interval(lowerPoint, sample[i])));
	    if(local > discrepancy)
	      discrepancy = local;
	  }
	return discrepancy;
      }


      /* String converter */
      String LowDiscrepancySequenceImplementation::__repr__() const
      {
	OSS oss;
	oss << "class=" << LowDiscrepancySequenceImplementation::GetClassName()
	    << " dimension=" << dimension_;
	return oss;
      }


      /* Compute the local discrepancy of a sample, given a multidimensionnal interval */
      NumericalScalar LowDiscrepancySequenceImplementation::ComputeLocalDiscrepancy(const NumericalSample & sample,
										    const Interval & interval)
      {
	if (sample.getDimension() != interval.getDimension()) throw InvalidArgumentException(HERE) << "Error: the sample must have the same dimension as the given interval.";
	// calculate number of inner points
	const UnsignedLong size(sample.getSize());
	UnsignedLong inPoints(0);
	for(UnsignedLong j = 0; j < size; ++j)
	  if (interval.isNumericallyInside(sample[j])) ++inPoints;
	// The local discrepancy is the absolute difference between the fraction of points
	// that fall into the given interval and its volume
	return fabs(static_cast<NumericalScalar>(inPoints) / size - interval.getNumericalVolume());
      }


    } /* namespace Stat */
  } /* namespace Base */
} /* namespace OpenTURNS */
