//                                               -*- C++ -*-
/**
 *  @file  MarginalTransformationGradient.cxx
 *  @brief Class for the Nataf transformationGradient evaluation for elliptical
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: MarginalTransformationGradient.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "MarginalTransformationGradient.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {



      CLASSNAMEINIT(MarginalTransformationGradient);

      /* Parameter constructor */
      MarginalTransformationGradient::MarginalTransformationGradient(const MarginalTransformationEvaluation & evaluation):
        NumericalMathGradientImplementation(),
	evaluation_(evaluation)
      {
        // Nothing to do
      }

      /* Virtual constructor */
      MarginalTransformationGradient * MarginalTransformationGradient::clone() const
      {
        return new MarginalTransformationGradient(*this);
      }

      /* Gradient */
      MarginalTransformationGradient::Matrix MarginalTransformationGradient::gradient(const NumericalPoint & in) const
      /* throw (InvalidArgumentException, InternalException) */
      {
        const UnsignedLong dimension(getOutputDimension());
        Matrix result(dimension, dimension);
	// (G^{-1} o F)' = F' . G^{-1}' o F = F' / (G' o G^{-1} o F)
        for (UnsignedLong i = 0; i < dimension; ++i)
          {
	    const NumericalScalar inputPDF(evaluation_.inputDistributionCollection_[i].computePDF(in[i]));
	    // Quick rejection step: if the input PDF is zero, the result will be zero, so continue only if the value is > 0
	    if (inputPDF > 0.0)
	      {
		NumericalScalar inputCDF(evaluation_.inputDistributionCollection_[i].computeCDF(in[i]));
		// For accuracy reason, check if we are in the upper tail of the distribution
		const Bool upperTail(inputCDF > 0.5);
		if (upperTail) inputCDF = evaluation_.inputDistributionCollection_[i].computeCDF(in[i], upperTail);
		// The upper tail CDF is defined by CDF(x, upper) = P(X>x)
		// The upper tail quantile is defined by Quantile(CDF(x, upper), upper) = x
		const NumericalPoint  outputQuantile(evaluation_.outputDistributionCollection_[i].computeQuantile(inputCDF, upperTail));
		const NumericalScalar outputPDF(evaluation_.outputDistributionCollection_[i].computePDF(outputQuantile));
		// The output PDF should never be zero here, be it can occure due to some strange rounding error
		if (outputPDF > 0.0) result(i, i) = inputPDF / outputPDF;
	      }
          }
        return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong MarginalTransformationGradient::getInputDimension() const
      /* throw(InternalException) */
      {
        return evaluation_.inputDistributionCollection_.getSize();
      }

      /* Accessor for output point dimension */
      UnsignedLong MarginalTransformationGradient::getOutputDimension() const
      /* throw(InternalException) */
      {
        return evaluation_.outputDistributionCollection_.getSize();
      }

      /* String converter */
      String MarginalTransformationGradient::__repr__() const
      {
        OSS oss;
        oss << "class=" << MarginalTransformationGradient::GetClassName()
	    << " evaluation=" << evaluation_;
        return oss;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

