//                                               -*- C++ -*-
/**
 *  @file  InverseNatafIndependentCopulaHessian.cxx
 *  @brief Class for the InverseNataf transformation evaluation for independent
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: InverseNatafIndependentCopulaHessian.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "InverseNatafIndependentCopulaHessian.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class InverseNatafIndependentCopulaHessian
       *
       * This class offers an interface for the InverseNataf function for independent copula
       */

      CLASSNAMEINIT(InverseNatafIndependentCopulaHessian);

      /* Parameter constructor */
      InverseNatafIndependentCopulaHessian::InverseNatafIndependentCopulaHessian(const UnsignedLong dimension):
	NumericalMathHessianImplementation(),
	dimension_(dimension)
      {
	// Nothing to do
      }

      /* Virtual constructor */
      InverseNatafIndependentCopulaHessian * InverseNatafIndependentCopulaHessian::clone() const
      {
	return new InverseNatafIndependentCopulaHessian(*this);
      }

      /* String converter */
      String InverseNatafIndependentCopulaHessian::__repr__() const
      {
	OSS oss;
	oss << "class=" << InverseNatafIndependentCopulaHessian::GetClassName()
            << " dimension=" << dimension_;
                   
	return oss;
      }

      /*
       * Evaluation
       * The inverse Nataf transform S reads:
       * Si(u) = Phi(ui), where Phi is the CDF of the standard normal distribution
       * Hijk = d2Si/dujduk = Phi''(ui) if i = j = k
       *                    = 0 else
       */
      InverseNatafIndependentCopulaHessian::SymmetricTensor InverseNatafIndependentCopulaHessian::hessian(const NumericalPoint & in) const
	/* throw(InvalidArgumentException, InternalException) */
      {
	SymmetricTensor result(dimension_, dimension_);
	for (UnsignedLong i = 0; i < dimension_; ++i)
	  {
	    NumericalScalar x(in[i]);
	    // 0.3989422804014326779399462 = 1/sqrt(2*Pi)
	    result(i, i, i) = -0.3989422804014326779399462 * x * exp(-0.5 * x * x);
	  }
	return result;
      }

      /* Accessor for input point dimension */
      UnsignedLong InverseNatafIndependentCopulaHessian::getInputDimension() const
	/* throw(InternalException) */
      {
	return dimension_;
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseNatafIndependentCopulaHessian::getOutputDimension() const
	/* throw(InternalException) */
      {
	return dimension_;
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
