//                                               -*- C++ -*-
/**
 *  @file  ComposedCopula.hxx
 *  @brief The class that implements assembly distributions
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: ComposedCopula.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_COMPOSEDCOPULA_HXX
#define OPENTURNS_COMPOSEDCOPULA_HXX

#include "CopulaImplementation.hxx"
#include "Exception.hxx"
#include "PersistentCollection.hxx"
#include "Copula.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class ComposedCopula
       *
       * The class describes the probabilistic concept of copulas
       * made from a collection of copulas joined by an independent copula
       */
      class ComposedCopula 
	: public Model::CopulaImplementation
      {
	CLASSNAME;
      public:

	/** A type for distribution collection */
	typedef Model::CopulaImplementation                    CopulaImplementation;  // required by SWIG
	typedef Model::Copula                                  Copula; // required by SWIG
	typedef Base::Type::Collection<Copula>                 CopulaCollection;
	typedef Base::Type::PersistentCollection<Copula>       CopulaPersistentCollection;
	typedef CopulaImplementation::InvalidArgumentException InvalidArgumentException;
	typedef CopulaImplementation::Indices                  Indices;
	typedef CopulaImplementation::Interval                 Interval;
	typedef CopulaImplementation::NumericalPoint           NumericalPoint;
	typedef CopulaImplementation::NumericalSample          NumericalSample;
	typedef CopulaImplementation::Implementation           Implementation;
	typedef CopulaImplementation::NumericalPointCollection NumericalPointCollection;
	typedef CopulaImplementation::NumericalPointWithDescriptionCollection NumericalPointWithDescriptionCollection;
	typedef CopulaImplementation::IsoProbabilisticTransformation        IsoProbabilisticTransformation;
	typedef CopulaImplementation::InverseIsoProbabilisticTransformation InverseIsoProbabilisticTransformation;
	typedef CopulaImplementation::StorageManager           StorageManager;

	/** Default constructor for save/load methods : 1D distribution with default Uniform marginal and IndependentCopula */
	ComposedCopula();

	/** Default constructor */
	explicit ComposedCopula(const CopulaCollection & coll)
	  /* throw (InvalidArgumentException) */;

	/** Comparison operator */
	Bool operator ==(const ComposedCopula & other) const;

	/** String converter */
	String __repr__() const;


	/** Distribution collection accessor */
	void setCopulaCollection(const CopulaCollection & coll)
	  /* throw (InvalidArgumentException) */;
	const CopulaCollection & getCopulaCollection() const;

	/* Here is the interface that all derived class must implement */

	/** Virtual constructor */
	ComposedCopula * clone() const;

	/** Get one realization of the ComposedCopula */
	NumericalPoint getRealization() const;

	/** Get the DDF of the ComposedCopula */
	using CopulaImplementation::computeDDF;
	NumericalPoint computeDDF(const NumericalPoint & point) const;

	/** Get the PDF of the ComposedCopula */
	using CopulaImplementation::computePDF;
	NumericalScalar computePDF(const NumericalPoint & point) const;

	/** Get the CDF of the ComposedCopula */
	using CopulaImplementation::computeCDF;
	NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

	/** Get the probability content of an interval */
	NumericalScalar computeProbability(const Interval & interval) const;

	/** Get the PDF gradient of the distribution */
	NumericalPoint computePDFGradient(const NumericalPoint & point) const;

	/** Get the CDF gradient of the distribution */
	NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

	/** Get the distribution of the marginal distribution corresponding to indices dimensions */
	using CopulaImplementation::getMarginal;
	Implementation getMarginal(const Indices & indices) const /* throw(InvalidArgumentException) */;

	/** Compute the PDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
	virtual NumericalScalar computeConditionalPDF(const NumericalScalar x, const NumericalPoint & y) const;

	/** Compute the CDF of Xi | X1, ..., Xi-1. x = Xi, y = (X1,...,Xi-1) */
	virtual NumericalScalar computeConditionalCDF(const NumericalScalar x, const NumericalPoint & y) const;

	/** Compute the quantile of Xi | X1, ..., Xi-1, i.e. x such that CDF(x|y) = q with x = Xi, y = (X1,...,Xi-1) */
	virtual NumericalScalar computeConditionalQuantile(const NumericalScalar q, const NumericalPoint & y) const;

	/** Parameters value and description accessor */
	NumericalPointWithDescriptionCollection getParametersCollection() const;

	/** Tell if the distribution has elliptical copula */
	Bool hasEllipticalCopula() const;

	/** Tell if the distribution has independent copula */
	Bool hasIndependentCopula() const;

	/** Method save() stores the object through the StorageManager */
	void save(StorageManager::Advocate & adv) const;

	/** Method load() reloads the object from the StorageManager */
	void load(StorageManager::Advocate & adv);

    
      private:
	/** The collection of distribution of the ComposedCopula */
	CopulaPersistentCollection copulaCollection_;
    
	/** The copula of the ComposedCopula */
	Copula copula_;

      }; /* class ComposedCopula */


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_COMPOSEDCOPULA_HXX */
