//                                               -*- C++ -*-
/**
 *  @file  MultiNomialFactory.cxx
 *  @brief Factory for MultiNomial distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: MultiNomialFactory.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "MultiNomialFactory.hxx"
#include "MultiNomial.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      MultiNomialFactory::MultiNomialFactory():
	DistributionImplementationFactory()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      MultiNomialFactory * MultiNomialFactory::clone() const
      {
	return new MultiNomialFactory(*this);
      }


      /* Here is the interface that all derived class must implement */

      MultiNomialFactory::DistributionImplementation * MultiNomialFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a MultiNomial distribution from an empty sample";
	// Check if each component of the sample is an integer, compute the mean and extract the maximum value
	UnsignedLong size(sample.getSize());
	UnsignedLong dimension(sample.getDimension());
	NumericalPoint p(dimension, 0.0);
	NumericalScalar max(sample[0][0]);
	for (UnsignedLong i = 0; i < size; i++)
	  {
	    for (UnsignedLong j = 0; j < dimension; j++)
	      {
		NumericalScalar x(sample[i][j]);
		if ((x != trunc(x)) || (x < 0.0)) throw InternalException(HERE) << "Error: can build a MultiNomial distribution only from a sample with positive integer components, here sample[" << i << "][" << j << "]=" << x;
		if (x > max) max = x;
		p[j] += x;
	      }
	  }
	UnsignedLong n((UnsignedLong)max);
	p *= 1.0 / (max * size);
	return MultiNomial(p, n).clone();
      }

      MultiNomialFactory::DistributionImplementation * MultiNomialFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
	return MultiNomial().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
