//                                               -*- C++ -*-
/**
 *  @file  Triangular.cxx
 *  @brief The Triangular distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Triangular.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "Triangular.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Base::Stat::RandomGenerator RandomGenerator;

      CLASSNAMEINIT(Triangular);

      static Base::Common::Factory<Triangular> RegisteredFactory("Triangular");

      /* Default constructor */
      Triangular::Triangular()
        : NonEllipticalDistribution("Triangular"),
          a_(-1.0), m_(0.0), b_(1.0)
      {
        setDimension( 1 );
        computeRange();
      }

      /* Parameters constructor */
      Triangular::Triangular(const NumericalScalar a,
                             const NumericalScalar m,
                             const NumericalScalar b)
        /* throw (InvalidArgumentException) */
        : NonEllipticalDistribution("Triangular"),
          a_(0.), m_(0.), b_(0.)
      {
        setAMB(a, m, b);
        setDimension( 1 );
        computeRange();
      }

      /* Comparison operator */
      Bool Triangular::operator ==(const Triangular & other) const {
        Bool sameObject = false;

        if (this != &other) { // Other is NOT me, so I have to realize the comparison
          // sameObject = ...
          // TODO: Write Triangular::operator ==(...)
          sameObject = (a_ == other.a_) && (m_ == other.m_) && (b_ == other.b_);
        } else sameObject = true;

        return sameObject;
      }

      /* String converter */
      String Triangular::__repr__() const {
        OSS oss;
        oss << "class=" << Triangular::GetClassName()
            << " name=" << getName()
            << " dimension=" << getDimension()
            << " a=" << a_
            << " m=" << m_
            << " b=" << b_;
        return oss;
      }

      /* Virtual constructor */
      Triangular * Triangular::clone() const
      {
        return new Triangular(*this);
      }

      /* Compute the numerical range of the distribution given the parameters values */
      void Triangular::computeRange()
      {
        setRange(Interval(a_, b_));
      }

      /* Get one realization of the distribution */
      Triangular::NumericalPoint Triangular::getRealization() const
      {
        const NumericalScalar ma(m_ - a_);
        const NumericalScalar ba(b_ - a_);
        const NumericalScalar prob(RandomGenerator::Generate());
        if (ba * prob < ma) return NumericalPoint(1, a_ + sqrt(prob * ba * ma));
        return NumericalPoint(1, b_ - sqrt((1.0 - prob) * ba * (b_ - m_)));
      }


      /* Get the DDF of the distribution */
      Triangular::NumericalPoint Triangular::computeDDF(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x <= a_) || (x > b_)) return NumericalPoint(1, 0.0);
        const NumericalScalar ddf(2.0 / (b_ - a_));
        if (x < m_) return NumericalPoint(1, ddf / (m_ - a_));
        return NumericalPoint(1, ddf / (m_ - b_));
      }


      /* Get the PDF of the distribution */
      NumericalScalar Triangular::computePDF(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x <= a_) || (x > b_)) return 0.0;
        const NumericalScalar pdf(2.0 / (b_ - a_));
        if (x < m_) return pdf * (x - a_) / (m_ - a_);
        return pdf * (x - b_) / (m_ - b_);
      }


      /* Get the CDF of the distribution */
      NumericalScalar Triangular::computeCDF(const NumericalPoint & point, const Bool tail) const
      {
        const NumericalScalar x(point[0]);
        if (x <= a_) return (tail ? 1.0 : 0.0);
        if (x > b_) return (tail ? 0.0 : 1.0);
        const NumericalScalar cdf(1.0 / (b_ - a_));
	if (tail)
	  {
	    if (x < m_) return 1.0 - (x - a_) * (x - a_) * cdf / (m_ - a_);
	    return (x - b_) * (x - b_) * cdf / (b_ - m_);
	  }
        if (x < m_) return (x - a_) * (x - a_) * cdf / (m_ - a_);
        return 1.0 - (x - b_) * (x - b_) * cdf / (b_ - m_);
      }

      /* Get the characteristic function of the distribution, i.e. phi(u) = E(exp(I*u*X)) */
      NumericalComplex Triangular::computeCharacteristicFunction(const NumericalScalar x,
                                                                 const Bool logScale) const
      {
	if (fabs(x) < pdfEpsilon_) return (logScale ? 0.0 : 1.0);
        const NumericalScalar ba(b_ - a_);
        const NumericalScalar bm(b_ - m_);
        const NumericalScalar ma(m_ - a_);
        if (logScale) return  M_LN2 - 2.0 * log(fabs(x)) + log(-exp(NumericalComplex(0.0, a_ * x)) / (ba * ma) + exp(NumericalComplex(0.0, m_ * x)) / (bm * ma) - exp(NumericalComplex(0.0, b_ * x)) / (ba * bm));
        return 2.0 / (x * x) * (-exp(NumericalComplex(0.0, a_ * x)) / (ba * ma) + exp(NumericalComplex(0.0, m_ * x)) / (bm * ma) - exp(NumericalComplex(0.0, b_ * x)) / (ba * bm));
      }

      /* Get the PDFGradient of the distribution */
      Triangular::NumericalPoint Triangular::computePDFGradient(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x <= a_) || (x > b_)) return NumericalPoint(1, 0.0);
        NumericalPoint pdfGradient(3);
        const NumericalScalar ba(b_ - a_);
        const NumericalScalar ma(m_ - a_);
        const NumericalScalar bm(b_ - m_);
        const NumericalScalar bx(b_ - x);
        const NumericalScalar xa(x - a_);
        const NumericalScalar fact(2.0 / ba);
        if (x < m_)
          {
            pdfGradient[0] = fact * (-a_ * xa + x * ba - m_ * bx) / (ba * ma * ma);
            pdfGradient[1] = -fact * xa / (ma * ma);
            pdfGradient[2] = -fact * xa / (ba * ma);
            return pdfGradient;
          }
        pdfGradient[0] = fact * bx / (bm * ba);
        pdfGradient[1] = fact * bx / (bm * bm);
        pdfGradient[2] = fact * (-b_ * bx - m_ * xa + x * ba) / (ba * bm * bm);
        return pdfGradient;
      }

      /* Get the CDFGradient of the distribution */
      Triangular::NumericalPoint Triangular::computeCDFGradient(const NumericalPoint & point) const
      {
        const NumericalScalar x(point[0]);
        if ((x <= a_) || (x > b_)) return NumericalPoint(1, 0.0);
        NumericalPoint cdfGradient(3);
        const NumericalScalar ba(b_ - a_);
        const NumericalScalar ma(m_ - a_);
        const NumericalScalar bm(b_ - m_);
        const NumericalScalar bx(b_ - x);
        const NumericalScalar xa(x - a_);
        const NumericalScalar xm(x - m_);
        if (x < m_)
          {
            const NumericalScalar fact(xa / (ba * ma));
            cdfGradient[0] = fact * (xm * ba - bx * ma) / (ma * ba);
            cdfGradient[1] = -fact * xa / ma;
            cdfGradient[2] = -xa / ba;
            return cdfGradient;
          }
        const NumericalScalar fact(bx / (ba * bm));
        cdfGradient[0] = -fact * bx / ba;
        cdfGradient[1] = -fact * bx / bm;
        cdfGradient[2] = -fact * (ba * xm + xa * bm) / (bm * ba);
        return cdfGradient;
      }

      /* Get the quantile of the distribution */
      NumericalScalar Triangular::computeScalarQuantile(const NumericalScalar prob,
							const Bool tail,
                                                        const NumericalScalar precision) const
      {
        const NumericalScalar ma(m_ - a_);
        const NumericalScalar ba(b_ - a_);
        const NumericalScalar bm(b_ - m_);
	if (tail)
	  {
	    if (bm < prob * ba) return a_ + sqrt((1.0 - prob) * ba * ma);
	    return b_ - sqrt(prob * ba * (b_ - m_));
	  }
        if (ba * prob < ma) return a_ + sqrt(prob * ba * ma);
        return b_ - sqrt((1.0 - prob) * ba * bm);
      }

      /* Get the roughness, i.e. the L2-norm of the PDF */
      NumericalScalar Triangular::getRoughness() const
      {
        // 1.333333333333333333333333 = 4/3
        return 1.333333333333333333333333 / (b_ - a_);
      }

      /* Get the mean of the distribution */
      Triangular::NumericalPoint Triangular::getMean() const /* throw(NotDefinedException) */
      {
        return NumericalPoint(1, (a_ + m_ + b_) / 3.0);
      }

      /* Get the standard deviation of the distribution */
      Triangular::NumericalPoint Triangular::getStandardDeviation() const /* throw(NotDefinedException) */
      {
        const NumericalScalar ma(m_ - a_);
        const NumericalScalar bm(b_ - m_);
        return NumericalPoint(1, sqrt((bm * bm + bm * ma + ma * ma) / 18.0));
      }

      /* Get the skewness of the distribution */
      Triangular::NumericalPoint Triangular::getSkewness() const /* throw(NotDefinedException) */
      {
        const NumericalScalar ma(m_ - a_);
        const NumericalScalar bm(b_ - m_);
        const NumericalScalar ba(b_ - a_);
        const NumericalScalar den(pow(bm * bm + bm * ma + ma * ma, 1.5));
        NumericalScalar num((ba + ma) * (bm - ma) * (bm + ba));
        // 0.2828427124746190097603378 = sqrt(2) / 5
        return NumericalPoint(1, 0.2828427124746190097603378 * num / den);
      }

      /* Get the kurtosis of the distribution */
      Triangular::NumericalPoint Triangular::getKurtosis() const /* throw(NotDefinedException) */
      {
        // 2.4 = 12/5
        return NumericalPoint(1, 2.4);
      }

      /* Get the covariance of the distribution */
      Triangular::CovarianceMatrix Triangular::getCovariance() const /* throw(NotDefinedException) */
      {
        CovarianceMatrix covariance(1);
        const NumericalScalar ma(m_ - a_);
        const NumericalScalar bm(b_ - m_);
        covariance(0, 0) = (bm * bm + bm * ma + ma * ma) / 18.0;
        return covariance;
      }

      /* Get the moments of the standardized distribution */
      Triangular::NumericalPoint Triangular::getStandardMoment(const UnsignedLong n) const
      {
        const NumericalScalar mu(((m_ - a_) + (m_ - b_)) / (b_ - a_));
        // Even order
        if (n % 2 == 0)
          {
            // Vertical part?
            if (1.0 - fabs(mu) < DefaultPDFEpsilon) return NumericalPoint(1, 1.0 / (n + 1.0));
            // Usual case
            return NumericalPoint(1, 2.0 * (1.0 - pow(mu, n + 2)) / ((n + 1.0) * (n + 2.0) * (1.0 - mu) * (1.0 + mu)));
          }
        // Odd order
        // Vertical part?
        if (1.0 - fabs(mu) < DefaultPDFEpsilon) return NumericalPoint(1, 1.0 / (n + 2.0));
        return NumericalPoint(1, 2.0 * mu * (1.0 - pow(mu, n + 1)) / ((n + 1.0) * (n + 2.0) * (1.0 - mu) * (1.0 + mu)));
      }

      /* Parameters value and description accessor */
      Triangular::NumericalPointWithDescriptionCollection Triangular::getParametersCollection() const
      {
        NumericalPointWithDescriptionCollection parameters(1);
        NumericalPointWithDescription point(3);
        Description description(point.getDimension());
        point[0] = a_;
        point[1] = m_;
        point[2] = b_;
        description[0] = "a";
        description[1] = "m";
        description[2] = "b";
        point.setDescription(description);
        point.setName(getDescription()[0]);
        parameters[0] = point;
        return parameters;
      }

      void Triangular::setParametersCollection(const NumericalPointCollection & parametersCollection)
      {
        *this = Triangular(parametersCollection[0][0], parametersCollection[0][1], parametersCollection[0][2]);
      }

      /* Interface specific to Triangular */

      /* A accessor */
      void Triangular::setAMB(const NumericalScalar a,
                              const NumericalScalar m,
                              const NumericalScalar b)
        /* throw (InvalidArgumentException) */
      {
        if ( (m < a) || (m > b) || (b < a) ) throw InvalidArgumentException(HERE) << "M MUST be inclosed between a and b";
        a_ = a;
        m_ = m;
        b_ = b;
        computeRange();
      }

      NumericalScalar Triangular::getA() const
      {
        return a_;
      }


      /* M accessor */
      NumericalScalar Triangular::getM() const
      {
        return m_;
      }


      /* B accessor */
      NumericalScalar Triangular::getB() const
      {
        return b_;
      }


      /* Method save() stores the object through the StorageManager */
      void Triangular::save(StorageManager::Advocate & adv) const
      {
        NonEllipticalDistribution::save(adv);
        adv.saveAttribute( "a_", a_ );
        adv.saveAttribute( "m_", m_ );
        adv.saveAttribute( "b_", b_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void Triangular::load(StorageManager::Advocate & adv)
      {
        NonEllipticalDistribution::load(adv);
        adv.loadAttribute( "a_", a_ );
        adv.loadAttribute( "m_", m_ );
        adv.loadAttribute( "b_", b_ );
        computeRange();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
