//                                               -*- C++ -*-
/**
 *  @file  TemporalFunction.cxx
 *  @brief Abstract top-level class for all distributions
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2011-05-24 19:30:41 +0200 (mar. 24 mai 2011) $
 *  Id:      $Id: TemporalFunction.cxx 1910 2011-05-24 17:30:41Z schueller $
 */
#include "TemporalFunction.hxx"
#include "PersistentObjectFactory.hxx"
#include "NumericalMathEvaluationImplementation.hxx"
#include "NoNumericalMathEvaluationImplementation.hxx"

BEGIN_NAMESPACE_OPENTURNS



CLASSNAMEINIT(TemporalFunction);

static Factory<TemporalFunction> RegisteredFactory("TemporalFunction");

/* Default constructor */
TemporalFunction::TemporalFunction()
  : DynamicalFunctionImplementation(),
    p_evaluation_(new NoNumericalMathEvaluationImplementation)
{
  // Nothing to do
}

/* Parameter constructor */
TemporalFunction::TemporalFunction(const NumericalMathFunction & function)
  : DynamicalFunctionImplementation(),
    p_evaluation_(function.getEvaluationImplementation())
{
  // Set the descriptions
  setInputDescription(p_evaluation_->getInputDescription());
  setOutputDescription(p_evaluation_->getOutputDescription());
}

/* Parameter constructor */
TemporalFunction::TemporalFunction(const EvaluationImplementation & p_evaluation)
  : DynamicalFunctionImplementation(),
    p_evaluation_(p_evaluation)
{
  // Set the descriptions
  setInputDescription(p_evaluation_->getInputDescription());
  setOutputDescription(p_evaluation_->getOutputDescription());
}

/* Parameter constructor */
TemporalFunction::TemporalFunction(const NumericalMathEvaluationImplementation & evaluation)
  : DynamicalFunctionImplementation(),
    p_evaluation_(evaluation.clone())
{
  // Set the descriptions
  setInputDescription(p_evaluation_->getInputDescription());
  setOutputDescription(p_evaluation_->getOutputDescription());
}

/* Virtual constructor */
TemporalFunction * TemporalFunction::clone() const
{
  return new TemporalFunction(*this);
}

/* Comparison operator */
Bool TemporalFunction::operator ==(const TemporalFunction & other) const
{
  return true;
}

/* String converter */
String TemporalFunction::__repr__() const
{
  OSS oss;
  oss << "class=" << TemporalFunction::GetClassName()
      << " evaluation=" << p_evaluation_->__repr__();
  return oss;
}

/* String converter */
String TemporalFunction::__str__(const String & offset) const
{
  return OSS(false) << offset << p_evaluation_->__str__();
}

/* Operator () */
TimeSeries TemporalFunction::operator() (const TimeSeries & inTS) const
{
  ++callsNumber_;
  return TimeSeries(inTS.getTimeGrid(), (*p_evaluation_)(inTS.getImplementation()->asNumericalSample()));
}

/* Get the i-th marginal function */
TemporalFunction * TemporalFunction::getMarginal(const UnsignedLong i) const
{
  if (i >= getOutputDimension()) throw InvalidArgumentException(HERE) << "Error: the index of a marginal function must be in the range [0, outputDimension-1]";
  return new TemporalFunction(p_evaluation_->getMarginal(i));
}

/* Get the function corresponding to indices components */
TemporalFunction * TemporalFunction::getMarginal(const Indices & indices) const
{
  if (!indices.check(getOutputDimension() - 1)) throw InvalidArgumentException(HERE) << "Error: the indices of a marginal function must be in the range [0, outputDimension-1] and  must be different";
  return new TemporalFunction(p_evaluation_->getMarginal(indices));
}

/* Method save() stores the object through the StorageManager */
void TemporalFunction::save(Advocate & adv) const
{
  PersistentObject::save(adv);
  adv.saveAttribute( "evaluation_", *p_evaluation_ );
}

/* Evaluation accessor */
TemporalFunction::EvaluationImplementation TemporalFunction::getEvaluation() const
{
  return p_evaluation_;
}

/* Method load() reloads the object from the StorageManager */
void TemporalFunction::load(Advocate & adv)
{
  TypedInterfaceObject<NumericalMathEvaluationImplementation> evaluationValue;
  PersistentObject::load(adv);
  adv.loadAttribute( "evaluation_", evaluationValue );
  p_evaluation_ = evaluationValue.getImplementation();
}




END_NAMESPACE_OPENTURNS
