//                                               -*- C++ -*-
/**
 *  @file  Curve.cxx
 *  @brief Curve class for curve plots
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: Curve.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include "Curve.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



CLASSNAMEINIT(Curve);

static Factory<Curve> RegisteredFactory("Curve");

/* Default constructor */
Curve::Curve(const String & legend):
  DrawableImplementation(NumericalSample(2, 0), legend),
  showPoints_(false)
{
  // Nothing to do
}

/* Default constructor */
Curve::Curve(const NumericalSample & data,
             const String & legend):
  DrawableImplementation(NumericalSample(2, 0), legend),
  showPoints_(false)
{
  NumericalSample dataFull;
  // If data is unidimensional, assume that it means Y values with indices as X values
  if (data.getDimension() == 1)
    {
      const UnsignedLong size(data.getSize());
      dataFull = NumericalSample(size, 2);
      for (UnsignedLong i = 0; i < size; ++i)
        {
          dataFull[i][0] = i;
          dataFull[i][1] = data[i][0];
        }
    }
  else dataFull = data;
  // Check data validity
  setData(dataFull);
}

/* Contructor from 2 data sets */
Curve::Curve(const NumericalSample & dataX,
             const NumericalSample & dataY,
             const String & legend):
  DrawableImplementation(NumericalSample(2, 0), legend),
  showPoints_(false)
{
  const UnsignedLong size(dataX.getSize());
  if (dataY.getSize() != size) throw InvalidArgumentException(HERE) << "Error: cannot build a Curve based on two numerical samples with different size.";
  if ((dataX.getDimension() != 1) || (dataY.getDimension() != 1)) throw InvalidArgumentException(HERE) << "Error: cannot build a Curve based on two numerical samples of dimension greater than 1.";
  NumericalSample dataFull(size, 2);
  for (UnsignedLong i = 0; i < size; ++i)
    {
      dataFull[i][0] = dataX[i][0];
      dataFull[i][1] = dataY[i][0];
    }
  // Check data validity
  setData(dataFull);
}

/* Constructor with parameters */
Curve::Curve(const NumericalSample & data,
             const String & color,
             const String & lineStyle,
             const UnsignedLong lineWidth,
             const String & legend,
             const Bool & showPoints)
  : DrawableImplementation(NumericalSample(2, 0), legend),
    showPoints_(showPoints)
{
  NumericalSample dataFull;
  // If data is unidimensional, assume that it means Y values with indices as X values
  if (data.getDimension() == 1)
    {
      const UnsignedLong size(data.getSize());
      dataFull = NumericalSample(size, 2);
      for (UnsignedLong i = 0; i < size; ++i)
        {
          dataFull[i][0] = i;
          dataFull[i][1] = data[i][0];
        }
    }
  else dataFull = data;
  // Check data validity
  setData(dataFull);
  setLineStyle(lineStyle);
  setLineWidth(lineWidth);
  setColor(color);
}

/* String converter */
String Curve::__repr__() const
{
  OSS oss;
  oss << "class=" << Curve::GetClassName()
      << " name=" << getName()
      << " derived from " << DrawableImplementation::__repr__();
  return oss;
}

/* Draw method */
String Curve::draw() const
{
  OSS oss;
  // Stores the data in a temporary file
  oss << DrawableImplementation::draw() << "\n";
  // The specific R command for drawing
  oss << "lines(dataOT[,1], dataOT[,2]"
      << ", lty=\"" << lineStyle_
      << "\", col=\"" << color_
      << "\", lwd=" << lineWidth_;
  if (showPoints_)
    {
      const String code((OSS() << getPointCode(pointStyle_)));
      oss << ", type=\"b\""
          << ", pch=" << (pointStyle_ == "dot" ? "\".\"" : code);
    }
  else
    {
      oss << ", type=\"l\"";
    }
  oss << ")";

  return oss;
}

/* Clone method */
Curve * Curve::clone() const
{
  return new Curve(*this);
}
/* Check validity of data */
Bool Curve::isValidData(const NumericalSample & data) const
{
  return (data.getDimension() == 2);
}

/* Show points accessors */
void Curve::setShowPoints(const Bool showPoints)
{
  showPoints_ = showPoints;
}

Bool Curve::getShowPoints() const
{
  return showPoints_;
}

/* Method save() stores the object through the StorageManager */
void Curve::save(Advocate & adv) const
{
  DrawableImplementation::save(adv);
  adv.saveAttribute( "showPoints_", showPoints_ );
}

/* Method load() reloads the object from the StorageManager */
void Curve::load(Advocate & adv)
{
  DrawableImplementation::load(adv);
  adv.loadAttribute( "showPoints_", showPoints_ );
}



END_NAMESPACE_OPENTURNS
