//                                               -*- C++ -*-
/**
 *  @file  Bisection.cxx
 *  @brief Implementation class of the scalar nonlinear solver based on
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: Bisection.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include "Bisection.hxx"
#include <cmath>

BEGIN_NAMESPACE_OPENTURNS




/**
 * @class Bisection
 *
 * This class is an interface for the 1D nonlinear Bisections
 */

CLASSNAMEINIT(Bisection);

/* Default constructor */
Bisection::Bisection():
  SolverImplementation()
{
  // Nothing to do
}

/* Parameter constructor */
Bisection::Bisection(const NumericalScalar absoluteError,
                     const NumericalScalar relativeError,
                     const UnsignedLong maximumFunctionEvaluation):
  SolverImplementation(absoluteError, relativeError, maximumFunctionEvaluation)
{
  // Nothing to do
}

/* Virtual constructor */
Bisection * Bisection::clone() const
{
  return new Bisection(*this);
}

/* String converter */
String Bisection::__repr__() const
{
  OSS oss;
  oss << "class=" << Bisection::GetClassName()
      << " derived from " << SolverImplementation::__repr__();
  return oss;
}

/* Solve attempt to find one root to the equation function(x) = value in [infPoint, supPoint] given function(infPoint) and function(supPoint) with the bisection method */
NumericalScalar Bisection::solve(const NumericalMathFunction & function,
                                 const NumericalScalar value,
                                 const NumericalScalar infPoint,
                                 const NumericalScalar supPoint,
                                 const NumericalScalar infValue,
                                 const NumericalScalar supValue)
{
  if ((function.getInputDimension() != 1) || (function.getOutputDimension() != 1)) throw InvalidDimensionException(HERE) << "Error: bisection method requires a scalar function, here input dimension=" << function.getInputDimension() << " and output dimension=" << function.getOutputDimension();
  UnsignedLong usedFunctionEvaluation(0);
  const UnsignedLong maximumFunctionEvaluation(getMaximumFunctionEvaluation());
  /* We transform function(x) = value into function(x) - value = 0 */
  NumericalScalar a(infPoint);
  NumericalScalar fA(infValue - value);
  NumericalScalar b(supPoint);
  NumericalScalar fB(supValue - value);
  if (fA * fB > 0.0) throw InternalException(HERE) << "Error: bisection method requires that the function takes different signs at the endpoints of the given starting interval, here f(infPoint) - value=" << fA << ", f(supPoint) - value=" << fB;
  if (fA == 0.0) return a;
  if (fB == 0.0) return b;
  NumericalScalar c(a);
  NumericalScalar fC(fA);
  // Main loop
  for (;;)
    {
      // Current error on the root
      const NumericalScalar error(2.0 * getRelativeError() * fabs(c) + 0.5 * getAbsoluteError());
      // Mid-point step
      NumericalScalar delta(0.5 * (b - a));
      // If the current approximation of the root is good enough, return it
      if ((fabs(delta) <= error) || (fC == 0)) break;
      // c is now the mid-point
      c = a + delta;
      // If all the evaluation budget has been spent, return the approximation
      if (usedFunctionEvaluation == maximumFunctionEvaluation) break;
      // New evaluation
      fC = function(NumericalPoint(1, c))[0] - value;
      ++usedFunctionEvaluation;
      // If the function takes a value at middle on the same side of value that at left
      if (fC * fA > 0.0)
        {
          a = c;
          fA = fC;
        }
      else
        {
          b = c;
          fB = fC;
        }
    } // end Bisection loop
  setUsedFunctionEvaluation(usedFunctionEvaluation);
  return c;
}

END_NAMESPACE_OPENTURNS
