//                                               -*- C++ -*-
/**
 *  @file  MeixnerFactory.cxx
 *  @brief Meixner polynomial factory
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2008-05-21 11:21:38 +0200 (Wed, 21 May 2008) $
 *  Id:      $Id: Object.cxx 815 2008-05-21 09:21:38Z dutka $
 */
#include "MeixnerFactory.hxx"
#include "PersistentObjectFactory.hxx"
#include "NegativeBinomial.hxx"
#include "Exception.hxx"

BEGIN_NAMESPACE_OPENTURNS



CLASSNAMEINIT(MeixnerFactory);

static Factory<MeixnerFactory> RegisteredFactory("MeixnerFactory");



/* Default constructor, associated with the default Negative Binomial distribution of parameter 1, 1/2. */
MeixnerFactory::MeixnerFactory()
  : OrthogonalUniVariatePolynomialFactory(NegativeBinomial()),
    r_(1.0),
    p_(0.5)
{
  initializeCache();
}


/* Parameter constructor: lambda is the order of the generalized Meixner polynomial, associated with the Poisson(lambda) distribution */
MeixnerFactory::MeixnerFactory(const NumericalScalar r,
                               const NumericalScalar p)
  : OrthogonalUniVariatePolynomialFactory( NegativeBinomial(r, p) ),
    r_(r),
    p_(p)
{
  if (r <= 0.0) throw InvalidArgumentException(HERE) << "Error: must have r>0 to build Meixner polynomials.";
  if ((p <= 0.0) || (p >= 1.0)) throw InvalidArgumentException(HERE) << "Error: p must be in [0, 1] to build Meixner polynomials.";
  initializeCache();
}

/* Virtual constructor */
MeixnerFactory * MeixnerFactory::clone() const
{
  return new MeixnerFactory(*this);
}


/* Calculate the coefficients of recurrence a0n, a1n, a2n such that
   Pn+1(x) = (a0n * x + a1n) * Pn(x) + a2n * Pn-1(x) */
MeixnerFactory::Coefficients MeixnerFactory::getRecurrenceCoefficients(const UnsignedLong n) const
{
  Coefficients recurrenceCoefficients(3, 0.0);
  if (n == 0)
    {
      const NumericalScalar factor(sqrt(r_ * p_));
      recurrenceCoefficients[0] = (p_ - 1.0) / factor;
      recurrenceCoefficients[1] = factor;
      // Conventional value of 0.0 for recurrenceCoefficients[2]
      return recurrenceCoefficients;
    }
  const NumericalScalar denominator(sqrt(p_ * (n + 1) * (n + r_)));
  recurrenceCoefficients[0] = (p_ - 1.0) / denominator;
  recurrenceCoefficients[1] = (p_ * (n + r_) + n) / denominator;
  recurrenceCoefficients[2] = -sqrt(p_ * n * (n + r_ - 1.0)) / denominator;
  return recurrenceCoefficients;
}

/* R accessor */
NumericalScalar MeixnerFactory::getR() const
{
  return r_;
}

/* P accessor */
NumericalScalar MeixnerFactory::getP() const
{
  return p_;
}

/* String converter */
String MeixnerFactory::__repr__() const
{
  return OSS() << "class=" << getClassName()
               << " r=" << r_
               << " p=" << p_
               << " measure=" << measure_;
}


/* Method save() stores the object through the StorageManager */
void MeixnerFactory::save(Advocate & adv) const
{
  OrthogonalUniVariatePolynomialFactory::save(adv);
  adv.saveAttribute( "r_", r_ );
  adv.saveAttribute( "p_", p_ );
}


/* Method load() reloads the object from the StorageManager */
void MeixnerFactory::load(Advocate & adv)
{
  OrthogonalUniVariatePolynomialFactory::load(adv);
  adv.loadAttribute( "r_", r_ );
  adv.loadAttribute( "p_", p_ );
}

END_NAMESPACE_OPENTURNS
