//                                               -*- C++ -*-
/**
 *  @file  NatafEllipticalCopulaGradient.cxx
 *  @brief Class for the Nataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: lebrun $
 *  @date:   $LastChangedDate: 2012-04-05 17:05:15 +0200 (Thu, 05 Apr 2012) $
 *  Id:      $Id: NatafEllipticalCopulaGradient.cxx 2447 2012-04-05 15:05:15Z lebrun $
 */
#include "NatafEllipticalCopulaGradient.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS

/*
 * @class NatafEllipticalCopulaGradient
 *
 * This class offers an interface for the Nataf function for elliptical copula
 */

CLASSNAMEINIT(NatafEllipticalCopulaGradient);

static Factory<NatafEllipticalCopulaGradient> RegisteredFactory("NatafEllipticalCopulaGradient");

/* Default constructor */
NatafEllipticalCopulaGradient::NatafEllipticalCopulaGradient():
  NumericalMathGradientImplementation(),
  standardDistribution_(),
  inverseCholesky_()
{
  // Nothing to do
}

/* Parameter constructor */
NatafEllipticalCopulaGradient::NatafEllipticalCopulaGradient(const Distribution & standardDistribution,
                                                             const SquareMatrix & inverseCholesky):
  NumericalMathGradientImplementation(),
  standardDistribution_(standardDistribution),
  inverseCholesky_(inverseCholesky)
{
  // Nothing to do
}

/* Virtual constructor */
NatafEllipticalCopulaGradient * NatafEllipticalCopulaGradient::clone() const
{
  return new NatafEllipticalCopulaGradient(*this);
}

/* String converter */
String NatafEllipticalCopulaGradient::__repr__() const
{
  OSS oss;
  oss << "class=" << NatafEllipticalCopulaGradient::GetClassName()
      << " standardDistribution=" << standardDistribution_
      << " inverseCholesky=" << inverseCholesky_;

  return oss;
}

/*
 * Evaluation
 * The elliptical copula has a correlation matrix R. The Nataf transform T reads:
 * Yi(x) = Q(xi), where Q = F^{-1} and F is the CDF of the standard elliptical distribution
 * T(x) = G.Y(x), where G = L^{-1} and L is the Cholesky factor of R: L.L^t = R, L is lower triangular
 * Its Jacobian J is:
 * Jij = dTi/dxj = (G.dY/dx)ij
 *               = (G.diag(Q'(xk)))ij
 *               = (G:,1Q'(x1)|...|G:,nQ'(xn))ij
 *               = GijQ'(xj)
 * thus, (DT)ij = Jji = GjiQ'(xi)
 */
Matrix NatafEllipticalCopulaGradient::gradient(const NumericalPoint & inP) const
{
  UnsignedLong dimension(getInputDimension());
  const Distribution standardMarginal(standardDistribution_.getMarginal(0));
  Matrix result(dimension, dimension);
  for (UnsignedLong i = 0; i < dimension; ++i)
    {
      NumericalScalar quantileDerivative(1.0 / standardMarginal.computePDF(standardMarginal.computeQuantile(inP[i])));
      for (UnsignedLong j = i; j < dimension; ++j) result(i, j) = inverseCholesky_(j, i) * quantileDerivative;
    }
  return result;
}

/* Accessor for input point dimension */
UnsignedLong NatafEllipticalCopulaGradient::getInputDimension() const
{
  return inverseCholesky_.getNbColumns();
}

/* Accessor for output point dimension */
UnsignedLong NatafEllipticalCopulaGradient::getOutputDimension() const
{
  return inverseCholesky_.getNbRows();
}

/* Method save() stores the object through the StorageManager */
void NatafEllipticalCopulaGradient::save(Advocate & adv) const
{
  NumericalMathGradientImplementation::save(adv);
  adv.saveAttribute( "standardDistribution_", standardDistribution_ );
  adv.saveAttribute( "inverseCholesky_", inverseCholesky_ );
}

/* Method load() reloads the object from the StorageManager */
void NatafEllipticalCopulaGradient::load(Advocate & adv)
{
  NumericalMathGradientImplementation::load(adv);
  adv.loadAttribute( "standardDistribution_", standardDistribution_ );
  adv.loadAttribute( "inverseCholesky_", inverseCholesky_ );
}

END_NAMESPACE_OPENTURNS
