//                                               -*- C++ -*-
/**
 *  @file  Burr.cxx
 *  @brief The Burr distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Publicdistributiolib
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: keller $
 *  @date:   @date:   $LastChangedDate: 2011-22-03 11:30:00 +0100 (mer. 23 mars 2011) $
 *  Id:      $Id: Burr.cxx 1473 2011-22-03 11:30:49Z keller $
 */
#include <cmath>
#include "Burr.hxx"
#include "SpecFunc.hxx"
#include "RandomGenerator.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS




CLASSNAMEINIT(Burr);

static Factory<Burr> RegisteredFactory("Burr");

/* Default constructor */
Burr::Burr()
  : NonEllipticalDistribution("Burr"),
    c_(1.0),
    k_(1.0)
{
  setDimension( 1 );
  computeRange();
}

/* Parameters constructor */
Burr::Burr(const NumericalScalar c,
           const NumericalScalar k)
  : NonEllipticalDistribution("Burr"),
    c_(c),
    k_(k)
{
  // We set the dimension of the Burr distribution
  setDimension( 1 );
  computeRange();
}

/* Comparison operator */
Bool Burr::operator ==(const Burr & other) const
{
  if (this == &other) return true;
  return (c_ == other.c_) && (k_ == other.k_);
}

/* String converter */
String Burr::__repr__() const
{
  OSS oss;
  oss << "class=" << Burr::GetClassName()
      << " name=" << getName()
      << " dimension=" << getDimension()
      << " c=" << c_
      << " k=" << k_;
  return oss;
}

String Burr::__str__(const String & offset) const
{
  OSS oss;
  oss << offset << getClassName() << "(c = " << c_ << ", k = " << k_ << ")";
  return oss;
}

/* Virtual constructor */
Burr * Burr::clone() const
{
  return new Burr(*this);
}

/* Compute the numerical range of the distribution given the parameters values */
void Burr::computeRange()
{
  const NumericalPoint lowerBound(1, 0.0);
  const NumericalPoint upperBound(computeUpperBound());
  const Interval::BoolCollection finiteLowerBound(1, true);
  const Interval::BoolCollection finiteUpperBound(1, false);
  setRange(Interval(lowerBound, upperBound, finiteLowerBound, finiteUpperBound));
}

/* Get one realization of the distribution */
NumericalPoint Burr::getRealization() const
{
  return NumericalPoint(1, computeScalarQuantile(RandomGenerator::Generate()));
}


/* Get the DDF of the distribution */
NumericalPoint Burr::computeDDF(const NumericalPoint & point) const
{
  const NumericalScalar x(point[0]);
  if (x < k_) return NumericalPoint(1, 0.0);
  const NumericalScalar xC(pow(x, c_));
  return NumericalPoint(1, -(xC * (c_ * k_ + 1.0) + 1.0 - c_) * computePDF(x) / (x * (1.0 + xC)));
}


/* Get the PDF of the distribution */
NumericalScalar Burr::computePDF(const NumericalPoint & point) const
{
  const NumericalScalar x(point[0]);
  if (x <= 0.0) return 0.0;
  const NumericalScalar logX(log(x));
  return c_ * k_ * exp((c_ - 1.0) * logX - (k_ + 1.0) * log(1.0 + exp(c_ * logX)));
}


/* Get the CDF of the distribution */
NumericalScalar Burr::computeCDF(const NumericalPoint & point, const Bool tail) const
{
  NumericalScalar x(point[0]);
  if (x < 0.0) return (tail ? 1.0 : 0.0);
  if (tail) return exp(-k_ * log(1.0 + exp(c_ * log(x))));
  return 1.0 - exp(-k_ * log(1.0 + exp(c_ * log(x))));
}

/* Get the PDFGradient of the distribution */
NumericalPoint Burr::computePDFGradient(const NumericalPoint & point) const
{
  const NumericalScalar x(point[0]);
  NumericalPoint pdfGradient(2, 0.0);
  if (x <= 0.0) return pdfGradient;
  const NumericalScalar pdf(computePDF(point));
  const NumericalScalar logX(log(x));
  pdfGradient[0] = ((1.0 - (1.0 + k_) / (1.0 + exp(-c_ * logX))) * logX + 1.0 / c_) * pdf;
  pdfGradient[1] = (1.0 / k_ - log(1.0 + pow(x, c_))) * pdf;
  return pdfGradient;
}

/* Get the CDFGradient of the distribution */
NumericalPoint Burr::computeCDFGradient(const NumericalPoint & point) const
{
  const NumericalScalar x(point[0]);
  NumericalPoint cdfGradient(2, 0.0);
  if (x <= 0.0) return cdfGradient;
  const NumericalScalar cdfC(computeCDF(point, true));
  const NumericalScalar xC(pow(x, c_));
  cdfGradient[0] = k_ * xC * log(x) * cdfC / (1.0 + xC);
  cdfGradient[1] = log(1.0 + xC) * cdfC;
  return cdfGradient;
}

/* Get the quantile of the distribution */
NumericalScalar Burr::computeScalarQuantile(const NumericalScalar prob,
                                            const Bool tail,
                                            const NumericalScalar precision) const
{
  if (tail) return exp( log( exp( -log( prob ) / k_ ) - 1.0) / c_ );
  return exp( log( exp( -log( 1.0 - prob ) / k_ ) - 1.0) / c_ );
}

/* Compute the mean of the distribution */
void Burr::computeMean() const
{
  mean_ = getStandardMoment(1);
  isAlreadyComputedMean_ = true;
}

/* Get the standard deviation of the distribution */
NumericalPoint Burr::getStandardDeviation() const
{
  return NumericalPoint(1, getCovariance()(0, 0));
}

/* Get the skewness of the distribution */
NumericalPoint Burr::getSkewness() const
{
  const NumericalScalar mu(getMean()[0]);
  const NumericalScalar sigma(getStandardDeviation()[0]);
  return NumericalPoint(1, (getStandardMoment(3)[0] - 3.0 * mu * sigma * sigma - mu * mu * mu) / (sigma * sigma * sigma));
}

/* Get the kurtosis of the distribution */
NumericalPoint Burr::getKurtosis() const
{
  const NumericalScalar mu(getMean()[0]);
  const NumericalScalar sigma(getStandardDeviation()[0]);
  return NumericalPoint(1, (getStandardMoment(4)[0] - 4.0 * mu * getStandardMoment(3)[0] + 6.0 * sigma * sigma * mu * mu + 3.0 * mu * mu * mu * mu) / (sigma * sigma * sigma * sigma));
}

/* Compute the covariance of the distribution */
void Burr::computeCovariance() const
{
  covariance_ = CovarianceMatrix(1);
  covariance_(0, 0) = getStandardMoment(2)[0] - pow(getMean()[0], 2);
  isAlreadyComputedCovariance_ = true;
}

/* Get the moments of the standardized distribution */
NumericalPoint Burr::getStandardMoment(const UnsignedLong n) const
{
  return NumericalPoint(1, exp(SpecFunc::LogGamma(k_ - n / c_) + SpecFunc::LogGamma(n / c_ + 1.0) - SpecFunc::LogGamma(k_)));
}

/* Parameters value and description accessor */
Burr::NumericalPointWithDescriptionCollection Burr::getParametersCollection() const
{
  NumericalPointWithDescriptionCollection parameters(1);
  NumericalPointWithDescription point(2);
  Description description(point.getDimension());
  point[0] = c_;
  point[1] = k_;
  description[0] = "c";
  description[1] = "k";
  point.setDescription(description);
  point.setName(getDescription()[0]);
  parameters[0] = point;
  return parameters;
}

void Burr::setParametersCollection(const NumericalPointCollection & parametersCollection)
{
  *this = Burr(parametersCollection[0][0], parametersCollection[0][1]);
}

/* C accessor */
void Burr::setC(const NumericalScalar c)
{
  if (c <= 0.0) throw InvalidArgumentException(HERE) << "C MUST be positive";
  if (c_ != c)
    {
      c_ = c;
      isAlreadyComputedMean_ = true;
      isAlreadyComputedCovariance_ = true;
      computeRange();
    }
}

/* C accessor */
NumericalScalar Burr::getC() const
{
  return c_;
}


/* K accessor */
void Burr::setK(const NumericalScalar k)
{
  if (k <= 0.0) throw InvalidArgumentException(HERE) << "K MUST be positive";
  if (k_ != k)
    {
      k_ = k;
      isAlreadyComputedMean_ = true;
      isAlreadyComputedCovariance_ = true;
      computeRange();
    }
}

/* K accessor */
NumericalScalar Burr::getK() const
{
  return k_;
}

/* Method save() stores the object through the StorageManager */
void Burr::save(Advocate & adv) const
{
  NonEllipticalDistribution::save(adv);
  adv.saveAttribute( "c_", c_ );
  adv.saveAttribute( "k_", k_ );
}

/* Method load() reloads the object from the StorageManager */
void Burr::load(Advocate & adv)
{
  NonEllipticalDistribution::load(adv);
  adv.loadAttribute( "c_", c_ );
  adv.loadAttribute( "k_", k_ );
  computeRange();
}

END_NAMESPACE_OPENTURNS
