//                                               -*- C++ -*-
/**
 *  @file  GeometricFactory.cxx
 *  @brief Factory for Geometric distribution
 *
 *  (C) Copyright 2005-2012 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: schueller $
 *  @date:   $LastChangedDate: 2012-02-17 19:35:43 +0100 (Fri, 17 Feb 2012) $
 *  Id:      $Id: GeometricFactory.cxx 2392 2012-02-17 18:35:43Z schueller $
 */
#include <cmath>
#include "GeometricFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS



/* Default constructor */
GeometricFactory::GeometricFactory():
  DistributionImplementationFactory()
{
  // Nothing to do
}

/* Virtual constructor */
GeometricFactory * GeometricFactory::clone() const
{
  return new GeometricFactory(*this);
}


/* Here is the interface that all derived class must implement */

Geometric * GeometricFactory::build(const NumericalSample & sample) const
{
  if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Geometric distribution from an empty sample";
  if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build a Geometric distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
  // Check if all the components of the sample are integers > 0
  UnsignedLong size(sample.getSize());
  NumericalScalar sum(0.0);
  for (UnsignedLong i = 0; i < size; i++)
    {
      NumericalScalar x(sample[i][0]);
      if ((x != trunc(x)) || (x < 1.0)) throw InternalException(HERE) << "Error: can build a Geometric distribution only from a sample with integer components >= 1, here sample[" << i << "][0]=" << x;
      sum += x;
    }
  return Geometric(size / sum).clone();
}

Geometric * GeometricFactory::build(const NumericalPointCollection & parameters) const
{
  try {
    Geometric distribution;
    distribution.setParametersCollection(parameters);
    return distribution.clone();
  }
  catch (InvalidArgumentException & ex)
    {
      throw InternalException(HERE) << "Error: cannot build a Geometric distribution from the given parameters";
    }
}

Geometric * GeometricFactory::build() const
{
  return Geometric().clone();
}

END_NAMESPACE_OPENTURNS
