//                                               -*- C++ -*-
/**
 *  @file  KrigingResult.hxx
 *  @brief The result of a chaos expansion
 *
 *  Copyright (C) 2005-2014 Airbus-EDF-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author dutka
 *  @date   2008-05-21 17:44:02 +0200 (Wed, 21 May 2008)
 */
#ifndef OPENTURNS_KRIGINGRESULT_HXX
#define OPENTURNS_KRIGINGRESULT_HXX

#include "MetaModelResult.hxx"
#include "NumericalPoint.hxx"
#include "NumericalSample.hxx"
#include "Indices.hxx"
#include "Collection.hxx"
#include "PersistentCollection.hxx"
#include "NumericalMathFunction.hxx"
#include "Distribution.hxx"
#include "OrthogonalBasis.hxx"

BEGIN_NAMESPACE_OPENTURNS

/**
 * @class KrigingResult
 *
 * The result of a chaos expansion
 */

class KrigingResult
  : public MetaModelResult
{
  CLASSNAME;

public:

  // friend class Factory<KrigingResult>;

  /** Default constructor */
  KrigingResult();

  /** Parameter constructor */
  KrigingResult(const NumericalMathFunction & model,
                const Distribution & distribution,
                const NumericalMathFunction & transformation,
                const NumericalMathFunction & inverseTransformation,
                const NumericalMathFunction & composedModel,
                const OrthogonalBasis & orthogonalBasis,
                const Indices & I,
                const NumericalSample & alpha_k,
                const NumericalMathFunctionCollection & Psi_k,
                const NumericalPoint & residuals,
                const NumericalPoint & relativeErrors);

  /** Virtual constructor */
  virtual KrigingResult * clone() const;

  /** String converter */
  virtual String __repr__() const;
  virtual String __str__(const String & offset = "") const;

  /** Distribution accessor */
  virtual Distribution getDistribution() const;

  /** IsoProbabilisticTransformation accessor */
  virtual NumericalMathFunction getTransformation() const;

  /** InverseIsoProbabilisticTransformation accessor */
  virtual NumericalMathFunction getInverseTransformation() const;

  /** Composed model accessor */
  virtual NumericalMathFunction getComposedModel() const;

  /** Orthogonal basis accessor */
  virtual OrthogonalBasis getOrthogonalBasis() const;

  /** Indices accessor */
  virtual Indices getIndices() const;

  /** Coefficients accessor */
  virtual NumericalSample getCoefficients() const;

  /** Reduced basis accessor */
  virtual NumericalMathFunctionCollection getReducedBasis() const;

  /** Composed meta model accessor */
  virtual NumericalMathFunction getComposedMetaModel() const;

  /** Method save() stores the object through the StorageManager */
  virtual void save(Advocate & adv) const;

  /** Method load() reloads the object from the StorageManager */
  virtual void load(Advocate & adv);


protected:

private:
  /** The input vector distribution */
  Distribution distribution_;

  /** The isoprobabilistic transformation maps the distribution into the orthogonal measure */
  NumericalMathFunction transformation_;

  /** The inverse isoprobabilistic transformation */
  NumericalMathFunction inverseTransformation_;

  /** The composed model */
  NumericalMathFunction composedModel_;

  /** The orthogonal basis */
  OrthogonalBasis orthogonalBasis_;

  /** The collection of indices */
  Indices I_;

  /** The collection of Alpha_k coefficients */
  NumericalSample alpha_k_;

  /** The collection of vectors upon which we project the (composed) model */
  NumericalMathFunctionPersistentCollection Psi_k_;

  /** Composed meta model */
  NumericalMathFunction composedMetaModel_;

} ; /* class KrigingResult */


END_NAMESPACE_OPENTURNS

#endif /* OPENTURNS_KRIGINGRESULT_HXX */
