/*
 * Copyright (C) 1999-2001  Richard Hult 
 * 
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the 
 * License, or (at your option) any later version. 
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include <gnome.h>
#include <locale.h>
#include <glade/glade.h>
#include "main.h"
#include "simulation.h"
#include "schematic.h"
#include "schematic-view.h" 
#include "dialogs.h"
#include "oregano-utils.h"
#include "oregano-config.h"
#include "sim-engine.h"
#include "plot.h"

typedef struct {
	Schematic      *sm;
	SchematicView  *sv;
	GnomeDialog    *dialog;
	SimEngine      *engine;
	GtkProgressBar *progress;
	int             progress_timeout_id;
} Simulation;

static int      progress_bar_timeout_callback (Simulation *s);
static void     cancel_callback (GtkWidget *widget, Simulation *s);
static void     input_done_callback (SimEngine *engine, Simulation *s);
static void     input_aborted_callback (SimEngine *engine, Simulation *s);
static gboolean simulate_cmd (Simulation *s);

static int 
delete_event_callback (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	return FALSE;
}

gpointer
simulation_new (Schematic *sm)
{
	Simulation *s;

	s = g_new0 (Simulation, 1);
	s->sm = sm;
	s->sv = NULL;

	return s;
}

void
simulation_show (GtkWidget *widget, SchematicView *sv)
{
	GtkWidget *w;
	GladeXML *gui;
	Simulation *s;
	Schematic *sm;
	gchar *msg;

	g_return_if_fail (sv != NULL);

	sm = schematic_view_get_schematic (sv);
	s = schematic_get_simulation (sm);

	/*
	 * Only allow one instance of the dialog box per schematic.
	 */
	if (s->dialog){
		gdk_window_raise (GTK_WIDGET (s->dialog)->window);
 		return;
	}		

	if (oregano.simexec && !g_file_exists (oregano.simexec)) {
		msg = g_strdup_printf (_("Could not find the simulation executable:\n%s\n\n"
					 "This probably means that you have not configured\n"
					 "Oregano properly. Please choose Settings and specify\n"
					 "the path to the simulation executable (e.g. spice),"),
				       oregano.simexec);
		oregano_error (msg);
		g_free (msg);
		return;
	} else if (!oregano.simexec) {
		oregano_error (_("You have not entered a simulation executable.\n"
				 "Please choose Settings and specify which\n"
				 "program to use for simulations."));
		return;
	}

	if (!g_file_exists (OREGANO_GLADEDIR "/simulation.glade")) {
		oregano_error (_("Could not create simulation dialog."));
		return;
	}

	gui = glade_xml_new (OREGANO_GLADEDIR "/simulation.glade", "toplevel");
	if (!gui) {
		oregano_error (_("Could not create simulation dialog."));
		return;
	}

	w = glade_xml_get_widget (gui, "toplevel");
	if (!w) {
		oregano_error (_("Could not create simulation dialog."));
		return;
	}

	s->dialog = GNOME_DIALOG (w);
	gtk_signal_connect (GTK_OBJECT (w), "delete_event",
			    GTK_SIGNAL_FUNC (delete_event_callback), s);

	w = glade_xml_get_widget (gui, "progressbar");
	s->progress = GTK_PROGRESS_BAR (w);
	gtk_progress_bar_update (s->progress, 0.0);
	gtk_widget_draw (w, NULL);	

        gnome_dialog_button_connect (GNOME_DIALOG (s->dialog),
				     0,
				     cancel_callback,
				     s);

	gnome_dialog_set_parent (GNOME_DIALOG (s->dialog), GTK_WINDOW (sv->toplevel));
	gtk_widget_show_all (GTK_WIDGET (s->dialog));

	s->sv = sv;
	simulate_cmd (s);
}

static int
progress_bar_timeout_callback (Simulation *s)
{
	g_return_val_if_fail (s != NULL, FALSE);

	gtk_progress_bar_update (s->progress, s->engine->progress);
	gtk_widget_draw (GTK_WIDGET (s->progress), NULL);

	return TRUE;
}

static void
input_done_callback (SimEngine *engine, Simulation *s)
{
	if (s->progress_timeout_id != 0) {
		gtk_timeout_remove (s->progress_timeout_id);
		s->progress_timeout_id = 0;

		/* Make sure that the progress bar is completed, just for good looks. */
		gtk_progress_bar_update (s->progress, 1.0);
		gtk_widget_draw (GTK_WIDGET (s->progress), NULL);
	}

	gtk_widget_destroy (GTK_WIDGET (s->dialog));
	s->dialog = NULL;

	plot_show (s->engine);

	if (s->engine->has_warnings) {
		schematic_view_log_show (s->sv, FALSE);
	}

	schematic_view_clear_op_values (s->sv);
	schematic_view_show_op_values (s->sv, s->engine);
}

static void
input_aborted_callback (SimEngine *engine, Simulation *s)
{
	GtkWidget *dialog;
	int answer;

	if (s->progress_timeout_id != 0) {
		gtk_timeout_remove (s->progress_timeout_id);
		s->progress_timeout_id = 0;
	}

	gtk_widget_destroy (GTK_WIDGET (s->dialog));
	s->dialog = NULL;

	if (!schematic_view_log_window_exists (s->sv)) {
		dialog = gnome_message_box_new (
			_("The simulation was aborted due to an error.\n"
			  "Would you like to view the error log?"),
			GNOME_MESSAGE_BOX_ERROR, 
			GNOME_STOCK_BUTTON_YES,
			GNOME_STOCK_BUTTON_NO,
			NULL);
		
		answer = gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		
		if (answer == 0) {
			/* 
			 * Show logs. 
			 */
			schematic_view_log_show (s->sv, TRUE);
		}
	} else {
		oregano_error (_("The simulation was aborted due to an error."));
		schematic_view_log_show (s->sv, FALSE);
	}
}

static void
cancel_callback (GtkWidget *widget, Simulation *s)
{
	g_return_if_fail (s != NULL);

	if (s->progress_timeout_id != 0) {
		gtk_timeout_remove (s->progress_timeout_id);
		s->progress_timeout_id = 0;
	}

	if (s->engine)
		sim_engine_stop (s->engine);

	gtk_widget_destroy (GTK_WIDGET (s->dialog));
	s->dialog = NULL;
	s->sv = NULL;
}

static gboolean
simulate_cmd (Simulation *s)
{
	SimEngine *engine;
	char *netlist_filename;

	if (s->engine != NULL) {
		gtk_object_destroy (GTK_OBJECT (s->engine));
		s->engine = NULL;
	}

	netlist_filename = schematic_get_netlist_filename (s->sm);
	if (netlist_filename == NULL) {
		return FALSE;
	}
	
	engine = sim_engine_new (s->sm);
	s->engine = engine;

	s->progress_timeout_id = gtk_timeout_add (
		50,
		(gpointer) progress_bar_timeout_callback,
		(gpointer) s);
	
	gtk_signal_connect (GTK_OBJECT (engine), "done", input_done_callback, s);
	gtk_signal_connect (GTK_OBJECT (engine), "aborted", input_aborted_callback, s);

	sim_engine_start_with_file (engine, netlist_filename);

	return TRUE;
}
