/////////////////////////////////////////////////////////////
//                                                         //
// Copyright (c) 2007-2011 by The University of Queensland //
// Earth Systems Science Computational Centre (ESSCC)      //
// http://www.uq.edu.au/esscc                              //
//                                                         //
// Primary Business: Brisbane, Queensland, Australia       //
// Licensed under the Open Software License version 3.0    //
// http://www.opensource.org/licenses/osl-3.0.php          //
//                                                         //
/////////////////////////////////////////////////////////////

#include "FullCircMNTable3D.h"

// --- System includes ---
#include <cmath>

// --- IO includes ---
#include <iostream>

using std::endl;

using std::floor;


FullCircMNTable3D::FullCircMNTable3D()
{}

/*!
  Construct FullCircMNTable3D. 

  \param MinPt minimum point 
  \param MaxPt maximum point 
  \param cd cell dimension
  \param ngroups initial number of particle groups
*/
FullCircMNTable3D::FullCircMNTable3D(const Vector3& MinPt,const Vector3& MaxPt,double cd,unsigned int ngroups):
  CircMNTable3D(MinPt, MaxPt, cd, ngroups)
{
  m_shift_y=Vector3(0.0,(m_max_pt-m_min_pt).Y(),0.0);
  m_shift_z=Vector3(0.0,0.0,(m_max_pt-m_min_pt).Z());
  
  // check if grid spacing fits size in circular directions:
  double ny=(MaxPt-MinPt).Y()/m_celldim;
  double nz=(MaxPt-MinPt).Z()/m_celldim;
  // error message if not
  if(ny!=floor(ny)){
    std::cerr << "WARNING! grid spacing " << m_celldim << " doesn't fit periodic y-dimension " << (MaxPt-MinPt).Y() << std::endl;
  }
  if(nz!=floor(nz)){
    std::cerr << "WARNING! grid spacing " << m_celldim << " doesn't fit periodic z-dimension " << (MaxPt-MinPt).Z() << std::endl;
  }
}
  
/*!
  Destruct FullCircMNTable3D. 
*/
FullCircMNTable3D::~FullCircMNTable3D()
{}
  
/*!
  set circularity of y-dimension to 1
*/
void FullCircMNTable3D::set_y_circ()
{
  m_y_periodic=1;
} 

/*!
  set circularity of z-dimension to 1
*/
void FullCircMNTable3D::set_z_circ()
{
  m_z_periodic=1;
} 


/*!
  get the cell index for a given position

  \param Pos the position
  \return the cell index if Pos is inside the table, -1 otherwise
*/
int FullCircMNTable3D::getIndex(const Vector3& Pos) const
{
  int ret;

  int ix=int(floor((Pos.x()-m_origin.x())/m_celldim));
  int iy=int(floor((Pos.y()-m_origin.y())/m_celldim));
  int iz=int(floor((Pos.z()-m_origin.z())/m_celldim));

  // check if pos is in table excl. padding
  if((ix>=0) && (ix<=m_nx-1) && (iy>=0) && (iy<=m_ny-1) && (iz>=0) && (iz<=m_nz-1)){
    ret=idx(ix,iy,iz);
  } else {
    ret=-1;
  }
  
  return ret;
}

/*!
  Insert sphere. Insert clone into other side of Table.

  \param S the Sphere
  \param gid the group id
*/
bool FullCircMNTable3D::insert(const Sphere& S,unsigned int gid)
{  bool res;

  int id=getIndex(S.Center());
  int xidx=getXIndex(S.Center());
  int yidx=getYIndex(S.Center());
  int zidx=getZIndex(S.Center());
  
  if((id!=-1) && (xidx!=0) && (xidx!=m_nx-1)  && (yidx!=0) && (yidx!=m_ny-1)  && (zidx!=0) && (zidx!=m_nz-1) && (gid<m_ngroups)){ // valid index
    // insert sphere
    m_data[id].insert(S,gid);
    res=true;
    // insert x-clone
    if (xidx==1){
      Sphere SClone=S;
      SClone.shift(m_shift_x);
      int clone_id=getFullIndex(SClone.Center());
      m_data[clone_id].insert(SClone,gid);
    } else if  (xidx==m_nx-2){
      Sphere SClone=S;
      SClone.shift(-1.0*m_shift_x);
      int clone_id=getFullIndex(SClone.Center());
      m_data[clone_id].insert(SClone,gid);
    } 
    // insert y-clone
    if (yidx==1){
      Sphere SClone=S;
      SClone.shift(m_shift_y);
      int clone_id=getFullIndex(SClone.Center());
      m_data[clone_id].insert(SClone,gid);
    } else if  (yidx==m_ny-2){
      Sphere SClone=S;
      SClone.shift(-1.0*m_shift_y);
      int clone_id=getFullIndex(SClone.Center());
      m_data[clone_id].insert(SClone,gid);
    } 
    // insert z-clone
    if (zidx==1){
      Sphere SClone=S;
      SClone.shift(m_shift_z);
      int clone_id=getFullIndex(SClone.Center());
      m_data[clone_id].insert(SClone,gid);
    } else if  (zidx==m_nz-2){
      Sphere SClone=S;
      SClone.shift(-1.0*m_shift_z);
      int clone_id=getFullIndex(SClone.Center());
      m_data[clone_id].insert(SClone,gid);
    } 
  } else {
    res=false;
  }

  return res;
}

/*!
  Insert sphere if it doesn't collide with other spheres. Insert clone into other side of Table.

  \param S the Sphere
  \param gid the group id
*/
bool FullCircMNTable3D::insertChecked(const Sphere& S,unsigned int gid,double tol)
{
  bool res;
  int id=getIndex(S.Center());
  int xidx=getXIndex(S.Center());
  int yidx=getYIndex(S.Center());
  int zidx=getZIndex(S.Center());

  tol+=s_small_value;
  if((id!=-1) && (xidx!=0) && (xidx!=m_nx-1) && (gid<m_ngroups)){
    // insert original
    multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(S.Center(),S.Radius()-tol,gid);
    if(close_spheres.size()==0){
      m_data[id].insert(S,gid);
      res=true;
    } else res=false;
    // insert x-clone
    if (xidx==1){
      Sphere SClone=S;
      SClone.shift(m_shift_x);
      multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(SClone.Center(),SClone.Radius()-tol,gid);
      if(close_spheres.size()==0){
	int clone_id=getFullIndex(SClone.Center());
	m_data[clone_id].insert(SClone,gid);
      }
    } else if  (xidx==m_nx-2){
      Sphere SClone=S;
      SClone.shift(-1.0*m_shift_x);
      multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(SClone.Center(),SClone.Radius()-tol,gid);
      if(close_spheres.size()==0){
	int clone_id=getFullIndex(SClone.Center());
	m_data[clone_id].insert(SClone,gid);
      }
    } 
    // insert y-clone
    if (yidx==1){
      Sphere SClone=S;
      SClone.shift(m_shift_y);
      multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(SClone.Center(),SClone.Radius()-tol,gid);
      if(close_spheres.size()==0){
	int clone_id=getFullIndex(SClone.Center());
	m_data[clone_id].insert(SClone,gid);
      }
    } else if  (yidx==m_ny-2){
      Sphere SClone=S;
      SClone.shift(-1.0*m_shift_y);
      multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(SClone.Center(),SClone.Radius()-tol,gid);
      if(close_spheres.size()==0){
	int clone_id=getFullIndex(SClone.Center());
	m_data[clone_id].insert(SClone,gid);
      }
    } 
    // insert z-clone
    if (zidx==1){
      Sphere SClone=S;
      SClone.shift(m_shift_z);
      multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(SClone.Center(),SClone.Radius()-tol,gid);
      if(close_spheres.size()==0){
	int clone_id=getFullIndex(SClone.Center());
	m_data[clone_id].insert(SClone,gid);
      }
    } else if  (zidx==m_nz-2){
      Sphere SClone=S;
      SClone.shift(-1.0*m_shift_z);
      multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(SClone.Center(),SClone.Radius()-tol,gid);
      if(close_spheres.size()==0){
	int clone_id=getFullIndex(SClone.Center());
	m_data[clone_id].insert(SClone,gid);
      }
    } 
  } else {
    res=false;
  }

  return res;
}

bool FullCircMNTable3D::checkInsertable(const Sphere& S,unsigned int gid)
{
  bool res;

  int id=this->getIndex(S.Center());
  int xidx=getXIndex(S.Center());
  int yidx=getYIndex(S.Center());
  int zidx=getZIndex(S.Center());
  
  if((id!=-1) && (xidx!=0) && (xidx!=m_nx-1)  && (yidx!=0) && (yidx!=m_ny-1)  && (zidx!=0) && (zidx!=m_nz-1)  && (gid<m_ngroups)){
    multimap<double,const Sphere*> close_spheres=getSpheresFromGroupNear(S.Center(),S.Radius()-s_small_value,gid);
    if(close_spheres.size()==0){ 
      res=true;
    } else {
      res=false;
//       for(map<double,const Sphere*>::const_iterator iter=close_spheres.begin();
// 	  iter!=close_spheres.end();
// 	  iter++){
// 	std::cerr << iter->first << "  |  " << *(iter->second) << std::endl; 
//       }
    }
  } else {
    res=false;
  }

  return res;}

/*!
  Generate bonds between particles of a group. Takes cloned particles into account.

  \param gid the group ID
  \param tol max. difference between bond length and equilibrium dist.
  \param btag bond tag
*/
void FullCircMNTable3D::generateBonds(int gid,double tol,int btag)
{
  std::cerr << "FullCircMNTable3D::generateBonds( " << gid << " , " << tol << " , " << btag << " )" << std::endl;
  // loop over all inner cells 
  for(int i=0;i<m_nx-1;i++){
    for(int j=0;j<m_ny-1;j++){
      for(int k=0;k<m_nz-1;k++){
	int id=idx(i,j,k);
	// loop over "upper" neighbors of each cell
	for(int ii=-1;ii<=1;ii++){
	  for(int jj=-1;jj<=1;jj++){
	    for(int kk=-1;kk<=1;kk++){
	      int id2=idx(i+ii,j+jj,k+kk);
	      vector<pair<int,int> > bonds;
	      if((ii+jj+kk)==0){ // intra-cell, not for boundary 
		// std::cerr << id << " - " << id << std::endl;
		bonds=m_data[id].getBonds(gid,tol);
	      } else if(id2>id){ // inter-cell
		//std::cerr << id << " - " << id2 << std::endl;
		bonds=m_data[id].getBonds(gid,tol,m_data[id2]);
	      }
	      for(vector<pair<int,int> >::iterator iter=bonds.begin();
		  iter!=bonds.end();
		  iter++){
		//std::cerr << iter->first << " | " << iter->second << "   "; 
		if(iter->second > iter->first){
		  m_bonds[btag].insert(*iter);
		}
	      }
	      //std::cerr << std::endl;
	    }
	  }
	}
	//std::cerr << std::endl;
      }
    }
  }
}
  
// ostream& FullCircMNTable3D::operator << (ostream&,const FullCircMNTable3D&)
// {}
