#  Copyright (C) 1999-2004
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

# special kludge to work around a infinit loop caused by cube.tcl and idletasks

proc ContourUpdate {} {
    SetWatchCursor
    ContourUpdateNow
    UnsetWatchCursor
}

proc ContourUpdateNow {} {
    global contour
    global current

    if {$current(frame) != ""} {
	if {$contour(view) && [$current(frame) has fits]} {
	    ContourCheckParams
	    if {[winfo exist $contour(top)]} {
		# remove endl
		set levels ""
		regsub -all "\n" "[$contour(txt) get 1.0 end]" " " levels
		# and trim any trailing spaces
		set levels [string trimright $levels " "]
		if {$levels != ""} {
		    $current(frame) contour create \
			$contour(color) $contour(width) \
			$contour(smooth) $contour(method) "\"$levels\""
		}
		UpdateContourDialogMenu
	    } else {
		$current(frame) contour create \
		    $contour(color) $contour(width) \
		    $contour(levels) $contour(smooth) $contour(method)
	    }
	} else {
	    $current(frame) contour delete
	}
    }
}

proc ContourCheckParams {} {
    global contour

    if {$contour(smooth) < 1} {
	set contour(smooth) 1
    }
    if {$contour(levels) < 1} {
	set contour(levels) 1
    }
}

proc ContourDialog {} {
    global contour
    global current
    global menu
    global ds9

    # see if we already have a ctr window visible
    if [winfo exist $contour(top)] {
	raise $contour(top)
	return
    }

    set w $contour(top)
    set title "Contours"

    # create the contour window

    toplevel $w -colormap $ds9(main)
    wm title $w $title
    wm iconname $w $title
    wm group $w $ds9(top)
    wm protocol $w WM_DELETE_WINDOW ContourDestroyDialog

    # local variables
    set contour(txt) $w.param.right.level.value.text
    set txtscr $w.param.right.level.value.yscroll

    $w configure -menu $contour(mb)

    menu $contour(mb) -tearoff 0
    $contour(mb) add cascade -label File -menu $contour(mb).file
    $contour(mb) add cascade -label Edit -menu $contour(mb).edit
    $contour(mb) add cascade -label Color -menu $contour(mb).color
    $contour(mb) add cascade -label Width -menu $contour(mb).width
    $contour(mb) add cascade -label Scale -menu $contour(mb).scale
    $contour(mb) add cascade -label Limits -menu $contour(mb).limit
    $contour(mb) add cascade -label Method -menu $contour(mb).method

    menu $contour(mb).file -tearoff 0 -selectcolor $menu(selectcolor)
    $contour(mb).file add command -label "Apply" -command ContourApplyDialog
    $contour(mb).file add command -label "Generate" \
	-command ContourGenerateDialog
    $contour(mb).file add command -label "Clear" -command ContourClearDialog
    $contour(mb).file add separator
    $contour(mb).file add command -label "Load Contours..." \
	-command ContourLoadDialog
    $contour(mb).file add command -label "Save Contours..." \
	-command ContourSaveDialog
    $contour(mb).file add separator
    $contour(mb).file add command -label "Convert to Polygons" \
	-command Contour2Polygons
    $contour(mb).file add separator
    $contour(mb).file add command -label "Load Contour Levels..." \
	-command ContourLoadLevels
    $contour(mb).file add command -label "Save Contour Levels..." \
	-command ContourSaveLevels
    $contour(mb).file add separator
    $contour(mb).file add command -label "Close" -command ContourDestroyDialog

    menu $contour(mb).edit -tearoff 0 -selectcolor $menu(selectcolor)
    $contour(mb).edit add command -label "Copy" -command ContourCopyDialog
    $contour(mb).edit add command -label "Paste..." -command ContourPasteDialog

    menu $contour(mb).color -tearoff 0 -selectcolor $menu(selectcolor)
    $contour(mb).color add radiobutton -label "Black" \
	-variable contour(color) -value black -command ContourColorDialog
    $contour(mb).color add radiobutton -label "White" \
	-variable contour(color) -value white -command ContourColorDialog
    $contour(mb).color add radiobutton -label "Red" \
	-variable contour(color) -value red -command ContourColorDialog
    $contour(mb).color add radiobutton -label "Green" \
	-variable contour(color) -value green -command ContourColorDialog
    $contour(mb).color add radiobutton -label "Blue" \
	-variable contour(color) -value blue -command ContourColorDialog
    $contour(mb).color add radiobutton -label "Cyan" \
	-variable contour(color) -value cyan -command ContourColorDialog
    $contour(mb).color add radiobutton -label "Magenta" \
	-variable contour(color) -value magenta -command ContourColorDialog
    $contour(mb).color add radiobutton -label "Yellow" \
	-variable contour(color) -value yellow -command ContourColorDialog

    menu $contour(mb).width -tearoff 0 -selectcolor $menu(selectcolor)
    $contour(mb).width add radiobutton -label "Thin" \
	-variable contour(width) -value 0 -command ContourWidthDialog
    $contour(mb).width add radiobutton -label "1" \
	-variable contour(width) -value 1 -command ContourWidthDialog
    $contour(mb).width add radiobutton -label "2" \
	-variable contour(width) -value 2 -command ContourWidthDialog
    $contour(mb).width add radiobutton -label "3" \
	-variable contour(width) -value 3 -command ContourWidthDialog
    $contour(mb).width add radiobutton -label "4" \
	-variable contour(width) -value 4 -command ContourWidthDialog

    menu $contour(mb).scale -tearoff 0 -selectcolor $menu(selectcolor)
    $contour(mb).scale add radiobutton -label Linear \
	-variable contour(scale) -value linear
    $contour(mb).scale add radiobutton -label Log \
	-variable contour(scale) -value log
    $contour(mb).scale add radiobutton -label Squared \
	-variable contour(scale) -value squared
    $contour(mb).scale add radiobutton -label "Square Root" \
	-variable contour(scale) -value sqrt
    $contour(mb).scale add radiobutton -label "Histogram Equalization" \
	-variable contour(scale) -value histequ

    menu $contour(mb).limit -tearoff 0 -selectcolor $menu(selectcolor)
    $contour(mb).limit add radiobutton -label MinMax \
	-variable contour(limit)  -value minmax -command ContourLimitDialog
    $contour(mb).limit add separator
    $contour(mb).limit add radiobutton -label "99.5%" \
	-variable contour(limit) -value 99.5 -command ContourLimitDialog
    $contour(mb).limit add radiobutton -label "99%" \
	-variable contour(limit) -value 99 -command ContourLimitDialog
    $contour(mb).limit add radiobutton -label "98%" \
	-variable contour(limit) -value 98 -command ContourLimitDialog
    $contour(mb).limit add radiobutton -label "95%" \
	-variable contour(limit) -value 95 -command ContourLimitDialog
    $contour(mb).limit add radiobutton -label "90%" \
	-variable contour(limit) -value 90 -command ContourLimitDialog
    $contour(mb).limit add separator
    $contour(mb).limit add radiobutton -label "ZScale" \
	-variable contour(limit) -value zscale -command ContourLimitDialog 
    $contour(mb).limit add radiobutton -label "ZMax" \
	-variable contour(limit) -value zmax -command ContourLimitDialog 

    menu $contour(mb).method -tearoff 0 -selectcolor $menu(selectcolor)
    $contour(mb).method add radiobutton -label Block \
	-variable contour(method) -value block
    $contour(mb).method add radiobutton -label Smooth \
	-variable contour(method) -value smooth

    set length 300

    frame $w.param
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param -fill x -expand true
    pack $w.buttons -fill x -expand true -ipadx 4 -ipady 4

    frame $w.param.left
    frame $w.param.left.slide -relief groove -borderwidth 2
    frame $w.param.left.clip -relief groove -borderwidth 2
    frame $w.param.right
    frame $w.param.right.level -relief groove -borderwidth 2

    pack $w.param.left.slide $w.param.left.clip \
	-fill x -expand true -ipadx 4 -ipady 4
    pack $w.param.right.level -fill both -expand true
    pack $w.param.left $w.param.right -side left -fill both -expand true

    scale $w.param.left.slide.num -from 0 -to 50 -length $length \
	-variable contour(levels) -orient horizontal -label "Contour Levels" \
	-tickinterval 10 -showvalue true -resolution 1
    scale $w.param.left.slide.res -from 0 -to 32 -length $length \
	-variable contour(smooth) -orient horizontal \
	-label "Contour Smoothness" \
	-tickinterval 4 -showvalue true -resolution 1
    pack $w.param.left.slide.num $w.param.left.slide.res \
	-side top -ipadx 4 -ipady 4 -anchor w

    label $w.param.left.clip.title -text "Limits"

    label $w.param.left.clip.lowtitle -text "Low"
    entry $w.param.left.clip.low -textvariable contour(low) -width 10

    label $w.param.left.clip.hightitle -text "High"
    entry $w.param.left.clip.high -textvariable contour(high) -width 10

    grid $w.param.left.clip.title -row 0 -column 0 -padx 4 -pady 1 -sticky w
    grid $w.param.left.clip.lowtitle -row 0 -column 1 -padx 4 -pady 1 -sticky w
    grid $w.param.left.clip.low -row 0 -column 2 -padx 4 -pady 1 -sticky w
    grid $w.param.left.clip.hightitle -row 0 -column 3 \
	-padx 4 -pady 1 -sticky w
    grid $w.param.left.clip.high -row 0 -column 4 -padx 4 -pady 1 -sticky w

    label $w.param.right.level.title -text "Levels"
    frame $w.param.right.level.value
    text $contour(txt) -height 13 -width 15 -wrap none \
	-font {courier 12} -yscrollcommand "$txtscr set"
    scrollbar $txtscr -command [list $contour(txt) yview] -orient vertical

    grid $w.param.right.level.value.text \
	$w.param.right.level.value.yscroll -sticky news
    grid $w.param.right.level.title -row 0 -column 0 -padx 4 -sticky w
    grid $w.param.right.level.value -row 1 -column 0 -padx 4 -sticky w

    button $w.buttons.apply -text "Apply" -command ContourApplyDialog
    button $w.buttons.generate -text "Generate" -command ContourGenerateDialog
    button $w.buttons.clear -text "Clear" -command ContourClearDialog
    button $w.buttons.close -text "Close" -command ContourDestroyDialog
    pack $w.buttons.apply $w.buttons.generate $w.buttons.clear \
	$w.buttons.close -side left -padx 10 -expand true

    UpdateContourMenu
    UpdateContourDialog
}

proc ContourApplyDialog {} {
    global contour

    set contour(view) 1
    ContourUpdate
}

proc ContourDestroyDialog {} {
    global contour

    destroy $contour(top)
    destroy $contour(mb)

    unset contour(txt)
}

proc ContourGenerateDialog {} {
    global current
    global contour

    ContourCheckParams

    $contour(txt) delete 1.0 end
    if {($current(frame) != "") && ([$current(frame) has fits]) && \
	    ($contour(low) != "") && ($contour(high) != "")} {
	$contour(txt) insert end \
	    [$current(frame) get contour level $contour(levels) \
		 $contour(low) $contour(high) $contour(scale)]
    }
}

proc ContourClearDialog {} {
    global contour
    global current

    set contour(view) 0
    if {$current(frame) != ""} {
	$current(frame) contour delete all
    }
    UpdateContourDialog
}

proc ContourCopyDialog {} {
    global contour
    global current

    set contour(copy) $current(frame)
    UpdateContourDialog
}

proc ContourPasteDialog {} {
    global contour
    global current

    if {$current(frame) != "" && $contour(copy) != ""} {
	ContourParamsDialog paste {}
    }
}

proc ContourColorDialog {} {
    global contour
    global current
    
    if {$current(frame) != ""} {
	$current(frame) contour color $contour(color)
    }
}

proc ContourWidthDialog {} {
    global contour
    global current
    
    if {$current(frame) != ""} {
	$current(frame) contour width $contour(width)
    }
}

proc ContourLimitDialog {} {
    global current
    global contour

    if {($current(frame) != "") && [$current(frame) has fits]} {
	set limits [$current(frame) get clip $contour(limit)]
	set contour(low) [lindex $limits 0]
	set contour(high) [lindex $limits 1]
    } else {
	set contour(low) ""
	set contour(high) ""
    }
}

proc ContourLoadLevels {} {
    global contour

    set filename [OpenFileDialog contourlevfbox]

    if {$filename != {}} {
	set id [open $filename r]
	$contour(txt) delete 1.0 end
	$contour(txt) insert end [read $id]
	close $id
    }
}

proc ContourSaveLevels {} {
    global contour

    set filename [SaveFileDialog contourlevfbox]

    if {$filename != {}} {
	set id [open $filename w]
	puts -nonewline $id "[$contour(txt) get 1.0 end]"
	close $id
    }
}

proc ContourSaveDialog {} {
    global current

    set filename [SaveFileDialog contourfbox]

    if {$filename != ""} {
	if {$current(frame) != ""} {
	    ContourParamsDialog save $filename
	}
    }
}

proc ContourLoadDialog {} {
    global current

    set filename [OpenFileDialog contourfbox]

    if {$filename != {}} {
	if {$current(frame) != ""} {
	    ContourParamsDialog load $filename
	}
    }
}

proc Contour2Polygons {} {
    global current
    global contour

    if {$current(frame) != ""} {
	$current(frame) contour create polygon \
	    color = $contour(color) width = $contour(width)
	$current(frame) contour delete

    }    
}

proc UpdateContourMenu {} {
    global contour
    global current

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateContourMenu"
    }

    if {($current(frame) != "") && [$current(frame) has fits]} {
	set contour(view) [$current(frame) has contour]

	set contour(scale) [$current(frame) get colorscale]
	set contour(limit) [$current(frame) get clip mode]
	set limits [$current(frame) get clip $contour(limit)]
	set contour(low) [lindex $limits 0]
	set contour(high) [lindex $limits 1]
    } else {
	set contour(low) ""
	set contour(high) ""
    }
}

proc UpdateContourDialog {} {
    global contour
    global current

    global debug
    if {$debug(tcl,update)} {
	puts "UpdateContourDialog"
    }

    if {[winfo exist $contour(top)] && $current(frame) != ""} {
	set levels [$current(frame) get contour level]
	if {$levels != ""} {
	    $contour(txt) delete 1.0 end
	    $contour(txt) insert end $levels
	    UpdateContourDialogMenu
	} else {
	    ContourGenerateDialog
	}
    }
}

proc UpdateContourDialogMenu {} {
    global current
    global contour

    if [winfo exist $contour(top)] {
	if {$current(frame) != {}} {
	    if {[$current(frame) has contour]} {
		set contour(method) [$current(frame) get contour method]
		set contour(color) [$current(frame) get contour color]
		set contour(width) [$current(frame) get contour width]

		$contour(mb).edit entryconfig Copy -state normal
	    } else {
		$contour(mb).edit entryconfig Copy -state disabled
	    }

	    if {$contour(copy) != {}} {
		if {[$contour(copy) has contour]} {
		    $contour(mb).edit entryconfig "Paste..." -state normal
		} else {
		    $contour(mb).edit entryconfig "Paste..." -state disabled
		}
		
	    } else {
		$contour(mb).edit entryconfig "Paste..." -state disabled
	    }
	}
    }
}

proc ContourParamsDialog {action fn} {
    global ctld
    global current
    global contour
    global menu
    global ds9

    set w ".ctld"

    set ctld(ok) 0
    set ctld(system) WCS
    set ctld(sky) fk5
    set ctld(color) Green
    set ctld(width) 1

    DialogCreate $w "Contour Parameters" -borderwidth 2
    frame $w.param -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.param $w.buttons  -side top -ipadx 4 -ipady 4 -fill x -expand true

    label $w.param.dummy -text {} -width 40
    label $w.param.coordtitle -text "Coord System: "
    menubutton $w.param.coordbutton -relief raised \
	-menu $w.param.coordbutton.menu -textvariable ctld(system)

    menu $w.param.coordbutton.menu -tearoff 0 -selectcolor $menu(selectcolor)
    $w.param.coordbutton.menu add radiobutton -label "WCS" \
	-variable ctld(system) -value "WCS"
    $w.param.coordbutton.menu add cascade -label "Multiple WCS" \
	-menu $w.param.coordbutton.menu.wcs
    $w.param.coordbutton.menu add separator
    $w.param.coordbutton.menu add radiobutton -label "Image" \
	-variable ctld(system) -value "Image"
    $w.param.coordbutton.menu add radiobutton -label "Physical" \
	-variable ctld(system) -value "Physical"
    if {$ds9(amp,det)} {
	$w.param.coordbutton.menu add radiobutton -label "Amplifier" \
	    -variable ctld(system) -value "Amplifier"
	$w.param.coordbutton.menu add radiobutton -label "Detector" \
	    -variable ctld(system) -value "Detector"
    }
    $w.param.coordbutton.menu add separator
    $w.param.coordbutton.menu add radiobutton -label "Equatorial B1950" \
	-variable ctld(sky) -value fk4
    $w.param.coordbutton.menu add radiobutton -label "Equatorial J2000" \
	-variable ctld(sky) -value fk5
    $w.param.coordbutton.menu add radiobutton -label "ICRS" \
	-variable ctld(sky) -value icrs
    $w.param.coordbutton.menu add radiobutton -label "Galactic" \
	-variable ctld(sky) -value galactic
    $w.param.coordbutton.menu add radiobutton -label "Ecliptic" \
	-variable ctld(sky) -value ecliptic

    menu $w.param.coordbutton.menu.wcs -tearoff $menu(tearoff) \
	-selectcolor $menu(selectcolor)
    foreach l {A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$w.param.coordbutton.menu.wcs add radiobutton -label "WCS $l" \
	    -variable ctld(system) -value "wcs$l"
    }

    label $w.param.colortitle -text "Color: "
    menubutton $w.param.colorbutton -relief raised \
	-menu $w.param.colorbutton.menu -textvariable ctld(color)

    menu $w.param.colorbutton.menu -tearoff 0 -selectcolor $menu(selectcolor)
    $w.param.colorbutton.menu add radiobutton -label "Black" \
	-variable ctld(color) -value "Black"
    $w.param.colorbutton.menu add radiobutton -label "White" \
	-variable ctld(color) -value "White"
    $w.param.colorbutton.menu add radiobutton -label "Red" \
	-variable ctld(color) -value "Red"
    $w.param.colorbutton.menu add radiobutton -label "Green" \
	-variable ctld(color) -value "Green"
    $w.param.colorbutton.menu add radiobutton -label "Blue" \
	-variable ctld(color) -value "Blue"
    $w.param.colorbutton.menu add radiobutton -label "Cyan" \
	-variable ctld(color) -value "Cyan"
    $w.param.colorbutton.menu add radiobutton -label "Magenta" \
	-variable ctld(color) -value "Magenta"
    $w.param.colorbutton.menu add radiobutton -label "Yellow" \
	-variable ctld(color) -value "Yellow"

    label $w.param.widthtitle -text "Width: "
    menubutton $w.param.widthbutton -relief raised \
	-menu $w.param.widthbutton.menu -textvariable ctld(width)

    menu $w.param.widthbutton.menu -tearoff 0 -selectcolor $menu(selectcolor)
    $w.param.widthbutton.menu add radiobutton -label "Thin" \
	-variable ctld(width) -value "Thin"
    $w.param.widthbutton.menu add radiobutton -label "1" \
	-variable ctld(width) -value "1"
    $w.param.widthbutton.menu add radiobutton -label "2" \
	-variable ctld(width) -value "2"
    $w.param.widthbutton.menu add radiobutton -label "3" \
	-variable ctld(width) -value "3"
    $w.param.widthbutton.menu add radiobutton -label "4" \
	-variable ctld(width) -value "4"

    grid rowconfigure $w.param 0 -pad 0
    grid rowconfigure $w.param 1 -pad 2
    grid rowconfigure $w.param 2 -pad 2
    grid rowconfigure $w.param 3 -pad 2

    grid $w.param.dummy -columnspan 2
    grid $w.param.coordtitle $w.param.coordbutton -padx 4 -sticky w
    if {$action != "save"} {
	grid $w.param.colortitle $w.param.colorbutton -padx 4 -sticky w
	grid $w.param.widthtitle $w.param.widthbutton -padx 4 -sticky w
    }

    button $w.buttons.ok -text "OK" -default active -command {set ctld(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ctld(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ctld(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    DialogWait $w ctld(ok)
    DialogDismiss $w

    if {$ctld(ok)} {
	set ctld(color) [string tolower $ctld(color)]
	switch -- $ctld(width) {
	    "Thin" {set ctld(width) 0}
	}

	SetWatchCursor
	switch -- $action {
	    paste {
		set ptr [$contour(copy) contour copy $ctld(system) $ctld(sky)]
		if {$ptr != {}} {
		    $current(frame) contour paste \
			$ctld(color) $ctld(width) $ptr $ctld(system) $ctld(sky)
		}
	    }
	    load {
		$current(frame) contour load \
		    $ctld(color) $ctld(width) "\{$fn\}" \
		    $ctld(system) $ctld(sky)
	    }
	    save {
		$current(frame) contour save "\{$fn\}" $ctld(system) $ctld(sky)
	    }
	}
	UnsetWatchCursor
	UpdateContourDialog
    }

    # we want to destroy this window

    destroy $w 

    unset ctld
}

proc ProcessContourCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global contour
    global current

    switch -- [string tolower [lindex $var $i]] {
	copy {
	    set contour(copy) $current(frame)
	    UpdateContourDialog
	}
	paste {
	    incr i
	    if {$current(frame) != "" && $contour(copy) != ""} {
		set sys [lindex $var $i]
		incr i
		set sky [lindex $var $i]
		incr i
		set color [lindex $var $i]
		incr i
		set width [lindex $var $i]

		ProcessContourFix sys sky color width

		set ptr [$contour(copy) contour copy $sys $sky]
		if {$ptr != {}} {
		    $current(frame) contour paste $color $width $ptr $sys $sky
		}
	    }
	}
	load {
	    incr i
	    set filename [lindex $var $i]
	    incr i
	    set sys [lindex $var $i]
	    incr i
	    set sky [lindex $var $i]
	    incr i
	    set color [lindex $var $i]
	    incr i
	    set width [lindex $var $i]

	    ProcessContourFix sys sky color width

	    $current(frame) contour load $color $width $filename $sys $sky

	    UpdateContourDialog
	}
	save {
	    incr i
	    set filename [lindex $var $i]
	    incr i
	    set sys [lindex $var $i]
	    incr i
	    set sky [lindex $var $i]
	    set color green
	    set width 1

	    ProcessContourFix sys sky color width

	    if {$filename != {}} {
		$current(frame) contour save $filename $sys $sky
	    }
	}
	clear {
	    set contour(view) 0
	    $current(frame) contour delete all
	}
	yes -
	true -
	1 -
	no -
	false -
	0 {
	    set contour(view) [FromYesNo [lindex $var $i]]
	    ContourUpdate
	}
	default {
	    set contour(view) 1
	    ContourUpdate
	    incr i -1
	}
    }
}

proc ProcessContourFix {sysname skyname colorname widthname} {
    upvar $sysname sys
    upvar $skyname sky
    upvar $colorname color
    upvar $widthname width

    switch -- $sys {
	{} {set sys image}
	fk4 -
	fk5 -
	icrs -
	galactic -
	ecliptic {
	    set width $color;
	    set color $sky;
	    set sky $sys;
	    set sys wcs
	}
    }

    switch -- $sky {
	{} {set sky fk5}
	fk4 -
	fk5 -
	icrs -
	galactic -
	ecliptic {}
	default {
	    set width $color
	    set color $sky
	    set sky fk5
	}
    }

    switch -- $color {
	{} {set color green}
	white -
	black -
	red -
	green -
	blue -
	cyan -
	magenta -
	yellow {}
	default {
	    set width $color
	    set color green
	}
    }

    if {$width == {}} {
	set width 1
    }
}
