// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "box.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Box::Box(const Box& a) : BaseBox(a) {}

Box::Box(FrameBase* p, const Vector& ctr, const Vector& s, double a,
	 const char* clr, int w, const char* f, 
	 const char* t, unsigned short prop, const char* c, 
	 const List<Tag>& tag)
  : BaseBox(p, ctr, a, clr, w, f, t, prop, c, tag)
{
  annuli = 1;
  size = new Vector[1];
  size[0] = s;

  strcpy(type,"box");
  handle = new Vector[4];
  numHandle = 4;

  updateBBox();
}

void Box::editBegin(int h)
{
  switch (h) {
  case 1:
    return;
  case 2:
    size[0] = Vector(-size[0][0],size[0][1]);
    return;
  case 3:
    size[0] = -size[0];
    return;
  case 4:
    size[0] = Vector(size[0][0],-size[0][1]);
    return;
  }

  doCallBack(&editBeginCB);
}

void Box::edit(const Vector& v, int h)
{
  // This sizes about the opposite node

  size[0] = center+size[0]/2 - 
    (v * Translate(-center) * flip * Rotate(-angle) * Translate(center));
  center = v+((size[0]/2) * (Rotate(angle) * flip));

  updateBBox();
  doCallBack(&editCB);
}

void Box::editEnd()
{
  size[0] = size[0].abs();

  updateBBox();
  doCallBack(&editEndCB);
}

void Box::setSize(const Vector& s)
{
  size[0] = s;

  updateBBox();
  doCallBack(&editCB);
}

// private

void Box::updateHandles()
{
  Vector s = size[0];
  if (properties & FIXED)
    s /= parent->getZoom();

  Matrix m = Rotate(angle) * flip * Translate(center) * parent->refToCanvas;

  handle[0] = Vector(-s[0],-s[1])/2 * m;
  handle[1] = Vector( s[0],-s[1])/2 * m;
  handle[2] = Vector( s[0], s[1])/2 * m;
  handle[3] = Vector(-s[0], s[1])/2 * m;
}

// list

void Box::list(ostream& str, CoordSystem sys, SkyFrame sky, SkyFormat format,
	       char delim)
{
  FitsImage* ptr = parent->findFits(center);

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listPre(str,sys,sky,ptr);

      Vector s = ptr->mapLenFromRef(size[0],sys);
      Vector v = ptr->mapFromRef(center,sys);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << s[0] << ',' << s[1] << ',' 
	  << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';

      listPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCS(sys)) {
      listPre(str,sys,sky,ptr);

      if (ptr->hasWCSEqu(sys)) {
	Vector s = ptr->mapLenFromRef(size[0],sys,ARCSEC);
	switch (format) {
	case DEGREES:
	  {
	    Vector v = ptr->mapFromRef(center,sys,sky);
	    str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<',' 
		<< s[0] << "\"" << ',' << s[1] << "\"" << ',' 
		<< radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	case SEXAGESIMAL:
	  {
	    char buf[64];
	    ptr->mapFromRef(center,sys,sky,format,buf,64);
	    char ra[16];
	    char dec[16];
#if __GNUC__ >= 3
	    string x(buf);
	    istringstream wcs(x);
#else
	    istrstream wcs(buf,64);
#endif
	    wcs >> ra >> dec;
	    str << type << '(' << ra << ',' << dec << ',' 
		<< s[0] << "\"" << ',' << s[1] << "\"" << ',' 
		<< radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
	  }
	  break;
	}
      }
      else {
	Vector v = ptr->mapFromRef(center,sys);
	Vector s = ptr->mapLenFromRef(size[0],sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] <<',' 
	    << s[0] << ',' << s[1] << ',' 
	    << radToDeg(parent->mapAngleFromRef(angle,sys)) << ')';
      }

      listPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Box::listCiao(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // radius is always in image coords

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listCiaoPre(str);

      Vector v = ptr->mapFromRef(center,PHYSICAL);
      Vector s = ptr->mapLenFromRef(size[0],PHYSICAL);
      str << "rotbox(" << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')'
	  << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listCiaoPre(str);

      Vector s = ptr->mapLenFromRef(size[0],sys,ARCMIN);
      char buf[64];
      ptr->mapFromRef(center,sys,FK5,SEXAGESIMAL,buf,64);
      char ra[16];
      char dec[16];
#if __GNUC__ >= 3
      string x(buf);
      istringstream wcs(x);
#else
      istrstream wcs(buf,64);
#endif
      wcs >> ra >> dec;
      str << "rotbox(" << ra << ',' << dec << ',' 
	  << s[0] << '\'' << ',' << s[1] << '\'' << ',' 
	  << radToDeg(angle) << ')'
	  << delim;
    }
    else
      str << "";
    break;
  }
}

void Box::listPros(ostream& str, CoordSystem sys, SkyFrame sky,
		   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  switch (sys) {
  case IMAGE:
  case DETECTOR:
  case AMPLIFIER:
    sys = IMAGE;
  case PHYSICAL:
    {
      listProsCoordSystem(str,sys,sky);
      str << "; ";
      Vector s = ptr->mapLenFromRef(size[0],IMAGE);
      Vector v = ptr->mapFromRef(center,sys);
      str << "rotbox " << setprecision(8) << v << s << radToDeg(angle)
	  << delim;
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listProsCoordSystem(str,sys,sky);
      str << "; ";

      // display wcs radius in ARCSEC
      Vector s = ptr->mapLenFromRef(size[0],sys,ARCSEC);

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  str << "rotbox " << setprecision(8) << v[0] << "d " << v[1] << "d " 
	      << s[0] << "\" " << s[1] << "\" " << radToDeg(angle);
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  if (dec[0]=='+')
	    str << "rotbox " << ra << ' ' << dec+1 << ' '
		<< s[0] << "\" " << s[1] << "\" " << radToDeg(angle);
	  else
	    str << "rotbox " << ra << ' ' << dec << ' '
		<< s[0] << "\" " << s[1] << "\" " << radToDeg(angle);
	}
	break;
      }

      str << delim;
    }
    else
      str << "";
    break;
  }
}

void Box::listSAOtng(ostream& str, CoordSystem sys, SkyFrame sky,
		     SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // radius is always in image coords

  switch (sys) {
  case IMAGE:
  case PHYSICAL:
  case DETECTOR:
  case AMPLIFIER:
    {
      listSAOtngPre(str,delim);

      Vector v = ptr->mapFromRef(center,IMAGE);
      Vector s = ptr->mapLenFromRef(size[0],IMAGE);
      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	  << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')';

      listSAOtngPost(str,delim);
    }
    break;
  default:
    if (ptr->hasWCSEqu(sys)) {
      listSAOtngPre(str,delim);

      switch (format) {
      case DEGREES:
	{
	  Vector v = ptr->mapFromRef(center,sys,sky);
	  Vector s = ptr->mapLenFromRef(size[0],IMAGE);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	      << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')';
	}
	break;
      case SEXAGESIMAL:
	{
	  char buf[64];
	  ptr->mapFromRef(center,sys,sky,format,buf,64);
	  Vector s = ptr->mapLenFromRef(size[0],IMAGE);
	  char ra[16];
	  char dec[16];
#if __GNUC__ >= 3
	  string x(buf);
	  istringstream wcs(x);
#else
	  istrstream wcs(buf,64);
#endif
	  wcs >> ra >> dec;
	  str << type << '(' << ra << ',' << dec << ',' 
	      << s[0] << ',' << s[1] << ',' << radToDeg(angle) << ')';
	}
	break;
      }

      listSAOtngPost(str,delim);
    }
    else
      str << "";
    break;
  }
}

void Box::listSAOimage(ostream& str, CoordSystem sys, SkyFrame sky,
			   SkyFormat format, char delim)
{
  FitsImage* ptr = parent->findFits(1);

  // all coords are in image coords

  if (!(properties&INCLUDE))
    str << '-';

  Vector v = ptr->mapFromRef(center,IMAGE);
  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
      << size[0][0] << ',' << size[0][1] << ',' << radToDeg(angle) << ')';

  str << delim;
}
