// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <stdlib.h>
#include <math.h>
#include <float.h>

#include "tcl.h"
#include <Xlib.h>
#include <Xutil.h>

#include "framebase.h"
#include "util.h"
#include "fitsimage.h"
#include "nan.h"

#include "contour.h"
#include "grid.h"

#include "hist.h"

#define BLT 1

#if BLT == 1
#include "blt.h"
#endif

// Debug

int DebugAST= 0;
int DebugMosaic= 0;
int DebugPerf= 0;
int DebugWCS= 0;
int DebugBin= 0;
int DebugGZ= 0;
int DebugRGB= 0;

// Parser Stuff

#undef yyFlexLexer
#define yyFlexLexer frFlexLexer
#include <FlexLexer.h>

frFlexLexer* frlexx = NULL;    // used by frlex
FrameBase* fr = NULL;           // used by frerror
extern int frparse(void*);

int frlex()
{
  return (frlexx ? frlexx->yylex() : 0);
}

void frerror(const char* m)
{
  if (fr) {
    fr->error(m);
    const char* cmd = frlexx ? frlexx->YYText() : (const char*)NULL;
    if (cmd && cmd[0] != '\n') {
      fr->error(": ");
      fr->error(cmd);
    }
  }
}

// Fits Group Functions

// Frame Member Functions

FrameBase::FrameBase(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item) 
  : Widget(i, c, item)
{
  // these should all be set in the derived classes
  channelFits = NULL;
  currentFits = NULL;
  currentCount = NULL;
  currentMode = NULL;
  currentMosaic = NULL;

  // WCS params
  currentWCScdelt = NULL;
  currentWCSrot = NULL;
  currentWCSorientation = NULL;
  currentWCSbb = NULL;
  currentWCSmatrix = NULL;

  imageToData = Translate(-.5, -.5);
  dataToImage = Translate( .5,  .5);

  // the rest
  mosaicFast = 1;
  slice = 1;

  baseXImage = NULL;
  basePixmap = 0;
  needsUpdate = NOUPDATE;

  panPM = 0;
  panGCXOR = NULL;
  pannerPixmap = 0;
  pannerXImage = NULL;
  pannerWidth = 0;
  pannerHeight = 0;
  pannerName[0] = '\0';
  pannerSystem = WCS;
  pannerSky = FK5;
  usePanner = 0;

  magnifierPixmap = 0;
  magnifierXImage = NULL;
  magnifierWidth = 0;
  magnifierHeight = 0;
  magnifierZoom = 4;
  magnifierName[0] = '\0';
  useMagnifier = 0;
  useMagnifierGraphics = 1;
  useMagnifierCursor = 1;
  magnifierGC = NULL;

  zoom = 1;
  rotation = 0;
  orientation = NORMAL;
  rotateRotation = 0;
  rotateSrcXM = NULL;
  rotateDestXM = NULL;
  rotatePM = 0;
  rotateGCXOR = NULL;

  preservePan = 0;

  wcsAlignment = 0;
  wcsSystem = WCS;
  wcsSky = FK5;
  wcsRotation = 0;
  wcsOrientation = NORMAL;
  
  // this should all be set in the derived classes
  currentcontour = NULL;
  contourGC = NULL;

  grid = NULL;
  gridGC = NULL;

  cmapArea = 0;
  cmapMode = CENTER;

  useCrosshair = 0;
  crosshairGC = NULL;
  crosshairGCXOR = NULL;

  highliteGC = NULL;
  useHighlite = 0;

  markers = &fgMarkers;
  undoMarkers = &undofgMarkers;
  pasteMarkers = &pastefgMarkers;
  undoMarkerType = NONE;
  markerUpdate = PIXMAP;

  showMarkers = &showfgMarkers;
  showfgMarkers = 1;
  showbgMarkers = 1;
  preserveMarkers = 0;
  markerGC = NULL;
  selectGCXOR = NULL;
  editMarker = NULL;
  rotateMarker = NULL;
  markerDefaultSystem = IMAGE;
  markerDefaultSky = FK5;

  psLevel = 2;
  psResolution = 300;
  psInterpolate = 0;
  psColorSpace = RGB;

  binFunction_ = FitsHist::SUM;
  binFactor_ = Vector(1,1);
  binBufferSize_ = 1024;
  binDepth_ = 1;
  binSmooth_ = 0;
  binSmoothFunc_ = FitsHist::GAUSSIAN;
  binSmoothRadius_ = 3;

  bgColorName = dupstr("white");
  bgColor = whiteColor;
  nanColorName = dupstr("white");
  nanColor = whiteColor;
}

FrameBase::~FrameBase()
{
  if (basePixmap)
    Tk_FreePixmap(display, basePixmap);

  if (baseXImage)
    XDestroyImage(baseXImage);

  if (panPM)
    Tk_FreePixmap(display, panPM);

  if (crosshairGC)
    XFreeGC(display, crosshairGC);

  if (crosshairGCXOR)
    XFreeGC(display, crosshairGCXOR);

  if (highliteGC)
    XFreeGC(display, highliteGC);

  if (markerGC)
    XFreeGC(display, markerGC);

  if (selectGCXOR)
    XFreeGC(display, selectGCXOR);

  if (panGCXOR)
    XFreeGC(display, panGCXOR);

  if (rotateGCXOR)
    XFreeGC(display, rotateGCXOR);

  if (pannerPixmap)
    Tk_FreePixmap(display, pannerPixmap);

  if (pannerXImage)
    XDestroyImage(pannerXImage);

  if (magnifierPixmap)
    Tk_FreePixmap(display, magnifierPixmap);

  if (magnifierXImage)
    XDestroyImage(magnifierXImage);

  if (magnifierGC)
    XFreeGC(display, magnifierGC);

  if (contourGC)
    XFreeGC(display, contourGC);

  if (gridGC)
    XFreeGC(display, gridGC);

  if (bgColorName)
    delete [] bgColorName;

  if (nanColorName)
    delete [] nanColorName;
}

#if __GNUC__ >= 3
int FrameBase::parse(istringstream& istr)
#else
int FrameBase::parse(istrstream& istr)
#endif
{
  result = TCL_OK;
  frlexx = new frFlexLexer(&istr);
  fr = this;
  frparse(this);

  delete frlexx;
  frlexx = NULL;
  return result;
}

// ***Protected

// Virtual routines with base component

void FrameBase::reset()
{
  zoom = 1;
  rotation = 0;
  orientation = NORMAL;
  orientationMatrix.identity();

  wcsAlignment = 0;
  wcsSystem = WCS;
  wcsSky = FK5;
  wcsRotation = 0;
  wcsOrientation = NORMAL;
  wcsOrientationMatrix.identity();
  
  centerImage();

  markerbg();
  unselectMarkers();
  markerfg();
  unselectMarkers();

  crosshair = cursor;

  update(MATRIX);
}

void FrameBase::unloadFits()
{
  if (DebugPerf)
    cerr << "FrameBase::unloadFits" << endl;

  if (!preserveMarkers) {
    // delete markers
    fgMarkers.deleteAll();
    undofgMarkers.deleteAll();
    pastefgMarkers.deleteAll();

    bgMarkers.deleteAll();
    undobgMarkers.deleteAll();
    pastebgMarkers.deleteAll();
  }

  // delete any auxcontour
  auxcontours.deleteAll();

  // delete any grid
  if (grid)
    delete grid;
  grid = NULL;

  slice = 1;
  setCurrentFits(NULL);
  *currentCount = 0;
  *currentMode = SINGLE;
  *currentMosaic = NOMOSAIC;

  resetClip(0);
}

void FrameBase::updateMatrices()
{
  if (DebugPerf)
    cerr << "updateMatrices...";

  // These are the basic tranformation matrices

  // Note: imageCenter() is in IMAGE coords
  Vector center = imageCenter() * imageToData;

  if (!isIIS()) {
    refToUser = 
      Translate(-center) *                // translate to center image
      FlipY() *                           // flip y axis for X Windows
      Translate(center);                  // translate back from center
    userToRef = refToUser.invert();
  }
  else {
    refToUser.identity();
    userToRef.identity();
  }

  userToWidget =
    Translate(-cursor) *                // translate to cursor position
    Rotate(rotation) *                  // rotate about cursor position
    Rotate(wcsRotation) *               // rotate about center position
    orientationMatrix *                 // flip x/y axis about cursor position
    wcsOrientationMatrix *              // flip x/y axis about center
    Scale(zoom) *                       // scale about cursor position
    //    Translate(cursor) *
    //    Translate(-cursor) *
    // translate to center of widget. needs to be integer
    Translate((int)(options->width/2.), (int)(options->height/2.));

  widgetToUser = userToWidget.invert();

  widgetToCanvas = Translate(originX, originY); // translate by widget location
  canvasToWidget = widgetToCanvas.invert();

  short x, y;
  Tk_CanvasWindowCoords(canvas, 0, 0, &x, &y);
  canvasToWindow = Translate(x, y);
  windowToCanvas = canvasToWindow.invert();

  // These are derived Transformation Matrices

  refToWidget = refToUser * userToWidget;
  widgetToRef = refToWidget.invert();

  refToCanvas = refToUser * userToWidget * widgetToCanvas;
  canvasToRef = refToCanvas.invert();

  refToWindow = refToUser * userToWidget * widgetToCanvas * canvasToWindow;
  windowToRef = refToWindow.invert();

  userToCanvas = userToWidget * widgetToCanvas;
  canvasToUser = userToCanvas.invert();

  widgetToWindow = widgetToCanvas * canvasToWindow;
  windowToWidget = widgetToWindow.invert();

    // calculate panner matrices

  userToPanner =
    Translate(-center) *
    Rotate(rotation) *                  // rotate about cursor position
    Rotate(wcsRotation) *               // rotate about center position
    orientationMatrix *                 // flip x/y axis about cursor position
    wcsOrientationMatrix *              // flip x/y axis about center
    Scale(calcPannerZoom()) *
    Translate(pannerWidth/2., pannerHeight/2.);
  pannerToUser = userToPanner.invert();

  refToPanner = refToUser * userToPanner;
  pannerToRef = refToPanner.invert();

  pannerToWidget = pannerToUser * userToWidget;
  widgetToPanner = pannerToWidget.invert();

  updateMagnifierMatrix();

  if (DebugPerf)
    cerr << "end" << endl;
}

// Other routines

void FrameBase::update(UpdateType flag)
{
  if (DebugPerf)
    cerr << "update" << endl;

  // Be careful, someone may have already set the flag at a lower level
  // therefor, only change the flag if we need more to be done

  if (flag < needsUpdate)
    needsUpdate = flag;
  redraw();
}

void FrameBase::update(UpdateType flag, BBox bb)
{
  if (DebugPerf)
    cerr << "update BB" << endl;

  // bb is in canvas coords

  if (flag < needsUpdate)
    needsUpdate = flag;
  redraw(bb);
}

void FrameBase::updateNow(UpdateType flag)
{
  if (DebugPerf)
    cerr << "updateNow" << endl;

  if (flag < needsUpdate)
    needsUpdate = flag;
  redrawNow();
}

void FrameBase::updateNow(UpdateType flag, BBox bb)
{
  if (DebugPerf)
    cerr << "updateNow BB" << endl;

  // bb is in canvas coords

  if (flag < needsUpdate)
    needsUpdate = flag;
  redrawNow(bb);
}

void FrameBase::setColorScale(FrScale::ColorScaleType s)
{
  currentScale->setColorScaleType(s);
  updateColorScale();
}

ContourScale* FrameBase::contourScale(int cnt, float l, float h, 
				  FrScale::ColorScaleType type)
{
  switch (type) {
  case FrScale::IISSCALE:
  case FrScale::LINEARSCALE:
    return new LinearContourScale(cnt, l, h);
  case FrScale::LOGSCALE:
    return new LogContourScale(cnt, l, h);
  case FrScale::SQUAREDSCALE:
    return new SquaredContourScale(cnt, l, h);
  case FrScale::SQRTSCALE:
    return new SqrtContourScale(cnt, l, h);
  case FrScale::HISTEQUSCALE:
    calcHistEqu();
    return new HistEquContourScale(cnt, currentScale->min(), 
				   currentScale->max(), 
				   currentScale->histequ(), HISTEQUSIZE);
  }
}

void FrameBase::resetValues()
{
  if (DebugPerf)
    cerr << "resetValues" << endl;

  align();

  if (!preservePan) {
    centerImage();
    crosshair = cursor;
  }
}

void FrameBase::centerImage()
{
  // always center to center of pixel
  //  cursor = imageCenter().floor() + Vector(.5,.5);

  // Note: cursor is in USER coords, imageCenter() in IMAGE coords
  Vector center = imageCenter() * imageToData;
  cursor = center.floor() * dataToImage;
}

// ***Private

Vector FrameBase::getClip(FrScale::ClipMode cm, float ac)
{
  // save current scale
  FrScale* sav = currentScale;
  currentScale = new FrScale(*sav);

  // change values
  currentScale->setClipMode(cm);
  currentScale->setAutoCutPer(ac);
  updateClip(0);

  // this routine will change current settings, calculate, then reset
  double ll = currentScale->low();
  double hh = currentScale->high();

  // restore currentScale
  delete currentScale;
  currentScale = sav;
  updateClip(0);

  return Vector(ll,hh);
}

void FrameBase::resetClip(int force)
{
  if (DebugPerf)
    cerr << "resetClip" << endl;

  updateClip(force);

  // init histogram
  currentScale->deleteHistogramX();
  currentScale->deleteHistogramY();
  currentScale->deleteHistequ();

  // update histogram
  if (currentScale->colorScaleType() == FrScale::HISTEQUSCALE)
    updateColorScale();
}

void FrameBase::updateClip(int force)
{
  if (DebugPerf)
    cerr << "updateClip" << endl;

  if (!currentScale->preserve() || force) {
    currentScale->setMin(DBL_MAX);
    currentScale->setMax(-DBL_MAX);

    currentScale->setLow(DBL_MAX);
    currentScale->setHigh(-DBL_MAX);

    if (*channelFits) {
      // find min/max
      {
	FitsImage* ptr = *channelFits;
	while (ptr) {
	  ptr->updateClip(currentScale);

	  // find over-all min/max
	  if (currentScale->min() > ptr->getMinDouble())
	    currentScale->setMin(ptr->getMinDouble());
      
	  if (currentScale->max() <= ptr->getMaxDouble())
	    currentScale->setMax(ptr->getMaxDouble());

	  // find low/high
	  if (currentScale->low() > ptr->getLowDouble())
	    currentScale->setLow(ptr->getLowDouble());

	  if (currentScale->high() <= ptr->getHighDouble())
	    currentScale->setHigh(ptr->getHighDouble());

	  ptr = ptr->next();
	}
      }

      // set global
      {
	if ((hasFitsCube() || hasFitsMosaic()) && 
	    currentScale->clipScope() == FrScale::GLOBAL) {
	  FitsImage* ptr = *channelFits;
	  while (ptr) {
	    ptr->setClip(currentScale->low(), currentScale->high());
	    ptr = ptr->next();
	  }
	}
      }
    }
    else {
      if (currentScale->clipMode() != FrScale::USERCLIP) {
	currentScale->setLow(DEFAULTLOW);
	currentScale->setHigh(DEFAULTHIGH);
      }
      else {
	currentScale->setLow(currentScale->uLow());
	currentScale->setHigh(currentScale->uHigh());
      }
    }
  }
  else {
    FitsImage* ptr = *channelFits;
    while (ptr) {
      ptr->updateClip(currentScale);
      ptr->setClip(currentScale->low(), currentScale->high());
      ptr = ptr->next();
    }
  }
}

void FrameBase::updateWCSbb()
{
  // we may need to update WCSbb

  switch (*currentMosaic) {
  case FrameBase::NOMOSAIC:
  case FrameBase::IRAF:
    break;
  case FrameBase::WCSMOSAIC:
  case FrameBase::WFPC2:
    if (*channelFits) {
      FitsImage* ptr = *channelFits;
      while (ptr) {
	ptr->updateWCSbb(*currentMosaicSystem, (*channelFits == ptr));
	ptr = ptr->next();
      }
    }
    break;
  }
}

void FrameBase::setScanModeIncr(LoadMethod lm)
{
  if (lm == INCR)
    switch (currentScale->scanMode()) {
    case FrScale::NODATASEC:
    case FrScale::UNODATASEC:
    case FrScale::RESET:
      currentScale->setScanMode(FrScale::UNODATASEC);
      break;
    case FrScale::DATASEC:
    case FrScale::UDATASEC:
      currentScale->setScanMode(FrScale::UDATASEC);
      break;
    }
}

void FrameBase::calcHistEqu()
{
  // if we don't have any data, bail
  if (!*channelFits) {
    currentScale->deleteHistequ();
    return;
  }

  // if we already have it, bail
  if (currentScale->histequ())
    return;

  if (DebugPerf)
    cerr << "calcHistEqu...";

  // create pdf or histogram
  double* pdf = new double[HISTEQUSIZE];
  memset(pdf,0,HISTEQUSIZE*sizeof(double));

  FitsImage* ptr = *channelFits;
  while (ptr) {
    ptr->bin(pdf, HISTEQUSIZE, currentScale->min(), currentScale->max(), 
	     currentScale->scanMode());
    ptr = ptr->next();
  }

  // find a total/average
  double total, average;
  {
    total = 0;
    for (int i=0; i<HISTEQUSIZE; i++)
      total += pdf[i];
    average = total/HISTEQUSIZE;
  }

  // knock down peaks
  {
    for (int i=0; i<HISTEQUSIZE; i++)
      if (pdf[i] > average*3)
	pdf[i] = average*3;
  }

  // recalc total/average
  {
    total = 0;
    for (int i=0; i<HISTEQUSIZE; i++)
      total += pdf[i];
    average = total/HISTEQUSIZE;
  }

  // build transfer function (cdf)
  double* histequ = currentScale->initHistequ(HISTEQUSIZE);

  double bin = 0;
  int color,level;
  for (color=level=0; level<HISTEQUSIZE && color<HISTEQUSIZE; level++) {
    histequ[level] = (double)color/HISTEQUSIZE;
    bin += pdf[level];
    while (bin>=average && color<HISTEQUSIZE) {
      bin -= average;
      color++;
    }
  }
  while (level<HISTEQUSIZE)
    histequ[level++] = (double)(HISTEQUSIZE-1)/HISTEQUSIZE;

  delete pdf;

  if (DebugPerf)
    cerr << "end" << endl;
}

void FrameBase::updateBinFileNames()
{
  // we only want to do this to binn'd 3D cubes
  if (!(*channelFits)->isTable())
    return;

  char* zcol = (char*)(*channelFits)->getHistZ();
  int bd = (*channelFits)->binDepth();
  if (bd>1 && zcol) {
    Vector zlim = (*channelFits)->getHistZlim();
    double zlen = zlim[1]-zlim[0];
    double zdelta = zlen/bd;

    double zptr = zlim[0];
    FitsImage* ptr = *channelFits;
    for (int i=0; i<(*channelFits)->depth();i++,ptr=ptr->next(),zptr+=zdelta) {
#if __GNUC__ >= 3
      ostringstream str;
      str << zcol << ">=" << zptr << '&' << zcol << '<' << zptr+zdelta << ends;
      ptr->setBinSliceFilter(str.str().c_str());
#else
      char buf[256];
      ostrstream str(buf,256);
      str << zcol << ">=" << zptr << '&' << zcol << '<' << zptr+zdelta << ends;
      ptr->setBinSliceFilter(buf);
#endif
      ptr->updateFileName();
    } 
  }
  else {
    (*channelFits)->setBinSliceFilter(NULL);
    (*channelFits)->updateFileName();
  }
}

double FrameBase::calcZoom(int width, int height, Vector ur)
{
  // we need to calculate the width and height of the rotated image
  // so we can derived a zoom factor to shrink it to fit the requested size

  Vector ll(0,0);
  Vector ul(ll[0],ur[1]);
  Matrix m = Translate(-ur/2.) * Rotate(wcsRotation) * Rotate(rotation);

  Vector cc = ll * m;
  Vector dd = ul * m;

  float z1 = fabs(cc[0]) > fabs(cc[1]) ? fabs(cc[0]) : fabs(cc[1]);
  float z2 = fabs(dd[0]) > fabs(dd[1]) ? fabs(dd[0]) : fabs(dd[1]);

  return ((width<height ? width:height) / ((z1>z2 ? z1:z2)*2));
}

void FrameBase::invalidPixmap()
{
  Widget::invalidPixmap();

  if (basePixmap)
    Tk_FreePixmap(display, basePixmap);
  basePixmap = 0;

  if (baseXImage)
    XDestroyImage(baseXImage);
  baseXImage = NULL;

  needsUpdate = MATRIX;
}

int FrameBase::updatePixmap(const BBox& bb)
{

  // Note: lack of breaks-- on purpose. 
  // If Matrices are update, both Base and Pixmap
  // also need to be updated. Same for Base-- ie, pixmap is also updated.
  // bb is in canvas coords

  switch (needsUpdate) {
  case MATRIX:
    updateMatrices();
    markerbg();
    updateMarkers();
    markerfg();
    updateMarkers();
  case BASE:
    updateBase(bb);
  case PIXMAP:
    updatePM(bb);
    break;
  }

  needsUpdate = NOUPDATE;
  return TCL_OK;
}

void FrameBase::updateBase(const BBox& bbox)
{
  //  XSync(display,0);
  if (DebugPerf)
    cerr << "updateBase... ";

  int& width = options->width;
  int& height = options->height;

  if (!basePixmap) {
    if (!(basePixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin), 
				    width, height, depth))){
      cerr << "Frame Internal Error: Unable to Create Pixmap" << endl;
      exit(1);
    }

    // Geometry has changed, redefine our marker GCs including clip regions

    updateGCs();
  }

  if (!baseXImage) {
    if (!(baseXImage = XGETIMAGE(display, basePixmap, 0, 0, width, height, 
				 AllPlanes, ZPixmap))) {
      cerr << "Frame Internal Error: Unable to Create XImage" << endl;
      exit(1);
    }

    // we have a race condition. Some Truecolor ColorScales need to know the 
    // bytes per pixel, RGB masks, and byte order, from XImage struct.
    // yet, we receive colormap commands well before we render to the 
    // screen and have a valid XImage. 
    // So, we put off creating a colorscale until we are ready to render.

    if (!validColorScale())
      updateColorScale();
  }

  if (!bbox.isEmpty()) {
    BBox bb = bbox * canvasToWidget;
    int x0 = (int)bb.ll[0] > 0 ? (int)bb.ll[0] : 0;
    int y0 = (int)bb.ll[1] > 0 ? (int)bb.ll[1] : 0;
    int x1 = (int)bb.ur[0] < width ? (int)bb.ur[0] : width;
    int y1 = (int)bb.ur[1] < height ? (int)bb.ur[1] : height;

    if (DebugPerf)
      cerr << ' ' << x0 << ' ' << y0 << ' ' << x1 << ' ' << y1 << ' ';
    
    if (doRender())
      ximageToPixmap(basePixmap, baseXImage, x0, y0, x1, y1,
		     &FitsImage::getmWidgetToData);
    else {
      XSetForeground(display, gc, bgColor->pixel);
      XFillRectangle(display, basePixmap, gc, x0, y0, x1-x0, y1-y0);
    }
  }
  
  // contours
  renderContours(basePixmap, refToWidget, bbox*canvasToRef);

  // grid
  if (grid)
    if (!(grid->render()))
	cerr << "Frame Internal Error: Unable to render Grid" << endl;

  // background markers
  if (showbgMarkers)
    renderbgMarkers(bbox);

  // panner
  updatePanner();

  if (DebugPerf)
    cerr << "end" << endl;
}

void FrameBase::updatePM(const BBox& bbox)
{
  // bbox is in Canvas Coords

  if (DebugPerf)
    cerr << "updatePM...";

  int& width = options->width;
  int& height = options->height;

  if (!pixmap) {
    if (!(pixmap = Tk_GetPixmap(display, Tk_WindowId(tkwin), 
				width, height, depth))) {
      cerr << "Frame Internal Error: Unable to Create Pixmap" << endl;
      exit(1);
    }
  }

  if (!bbox.isEmpty()) {
    BBox bb = bbox * canvasToWidget;
    int x0 = (int)bb.ll[0] > 0 ? (int)bb.ll[0] : 0;
    int y0 = (int)bb.ll[1] > 0 ? (int)bb.ll[1] : 0;
    int x1 = (int)bb.ur[0] < width ? (int)bb.ur[0] : width;
    int y1 = (int)bb.ur[1] < height ? (int)bb.ur[1] : height;
    int sx = x1-x0;
    int sy = y1-y0;

    if (DebugPerf)
      cerr << ' ' << x0 << ' ' << y0 << ' ' << x1 << ' ' << y1 << ' ';

    XCopyArea(display, basePixmap, pixmap, gc, x0, y0, sx, sy, x0, y0);
  }

  // foreground markers
  if (showfgMarkers)
    renderfgMarkers(bbox);

  // crosshair
  renderCrosshair();

  // hightlite bbox
  renderHighLite();

  if (DebugPerf)
    cerr << "end" << endl;
}

void FrameBase::updateGCs()
{
  // widget clip region

  BBox bbWidget = BBox(0, 0, options->width-1, options->height-1);
  Vector sizeWidget = bbWidget.size();

  rectWidget[0].x = (int)bbWidget.ll[0];
  rectWidget[0].y = (int)bbWidget.ll[1];
  rectWidget[0].width = (int)sizeWidget[0];
  rectWidget[0].height = (int)sizeWidget[1];

  // window clip region

  BBox bbWindow = bbWidget * widgetToWindow;
  Vector sizeWindow = bbWindow.size();

  rectWindow[0].x = (int)bbWindow.ll[0];
  rectWindow[0].y = (int)bbWindow.ll[1];
  rectWindow[0].width = (int)sizeWindow[0];
  rectWindow[0].height = (int)sizeWindow[1];

  // crosshairGC

  if (!crosshairGC) {
    crosshairGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
    XSetForeground(display, crosshairGC, greenColor->pixel);
    XSetLineAttributes(display, crosshairGC, 1, LineSolid, CapButt, JoinMiter);
  }
  XSetClipRectangles(display, crosshairGC, 0, 0, rectWidget, 1, Unsorted);

  // crosshairGCXOR

  if (!crosshairGCXOR) {
    crosshairGCXOR = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
    XSetForeground(display, crosshairGCXOR, greenColor->pixel);
    XSetLineAttributes(display, crosshairGCXOR, 1, LineSolid, 
		       CapButt, JoinMiter);
  }
  XSetClipRectangles(display, crosshairGCXOR, 0, 0, rectWindow, 1, Unsorted);

  if (!highliteGC) {
    highliteGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
    XSetForeground(display, highliteGC, blueColor->pixel);
    XSetLineAttributes(display, highliteGC, 2, LineSolid, CapButt, JoinMiter);
  }
  XSetClipRectangles(display, highliteGC, 0, 0, rectWidget, 1, Unsorted);

  // contourGC

  if (!contourGC) {
    contourGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
    XSetLineAttributes(display, contourGC, 1, LineSolid, CapButt, JoinMiter);
  }
  XSetClipRectangles(display, contourGC, 0, 0, rectWidget, 1, Unsorted);

  // gridGC

  if (!gridGC) {
    gridGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
  }
  XSetClipRectangles(display, gridGC, 0, 0, rectWidget, 1, Unsorted);

  // markerGC

  if (!markerGC)
    markerGC = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);

  if (!selectGCXOR) {
    selectGCXOR = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
    XSetForeground(display, selectGCXOR, getWhiteColor());
    char dlist[] = {8,3};
    XSetLineAttributes(display, selectGCXOR, 1, LineOnOffDash, 
                     CapButt, JoinMiter);
    XSetDashes(display, selectGCXOR, 0, dlist, 2);
  }
  XSetClipRectangles(display, selectGCXOR, 0, 0, rectWindow, 1, Unsorted);

  // panGCXOR

  if (!panGCXOR)
    panGCXOR = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
  XSetClipRectangles(display, panGCXOR, 0, 0, rectWindow, 1, Unsorted);

  // rotateGCXOR

  if (!rotateGCXOR)
    rotateGCXOR = XCreateGC(display, Tk_WindowId(tkwin), 0, NULL);
  XSetClipRectangles(display, rotateGCXOR, 0, 0, rectWindow, 1, Unsorted);
}

void FrameBase::ximageToPixmap(Pixmap pmap, XImage* xmap, 
			       int x0, int y0, int x1, int y1, 
			       double* (FitsImage::*getMatrix)())
{
  buildXImage(xmap, x0, y0, x1, y1, getMatrix);
  XPutImage(display, pmap, gc, xmap, x0, y0, x0, y0, x1-x0, y1-y0);
}

void FrameBase::updatePanner()
{
  if (usePanner)
    if (doRender())
      ximageToPixmap(pannerPixmap, pannerXImage, 
		     0, 0, pannerWidth, pannerHeight,
		     &FitsImage::getmPannerToData);
    else {
      XSetForeground(display, gc, bgColor->pixel);
      XFillRectangle(display, pannerPixmap, gc, 
  		     0, 0, pannerWidth, pannerHeight);
    }  

  pannerPush();
}

double FrameBase::calcPannerZoom()
{
  if (pannerPixmap)
    return calcZoom(pannerWidth,pannerHeight,imageSize());
  else
    return 1;
}

void FrameBase::pannerPush()
{
  if (usePanner) {
#if __GNUC__ >= 3
    ostringstream str;
#else
    char buf[512];
    ostrstream str(buf,512);
#endif

    // always render (to update panner background color)
    // calculate bbox
    BBox bb = BBox(Vector(0,0), Vector(options->width, options->height)) * 
      widgetToUser * userToPanner;

    // calculate compass vectors
    Rotate orient = Rotate(wcsRotation) * Rotate(rotation) * 
      wcsOrientationMatrix * orientationMatrix;

    str << pannerName << " update " << pannerPixmap << ";";
    str << pannerName << " update bbox " << bb << ";";
    str << pannerName << " update image compass " << orient << ";";

    if (hasWCS(pannerSystem)) {
      Matrix m = currentFits->getImageToUser() * userToPanner;

      Vector orval = currentFits->pix2wcs(currentFits->getWCStan(pannerSystem),
					  pannerSystem, pannerSky);
      Vector orpix = currentFits->wcs2pix(orval,pannerSystem,pannerSky);
      Vector origin = orpix*m;

      // find normalized north
      Vector delta = currentFits->getWCScdelt(pannerSystem).abs();
      Vector north = currentFits->wcs2pix(Vector(orval[0],orval[1]+delta[1]), 
					  pannerSystem,pannerSky)*m;
      Vector nnorm = (north-origin).normalize();

      // find normalized east
      Vector east = currentFits->wcs2pix(Vector(orval[0]+delta[0],orval[1]), 
					 pannerSystem,pannerSky)*m;
      Vector enorm = (east-origin).normalize();

      // and update the panner
      str << pannerName << " update wcs compass " << nnorm << enorm << ends;
    }
    else
      str << pannerName << " update wcs compass invalid" << ends;

#if __GNUC__ >= 3
    Tcl_Eval(interp, str.str().c_str());
#else
    Tcl_Eval(interp, buf);
#endif
  }
}

// Magnifier Support

void FrameBase::updateMagnifier()
{
  updateMagnifier(magnifierCursor);
}

void FrameBase::updateMagnifier(const Vector& v)
{
  // v is in CANVAS coords
  // save it, we may need it later
  magnifierCursor = v;

  if (useMagnifier && magnifierXImage && magnifierGC && doRender()) {

    // clear XImage data

    memset(magnifierXImage->data, 0, magnifierXImage->bytes_per_line * 
	   magnifierHeight);

    // fill XImage

    updateMagnifierZoom(v); // pixmap is updated inside

    // render cursor
    
    if (useMagnifierCursor)
      renderMagnifierCursor(v);

    // notify the magnifier widget

#if __GNUC__ >= 3
    ostringstream str;
    str << magnifierName << " update " << magnifierPixmap << ends;
    Tcl_Eval(interp, str.str().c_str());
#else
    char buf[64];
    ostrstream str(buf,64);
    str << magnifierName << " update " << magnifierPixmap << ends;
    Tcl_Eval(interp, buf);
#endif
  }
}

void FrameBase::updateMagnifierMatrix()
{
  // calculate partial magnifier matrix
  // This is a partial matix, it lacks the Translate at the begining

  userToMagnifier =                     
    Rotate(rotation) *                  // rotate about cursor position
    Rotate(wcsRotation) *               // rotate about center position
    orientationMatrix *                 // flip x/y axis about cursor position
    wcsOrientationMatrix *              // flip x/y axis about center
    Scale(zoom) *                       // scale about cursor position
    Scale(magnifierZoom) *              // scale
    Translate(magnifierWidth/2., magnifierHeight/2.);

  userToMagnifierCursor =                     
    Rotate(rotation) *
    Rotate(wcsRotation) *
    Scale(zoom) *
    Scale(magnifierZoom) *
    Translate(magnifierWidth/2., magnifierHeight/2.);

  Vector magCorrection = Vector(.5,-.5) *
    Rotate(rotation) *
    Rotate(wcsRotation) *
    orientationMatrix *
    wcsOrientationMatrix;

  ximageToMagnifier = 
    Translate(magCorrection) * // we need to offset by this correction
    Scale(magnifierZoom) *
    Translate(magnifierWidth/2.,magnifierHeight/2.);
}

void FrameBase::updateMagnifierZoom(const Vector& v)
{
  // v is in CANVAS coords
  Matrix refToMagnifier = refToUser * Translate(-(v * canvasToUser)) * 
    userToMagnifier;
  Matrix magnifierToRef = refToMagnifier.invert();

  updateMagnifierMatrices(refToMagnifier);

  // Update ximage

  buildXImage(magnifierXImage, 0, 0, magnifierWidth, magnifierHeight, 
	      &FitsImage::getmMagnifierToData);

  // copy XImage to Pixmap

  XPutImage(display, magnifierPixmap, gc, magnifierXImage, 0, 0, 0, 0, 
	    magnifierWidth, magnifierHeight);

  if (useMagnifierGraphics) {

    // render markers
    // markers bounding box is in canvas coords
    // map the magnifier to a bounding box in canvas coords
    Matrix m = magnifierToRef * refToCanvas;
    BBox bb = BBox(v,v);
    bb.bound(Vector(0,0) * m);
    bb.bound(Vector(magnifierWidth,magnifierHeight) * m);

    if (showbgMarkers) {
      markerbg();
      renderMagnifierMarkers(refToMagnifier, bb);
    }
    if (showfgMarkers) {
      markerfg();
      renderMagnifierMarkers(refToMagnifier, bb);
    }
    markerfg();

    // render crosshair
    if (useCrosshair) {
      Vector r = crosshair * refToMagnifier;
      if (r[0]>=0 && r[0]<magnifierWidth)
	XDRAWLINE(display, magnifierPixmap, crosshairGC, 
		  (int)r[0], 1, (int)r[0], magnifierHeight);

      if (r[1]>=0 && r[1]<magnifierHeight)
	// bump Y by one, I don't know why, it just works
	XDRAWLINE(display, magnifierPixmap, crosshairGC,
		  1, (int)r[1], magnifierWidth, (int)r[1]);

    }

    // render contours
    renderContours(magnifierPixmap, refToMagnifier, bb*canvasToRef);
  }
}

void FrameBase::renderMagnifierCursor(const Vector& v)
{
    // first, the inner white box

    Vector u = v * canvasToUser;
    Matrix m = Translate(-u) * userToMagnifierCursor;

    Vector c[5];
    c[0] = (u + Vector(-.5,-.5)) * m;
    c[1] = (u + Vector( .5,-.5)) * m;
    c[2] = (u + Vector( .5, .5)) * m;
    c[3] = (u + Vector(-.5, .5)) * m;
    c[4] = c[0];
    
    XPoint pts[5];
    for (int i=0; i<5; i++) {
      Vector z = c[i].round();
      pts[i].x = (short)z[0];
      pts[i].y = (short)z[1];
    }
    XSetForeground(display, magnifierGC, whiteColor->pixel);
    XDRAWLINES(display, magnifierPixmap, magnifierGC, pts, 5, CoordModeOrigin);

    // now, the outer black box

    Matrix n =
      Translate(-(u * userToMagnifierCursor)) *
      Rotate(-rotation) *
      Rotate(-wcsRotation);
    Matrix o = n.invert();

    Vector d[5];
    {
      for (int i=0; i<5; i++)
	d[i] = c[i] * n;
    }

    d[0]+=Vector(-1,-1);
    d[1]+=Vector( 1,-1);
    d[2]+=Vector( 1, 1);
    d[3]+=Vector(-1, 1);
    d[4]=d[0];

    {
      for (int i=0; i<5; i++) {
	d[i] = d[i] * o;
	Vector z = d[i].round();
	pts[i].x = (int)z[0];
	pts[i].y = (int)z[1];
      }
    }
    XSetForeground(display, magnifierGC, blackColor->pixel);
    XDRAWLINES(display, magnifierPixmap, magnifierGC, pts, 5, CoordModeOrigin);
}

double FrameBase::getWCSRotation(CoordSystem sys, SkyFrame sky)
{
  return currentFits ? currentFits->getWCSRotation(sys,sky) : 0;
}

Orientation FrameBase::getWCSOrientation(CoordSystem sys, SkyFrame sky)
{
  return currentFits ? currentFits->getWCSOrientation(sys, sky) : NORMAL;
}

int FrameBase::hasLTMV()
{
  return currentFits && currentFits->hasLTMV();
}

int FrameBase::hasATMV()
{
  return currentFits && currentFits->hasATMV();
}

int FrameBase::hasDTMV()
{
  return currentFits && currentFits->hasDTMV();
}

int FrameBase::hasWCS(CoordSystem sys)
{ 
  return currentFits && currentFits->hasWCS(sys);
}

int FrameBase::hasWCSEqu(CoordSystem sys)
{ 
  return currentFits && currentFits->hasWCSEqu(sys);
}

int FrameBase::hasWCSLinear(CoordSystem sys)
{ 
  return currentFits && currentFits->hasWCSLinear(sys);
}

void FrameBase::renderCrosshair()
{
  Vector r = crosshair * refToWidget;

  if (useCrosshair) {
    XDRAWLINE(display, pixmap, crosshairGC, 
	      (int)r[0], 1, (int)r[0], options->height);
    XDRAWLINE(display, pixmap, crosshairGC,
	      1, (int)r[1], options->width, (int)r[1]);
  }
}

void FrameBase::drawCrosshair()
{
  if (visible) {
    Vector r = crosshair * refToWidget;

    // Vertical line

    BBox vv = BBox(Vector(r[0], 1), Vector(r[0], options->height)) * 
      widgetToWindow;
    XDRAWLINE(display, Tk_WindowId(tkwin), crosshairGCXOR, 
	      (int)vv.ll[0], (int)vv.ll[1], (int)vv.ur[0], (int)vv.ur[1]);

    // Horizontal line

    BBox hh = BBox(Vector(1, r[1]), Vector(options->width, r[1])) * 
      widgetToWindow;
    XDRAWLINE(display, Tk_WindowId(tkwin), crosshairGCXOR, 
	      (int)hh.ll[0], (int)hh.ll[1], (int)hh.ur[0], (int)hh.ur[1]);
  }
}

void FrameBase::eraseCrosshair()
{
  if (visible) {
    // This routine schedules an update for only the area written to previously
    // by drawCursor.

    Vector r = crosshair * refToWidget;

    // Vertical line

    BBox vv = BBox(Vector(r[0], 1), Vector(r[0], options->height)) * 
      widgetToCanvas;
    redrawNow(vv.expand(1));

    // Horizontal line

    BBox hh = BBox(Vector(1, r[1]), Vector(options->width, r[1])) * 
      widgetToCanvas;
    redrawNow(hh.expand(1));
  }
}

#if __GNUC__ >= 3
void FrameBase::psCrosshair(PSColorSpace mode)
{
  Vector r = crosshair * refToWidget;
  Vector a = Vector(r[0],1) * widgetToCanvas;
  Vector b = Vector(r[0],options->height) * widgetToCanvas;
  Vector c = Vector(1,r[1]) * widgetToCanvas;
  Vector d = Vector(options->width,r[1]) * widgetToCanvas;

  {
    ostringstream str;
    switch ((FrameBase::PSColorSpace)mode) {
    case FrameBase::BW:
    case FrameBase::GRAY:
      psColorGray("green", str);
      str << " setgray";
      break;
    case FrameBase::RGB:
      psColorRGB("green", str);
      str << " setrgbcolor";
      break;
    case FrameBase::CMYK:
      psColorCMYK("green", str);
      str << " setcmykcolor";
      break;
    }
    str << endl << ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }  

  { 
    ostringstream str;
    str << "newpath " 
	<< a.TkCanvasPs(canvas) << "moveto"
	<< b.TkCanvasPs(canvas) << "lineto"
	<< " stroke" << endl
	<< "newpath " 
	<< c.TkCanvasPs(canvas) << "moveto"
	<< d.TkCanvasPs(canvas) << "lineto"
	<< " stroke" << endl
	<< ends;
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }
}
#else
void FrameBase::psCrosshair(PSColorSpace mode)
{
  Vector r = crosshair * refToWidget;
  Vector a = Vector(r[0],1) * widgetToCanvas;
  Vector b = Vector(r[0],options->height) * widgetToCanvas;
  Vector c = Vector(1,r[1]) * widgetToCanvas;
  Vector d = Vector(options->width,r[1]) * widgetToCanvas;

  char buf[256];
  {
    ostrstream str(buf,256);
    switch ((FrameBase::PSColorSpace)mode) {
    case FrameBase::BW:
    case FrameBase::GRAY:
      psColorGray("green", str);
      str << " setgray";
      break;
    case FrameBase::RGB:
      psColorRGB("green", str);
      str << " setrgbcolor";
      break;
    case FrameBase::CMYK:
      psColorCMYK("green", str);
      str << " setcmykcolor";
      break;
    }
    str << endl << ends;
    Tcl_AppendResult(interp, buf, NULL);
  }  

  { 
    ostrstream str(buf,256);
    str << "newpath " 
	<< a.TkCanvasPs(canvas) << "moveto"
	<< b.TkCanvasPs(canvas) << "lineto"
	<< " stroke" << endl
	<< "newpath " 
	<< c.TkCanvasPs(canvas) << "moveto"
	<< d.TkCanvasPs(canvas) << "lineto"
	<< " stroke" << endl
	<< ends;
    Tcl_AppendResult(interp, buf, NULL);
  }
}
#endif

void FrameBase::renderContours(Pixmap pm, const Matrix& mm, const BBox& bb)
{
  // render from back to front
  // aux contours
  Contour* ptr=auxcontours.tail();
  while (ptr) {
    ptr->render(pm, mm, bb);
    ptr=ptr->previous();
  }

  // main contour
  if (hasContour())
    (*currentcontour)->render(pm, mm, bb);
}

void FrameBase::renderHighLite()
{
  if (useHighlite)
    XDRAWRECTANGLE(display, pixmap, highliteGC, 1, 1, 
		   options->width-2, options->height-2);
}


Vector FrameBase::mapFromRef(const Vector& v, InternalSystem sys)
{
  switch (sys) {
  case CANVAS:
    return v * refToCanvas;
  case PANNER:
    return v * refToPanner;
  }
}

Vector FrameBase::mapToRef(const Vector& v, InternalSystem sys)
{
  switch (sys) {
  case CANVAS:
    return v * canvasToRef;
  case PANNER:
    return v * pannerToRef;
  }
}

double FrameBase::mapLenFromRef(double d, InternalSystem sys)
{
  Vector r = mapLenFromRef(Vector(d,0),sys);
  return r[0];
}

Vector FrameBase::mapLenFromRef(const Vector& v, InternalSystem sys)
{
  switch (sys) {
  case CANVAS:
    {
      Vector v1 = Vector() * refToCanvas;
      Vector v2 = v * refToCanvas;
      return (v2-v1).abs();
    }
    break;
  case PANNER:
    {
      Vector v1 = Vector() * refToPanner;
      Vector v2 = v * refToPanner;
      return (v2-v1).abs();
    }
    break;
  }
}

double FrameBase::mapAngleFromRef(double angle, CoordSystem out)
{
  double r = angle;
  if (hasWCS(out)) {
    switch (getWCSOrientation(out,FK5)) {
    case NORMAL:
      r = angle + getWCSRotation(out,FK5);
      break;
    case XX:
      r = -angle - getWCSRotation(out,FK5);
      break;
    }
  }

  if (r>0)
    while (r>=2*M_PI)
      r -= 2*M_PI;
  else
    while (r<0)
      r += 2*M_PI;

  return r;
}

// special version, allows both 0 and 360
double FrameBase::mapAngleFromReff(double angle, CoordSystem out)
{
  double r = angle;
  if (hasWCS(out)) {
    switch (getWCSOrientation(out,FK5)) {
    case NORMAL:
      r = angle + getWCSRotation(out,FK5);
      break;
    case XX:
      r = -angle - getWCSRotation(out,FK5);
      break;
    }
  }

  if (r>0)
    while (r>2*M_PI)
      r -= 2*M_PI;
  else
    while (r<0)
      r += 2*M_PI;

  return r;
}

double FrameBase::mapAngleToRef(double angle, CoordSystem in)
{
  double r = angle;
  if (hasWCS(in)) {
    switch (getWCSOrientation(in,FK5)) {
    case NORMAL:
      r = angle - getWCSRotation(in,FK5);
      break;
    case XX:
      r = -angle - getWCSRotation(in,FK5);
      break;
    }
  }

  if (r>0)
    while (r>=2*M_PI)
      r -= 2*M_PI;
  else
    while (r<0)
      r += 2*M_PI;

  return r;
}

FitsImage* FrameBase::isInFits(const Vector& v, Matrix& (FitsImage::*getMatrix)(),
			   Vector* rv)
{
  switch (*currentMode) {
  case SINGLE:
    if (currentFits) {
      Vector r = v * (currentFits->*getMatrix)();
      int* params = currentFits->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      if (r[0]>=xmin && r[0]<xmax && r[1]>=ymin && r[1]<ymax) {
	if (rv)
	  *rv = r;
	return currentFits;
      }
      else
	return NULL;
    }
    else
      return NULL;

  case MOSAIC:
    FitsImage* ptr = *channelFits;
    while (ptr) {
      Vector r = v * (ptr->*getMatrix)();
      int* params = ptr->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      if (r[0]>=xmin && r[0]<xmax && r[1]>=ymin && r[1]<ymax) {
	if (rv)
	  *rv = r;
	return ptr;
      }
      ptr = ptr->next();
    }

    return NULL;
  }
}

FitsImage* FrameBase::findFits(const Vector& c)
{
  switch (*currentMode) {
  case SINGLE:
    return *keyFits;
    break;
  case MOSAIC:
    FitsImage* ptr = *channelFits;
    while (ptr) {
      Vector img = c * ptr->getRefToData();
      int* params = ptr->getDataParams(currentScale->scanMode());
      int& xmin = params[1];
      int& xmax = params[2];
      int& ymin = params[3];
      int& ymax = params[4];

      if (img[0]>=xmin && img[0]<xmax && img[1]>=ymin && img[1]<ymax)
	return ptr;

      ptr = ptr->next();
    }

    return currentFits;
    break;
  }
}

FitsImage* FrameBase::findFits(int which)
{
  switch (*currentMode) {
  case SINGLE:
    return *keyFits;
  case MOSAIC:
    FitsImage* ptr = *channelFits;
    for (int i=1; i<which; i++)
      if (ptr)
	ptr = ptr->next();

    if (ptr)
      return ptr;
    else
      return currentFits;
  }
}

int FrameBase::findFits(FitsImage* p)
{
  switch (*currentMode) {
  case SINGLE:
    return 1;
  case MOSAIC:
    FitsImage* ptr = *channelFits;
    int r = 0;
    while (ptr) {
      r++;
      if (ptr == p)
	return r;
      ptr = ptr->next();
    }

    return r;
  }
}

char* FrameBase::varcat(char* buf, char* var, char r)
{
  int l = strlen(var);
  strcpy(buf,var);
  buf[l++] = r;
  buf[l++] = NULL;
  return buf;
}

