// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "framepseudo8.h"
#include "colorscalepseudo8.h"
#include "fitsimage.h"
#include "nan.h"
#include "ps.h"
#include "util.h"

// Tk Canvas Widget Function Declarations

int FramePseudoColor8CreateProc(Tcl_Interp*, Tk_Canvas, Tk_Item*, int, 
				Tcl_Obj *const []);

// FramePseudoColor8 Specs

static Tk_CustomOption tagsOption = {
  Tk_CanvasTagsParseProc, Tk_CanvasTagsPrintProc, NULL
};

static Tk_ConfigSpec framePseudoColor8Specs[] = {

  {TK_CONFIG_STRING, "-command", NULL, NULL, "frame",
   Tk_Offset(WidgetOptions, cmdName), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-x", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, x), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-y", NULL, NULL, "1",
   Tk_Offset(WidgetOptions, y), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-width", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, width), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_INT, "-height", NULL, NULL, "512",
   Tk_Offset(WidgetOptions, height), TK_CONFIG_OPTION_SPECIFIED, NULL},
  {TK_CONFIG_ANCHOR, "-anchor", NULL, NULL, "nw",
   Tk_Offset(WidgetOptions, anchor), 0, NULL},
  {TK_CONFIG_CUSTOM, "-tags", NULL, NULL, NULL,
   0, TK_CONFIG_NULL_OK, &tagsOption},

  {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0, NULL},
};

// Tk Static Structure

static Tk_ItemType framePseudoColor8Type = {
  "framepseudocolor8",          // name
  sizeof(WidgetOptions),        // item size
  FramePseudoColor8CreateProc,  // configProc
  framePseudoColor8Specs,       // configSpecs
  WidgetConfigProc,             // configProc
  WidgetCoordProc,              // coordProc
  WidgetDeleteProc,             // deleteProc
  WidgetDisplayProc,            // displayProc
  0,                            // alwaysRedraw
  WidgetPointProc,              // pointProc
  WidgetAreaProc,               // areaProc
  WidgetPostscriptProc,         // postscriptProc
  WidgetScaleProc,              // scaleProc
  WidgetTranslateProc,          // translateProc
  (Tk_ItemIndexProc*)NULL,      // indexProc
  WidgetICursorProc,            // icursorProc
  (Tk_ItemSelectionProc*)NULL,  // selectionProc
  (Tk_ItemInsertProc*)NULL,     // insertProc
  (Tk_ItemDCharsProc*)NULL,     // dCharsProc
  (Tk_ItemType*)NULL            // nextPtr
};

// Non-Member Functions

int FramePseudoColor8_Init(Tcl_Interp* interp)
{
  Tk_CreateItemType(&framePseudoColor8Type);
  return TCL_OK;
}

int FramePseudoColor8CreateProc(Tcl_Interp* interp, Tk_Canvas canvas, 
				Tk_Item* item, int argc, Tcl_Obj *const argv[])
{
  FramePseudoColor8* frame = new FramePseudoColor8(interp, canvas, item);

  // and set default configuration

  if (frame->configure(argc, (const char**)argv, 0) != TCL_OK) {
    delete frame;
    Tcl_AppendResult(interp, " error occured while creating frame.", NULL);
    return TCL_ERROR;
  }

  return TCL_OK;
}

// FramePseudoColor8 Member Functions

FramePseudoColor8::FramePseudoColor8(Tcl_Interp* i, Tk_Canvas c, Tk_Item* item)
  : Frame(i, c, item)
{
  configSpecs = framePseudoColor8Specs;  // frame configure options
}

FramePseudoColor8::~FramePseudoColor8()
{
  // we must do this at this level, because updateColorScale is called
  unloadAllFits();
}

void FramePseudoColor8::bgColorCmd(const char* color)
{
  if (bgColorName)
    delete [] bgColorName;
  bgColorName = dupstr(color);
  bgColor = getXColor(bgColorName);
  update(BASE);
}

void FramePseudoColor8::updateColorScale()
{
  // we need colors before we can construct a scale

  if (!indexCells || !colorCells)
    return;

  if (colorScale)
    delete colorScale;

  switch (frScale.colorScaleType()) {
  case FrScale::LINEARSCALE:
    colorScale =
      new LinearScalePseudoColor8(colorCount, indexCells, colorCells, 
				  colorCount);
    break;
  case FrScale::LOGSCALE:
    colorScale = new LogScalePseudoColor8(SCALESIZE, indexCells, colorCells, 
					  colorCount);
    break;
  case FrScale::SQUAREDSCALE:
    colorScale =
      new SquaredScalePseudoColor8(SCALESIZE, indexCells, colorCells, 
				   colorCount);
    break;
  case FrScale::SQRTSCALE:
    colorScale = 
      new SqrtScalePseudoColor8(SCALESIZE, indexCells, colorCells, colorCount);
    break;
  case FrScale::IISSCALE:
    colorScale =
      new IISScalePseudoColor8(this,indexCells, colorCells, colorCount);
    break;
  case FrScale::HISTEQUSCALE:
    calcHistEqu();
    colorScale =
      new HistEquScalePseudoColor8(SCALESIZE, indexCells, colorCells, 
				   colorCount, currentScale->histequ(),
				   HISTEQUSIZE);
    break;
  }
}

void FramePseudoColor8::fillXImage(XImage* xmap, 
				   int x0, int y0, int x1, int y1,
				   double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render
  if (!colorScale)
    return;

  if (!currentFits)
    return;

  int& width = xmap->width;
  int& height = xmap->height;
  int& bytesPerLine = xmap->bytes_per_line;

  double* m = (currentFits->*getMatrix)();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->colors();

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  // clear ximage

#ifndef _WIN32
  memset(xmap->data, bgColor->pixel, bytesPerLine * height);
#else
  memset(xmap->data, 10, bytesPerLine * height);
#endif

  // test as doubles, convert to int's at last minute.
  // use double to maintain precision
  // This solves a host of problems.

  for (int j=y0; j<y1; j++) {
    // the line may be padded at the end
    char* dest = xmap->data + j*bytesPerLine + x0;

    for (int i=x0; i<x1; i++,dest++) {
      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isNaNd(value)) {
	  if (value <= ll)
	    *dest = table[0];
	  else if (value >= hh)
	    *dest = table[length];
	  else
	    *dest = table[(int)(((value - ll)/diff * length) + .5)];
	}
	else
	  *dest = nanColor->pixel;
      }
    }
  }
}

void FramePseudoColor8::fillXImageMosaic(XImage* xmap, 
					 int x0, int y0, int x1, int y1,
					 double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  int& width = xmap->width;
  int& height = xmap->height;
  int& bytesPerLine = xmap->bytes_per_line;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->colors();

  // clear ximage

#ifndef _WIN32
  memset(xmap->data, bgColor->pixel, bytesPerLine * height);
#else
  memset(xmap->data, 10, bytesPerLine * height);
#endif

  // test as doubles, convert to int's at last minute.
  // use double to maintain precision
  // This solves a host of problems.

  for (int j=y0; j<y1; j++) {
    // the line may be padded at the end
    char* dest = xmap->data + j*bytesPerLine + x0;

    for (int i=x0; i<x1; i++,dest++) {

      FitsImage* ptr = fits;
      while (ptr) {
	double* m = (ptr->*getMatrix)();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = ptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = ptr->getLowDouble();
	  double hh = ptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      *dest = table[0];
	    else if (value >= hh)
	      *dest = table[length];
	    else
	      *dest = table[(int)(((value - ll)/diff * length) + .5)];
	  }
	  else
	    *dest = nanColor->pixel;

	  break;
	}
	else
	  ptr = ptr->next();
      }
    }
  }
}

void FramePseudoColor8::fillXImageMosaicFast(XImage* xmap, 
					     int x0, int y0, int x1, int y1,
					     double* (FitsImage::*getMatrix)())
{
  // we need a colorScale before we can render

  if (!colorScale)
    return;

  if (!fits)
    return;

  int& width = xmap->width;
  int& height = xmap->height;
  int& bytesPerLine = xmap->bytes_per_line;
  int length = colorScale->size() - 1;
  const unsigned char* table = colorScale->colors();

  // clear ximage

#ifndef _WIN32
  memset(xmap->data, bgColor->pixel, bytesPerLine * height);
#else
  memset(xmap->data, 10, bytesPerLine * height);
#endif

  FitsImage* ptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  // test as doubles, convert to int's at last minute.
  // use double to maintain precision
  // This solves a host of problems.

  for (int j=y0; j<y1; j++) {
    // the line may be padded at the end
    char* dest = xmap->data + j*bytesPerLine + x0;

    for (int i=x0; i<x1; i++,dest++) {
      double x,y;

      if (ptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      *dest = table[0];
	    else if (value >= hh)
	      *dest = table[length];
	    else
	      *dest = table[(int)(((value - ll)/diff * length) + .5)];
	  }
	  else
	    *dest = nanColor->pixel;
	}
	else
	  ptr = NULL;
      }
      
      if (!ptr) {
	ptr = fits;
	while (ptr) {
	  m = (ptr->*getMatrix)();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = ptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = ptr->getLowDouble();
	    hh = ptr->getHighDouble();
	    diff = hh - ll;

	    if (!isNaNd(value)) {
	      if (value <= ll)
		*dest = table[0];
	      else if (value >= hh)
		*dest = table[length];
	      else
		*dest = table[(int)(((value - ll)/diff * length) + .5)];
	    }
	    else
	      *dest = nanColor->pixel;

	    break;
	  }
	  else
	    ptr = ptr->next();
	}
      }
    }
  }
}

void FramePseudoColor8::colormapCmd(int id, float b, float c, int i, 
				    unsigned short* index, 
				    unsigned char* cells, int cnt)
{
  cmapID = id;
  bias = b;
  contrast = c;
  invert = i;

  updateColorCells(index, cells, cnt);
  updateColorScale();
}

void FramePseudoColor8::rotateMotion()
{
  // Rotate from src to dest

  Vector c = Vector(options->width,options->height)/2;
  Matrix m = (Translate(-c) * 
	      Rotate(rotation-rotateRotation) * Translate(c)).invert();
  double* mm = m.mm();
  int bgc = bgColor->pixel;

  int& width = options->width;
  int& height = options->height;
  char* src = rotateSrcXM->data;

  for (int j=0; j<height; j++) {
    // the line may be padded at the end
    char* dest = rotateDestXM->data + j*rotateDestXM->bytes_per_line;

    for (int i=0; i<width; i++, dest++) {
      double x = i*mm[0] + j*mm[3] + mm[6];
      double y = i*mm[1] + j*mm[4] + mm[7];

      if (x >= 0 && x < width && y >= 0 && y < height)
	*dest = src[((int)y)*rotateSrcXM->bytes_per_line + (int)x];
      else
#ifndef _WIN32
	*dest = bgc;
#else
	*dest = 10;
#endif
    }
  }

  // XImage to Pixmap

  XPutImage(display, rotatePM, gc, rotateDestXM,
	    0, 0, 0, 0, options->width, options->height);

  // Display Pixmap

  Vector d = Vector(0,0) * widgetToWindow;
  XCopyArea(display, rotatePM, Tk_WindowId(tkwin), rotateGCXOR, 0, 0, 
	    options->width, options->height, (int)d[0], (int)d[1]);
}

void FramePseudoColor8::psLevel2Head(PSColorSpace mode, int width, int height)
{
#if __GNUC__ >= 3
  ostringstream str;
#else
  char buf[4096];
  ostrstream str(buf,4096);
#endif

  switch (mode) {
  case BW:
  case GRAY:
    {
      str << "[/Indexed /DeviceGray 255" << endl 
	  << '<' << endl;

      int count = 0;
      for (int i=0; i<256; i++) {
	unsigned short value = 0;

	unsigned char red;
	unsigned char green;
	unsigned char blue;
	if (psLevel2BaseColor(i, &red, &green, &blue))
	  value = RGB2Gray(red, green, blue);
	else
	  for (int j=0; j<colorCount; j++)
	    if (indexCells[j]==i) {
	      value = RGB2Gray(colorCells[j*3+2], colorCells[j*3+1], 
			       colorCells[j*3]);
	      break;
	    }

	str << hex << setfill('0') << setw(2) << value << ' ';

	if (++count == 16) {
	  str << endl;
	  count = 0;
	}
      }
    }
    break;
  case RGB:
    {
      str << "[/Indexed /DeviceRGB 255" << endl 
	  << '<' << endl;

      int count = 0;
      for (int i=0; i<256; i++) {
	unsigned short red = 0;
	unsigned short green = 0;
	unsigned short blue = 0;

	unsigned char r;
	unsigned char g;
	unsigned char b;
	if (psLevel2BaseColor(i, &r, &g, &b)) {
	  red = r;
	  green = g;
	  blue = b;
	}
	else
	  for (int j=0; j<colorCount; j++)
	    if (indexCells[j]==i) {
	      red = colorCells[j*3+2];
	      green = colorCells[j*3+1];
	      blue = colorCells[j*3];
	      break;
	    }

	str << hex << setfill('0') << setw(2) << red 
	    << hex << setfill('0') << setw(2) << green
	    << hex << setfill('0') << setw(2) << blue
	    << ' ';

	if (++count == 8) {
	  str << endl;
	  count = 0;
	}
      }
    }
    break;
  case CMYK:
    {
      str << "[/Indexed /DeviceCMYK 255" << endl 
	  << '<' << endl;

      int count = 0;
      for (int i=0; i<256; i++) {
	unsigned char c = 0;
	unsigned char m = 0;
	unsigned char y = 0;
	unsigned char b = 255;

	unsigned char red;
	unsigned char green;
	unsigned char blue;
	if (psLevel2BaseColor(i, &red, &green, &blue))
	  RGB2CMYK(red, green, blue, &c, &m, &y, &b);
	else
	  for (int j=0; j<colorCount; j++)
	    if (indexCells[j]==i) {
	      RGB2CMYK(colorCells[j*3+2], colorCells[j*3+1], colorCells[j*3],
		       &c, &m, &y, &b);
	      break;
	    }

	unsigned short cyan = c;
	unsigned short magenta = m;
	unsigned short yellow = y;
	unsigned short black = b;

	str << hex << setfill('0') << setw(2) << cyan 
	    << hex << setfill('0') << setw(2) << magenta
	    << hex << setfill('0') << setw(2) << yellow
	    << hex << setfill('0') << setw(2) << black
	    << ' ';

	if (++count == 8) {
	  str << endl;
	  count = 0;
	}
      }
    }
    break;
  }

  str << '>' << endl
      << "] setcolorspace" << endl
      << "<<" << endl
      << "/ImageType 1" << endl
      << "/Width " << dec << width << endl
      << "/Height " << dec << height << endl
      << "/BitsPerComponent 8" << endl
      << "/Decode [0 255]" << endl;

  if (psInterpolate)
    str << "/Interpolate true" << endl;

  str << "/ImageMatrix matrix" << endl
      << "/DataSource currentfile" << endl
      << "/ASCII85Decode filter" << endl
      << "/RunLengthDecode filter" << endl
      << ">>" << endl
      << "image" << endl
      << ends;

#if __GNUC__ >= 3
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
#else
  Tcl_AppendResult(interp, buf, NULL);
#endif
}

int FramePseudoColor8::psLevel2BaseColor(int i, unsigned char* red, 
				  unsigned char* green, unsigned char* blue)
{
  if (i == getBlackColor()) {
    *blue = 0;
    *green = 0;
    *red = 0;
  }
  else if (i == getWhiteColor()) {
    *blue = 255;
    *green = 255;
    *red = 255;
  }
  else if (i == getRedColor()) {
    *blue = 0;
    *green = 0;
    *red = 255;
  }
  else if (i == getGreenColor()) {
    *blue = 0;
    *green = 255;
    *red = 0;
  }
  else if (i == getBlueColor()) {
    *blue = 255;
    *green = 0;
    *red = 0;
  }
  else if (i == getCyanColor()) {
    *blue = 255;
    *green = 255;
    *red = 0;
  }
  else if (i == getMagentaColor()) {
    *blue = 255;
    *green = 0;
    *red = 255;
  }
  else if (i == getYellowColor()) {
    *blue = 0;
    *green = 255;
    *red = 255;
  }
  else
    return 0;

  return 1;
}

#if __GNUC__ >= 3
void FramePseudoColor8::psLevel2Single(PSColorSpace mode,
				       int width, int height, float scale)
{
  if (!currentFits)
    return;

  RLEAscii85 filter;
  currentFits->updatePS(psMatrix(scale, width, height));
  double* m = currentFits->getmPSToData();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  const unsigned short* table = colorScale->psIndex();

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isNaNd(value)) {
	  if (value <= ll)
	    filter << table[0];
	  else if (value >= hh)
	    filter << table[length];
	  else
	    filter << table[(int)(((value - ll)/diff * length) + .5)];
	}
	else
	  filter << nanColor->pixel;
      }
      else
	filter << bgColor->pixel;

      str << filter;
    }
    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FramePseudoColor8::psLevel2Single(PSColorSpace mode,
				       int width, int height, float scale)
{
  if (!currentFits)
    return;

  RLEAscii85 filter;

  // size = 2 char per byte * 3 bytes per pixel + estimate of endls + 
  // endl + ends
  int size = (int)(width*3*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  currentFits->updatePS(psMatrix(scale, width, height));
  double* m = currentFits->getmPSToData();
  int* params = currentFits->getDataParams(frScale.scanMode());
  int& srcw = params[0];
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int length = colorScale->size() - 1;
  const unsigned short* table = colorScale->psIndex();

  double ll = currentFits->getLowDouble();
  double hh = currentFits->getHighDouble();
  double diff = hh - ll;

  for (int j=0; j<height; j++) {
    ostrstream str(buf,size);

    for (int i=0; i<width; i++) {
      double x = i*m[0] + j*m[3] + m[6];
      double y = i*m[1] + j*m[4] + m[7];

      if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	double value = currentFits->getValueDouble(((int)y)*srcw + (int)x);
	
	if (!isNaNd(value)) {
	  if (value <= ll)
	    filter << table[0];
	  else if (value >= hh)
	    filter << table[length];
	  else
	    filter << table[(int)(((value - ll)/diff * length) + .5)];
	}
	else
	  filter << nanColor->pixel;
      }
      else
	filter << bgColor->pixel;

      str << filter;
    }
    str << ends;
    psFix(buf,size);
    Tcl_AppendResult(interp, buf, NULL);
  }

  ostrstream str(buf,size);
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(buf,size);
  Tcl_AppendResult(interp, buf, NULL);
  delete [] buf;
}
#endif

#if __GNUC__ >= 3
void FramePseudoColor8::psLevel2Mosaic(PSColorSpace mode,
				       int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;
  int length = colorScale->size() - 1;
  const unsigned short* table = colorScale->psIndex();
  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {

      ptr = fits;
      while (ptr) {
	double* m = ptr->getmPSToData();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = ptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = ptr->getLowDouble();
	  double hh = ptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      filter << table[0];
	    else if (value >= hh)
	      filter << table[length];
	    else
	      filter << table[(int)(((value - ll)/diff * length) + .5)];
	  }
	  else
	    filter << nanColor->pixel;

	  break;
	}
	else 
	  ptr = ptr->next();
      }

      if (!ptr)
	filter << bgColor->pixel;

      str << filter;
    }
    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FramePseudoColor8::psLevel2Mosaic(PSColorSpace mode,
				       int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;

  // size = 2 char per byte * 3 bytes per pixel + estimate of endls + 
  // endl + ends
  int size = (int)(width*3*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  int length = colorScale->size() - 1;
  const unsigned short* table = colorScale->psIndex();
  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  for (int j=0; j<height; j++) {
    ostrstream str(buf,size);

    for (int i=0; i<width; i++) {

      ptr = fits;
      while (ptr) {
	double* m = ptr->getmPSToData();
	double x = i*m[0] + j*m[3] + m[6];
	double y = i*m[1] + j*m[4] + m[7];

	int* params = ptr->getDataParams(frScale.scanMode());
	int& srcw = params[0];
	int& xmin = params[1];
	int& xmax = params[2];
	int& ymin = params[3];
	int& ymax = params[4];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  double ll = ptr->getLowDouble();
	  double hh = ptr->getHighDouble();
	  double diff = hh - ll;

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      filter << table[0];
	    else if (value >= hh)
	      filter << table[length];
	    else
	      filter << table[(int)(((value - ll)/diff * length) + .5)];
	  }
	  else
	    filter << nanColor->pixel;

	  break;
	}
	else 
	  ptr = ptr->next();
      }

      if (!ptr)
	filter << bgColor->pixel;

      str << filter;
    }
    str << ends;
    psFix(buf,size);
    Tcl_AppendResult(interp, buf, NULL);
  }

  ostrstream str(buf,size);
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(buf,size);
  Tcl_AppendResult(interp, buf, NULL);
  delete [] buf;
}
#endif

#if __GNUC__ >= 3
void FramePseudoColor8::psLevel2MosaicFast(PSColorSpace mode,
					   int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;
  int length = colorScale->size() - 1;
  const unsigned short* table = colorScale->psIndex();
  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  ptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  for (int j=0; j<height; j++) {
    ostringstream str;

    for (int i=0; i<width; i++) {
      double x,y;

      if (ptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      filter << table[0];
	    else if (value >= hh)
	      filter << table[length];
	    else
	      filter << table[(int)(((value - ll)/diff * length) + .5)];
	  }
	  else
	    filter << nanColor->pixel;
	}
	else 
	  ptr = NULL;
      }

      if (!ptr) {
	ptr = fits;
	while (ptr) {
	  m = ptr->getmPSToData();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = ptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = ptr->getLowDouble();
	    hh = ptr->getHighDouble();
	    diff = hh - ll;

	    if (!isNaNd(value)) {
	      if (value <= ll)
		filter << table[0];
	      else if (value >= hh)
		filter << table[length];
	      else
		filter << table[(int)(((value - ll)/diff * length) + .5)];
	    }
	    else
	      filter << nanColor->pixel;

	    break;
	  }
	  else 
	    ptr = ptr->next();
	}
      }

      if (!ptr)
	filter << bgColor->pixel;

      str << filter;
    }
    str << ends;
    psFix(str);
    Tcl_AppendResult(interp, str.str().c_str(), NULL);
  }

  ostringstream str;
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(str);
  Tcl_AppendResult(interp, str.str().c_str(), NULL);
}
#else
void FramePseudoColor8::psLevel2MosaicFast(PSColorSpace mode,
					   int width, int height, float scale)
{
  if (!fits)
    return;

  RLEAscii85 filter;

  // size = 2 char per byte * 3 bytes per pixel + estimate of endls + 
  // endl + ends
  int size = (int)(width*3*2*(1 + 1./80.) +.5) + 2;
  char* buf = new char[size];

  int length = colorScale->size() - 1;
  const unsigned short* table = colorScale->psIndex();
  FitsImage* ptr = fits;
  Matrix mm = psMatrix(scale, width, height);
  while (ptr) {
    ptr->updatePS(mm);
    ptr = ptr->next();
  }

  ptr = NULL;
  double* m;

  int* params;
  int srcw;
  int xmin;
  int xmax;
  int ymin;
  int ymax;

  double ll;
  double hh;
  double diff;

  for (int j=0; j<height; j++) {
    ostrstream str(buf,size);

    for (int i=0; i<width; i++) {
      double x,y;

      if (ptr) {
	x = i*m[0] + j*m[3] + m[6];
	y = i*m[1] + j*m[4] + m[7];

	if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	  double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	  if (!isNaNd(value)) {
	    if (value <= ll)
	      filter << table[0];
	    else if (value >= hh)
	      filter << table[length];
	    else
	      filter << table[(int)(((value - ll)/diff * length) + .5)];
	  }
	  else
	    filter << nanColor->pixel;
	}
	else 
	  ptr = NULL;
      }

      if (!ptr) {
	ptr = fits;
	while (ptr) {
	  m = ptr->getmPSToData();
	  x = i*m[0] + j*m[3] + m[6];
	  y = i*m[1] + j*m[4] + m[7];

	  params = ptr->getDataParams(frScale.scanMode());
	  srcw = params[0];
	  xmin = params[1];
	  xmax = params[2];
	  ymin = params[3];
	  ymax = params[4];

	  if (x>=xmin && x<xmax && y>=ymin && y<ymax) {
	    double value = ptr->getValueDouble(((int)y)*srcw + (int)x);

	    ll = ptr->getLowDouble();
	    hh = ptr->getHighDouble();
	    diff = hh - ll;

	    if (!isNaNd(value)) {
	      if (value <= ll)
		filter << table[0];
	      else if (value >= hh)
		filter << table[length];
	      else
		filter << table[(int)(((value - ll)/diff * length) + .5)];
	    }
	    else
	      filter << nanColor->pixel;

	    break;
	  }
	  else 
	    ptr = ptr->next();
	}
      }

      if (!ptr)
	filter << bgColor->pixel;

      str << filter;
    }
    str << ends;
    psFix(buf,size);
    Tcl_AppendResult(interp, buf, NULL);
  }

  ostrstream str(buf,size);
  filter.flush(str);
  str << endl << "~>" << endl << ends;
  psFix(buf,size);
  Tcl_AppendResult(interp, buf, NULL);
  delete [] buf;
}
#endif
