// Copyright (C) 1999-2004
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include <float.h>

#include "fitsimage.h"
#include "fvcontour.h"
#include "util.h"
#include "nan.h"

// It is a modified version of contour code found in Fv 2.4
// Fv may be obtained from the HEASARC (High Energy Astrophysics Science
// Archive Research Center) FTOOLS Web site at:
// http://heasarc.gsfc.nasa.gov/ftools/fv.html

// The original author is unknown.

FVContour::FVContour(FrameBase* p, FitsImage* fits, const char* c, int w,
		     float r, int m, ContourScale* s)
: resolution(r), Contour(p, c, w, m)
{
  scale = s;
  append(fits);
}

void FVContour::append(FitsImage* fits)
{
  if (resolution == 1.)
    unity(fits);
  else
    switch (method) {
    case SMOOTH:
      smooth(fits);
      break;
    case BLOCK:
      bin(fits);
      break;
    }
}

void FVContour::unity(FitsImage* fits)
{
  int* params = fits->getDataParams(parent->currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int width = fits->width();
  int height = fits->height();

  // blank img
  float* img = new float[width * height];
  {
    for(int j=0; j<height; j++)
      for(int i=0; i<width; i++)
	img[j*width + i] = FLT_MIN;
  }

  // fill img
  for(int j=ymin; j<ymax; j++)
    for(int i=xmin; i<xmax; i++) {
      int k = j*width + i;

      float v = fits->getValueFloat(k);
      if (!isNaNf(v))
	img[k] = v;
    }

  // do contours
  int status = build(width, height, img, fits->getDataToRef());

  // clean up
  delete img;

  if (status)
    cerr << "FVContour error" << endl;
}

void FVContour::smooth(FitsImage* fits)
{
  int width = fits->width();
  int height = fits->height();

  // blank img
  int size = width*height;
  float* img = new float[size];
  if (!img) {
    cerr << "FVContour Error: could not allocate enough memory" << endl;
    return;
  }
  for (int ii=0; ii<size; ii++)
    img[ii] = FLT_MIN;

  // generate kernal
  int r = resolution-1;
  //  float* kernal = tophat(r);
  float* kernal = gaussian(r);

  // convolve
  convolve(fits,kernal,img,r);
  
  // now, do contours
  int status = build(width, height, img, fits->getDataToRef());

  // cleanup
  delete kernal;
  delete img;

  if (status)
    cerr << "FVContour error" << endl;
}

void FVContour::convolve(FitsImage* fits, float* kernal, float* dest, int r)
{
  int* params = fits->getDataParams(parent->currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int width = fits->width();
  int rr = 2*r+1;

  for (int j=ymin; j<ymax; j++) {
    for (int i=xmin; i<xmax; i++) {
      int ir  = i-r;
      int irr = i+r;
      int jr = j-r;
      int jrr = j+r;

      for (int n=jr, nn=0; n<=jrr; n++, nn++) {
	if (n>=ymin && n<ymax) {
	  for (int m=ir, mm=0; m<=irr; m++, mm++) {
	    if (m>=xmin && m<xmax) {
	      float v = fits->getValueFloat(n*width+m);
	      if (!isNaNf(v)) {
		float k = kernal[nn*rr+mm];
		float* ptr = dest+(j*width+i);
		if (*ptr == FLT_MIN)
		  *ptr  = v*k;
		else
		  *ptr += v*k;
	      }
	    }
	  }
	}
      }
    }
  }
}

float* FVContour::tophat(int r)
{
  int rr = 2*r+1;
  int ksz = rr*rr;
  float* kernal = new float[ksz];
  memset(kernal, 0, ksz*sizeof(float));
  
  float kt = 0;
  for (int y=-r; y<=r; y++) {
    for (int x=-r; x<=r; x++) { 
      if ((x*x + y*y) <= r*r) {
	kernal[(y+r)*rr+(x+r)] = 1;
	kt++;
      }
    }
  }

  // normalize kernal
  for (int a=0; a<ksz; a++)
    kernal[a] /= kt;

  return kernal;
}

float* FVContour::gaussian(int r)
{
  int rr = 2*r+1;
  int ksz = rr*rr;
  float sigma = r/2.;
  float* kernal = new float[ksz];
  memset(kernal, 0, ksz*sizeof(float));
  
  float kt = 0;
  float a = 1./(sigma*sigma);
  float c = 1./(sigma*sigma);
  for (int y=-r; y<=r; y++) {
    for (int x=-r; x<=r; x++) { 
      if ((x*x + y*y) <= r*r) {
	float v = exp(-.5*(a*x*x + c*y*y));
	kernal[(y+r)*rr+(x+r)] = v;
	kt += v;
      }
    }
  }

  // normalize kernal
  for (int a=0; a<ksz; a++)
    kernal[a] /= kt;

  return kernal;
}

void FVContour::bin(FitsImage* fits)
{
  int* params = fits->getDataParams(parent->currentScale->scanMode());
  int& xmin = params[1];
  int& xmax = params[2];
  int& ymin = params[3];
  int& ymax = params[4];

  int width = fits->width();
  int height = fits->height();

  int rr = resolution;

  int w2 = (int)(width/rr);
  int h2 = (int)(height/rr);

  Matrix m = 
    Translate((Vector(-width,-height)/2).floor()) * 
    Scale(1./rr) * 
    Translate((Vector(w2,h2)/2).floor());
  Matrix n = m.invert();
  double* mm = m.mm();

  float* img = new float[w2 * h2];
  {
    for (int j=0; j<h2; j++)
      for (int i=0; i<w2; i++)
	img[j*w2 + i] = FLT_MIN;
  }

  short* count = new short[w2 * h2];
  memset(count, 0, w2*h2*sizeof(short));

  {
    for (int j=ymin; j<ymax; j++)
      for (int i=xmin; i<xmax; i++) {
	double x = i*mm[0] + j*mm[3] + mm[6];
	double y = i*mm[1] + j*mm[4] + mm[7];

	if (x >= 0 && x < w2 && y >= 0 && y < h2) {
	  int k = ((int)y)*w2 + (int)x;
	  float v = fits->getValueFloat(((int)j)*width + (int)i);
	  if (!isNaNf(v)) {
	    if (count[k])
	      img[k] += v;
	    else
	      img[k] = v;

	    count[k]++;
	  }
	}
      }
  }

  {
    for (int k=0; k<w2*h2; k++)
      if (count[k])
	img[k] /= count[k];
  }
  delete [] count;

  Matrix w = n * fits->getDataToRef();
  int status = build(w2, h2, img, w);

  delete [] img;

  if (status)
    cerr << "FVContour error" << endl;
}

int FVContour::build(int xdim, int ydim, float *image, Matrix& matrix)
{
  int status = 0;

  long nelem = xdim*ydim;
  char* usedGrid = new char[nelem];
  if (!usedGrid)
    return 1;

  float** rows = new float*[ydim];
  if (!rows)
    return 1;

  for (int j=0; j<ydim; j++)
    rows[j] = image + j*xdim;

  for (int c=0; c<scale->size() && !status; c++) {
    float cntour = scale->level(c);
    for (long elem=0; elem<nelem; elem++)
      usedGrid[elem] = 0;

    //  Search outer edge

    int i,j;

    //  Search top

    for (j=0, i=0; i<xdim-1 && !status; i++)
      if (rows[j][i]<cntour && cntour<=rows[j][i+1])
	status = trace(xdim, ydim, cntour, i, j, top, rows, usedGrid, matrix);

    //  Search right

    for (j=0; j<ydim-1 && !status; j++)
      if (rows[j][i]<cntour && cntour<=rows[j+1][i])
	status = trace(xdim, ydim, cntour, i-1, j, right, rows, usedGrid, 
		       matrix);

    //  Search Bottom

    for (i--; i>=0 && !status; i--)
      if (rows[j][i+1]<cntour && cntour<=rows[j][i])
	status = trace(xdim, ydim, cntour, i, j-1, bottom, rows, usedGrid, 
		       matrix);

    //  Search Left

    for (i=0, j--; j>=0 && !status; j--)
      if (rows[j+1][i]<cntour && cntour<=rows[j][i])
	status = trace(xdim, ydim, cntour, i, j, left, rows, usedGrid, matrix);

    //  Search each row of the image

    for (j=1; j<ydim-1 && !status; j++)
      for (i=0; i<xdim-1 && !status; i++)
	if (!usedGrid[j*xdim + i] && rows[j][i]<cntour && cntour<=rows[j][i+1])
	  status = trace(xdim, ydim, cntour, i, j, top, rows, usedGrid, 
			 matrix);
  }

  delete [] usedGrid;
  delete [] rows;

  return status;
}

int FVContour::trace(int xdim, int ydim, float cntr,
		     int xCell, int yCell, int side, 
		     float** rows, char* usedGrid, Matrix& matrix)
{
  int i = xCell;
  int j = yCell;
  int origSide = side;

  int init = 1;
  int done = (i<0 || i>=xdim-1 || j<0 && j>=ydim-1);

  while (!done) {
    int flag = 0;
    float a = rows[j][i];
    float b = rows[j][i+1];
    float c = rows[j+1][i+1];
    float d = rows[j+1][i];

    float X, Y;
    if (init) {
      init = 0;
      switch (side) {
      case top:
	X = (cntr-a) / (b-a) + i;
	Y = j;
	break;
      case right:
	X = i+1;
	Y = (cntr-b) / (c-b) + j;
	break;
      case bottom:
	X = (cntr-c) / (d-c) + i;
	Y = j+1;
	break;
      case left:
	X = i;
	Y = (cntr-a) / (d-a) + j;
	break;
      }

    }
    else {
      if (side==top)
	usedGrid[j*xdim + i] = 1;

      do {
	if (++side == none)
	  side = top;

	switch (side) {
	case top:
	  if (a>=cntr && cntr>b) {
	    flag = 1;
	    X = (cntr-a) / (b-a) + i;
	    Y = j;
	    j--;
	  }
	  break;
	case right:
	  if( b>=cntr && cntr>c ) {
	    flag = 1;
	    X = i+1;
	    Y = (cntr-b) / (c-b) + j;
	    i++;
	  }
	  break;
	case bottom:
	  if( c>=cntr && cntr>d ) {
	    flag = 1;
	    X = (cntr-d) / (c-d) + i;
	    Y = j+1;
	    j++;
	  }
	  break;
	case left:
	  if( d>=cntr && cntr>a ) {
	    flag = 1;
	    X = i;
	    Y = (cntr-a) / (d-a) + j;
	    i--;
	  }
	  break;
	}
      } while (!flag);

      if (++side == none)
	side = top;
      if (++side == none)
	side = top;
      if (i==xCell && j==yCell && side==origSide)
	done = 1;
      if (i<0 || i>=xdim-1 || j<0 || j>=ydim-1)
	done = 1;
    }

    contours_.append(new Vertex(Vector(X+.5,Y+.5)*matrix));

    if (done)
      contours_.append(new Vertex(DBL_MAX, DBL_MAX));
  }

  return 0;
}
