#  Copyright (C) 1999-2005
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc UpdateView {} {
    global debug
    if {$debug(tcl,layout)} {
	puts "UpdateView"
    }

    SetWatchCursor
    LayoutView
    UnsetWatchCursor
}

proc LayoutView {} {
    global ds9
    global view
    global graph
    global canvas
    global colorbar

    global debug
    if {$debug(tcl,layout)} {
	puts "LayoutView"
    }

    grid forget $ds9(panel)
    pack forget $ds9(info)
    pack forget $ds9(panner)
    pack forget $ds9(magnifier)
    grid forget $ds9(buttons)
    grid forget $ds9(graph,sp)
    grid forget $ds9(graph,horz)
    grid forget $ds9(graph,vert)

    # reset
    grid rowconfigure $ds9(main) 2 -weight 0
    grid columnconfigure $ds9(main) 0 -weight 0

    grid rowconfigure $ds9(main) 1 -weight 0
    grid columnconfigure $ds9(main) 1 -weight 0

    # layout
    switch $view(layout) {
	vertical {LayoutViewVert}
	horizontal -
	default {LayoutViewHorz}
    }

    # process colorbar/canvas issues
    if {$view(colorbar) != $view(colorbar,next)} {
	set view(colorbar) $view(colorbar,next)
	if {$view(colorbar,next)} {
	    CanvasUpdateSize $canvas(width) $canvas(height)
	} else {
	    CanvasUpdateSize $canvas(width) \
		[expr $canvas(height)-$colorbar(height)-$colorbar(gap)]
	}
    }

    # frames (for colorbar)
    LayoutFrames

    # info panel
    UpdateInfoPanel

    # adjust vertgraph for colorbar
    if {$view(graph,vert)} {
	if {$view(colorbar)} {
	    $ds9(graph,vert) configure -bottommargin \
		[expr $graph(sp,y)+$colorbar(height)+$colorbar(gap)]
	} else {
	    $ds9(graph,vert) configure -bottommargin $graph(sp,y)
	}
    }
}

proc LayoutViewHorz {} {
    global ds9
    global current
    global view

    grid rowconfigure $ds9(main) 2 -weight 1
    grid columnconfigure $ds9(main) 0 -weight 1

    grid $ds9(image) -row 2 -column 0 -sticky news
    pack $ds9(canvas) -fill both -expand true

    if {$view(info) || $view(magnifier) || $view(panner)} {
	grid $ds9(panel) -row 0 -column 0 -sticky ew -columnspan 3
    }
    if {$view(buttons)} {
	grid $ds9(buttons) -row 1 -sticky ew -columnspan 3
    }
    if {$view(graph,horz) || $view(graph,vert)} {
	grid $ds9(graph,sp) -row 3 -column 1
    }
    if {$view(graph,horz)} {
	grid $ds9(graph,horz) -row 4 -column 0 -sticky ew -columnspan 2 
    }
    if {$view(graph,vert)} {
	grid $ds9(graph,vert) -row 2 -column 2 -sticky ns -rowspan 2 
    }

    # info panel
    if {$view(info)} {
	pack $ds9(info) -side left -padx $ds9(internalpad) \
	    -pady $ds9(internalpad) -fill x -expand true
    }
    if {$view(panner)} {
	pack $ds9(panner) -side left -padx $ds9(internalpad) \
	    -pady $ds9(internalpad)
    }
    if {$view(magnifier)} {
	pack $ds9(magnifier) -side left -padx $ds9(internalpad) \
	    -pady $ds9(internalpad)
    }
}

proc LayoutViewVert {} {
    global ds9
    global current
    global view

    grid rowconfigure $ds9(main) 1 -weight 1
    grid columnconfigure $ds9(main) 1 -weight 1

    grid $ds9(image) -row 1 -column 1 -sticky news
    pack $ds9(canvas) -fill both -expand true

    if {$view(info) || $view(magnifier) || $view(panner)} {
	grid $ds9(panel) -row 1 -column 0 -sticky ns -rowspan 3
    }
#    if {$view(buttons)} {
#	grid $ds9(buttons) -row 0 -column 0 -sticky ew -columnspan 4
#    }
    if {$view(graph,horz) || $view(graph,vert)} {
	grid $ds9(graph,sp) -row 2 -column 2
    }
    if {$view(graph,horz)} {
	grid $ds9(graph,horz) -row 3 -column 1 -sticky ew -columnspan 2
    }
    if {$view(graph,vert)} {
	grid $ds9(graph,vert) -row 1 -column 3 -sticky ns -rowspan 2
    }

    # info panel
    if {$view(magnifier)} {
	pack $ds9(magnifier) -side top -padx $ds9(internalpad) \
	    -pady $ds9(internalpad)
    }
    if {$view(info)} {
	pack $ds9(info) -side top -padx $ds9(internalpad) \
	    -pady $ds9(internalpad) -fill y -expand true
    }
    if {$view(panner)} {
	pack $ds9(panner) -side top -padx $ds9(internalpad) \
	    -pady $ds9(internalpad)
    }
}

proc UpdateInfoPanel {} {
    global ds9
    global view
    global info

    global debug
    if {$debug(tcl,layout)} {
	puts "UpdateInfoPanel"
    }

    # reset
    grid columnconfigure $ds9(info) 5 -weight 0
    if {$info(row) > -1} {
	grid rowconfigure $ds9(info) $info(row) -weight 0
	set info(row) -1
    }

    # layout
    switch $view(layout) {
	vertical {UpdateInfoPanelVert}
	horizontal -
	default {UpdateInfoPanelHorz}
    }
}

proc UpdateInfoPanelHorz {} {
    global ds9
    global view

    set ww 13
    set xx 31
    set row 0

    grid columnconfigure $ds9(info) 5 -weight 1

    # filename
    if {$view(info,filename)} {
	$ds9(info).fileValue configure -width $xx

	grid $ds9(info).fileTitle -row $row -column 0 -sticky w
	grid $ds9(info).fileValue -row $row -column 2 -padx 2 \
	    -sticky ew -columnspan 4
	incr row
    } else {
	grid forget $ds9(info).fileTitle
	grid forget $ds9(info).fileValue
    }

    # object
    if {$view(info,object)} {
	$ds9(info).objValue configure -width $xx

	grid $ds9(info).objTitle -row $row -column 0 -sticky w
	grid $ds9(info).objValue -row $row -column 2 -padx 2 \
	    -sticky ew -columnspan 4
	incr row
    } else {
	grid forget $ds9(info).objTitle
	grid forget $ds9(info).objValue
    }

    # minmax
    if {$view(info,minmax)} {
	$ds9(info).minValue configure -width $ww
	$ds9(info).maxValue configure -width $ww
	$ds9(info).lowValue configure -width $ww
	$ds9(info).highValue configure -width $ww

	grid $ds9(info).minTitle -row $row -column 0 -sticky w
	grid $ds9(info).minValue -row $row -column 2 -padx 2 -sticky w
	grid $ds9(info).maxTitle -row $row -column 3 -sticky w
	grid $ds9(info).maxValue -row $row -column 4 -padx 2 -sticky w

	grid $ds9(info).lowTitle -row $row -column 0 -sticky w
	grid $ds9(info).lowValue -row $row -column 2 -padx 2 -sticky w
	grid $ds9(info).highTitle -row $row -column 3 -sticky w
	grid $ds9(info).highValue -row $row -column 4 -padx 2 -sticky w
	incr row
    } else {
	grid forget $ds9(info).minTitle
	grid forget $ds9(info).minValue
	grid forget $ds9(info).maxTitle
	grid forget $ds9(info).maxValue
	grid forget $ds9(info).lowTitle
	grid forget $ds9(info).lowValue
	grid forget $ds9(info).highTitle
	grid forget $ds9(info).highValue
    }

    # value
    $ds9(info).valueValue.v configure -width $ww
    $ds9(info).valueValue.red configure -width [expr $xx/3]
    $ds9(info).valueValue.green configure -width [expr $xx/3]
    $ds9(info).valueValue.blue configure -width [expr $xx/3-1]

    grid $ds9(info).valueTitle -row $row -column 0 -sticky w
    grid $ds9(info).valueValue -row $row -column 2 -padx 2 \
	-sticky w -columnspan 3

    grid $ds9(info).valueValue.v
    grid forget $ds9(info).valueValue.red
    grid forget $ds9(info).valueValue.green
    grid forget $ds9(info).valueValue.blue
    incr row

    # wcs
    foreach l {{} A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$ds9(info).wcsXValue$l configure -width $ww
	$ds9(info).wcsYValue$l configure -width $ww

	if {$view(info,wcs$l)} {
	    grid $ds9(info).wcsLabel$l  -row $row -column 0 -sticky w
	    grid $ds9(info).wcsXLabel$l -row $row -column 1 -sticky e
	    grid $ds9(info).wcsXValue$l -row $row -column 2 -padx 2
	    grid $ds9(info).wcsYLabel$l -row $row -column 3 -sticky e

	    grid $ds9(info).wcsYValue$l -row $row -column 4 -padx 2
	    incr row
	} else {
	    grid forget $ds9(info).wcsLabel$l
	    grid forget $ds9(info).wcsXLabel$l
	    grid forget $ds9(info).wcsXValue$l
	    grid forget $ds9(info).wcsYLabel$l
	    grid forget $ds9(info).wcsYValue$l
	}
    }

    # detector
    if {$view(info,detector)} {
	$ds9(info).detectorXValue configure -width $ww
	$ds9(info).detectorYValue configure -width $ww

	grid $ds9(info).detectorTitle -row $row -column 0 -sticky w
	grid $ds9(info).detectorXLabel -row $row -column 1 -sticky e
	grid $ds9(info).detectorXValue -row $row -column 2 -padx 2
	grid $ds9(info).detectorYLabel -row $row -column 3 -sticky e
	grid $ds9(info).detectorYValue -row $row -column 4 -padx 2
	incr row
    } else {
	grid forget $ds9(info).detectorTitle
	grid forget $ds9(info).detectorXLabel
	grid forget $ds9(info).detectorXValue
	grid forget $ds9(info).detectorYLabel
	grid forget $ds9(info).detectorYValue
    }

    # amplifier
    if {$view(info,amplifier)} {
	$ds9(info).amplifierXValue configure -width $ww
	$ds9(info).amplifierYValue configure -width $ww

	grid $ds9(info).amplifierTitle -row $row -column 0 -sticky w
	grid $ds9(info).amplifierXLabel -row $row -column 1 -sticky e
	grid $ds9(info).amplifierXValue -row $row -column 2 -padx 2
	grid $ds9(info).amplifierYLabel -row $row -column 3 -sticky e
	grid $ds9(info).amplifierYValue -row $row -column 4 -padx 2
	incr row
    } else {
	grid forget $ds9(info).amplifierTitle
	grid forget $ds9(info).amplifierXLabel
	grid forget $ds9(info).amplifierXValue
	grid forget $ds9(info).amplifierYLabel
	grid forget $ds9(info).amplifierYValue
    }

    # physical
    if {$view(info,physical)} {
	$ds9(info).physicalXValue configure -width $ww
	$ds9(info).physicalYValue configure -width $ww 

	grid $ds9(info).physicalTitle -row $row -column 0 -sticky w
	grid $ds9(info).physicalXLabel -row $row -column 1 -sticky e
	grid $ds9(info).physicalXValue -row $row -column 2 -padx 2
	grid $ds9(info).physicalYLabel -row $row -column 3 -sticky e
	grid $ds9(info).physicalYValue -row $row -column 4 -padx 2
	incr row
    } else {
	grid forget $ds9(info).physicalTitle
	grid forget $ds9(info).physicalXLabel
	grid forget $ds9(info).physicalXValue
	grid forget $ds9(info).physicalYLabel
	grid forget $ds9(info).physicalYValue
    }

    # image
    if {$view(info,image)} {
	$ds9(info).imageXValue configure -width $ww 
	$ds9(info).imageYValue configure -width $ww

	grid $ds9(info).imageTitle -row $row -column 0 -sticky w
	grid $ds9(info).imageXLabel -row $row -column 1 -sticky e
	grid $ds9(info).imageXValue -row $row -column 2 -padx 2
	grid $ds9(info).imageYLabel -row $row -column 3 -sticky e
	grid $ds9(info).imageYValue -row $row -column 4 -padx 2
	incr row
    } else {
	grid forget $ds9(info).imageTitle
	grid forget $ds9(info).imageXLabel
	grid forget $ds9(info).imageXValue
	grid forget $ds9(info).imageYLabel
	grid forget $ds9(info).imageYValue
    }

    # frame, zoom, angle
    if {$view(info,frame)} {
	$ds9(info).zoomValue configure -width $ww
	$ds9(info).angleValue configure -width $ww

	grid $ds9(info).frame -row $row -column 0 -sticky w
	grid $ds9(info).zoomtitle -row $row -column 1 -sticky w
	grid $ds9(info).zoomValue -row $row -column 2 -padx 2 -sticky w
	grid $ds9(info).angleTitle -row $row -column 3
	grid $ds9(info).angleValue -row $row -column 4 -padx 2 -sticky w
	incr row
    } else {
	grid forget $ds9(info).frame
	grid forget $ds9(info).zoomtitle
	grid forget $ds9(info).zoomValue
	grid forget $ds9(info).angleTitle
	grid forget $ds9(info).angleValue
    }

    # dummy
    grid forget $ds9(info).dummy
}

proc UpdateInfoPanelVert {} {
    global ds9
    global view

    set ww 11
    set row 0

    # filename
    if {$view(info,filename)} {
	$ds9(info).fileValue configure -width $ww

	grid $ds9(info).fileTitle -row $row -column 0 -sticky e 
	grid $ds9(info).fileValue -row $row -column 1 -padx 2 -sticky w
	incr row
    } else {
	grid forget $ds9(info).fileTitle
	grid forget $ds9(info).fileValue
    }

    # object
    if {$view(info,object)} {
	$ds9(info).objValue configure -width $ww

	grid $ds9(info).objTitle -row $row -column 0 -sticky e 
	grid $ds9(info).objValue -row $row -column 1 -padx 2 -sticky w
	incr row
    } else {
	grid forget $ds9(info).objTitle
	grid forget $ds9(info).objValue
    }

    # minmax
    if {$view(info,minmax)} {
	$ds9(info).minValue configure -width $ww
	$ds9(info).maxValue configure -width $ww
	$ds9(info).lowValue configure -width $ww
	$ds9(info).highValue configure -width $ww

	grid $ds9(info).minTitle -row $row -column 0 -sticky e
	grid $ds9(info).minValue -row $row -column 1 -padx 2 -sticky w
	incr row
	grid $ds9(info).maxTitle -row $row -column 0 -sticky e
	grid $ds9(info).maxValue -row $row -column 1 -padx 2 -sticky w
	incr row
	grid $ds9(info).lowTitle -row $row -column 0 -sticky e
	grid $ds9(info).lowValue -row $row -column 1 -padx 2 -sticky w
	incr row
	grid $ds9(info).highTitle -row $row -column 0 -sticky e
	grid $ds9(info).highValue -row $row -column 1 -padx 2 -sticky w
	incr row
    } else {
	grid forget $ds9(info).minTitle
	grid forget $ds9(info).minValue
	grid forget $ds9(info).maxTitle
	grid forget $ds9(info).maxValue
	grid forget $ds9(info).lowTitle
	grid forget $ds9(info).lowValue
	grid forget $ds9(info).highTitle
	grid forget $ds9(info).highValue
    }

    # value
    $ds9(info).valueValue.v configure -width $ww
    $ds9(info).valueValue.red configure -width [expr $ww/3]
    $ds9(info).valueValue.green configure -width [expr $ww/3]
    $ds9(info).valueValue.blue configure -width [expr $ww/3]

    grid $ds9(info).valueTitle -row $row -column 0 -sticky e
    grid $ds9(info).valueValue -row $row -column 1 -padx 2 -sticky w

    grid $ds9(info).valueValue.v
    grid forget $ds9(info).valueValue.red
    grid forget $ds9(info).valueValue.green
    grid forget $ds9(info).valueValue.blue
    incr row

    # wcs
    foreach l {{} A B C D E F G H I J K L M N O P Q R S T U V W X Y Z} {
	$ds9(info).wcsXValue$l configure -width $ww
	$ds9(info).wcsYValue$l configure -width $ww

	if {$view(info,wcs$l)} {
	    grid $ds9(info).wcsLabel$l  -row $row -column 1 -sticky ew
	    incr row
	    grid $ds9(info).wcsXLabel$l -row $row -column 0 -sticky e
	    grid $ds9(info).wcsXValue$l -row $row -column 1 -padx 2
	    incr row
	    grid $ds9(info).wcsYLabel$l -row $row -column 0 -sticky e
	    grid $ds9(info).wcsYValue$l -row $row -column 1 -padx 2
	    incr row
	} else {
	    grid forget $ds9(info).wcsLabel$l
	    grid forget $ds9(info).wcsXLabel$l
	    grid forget $ds9(info).wcsXValue$l
	    grid forget $ds9(info).wcsYLabel$l
	    grid forget $ds9(info).wcsYValue$l
	}
    }

    # detector
    if {$view(info,detector)} {
	$ds9(info).detectorXValue configure -width $ww
	$ds9(info).detectorYValue configure -width $ww

	grid $ds9(info).detectorTitle -row $row -column 1 -sticky ew
	incr row
	grid $ds9(info).detectorXLabel -row $row -column 0 -sticky e
	grid $ds9(info).detectorXValue -row $row -column 1 -padx 2
	incr row
	grid $ds9(info).detectorYLabel -row $row -column 0 -sticky e
	grid $ds9(info).detectorYValue -row $row -column 1 -padx 2
	incr row
    } else {
	grid forget $ds9(info).detectorTitle
	grid forget $ds9(info).detectorXLabel
	grid forget $ds9(info).detectorXValue
	grid forget $ds9(info).detectorYLabel
	grid forget $ds9(info).detectorYValue
    }

    # amplifier
    if {$view(info,amplifier)} {
	$ds9(info).amplifierXValue configure -width $ww
	$ds9(info).amplifierYValue configure -width $ww

	grid $ds9(info).amplifierTitle -row $row -column 1 -sticky ew
	incr row
	grid $ds9(info).amplifierXLabel -row $row -column 0 -sticky e
	grid $ds9(info).amplifierXValue -row $row -column 1 -padx 2
	incr row
	grid $ds9(info).amplifierYLabel -row $row -column 0 -sticky e
	grid $ds9(info).amplifierYValue -row $row -column 1 -padx 2
	incr row
    } else {
	grid forget $ds9(info).amplifierTitle
	grid forget $ds9(info).amplifierXLabel
	grid forget $ds9(info).amplifierXValue
	grid forget $ds9(info).amplifierYLabel
	grid forget $ds9(info).amplifierYValue
    }

    # physical
    if {$view(info,physical)} {
	$ds9(info).physicalXValue configure -width $ww
	$ds9(info).physicalYValue configure -width $ww 

	grid $ds9(info).physicalTitle -row $row -column 1 -sticky ew
	incr row
	grid $ds9(info).physicalXLabel -row $row -column 0 -sticky e
	grid $ds9(info).physicalXValue -row $row -column 1 -padx 2
	incr row
	grid $ds9(info).physicalYLabel -row $row -column 0 -sticky e
	grid $ds9(info).physicalYValue -row $row -column 1 -padx 2
	incr row
    } else {
	grid forget $ds9(info).physicalTitle
	grid forget $ds9(info).physicalXLabel
	grid forget $ds9(info).physicalXValue
	grid forget $ds9(info).physicalYLabel
	grid forget $ds9(info).physicalYValue
    }

    # image
    if {$view(info,image)} {
	$ds9(info).imageXValue configure -width $ww 
	$ds9(info).imageYValue configure -width $ww

	grid $ds9(info).imageTitle -row $row -column 1 -sticky ew
	incr row
	grid $ds9(info).imageXLabel -row $row -column 0 -sticky e
	grid $ds9(info).imageXValue -row $row -column 1 -padx 2
	incr row
	grid $ds9(info).imageYLabel -row $row -column 0 -sticky e
	grid $ds9(info).imageYValue -row $row -column 1 -padx 2
	incr row
    } else {
	grid forget $ds9(info).imageTitle
	grid forget $ds9(info).imageXLabel
	grid forget $ds9(info).imageXValue
	grid forget $ds9(info).imageYLabel
	grid forget $ds9(info).imageYValue
    }

    # frame, zoom, angle
    if {$view(info,frame)} {
	$ds9(info).zoomValue configure -width $ww
	$ds9(info).angleValue configure -width $ww

	grid $ds9(info).frame -row $row -column 1 -sticky ew
	incr row
	grid $ds9(info).zoomtitle -row $row -column 0 -sticky e
	grid $ds9(info).zoomValue -row $row -column 1 -padx 2
	incr row
	grid $ds9(info).angleTitle -row $row -column 0 -sticky e
	grid $ds9(info).angleValue -row $row -column 1 -padx 2
	incr row
    } else {
	grid forget $ds9(info).frame
	grid forget $ds9(info).zoomtitle
	grid forget $ds9(info).zoomValue
	grid forget $ds9(info).angleTitle
	grid forget $ds9(info).angleValue
    }

    # dummy
    global info
    set info(row) $row

    grid $ds9(info).dummy -row $row -column 1
    grid rowconfigure $ds9(info) $row -weight 1
}

proc UpdateInfoPanelValue {which} {
    global ds9

    switch -- [$which get type] {
	base {
	    grid forget $ds9(info).valueValue.red
	    grid forget $ds9(info).valueValue.green
	    grid forget $ds9(info).valueValue.blue
	    grid $ds9(info).valueValue.v
	}
	rgb {
	    grid forget $ds9(info).valueValue.v
	    grid $ds9(info).valueValue.red \
		$ds9(info).valueValue.green \
		$ds9(info).valueValue.blue
	}
    }
}

proc LayoutFrames {} {
    global ds9
    global current
    global canvas
    global tile
    global view
    global colorbar

    global debug
    if {$debug(tcl,layout)} {
	puts "LayoutFrames"
    }

    set canvas(width) [winfo width $ds9(canvas)]
    set canvas(height) [winfo height $ds9(canvas)]

    # turn everything off
    foreach f $ds9(frames) {
	$f hide
	$f highlite off
	$f panner off
	$f magnifier off
	UnBindEventsFrame $f
    }

    # colorbar
    LayoutColorbar

    # frames
    set n [llength $ds9(active)]
    if {$n > 0} {
	switch -- $ds9(display,mode) {
	    single {TileOne}
	    tile {
		switch -- $tile(mode) {
		    row {TileRect 1 $n $tile(grid,gap)}
		    column {TileRect $n 1 $tile(grid,gap)}
		    grid {
			switch -- $tile(grid,mode) {
			    automatic {
				TileRect [expr int(sqrt($n-1))+1] \
				    [expr int(sqrt($n)+.5)] \
				    $tile(grid,gap)
			    }
			    manual {
				TileRect $tile(grid,row) \
				    $tile(grid,col) \
				    $tile(grid,gap)
			    }
			}
		    }
		}
	    }
	    blink {TileOne}
	}
    } else {
	set current(frame) ""
	set ds9(next) ""

	# panner
	if {$view(panner)} {
	    panner clear
	}

	# magnifier
	if {$view(magnifier)} {
	    magnifier clear
	}

	# process proper colorbar
	switch -- $ds9(visual) {
	    pseudocolor {
		colorbar show
		set current(colorbar) colorbar
		set colorbar(map) [colorbar get name]
		set colorbar(invert) [colorbar get invert]
	    }
	    truecolor {
		colorbar show
		colorbarrgb hide
		set current(colorbar) colorbar
		set colorbar(map) [colorbar get name]
		set colorbar(invert) [colorbar get invert]
	    }
	}

	# update menus/dialogs
	UpdateDS9
    }
}

proc LayoutColorbar {} {
    global colorbar
    global canvas
    global ds9

    set colorbar(width) $canvas(width)

    switch -- $ds9(visual) {
	pseudocolor {
	    colorbar configure -x 0 -y $canvas(height) \
		-width $colorbar(width) -height $colorbar(height) -anchor sw
	}
	truecolor {
	    colorbar configure -x 0 -y $canvas(height) \
		-width $colorbar(width) -height $colorbar(height) -anchor sw
	    colorbarrgb configure -x 0 -y $canvas(height) \
		-width $colorbar(width) -height $colorbar(height) -anchor sw
	}
    }
}

# This procedure is called when we have only 1 frames to display
# Canvas Coordinate system is 1 to canvas(width), canvas(height)

proc TileOne {} {
    global ds9
    global current
    global canvas
    global view
    global colorbar

    global debug
    if {$debug(tcl,layout)} {
	puts "TileOne"
    }

    set ww $canvas(width)
    set hh [CanvasTrueHeight]

    set xx 0
    set yy 0

    foreach f $ds9(active) {
	$f configure -x $xx -y $yy -width $ww -height $hh -anchor nw
    }

    # only show the current frame
    $current(frame) show
    FrameToFront $current(frame)
}

proc TileRect {numx numy gap} {
    global canvas
    global view
    global colorbar

    global debug
    if {$debug(tcl,layout)} {
	puts "TileRect"
    }

    set ww $canvas(width)
    set hh [CanvasTrueHeight]

    set w [expr int(($ww-$gap*($numx-1))/$numx)]
    set h [expr int(($hh-$gap*($numy-1))/$numy)]
    
    for {set jj 0} {$jj<$numy} {incr jj} {
	for {set ii 0} {$ii<$numx} {incr ii} {
	    set c [expr $jj*$numx + $ii]
	    set x($c) [expr ($w+$gap)*$ii]
	    set y($c) [expr ($h+$gap)*$jj]
	}
    }

    TileIt $w $h x y [expr $numx*$numy]
}

proc TileIt {ww hh xx yy nn} {
    upvar $xx x
    upvar $yy y
    global ds9
    global current

    global debug
    if {$debug(tcl,layout)} {
	puts "TileIt"
    }

    set i 0
    foreach f $ds9(active) {
	if {$i<$nn} {
	    $f configure -x $x($i) -y $y($i) -width $ww -height $hh \
		-anchor nw
	    $f show
	    $ds9(canvas) raise $f
	    if {!$ds9(freeze)} {
		BindEventsFrame $f
	    }
	}
	incr i
    }

    # if manual grid, current frame could be not included
    if {[llength $ds9(active)] > $nn} {
	set current(frame) [lindex $ds9(active) 0]
    }
    FrameToFront $current(frame)
}

proc CanvasUpdateSize {width height} {
    global ds9
    global view
    global colorbar

    set wc [winfo width $ds9(canvas)]
    set hc [winfo height $ds9(canvas)]

    set wt [winfo width $ds9(top)]
    set ht [winfo height $ds9(top)]
    
    set w [expr $width  - $wc + $wt]
    if {$view(colorbar)} {
	set h [expr $height - $hc + $ht + $colorbar(height)+$colorbar(gap)]
    } else {
	set h [expr $height - $hc + $ht]
    }

    wm geometry $ds9(top) ${w}x${h}
}

proc CanvasTrueHeight {} {
    global view
    global colorbar
    global canvas

    if {$view(colorbar)} {
	return [expr $canvas(height)-$colorbar(height)-$colorbar(gap)]
    } else {
	return $canvas(height)
    }
}

proc DisplayDefaultDialog {} {
    global canvas
    global ed

    set w ".defdpy"

    set ed(ok) 0
    set ed(x) $canvas(width)
    set ed(y) [CanvasTrueHeight]

    DialogCreate $w "Display Size" -borderwidth 2
    frame $w.ed  -relief groove -borderwidth 2
    frame $w.buttons -relief groove -borderwidth 2
    pack $w.ed $w.buttons -fill x -ipadx 4 -ipady 4

    label $w.ed.xTitle -text "X"
    label $w.ed.yTitle -text "Y"
    entry $w.ed.x -textvariable ed(x) -width 10
    entry $w.ed.y -textvariable ed(y) -width 10
    label $w.ed.xunit -text "pixels" -relief groove -width 8
    label $w.ed.yunit -text "pixels" -relief groove -width 8
    
    grid $w.ed.xTitle $w.ed.x $w.ed.xunit -padx 4 -sticky w
    grid $w.ed.yTitle $w.ed.y $w.ed.yunit -padx 4 -sticky w

    button $w.buttons.ok -text "OK" -default active -command {set ed(ok) 1}
    button $w.buttons.cancel -text "Cancel" -command {set ed(ok) 0}
    pack $w.buttons.ok -side left -padx 10
    pack $w.buttons.cancel -side right -padx 10

    bind $w <Return> {set ed(ok) 1}
    bind $w <Alt-o> "tkButtonInvoke $w.buttons.ok"
    bind $w <Alt-c> "tkButtonInvoke $w.buttons.cancel"

    DialogCenter $w 
    $w.ed.x select range 0 end
    DialogWait $w ed(ok) $w.ed.x
    DialogDismiss $w

    if {$ed(ok)} {
	CanvasUpdateSize $ed(x) $ed(y)
    }

    unset ed
}

proc ProcessWidthCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    # we must have a window to change
    RealizeDS9

    CanvasUpdateSize [lindex $var $i] [CanvasTrueHeight]
}

proc ProcessHeightCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    # we must have a window to change
    RealizeDS9

    global canvas
    CanvasUpdateSize $canvas(width) [lindex $var $i] 
}

proc ProcessViewCmd {varname iname} {
    upvar $varname var
    upvar $iname i

    global view
    global rgb

    set item [string tolower [lindex $var $i]]

    switch  -- $item {
	horizontal -
	vertical {set view(layout) $item}

	default {
	    set yesno [lindex $var [expr $i+1]]

	    switch -- $item {
		info {set view(info) [FromYesNo $yesno]}
		panner {set view(panner) [FromYesNo $yesno]}
		magnifier {set view(magnifier) [FromYesNo $yesno]}
		buttons {set view(buttons) [FromYesNo $yesno]}
		colorbar {set view(colorbar,next) [FromYesNo $yesno]}
		horzgraph {set view(graph,horz) [FromYesNo $yesno]}
		vertgraph {set view(graph,vert) [FromYesNo $yesno]}

		filename {set view(info,filename) [FromYesNo $yesno]}
		object {set view(info,object) [FromYesNo $yesno]}
		minmax {set view(info,minmax) [FromYesNo $yesno]}
		frame {set view(info,frame) [FromYesNo $yesno]}

		detector {set view(info,detector) [FromYesNo $yesno]}
		amplifier {set view(info,amplifier) [FromYesNo $yesno]}
		physical {set view(info,physical) [FromYesNo $yesno]}
		image {set view(info,image) [FromYesNo $yesno]}
		wcs {set view(info,wcs) [FromYesNo $yesno]}
		wcsa {set view(info,wcsA) [FromYesNo $yesno]}
		wcsb {set view(info,wcsB) [FromYesNo $yesno]}
		wcsc {set view(info,wcsC) [FromYesNo $yesno]}
		wcsd {set view(info,wcsD) [FromYesNo $yesno]}
		wcse {set view(info,wcsE) [FromYesNo $yesno]}
		wcsf {set view(info,wcsF) [FromYesNo $yesno]}
		wcsg {set view(info,wcsG) [FromYesNo $yesno]}
		wcsh {set view(info,wcsH) [FromYesNo $yesno]}
		wcsi {set view(info,wcsI) [FromYesNo $yesno]}
		wcsj {set view(info,wcsJ) [FromYesNo $yesno]}
		wcsk {set view(info,wcsK) [FromYesNo $yesno]}
		wcsl {set view(info,wcsL) [FromYesNo $yesno]}
		wcsm {set view(info,wcsM) [FromYesNo $yesno]}
		wcsn {set view(info,wcsN) [FromYesNo $yesno]}
		wcso {set view(info,wcsO) [FromYesNo $yesno]}
		wcsp {set view(info,wcsP) [FromYesNo $yesno]}
		wcsq {set view(info,wcsQ) [FromYesNo $yesno]}
		wcsr {set view(info,wcsR) [FromYesNo $yesno]}
		wcss {set view(info,wcsS) [FromYesNo $yesno]}
		wcst {set view(info,wcsT) [FromYesNo $yesno]}
		wcsu {set view(info,wcsU) [FromYesNo $yesno]}
		wcsv {set view(info,wcsV) [FromYesNo $yesno]}
		wcsw {set view(info,wcsW) [FromYesNo $yesno]}
		wcsx {set view(info,wcsX) [FromYesNo $yesno]}
		wcsy {set view(info,wcsY) [FromYesNo $yesno]}
		wcsz {set view(info,wcsZ) [FromYesNo $yesno]}

		red {set rgb(red) [FromYesNo $yesno]; RGBView}
		green {set rgb(green) [FromYesNo $yesno]; RGBView}
		blue {set rgb(blue) [FromYesNo $yesno]; RGBView}
	    }
	    incr i
	}
    }
    UpdateView
}

