/*
 *	Copyright (c) 2005 Smithsonian Astrophysical Observatory
 */

/* code shameless ripped off from John Roll's search program in starbase
   (http://cfa-www.harvard.edu/~john/starbase/starbase.html) */

#include <math.h>
#include <funtools.h>
#include <filter.h>
#include <swap.h>
#include <word.h>
#include <xalloc.h>

#define MAXROW 8192
static int maxrow=MAXROW;

extern char *optarg;
extern int optind;

#define RA_DEFAULT_UNITS  "h"
#define DEC_DEFAULT_UNITS "d"

#define Abs(xx) 	(((xx) >= 0) ? (xx) : -(xx) )
#define Max(x, y)       (((x) > (y)) ? (x) : (y))

#define X__PI   3.14159265358979323846
#define X_2PI   ( 2 * X__PI )
#define X_R2D   (X_2PI / 360.0)
#define X_R2H   (X_2PI /  24.0)
#define X_H2D   (360.0 /  24.0)

#define r2h(r)  ( (r) / X_R2H )
#define h2r(d)  ( (d) * X_R2H )
#define r2d(r)  ( (r) / X_R2D )
#define d2r(d)  ( (d) * X_R2D )
#define h2d(r)  ( (r) * X_H2D )
#define d2h(d)  ( (d) / X_H2D )

typedef struct evstruct{
  double val0, val1;
} *Ev, EvRec;

typedef struct Axis {
  char *colname;
  double val;
  double lo;
  double hi;
} Axis;

typedef struct Csys {
	char	type[3];
	Axis	axis[2];
	double	radius;
} Csys;

#ifdef ANSI_FUNC
static void
slaDcs2c(double a, double b, double v[3]) 
#else
static void slaDcs2c(a, b, v) 
     double a;
     double b;
     double v[3];
#endif
{  
   double cosb;
   cosb = cos ( b );
   v[0] = cos ( a ) * cosb;
   v[1] = sin ( a ) * cosb;
   v[2] = sin ( b );
}  

#ifdef ANSI_FUNC
static double
slaDsep(double a1, double b1, double a2, double b2)
#else
static double slaDsep(a1, b1, a2, b2)
     double a1;
     double b1;
     double a2;
     double b2;
#endif
{
   int i;
   double d, v1[3], v2[3], s2, c2;
 
   slaDcs2c ( a1, b1, v1 );
   slaDcs2c ( a2, b2, v2 );
 
   s2 = 0.0;
   for ( i = 0; i < 3; i++ ) {
      d = v1[i] - v2[i];
      s2 += d * d;
   }
   s2 /= 4.0; 
 
   c2 = 1.0 - s2;
   return 2.0 * atan2 ( sqrt ( s2 ), sqrt ( Max ( 0.0, c2 )));
}

#ifdef ANSI_FUNC
static int
ConeFilter(Csys *csys, double val0, double val1)
#else
static int ConeFilter(csys, val0, val1)
     Csys *csys;
     double val0;
     double val1;
#endif
{
  double	r1, r2, d1, d2;
  double	radius;

  switch( csys->type[0] ) {
  case 'h':
    r1 = h2r(csys->axis[0].val);
    r2 = h2r(val0);
    break;
  case 'd':
    r1 = d2r(csys->axis[0].val);
    r2 = d2r(val0);
    break;
  case 'r':
    r1 = csys->axis[0].val;
    r2 = val0;
    break;
  }
  switch( csys->type[1] ) {
  case 'h':
    d1 = h2r(csys->axis[1].val);
    d2 = h2r(val1);
    break;
  case 'd':
    d1 = d2r(csys->axis[1].val);
    d2 = d2r(val1);
    break;
  case 'r':
    d1 = csys->axis[1].val;
    d2 = val1;
    break;
  }
  switch( csys->type[2] ) {
  case 'h':
    radius = h2r(csys->radius);
    break;
  case 'd':
    radius = d2r(csys->radius);
    break;
  case 'r':
    radius = csys->radius;
    break;
  }

  /* check angular separation */
  if ( slaDsep(r1, d1, r2, d2) <= radius )
    return 1;
  else
    return 0;
}

#ifdef ANSI_FUNC
static void 
radecbox(double ra, double dec, double width, 
	 double *r1, double *r2, double *d1, double *d2)
#else
static void radecbox(ra, dec, width, r1, r2, d1, d2)
     double	ra;
     double	dec;
     double	width;
     double	*r1, *r2, *d1, *d2;
#endif
{
  double	cosdec;

  *d1 = dec - width / 2.0;
  if ( *d1 <= -90.0 ) {
    *d1 = -90.0;
    *d2 = dec + width / 2.0;
    *r1 =  0.0;
    *r2 = 24.0;
  } else {
    *d2 = dec + width / 2.0;
    if ( *d2 >=  90.0 ) {
      *d1 = dec - width / 2.0;
      *d2 = 90.0;
      *r1  = 0.0;
      *r2  = 24.0;
    } else {
      if ( dec > 0.0 )	cosdec = Abs(cos(d2r(*d1)));
      else		cosdec = Abs(cos(d2r(*d2)));
      
      *r1 = ra - d2h(width) / 2 / cosdec;
      *r2 = ra + d2h(width) / 2 / cosdec;
      
      if ( *r1 <  0.0 ) *r1 += 24;
      if ( *r2 > 24.0 ) *r2 -= 24;
    }
  }
}

#ifdef ANSI_FUNC
static void
ConeLimits(Csys *csys, char *rastr, char *decstr, char *radstr)
#else
static void ConeLimits(csys, rastr, decstr, radstr)
     Csys *csys;
     char *rastr;
     char *decstr;
     char *radstr;
#endif
{
  int xtype;
  char *p;
  double dval;
  double ra, dec, radius;
  double r0, r1, d0, d1;

  /* Input units conversion. */
  /* process input ra */
  dval = SAOstrtod(rastr, &p);
  xtype = *p;
  if( (xtype != 'h') && (xtype != 'd') && (xtype != 'r') ) {
    if( SAOdtype == 'd' )
      xtype = 'd';
    else
      xtype = 'h';
  }
  switch( xtype ) {
  case 'h':
    ra = dval;
    break;
  case 'd':
    ra = d2h(dval);
    break;
  case 'r':
    ra = r2h(dval);
    break;
  }

  /* process input dec */
  dval = SAOstrtod(decstr, &p);
  xtype = *p;
  if( (xtype != 'h') && (xtype != 'd') && (xtype != 'r') ) {
    if( SAOdtype == 'h' )
      xtype = 'h';
    else
      xtype = 'd';
  }
  switch( xtype ) {
  case 'h':
    dec = h2d(dval);
    break;
  case 'd':
    dec = dval;
    break;
  case 'r':
    dec = r2d(dval);
    break;
  }

  /* process input radius */
  dval = SAOstrtod(radstr, &p);
  xtype = *p;
  if( (xtype != 'h')  && (xtype != 'd') && (xtype != 'r') &&
      (xtype != '\'') && (xtype != '"') ){
    if( SAOdtype == 'h' )
      xtype = 'h';
    else
      xtype = 'd';
  }
  switch( xtype ) {
  case 'h':
    radius = h2d(dval);
    break;
  case 'd':
    radius = dval;
    break;
  case 'r':
    radius = r2d(dval);
    break;
  case '\'':
    radius = dval/60.0;
    break;
  case '"':
    radius = dval/3600.0;
    break;
  default:
    radius = dval;
    break;
  }

  /* calculate box limits, units in degrees */
  radecbox(ra, dec, radius*2, &r0, &r1, &d0, &d1);

  /* Output units conversion  */
  switch( csys->type[0] ) {
  case 'h': 
    csys->axis[0].val =    ra;
    csys->axis[0].lo =     r0;
    csys->axis[0].hi =     r1;
    break;
  case 'd':
    csys->axis[0].val = h2d(ra);
    csys->axis[0].lo = h2d(r0);
    csys->axis[0].hi = h2d(r1);
    break;
  case 'r':
    csys->axis[0].val = h2r(ra);
    csys->axis[0].lo = h2r(r0);
    csys->axis[0].hi = h2r(r1);
    break;
  }
  switch( csys->type[1] ) {
  case 'h':
    csys->axis[1].val = d2h(dec);
    csys->axis[1].lo = d2h(d0);
    csys->axis[1].hi = d2h(d1);
    break;
  case 'd':
    csys->axis[1].val =    dec;
    csys->axis[1].lo =     d0;
    csys->axis[1].hi =     d1;
    break;
  case 'r':
    csys->axis[1].val = d2r(dec);
    csys->axis[1].lo = d2r(d0);
    csys->axis[1].hi = d2r(d1);
    break;
  }
  /* force separation to be in radians, as needed by angular sep code */
  csys->type[2] = 'r';
  csys->radius = d2r(radius);
}

#ifdef ANSI_FUNC
static int
AxisInfo(Csys *csys, char *colstr[2])
#else
static int AxisInfo(csys, colstr)
     Csys *csys;
     char *colstr[2];
#endif
{
  int i;
  int ip;
  int got=0;
  char tbuf[SZ_LINE];
  char tbuf2[SZ_LINE];

  newdtable(":");
  for(i=0; i<2; i++){
    ip = 0;
    /* name of column */
    if( !word(colstr[i], tbuf, &ip) ){
      gerror(stderr, "invalid column #%d specification\n", i+1);
      goto done;
    }
    if( !word(colstr[i], tbuf2, &ip) ){
      switch(i){
      case 0:
	strcpy(tbuf2, RA_DEFAULT_UNITS);
	break;
      case 1:
	strcpy(tbuf2, DEC_DEFAULT_UNITS);
	break;
      default:
	gerror(stderr, "internal funcone error: bad iter %d\n", i);
	break;
      }
    }
    /* axis info */
    csys->axis[i].colname = xstrdup(tbuf);
    switch(*tbuf2){
    case 'h':
      csys->type[i] = 'h';
      break;
    case 'd':
      csys->type[i] = 'd';
      break;
    case 'r':
      csys->type[i] = 'r';
      break;
    default:
      gerror(stderr, "invalid units specification '%s' for %s\n", tbuf2, tbuf);
      goto done;
    }
  }
  got = 1;

done:
  freedtable();
  return got;
}

#ifdef ANSI_FUNC
static void
usage (char *pname)
#else
static void usage(pname)
     char *pname;
#endif
{
  fprintf(stderr,
         "usage: %s <switches> iname oname ra[hdr] dec[hdr] radius[dr\"'] [columns]\n",
	 pname);
  fprintf(stderr, "\n");
  fprintf(stderr, "  iname:  input FITS file name\n");
  fprintf(stderr, "  oname:  output FITS file name\n");
  fprintf(stderr, "  ra:     RA  center position, optional units (def: hours)\n");
  fprintf(stderr, "  dec:    Dec center position, optional units (def: degrees)\n");
  fprintf(stderr, "  radius: Search radius, optional units (def: degrees)\n");
  fprintf(stderr, "  columns: optional columns to output\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "optional switches:\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "  -r  racol:[hdr]  # RA   column name, units (def: RA:h)\n");
  fprintf(stderr, "  -d deccol:[hdr]  # Dec. column name, units (def: DEC:d)\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "The default units for input center and radius are:");
  fprintf(stderr, " ra:hours, dec:degrees, radius: degrees.\n");
  fprintf(stderr, "Sexagesimal notation is supported.\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "examples:\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "# defaults: RA col in hours, DEC col in degrees\n");
  fprintf(stderr, "#           RA pos in hours, DEC pos and radius in degrees\n");
  fprintf(stderr, "%s in.fits out.fits 23.45 34.56 0.01\n",
	  pname);
  fprintf(stderr, "# defaults, but with RA position in degrees:\n");
  fprintf(stderr, "%s in.fits out.fits 23.45d 34.56 0.01\n",
	  pname);
  fprintf(stderr, "# user specified columns in degrees\n");
  fprintf(stderr, "# RA pos in hms, DEC pos in dms, radius in arcmin:\n");
  fprintf(stderr, "%s -r myRa:d -d myDec in.fits out.fits 12:30:15.5 30:12 15'\n",
	  pname);
  exit(1);
}

#ifdef ANSI_FUNC
int 
main (int argc, char **argv)
#else
int
main(argc, argv)
     int argc;
     char **argv;
#endif
{
  int c;
  int i;
  int debug=0;
  int len;
  int got;
  int args=0;
  int ioff=0;
  char *s;
  char *filestr=NULL;
  char *oname;
  char *rastr=NULL;
  char *decstr=NULL;
  char *radstr=NULL;
  char *params="merge=update";
  char *colstr[2];
  char tbuf[SZ_LINE];
  char iname[SZ_LINE];
  Csys *csys;
  Fun fun, ofun;
  Ev ev, ebuf=NULL;

  /* exit on gio errors */
  if( !getenv("GERROR")  )
    setgerror(2);
  
  /* initialize names of columns */
  colstr[0]="RA:h";
  colstr[1]="DEC:d";

  /* we want the args in the same order in which they arrived, and
     gnu getopt sometimes changes things without this */
  putenv("POSIXLY_CORRECT=true");

  /* process switch arguments */
  while ((c = getopt(argc, argv, "Dd:r:")) != -1){
    switch(c){
    case 'D':
      debug=1;
      break;
    case 'd':
      colstr[1] = optarg;
      break;
    case 'r':
      colstr[0] = optarg;
      break;
    }
  }

  /* get maxrow,if user-specified */
  if( (s=getenv("FUN_MAXROW")) != NULL )
    maxrow = atoi(s);

  /* check for required arguments */
  args = argc - optind;
  if( args < 5 ) usage(argv[0]);

  /* get required arguments */
  filestr = argv[optind+ioff++];
  oname = argv[optind+ioff++];
  rastr = argv[optind+ioff++];
  decstr = argv[optind+ioff++];
  radstr = argv[optind+ioff++];

  if( !(csys = (Csys *)xcalloc(sizeof(Csys), 1)) ){
    gerror(stderr, "can't allocate coordinate system record\n");
    exit(1);
  }

  /* get information for the ra, dec axes */
  if( !AxisInfo(csys, colstr) ){
    gerror(stderr, "can't get axis info for columns: %s %s\n",
	   colstr[0], colstr[1]);
    exit(1);
  }

  /* calculate limits from ra, dec, and radius */
  ConeLimits(csys, rastr, decstr, radstr);

  /* new filename containing the region through which we will filter */
  strncpy(iname, filestr, SZ_LINE-1);
  len = strlen(iname);
  if( iname[len-1] == ']' ){
    iname[len-1] = ',';
  }
  else{
    strncat(iname, "[", SZ_LINE-1);
  }
  if( csys->axis[0].lo < csys->axis[0].hi ){
    snprintf(tbuf, SZ_LINE-1, "%s=%f:%f&&",
	     csys->axis[0].colname, csys->axis[0].lo, csys->axis[0].hi);
  }
  else{
    snprintf(tbuf, SZ_LINE-1, "(%s>=%f||%s<=%f)&&",
	     csys->axis[0].colname, csys->axis[0].lo, 
	     csys->axis[0].colname, csys->axis[0].hi);
  }
  strncat(iname, tbuf, SZ_LINE-1);
  if( csys->axis[1].lo < csys->axis[1].hi ){
    snprintf(tbuf, SZ_LINE-1, "%s=%f:%f]",
	     csys->axis[1].colname, csys->axis[1].lo, csys->axis[1].hi);
  }
  else{
    snprintf(tbuf, SZ_LINE-1, "(%s>=%f||%s<=%f)]",
	     csys->axis[1].colname, csys->axis[1].lo, 
	     csys->axis[1].colname, csys->axis[1].hi);
  }
  strncat(iname, tbuf, SZ_LINE-1);

  if( debug ) fprintf(stderr, "searching: %s\n", iname);

  /* open input file */
  if( !(fun = FunOpen(iname, "rc", NULL)) ){
    gerror(stderr, "can't input FunOpen file (or find extension): %s\n", iname);
  }

  /* activate columns specified by user, if necessary */
  if( args >= ioff ) FunColumnActivate(fun, argv[optind+ioff], NULL);

  /* make sure columns are the table */
  for(i=0; i<2; i++){
    if( !FunColumnLookup(fun, csys->axis[i].colname, 0,
			 NULL, NULL, NULL, NULL, NULL, NULL) ){
      gerror(stderr, "column '%s' not found in data file\n", 
	     csys->axis[i].colname);
    }
  }

  /* open output file */
  if( !(ofun = FunOpen(oname, "w", fun)) ){
    gerror(stderr, "can't output FunOpen file: %s\n", oname);
  }

  /* we will work with the specified ra, de columns */
  FunColumnSelect(fun, sizeof(EvRec), params,
		  csys->axis[0].colname,  "D", "r", FUN_OFFSET(Ev, val0),
		  csys->axis[1].colname,  "D", "r", FUN_OFFSET(Ev, val1),
		  NULL);

  /* process events */
  while( (ebuf = (Ev)FunTableRowGet(fun, NULL, maxrow, NULL, &got)) ){
    /* process all rows */
    for(i=0; i<got; i++){
      /* point to the i'th row */
      ev = ebuf+i;
      if( ConeFilter(csys, ev->val0, ev->val1) ){
	if( FunTableRowPut(ofun, ev, 1, i, NULL) != 1 ){
	  gerror(stderr, "could not write row %d\n", i);
	}
      }
    }
    if( ebuf ) free(ebuf);
  }

  /* close files */
  FunClose(ofun);
  FunClose(fun);
  /* free up memory */
  if( csys ){
    for(i=0; i<2; i++){
      if( csys->axis[i].colname ) xfree(csys->axis[i].colname);
    }
    xfree(csys);
  }
  /* success */
  return(0);
}
