// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __fitsfile_h__
#define __fitsfile_h__

#include <tcl.h>

#include "head.h"

extern int DebugGZ;

class OutFitsStream;

class FitsFile {
public:
  enum FlushMode {NOFLUSH,FLUSH};
  enum ScanMode {RELAX, EXACT};
  enum ArchType {NATIVE,BIGENDIAN,LITTLEENDIAN};

  char* pName_;              // parsed file name

protected:
  FitsHead* primary_;        // pointer to primary header
  int managePrimary_;        // flag, true if we manage primary header

  FitsHead* head_;           // pointer to header
  int manageHead_;           // flag, true if we manage header

  void* data_;               // pointer to raw data

  int ext_;                  // extension number
  int inherit_;              // do we have inheritence?
  int byteswap_;             // flag, true if byteswap is needed
  int orgFits_;              // flag to indicate origin
  int valid_;                // flag, true if file is valid

  char* pExt_;               // parsed ext name
  int pIndex_;               // parsed ext number
  char* pFilter_;            // unparsed filter spec
  char* pSliceFilter_;       // unparsed slice filter spec
  char* pBinX_;              // parsed bin table x col name
  char* pBinY_;              // parsed bin table y col name
  char* pBinZ_;              // parsed bin table z col name
  Vector binZlim_;           // limits of z col
  int validZlim_;            // limits valid

  int pWidth_;               // parsed array width
  int pHeight_;              // parsed array height
  int pDepth_;               // parsed array depth
  int pBitpix_;              // parsed array bitpix
  int pSkip_;                // parsed array skip size
  ArchType pArch_;           // parsed array arch type

private:
  int saveFitsImage(OutFitsStream&);
  int saveFitsTable(OutFitsStream&);
  int saveArray(OutFitsStream&);

protected:
  void parse(const char*);
  int findEnd(const char*);
  void setArrayByteSwap();
  int validArrayParams();

public:
  FitsFile();
  virtual ~FitsFile();

  virtual void done() {}
  virtual char* page(char* ptr, off_t r) {return ptr;}
  virtual void resetpage() {}
  void error(const char*);
  void* data() {return data_;}
  FitsHead* head() {return head_;}
  FitsHead* primary() {return primary_;}
  int ext() {return ext_;}
  int inherit() {return inherit_;}
  int isValid() {return valid_;}
  int isImage() {return head_ ? head_->isImage() : 0;}
  int isTable() {return head_ ? head_->isTable() : 0;}
  int byteswap() {return byteswap_;}

  void setpName(const char*);
  void setpExt(const char*);
  void setpIndex(int i) {pIndex_ = i;}
  void setpFilter(const char*);
  void setpSliceFilter(const char*);
  void setpBinX(const char*);
  void setpBinY(const char*);
  void setpBinZ(const char*);
  void setpBinXY(const char* x, const char* y) {setpBinX(x); setpBinY(y);}
  void setpBinXYZ(const char* x, const char* y, const char* z)
    {setpBinX(x); setpBinY(y); setpBinZ(z);}
  void setBinZlim(const Vector& lim) {binZlim_=lim; validZlim_=1;}
  Vector getBinZlim();

  const char* pName() {return pName_;}
  const char* pExt() {return pExt_;}
  int pIndex() {return pIndex_;}
  const char* pFilter() {return pFilter_;}
  const char* pSliceFilter() {return pSliceFilter_;}
  const char* pBinX() {return pBinX_;}
  const char* pBinY() {return pBinY_;}
  const char* pBinZ() {return pBinZ_;}

  void setpXdim(int i) {pWidth_ = i;}
  void setpYdim(int i) {pHeight_ = i;}
  void setpZdim(int i) {pDepth_ = i;}
  void setpBitpix(int b) {pBitpix_ = b;}
  void setpSkip(int s) {pSkip_ = s;}
  void setpArch(ArchType a) {pArch_ = a;}

  Vector getColMinMax(const char*);

  int find(const char* name);
  int getLogical(const char* name, int def);
  int getInteger(const char* name, int def);
  double getReal(const char* name, double def);
  void getComplex(const char* name, double* real, double* img, 
		  double rdef, double idef);
  char* getString(const char* name);
  char* getComment(const char* name);
  char* getKeyword(const char* name);

  int saveFitsImageFile(const char*, int);
  int saveFitsImageChannel(Tcl_Interp*, const char*, int);
  int saveFitsImageSocket(int, int);

  int saveFitsTableFile(const char*, int);
  int saveFitsTableChannel(Tcl_Interp*, const char*, int);
  int saveFitsTableSocket(int, int);

  int saveArrayFile(const char*);
  int saveArrayChannel(Tcl_Interp*, const char*);
  int saveArraySocket(int);
};

#endif
