// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "baseellipse.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

BaseEllipse::BaseEllipse(const BaseEllipse& a) : BaseMarker(a) {}

BaseEllipse::BaseEllipse(FrameBase* p, const Vector& ctr, 
			 double ang,
			 const char* clr,  int w, const char* f, 
			 const char* t, unsigned short prop, const char* c,
			 const List<Tag>& tag, const List<CallBack>& cb)
  : BaseMarker(p, ctr, ang, 48, clr, w, f, t, prop, c, tag, cb) {}

int BaseEllipse::isIn(const Vector& v)
{
  // v is in canvas coords
  Vector r = annuli_[numAnnuli_-1];

  // zero radius
  if (!r[0] || !r[1])
    return 0;

  Matrix mm = bckRefMatrix();
  Vector p = v * mm;

  if ((p[0]*p[0])/(r[0]*r[0]) + (p[1]*p[1])/(r[1]*r[1]) <= 1)
    return 1;
  else
    return 0;
}

// protected

void BaseEllipse::updateHandles()
{
  // handles are in canvas coords
  // we can't garantee that the annuli_ have been sorted yet

  if (handle)
    delete [] handle;
  handle = new Vector[numHandle];

  Vector max;
  for(int i=0; i<numAnnuli_; i++)
    if (max[0]<annuli_[i][0])
      max = annuli_[i];
  Vector r = max;

  Matrix mm = fwdCanvasMatrix();

  handle[0] = Vector(-r[0],-r[1]) * mm;
  handle[1] = Vector( r[0],-r[1]) * mm;
  handle[2] = Vector( r[0], r[1]) * mm;
  handle[3] = Vector(-r[0], r[1]) * mm;

  // the rest are annuli_
  for (int i=4; i<numHandle; i++)
    handle[i] = Vector(annuli_[i-4][0],0) * mm;
}

void BaseEllipse::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  setGC(mode);

  Matrix mm = fwdRefMatrix() * mx;

  double ang = calcAngle();
  Vector r = annuli_[numAnnuli_-1];
  Vector z = parent->zoom();

  int isRound = r[0] == r[1];
  int isScale = z[0] == z[1];
  int isAngle = \
    (ang >         -FLT_EPSILON && ang <          FLT_EPSILON) ||
    (ang >   M_PI/2-FLT_EPSILON && ang <   M_PI/2+FLT_EPSILON) ||
    (ang >   M_PI  -FLT_EPSILON && ang <   M_PI  +FLT_EPSILON) ||
    (ang > 3*M_PI/2-FLT_EPSILON && ang < 3*M_PI/2+FLT_EPSILON);

  if (isRound && isScale)
    renderCircle(drawable, mx);
  else if (!isRound && isScale && isAngle)
    renderEllipse(drawable, mx);
  else
    BaseMarker::render(drawable, mx, mode);

  if ((!(properties & INCLUDE)) && ((isRound && isScale) || (!isRound && isScale && isAngle))) {
    double theta = degToRad(45);
    Vector r1 = (Vector(annuli_[numAnnuli_-1][0]*cos(theta), 
			annuli_[numAnnuli_-1][1]*sin(theta)) * mm).round();
    Vector r2 = (Vector(-annuli_[numAnnuli_-1][0]*cos(theta), 
			-annuli_[numAnnuli_-1][1]*sin(theta)) * mm).round();

    if (mode==SRC)
      XSetForeground(display, gc, parent->getRedColor());

    XDRAWLINE(display, drawable, gc, (int)r1[0], (int)r1[1], 
	      (int)r2[0], (int)r2[1]);    
  }
}

void BaseEllipse::renderCircle(Drawable drawable, const Matrix& mx)
{
  // this routine is only valid for circles with equal zoom in x & y

  Matrix mm = fwdRefMatrix() * mx;
  Vector c = center * mx;

  for (int i=0; i<numAnnuli_; i++) {
    Vector r = annuli_[i];

    Vector ur = (r * mm).round();
    double l = (ur-c).length() * cos(M_PI/4);

    Vector rr(l,l);
    Vector st = (c-rr).round();
    Vector size = (rr*2).round();

    int a1 = startAng_ + radToDeg(parent->getRotation());
    int a2 = stopAng_ + radToDeg(parent->getRotation());
    if (a2<=a1)
      a2 += 360;

    // Verify size is positive
    // XDrawArc is sensative to bad data, and may hang the XServer
    if (size[0]>0 && size[1]>0)
      XDRAWARC(display, drawable, gc, (int)st[0], (int)st[1], 
	       (int)size[0], (int)size[1], a1*64, (a2-a1)*64);
  }
}

void BaseEllipse::renderEllipse(Drawable drawable, const Matrix& mx)
{
  // this routine is only valid for equal zoom in x & y and 
  // angles of 0,90,180,and 270

  Matrix mm = fwdRefMatrix() * mx;

  for (int i=0; i<numAnnuli_; i++) {
    Vector r = annuli_[i];

    Vector ur = ( r * mm).round();
    Vector ll = (-r * mm).round();
    Vector size = (ur - ll).abs();
    Vector st(ur[0]>ll[0]?ll[0]:ur[0], ur[1]>ll[1]?ll[1]:ur[1]);

    int a1 = startAng_ + radToDeg(parent->getRotation());
    int a2 = stopAng_ + radToDeg(parent->getRotation());
    if (a2<=a1)
      a2 += 360;

    // Verify size is positive
    // XDrawArc is sensative to bad data, and may hang the XServer
    if (size[0]>0 && size[1]>0)
      XDRAWARC(display, drawable, gc, (int)st[0], (int)st[1],
	       (int)size[0], (int)size[1], a1*64, (a2-a1)*64);
  }
}

void BaseEllipse::newVertices()
{
  if (vertices_)
    deleteVertices();

  vertices_ = new Vector*[numAnnuli_+1];
  for (int i=0; i<numAnnuli_; i++)
    vertices_[i]= new Vector[numPoints_];
  vertices_[numAnnuli_] = new Vector[2];

  Matrix mm = fwdRefMatrix();

  double a1 = degToRad(startAng_);
  double a2 = degToRad(stopAng_);
  if (a2<=a1+FLT_EPSILON)
    a2 += 2*M_PI;

  double a2l = (a2-2*M_PI)-FLT_EPSILON;
  double a2h = (a2-2*M_PI)+FLT_EPSILON;
  int nn = numPoints_-1;
  for (int i=0; i<numAnnuli_; i++) {
    if (a1 >= a2l && a1 <= a2h) {
      for (int j=0; j<nn; j++) {
	double theta = ((a2-a1)/nn*j)+a1;
	vertices_[i][j] = 
	  Vector(annuli_[i][0]*cos(-theta), annuli_[i][1]*sin(-theta))*mm;
      }
      vertices_[i][nn] = vertices_[i][0];
    }
    else {
      for (int j=0; j<nn; j++) {
	double theta = ((a2-a1)/nn*j)+a1;
	vertices_[i][j] = 
	  Vector(annuli_[i][0]*cos(-theta), annuli_[i][1]*sin(-theta))*mm;
      }
      vertices_[i][nn] =
	  Vector(annuli_[i][0]*cos(-a2), annuli_[i][1]*sin(-a2))*mm;
    }
  }

  double theta = degToRad(45);
  vertices_[numAnnuli_][0] = 
    Vector(annuli_[numAnnuli_-1][0]*cos(theta), annuli_[numAnnuli_-1][1]*sin(theta))*mm;
  vertices_[numAnnuli_][1] = 
    Vector(-annuli_[numAnnuli_-1][0]*cos(theta), -annuli_[numAnnuli_-1][1]*sin(theta))*mm;
}

void BaseEllipse::ps(int mode) {
  Marker::ps(mode);

  Vector r = annuli_[numAnnuli_-1];
  Vector z = parent->zoom();
  int isRound = r[0] == r[1];
  int isScale = z[0] == z[1];

  if (isRound && isScale)
    psCircle(mode);
  else
    psEllipse(mode);

  if (!(properties & INCLUDE)) {
    Matrix mm = fwdCanvasMatrix();
    double theta = degToRad(45);
    Vector r1 = Vector(annuli_[numAnnuli_-1][0]*cos(theta), 
		       annuli_[numAnnuli_-1][1]*sin(theta)) * mm;
    Vector r2 = Vector(-annuli_[numAnnuli_-1][0]*cos(theta), 
		       -annuli_[numAnnuli_-1][1]*sin(theta)) * mm;

    psColor(mode, "red");

    ostringstream str;
    str << "newpath " 
	<< r1.TkCanvasPs(parent->canvas) << "moveto"
	<< r2.TkCanvasPs(parent->canvas) << "lineto"
	<< " stroke" << endl << ends;
    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

void BaseEllipse::psCircle(int mode)
{
  Vector cc = center * parent->refToCanvas;
  Matrix mm = fwdCanvasMatrix();

  for (int i=0; i<numAnnuli_; i++) {
    Vector r = annuli_[i];

    Vector ur = (r * mm).round();
    double l = (ur-cc).length() * cos(M_PI/4);

    double a1 = startAng_ + radToDeg(parent->getRotation());
    double a2 = stopAng_ + radToDeg(parent->getRotation());
    if (a2<=a1)
      a2 += 360;

    ostringstream str;
    str << "newpath " << cc.TkCanvasPs(parent->canvas) << l << ' '
	<< a1 << ' ' << a2 << " arc stroke " << endl << ends;

    Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
  }
}

void BaseEllipse::psEllipse(int mode)
{
  double a1 = degToRad(startAng_);
  double a2 = degToRad(stopAng_);

  if (a2 <= a1)
    a2 += 2*M_PI;

  for (int i=0; i<numAnnuli_; i++) {
    Vector r = annuli_[i];

    int s1,s2 =0;
    for (int i=0; i<8; i++) {
      double b1 = i*M_PI/2;
      double b2 = (i+1)*M_PI/2;
      if (!s1 && a1>=b1 && a1<b2)
	s1 =1;
      if (!s2 && a2>b1 && a2<=b2)
	s2 =1;

      if ((s1 && !s2) || (s1 && s2))
	psEllipsePrep(a1,a2,b1,b2,r);

      if (s1&&s2)
	s1=s2=0;
    }
  }
}

void BaseEllipse::psEllipsePrep(double a1, double a2, double ll, double ul, 
			       Vector& r)
{
  if (!(a1 >= ll && a1 <= ul))
    a1 = ll;
  if (!(a2 >= ll && a2 <= ul))
    a2 = ul;

  if (a1>a2) {
    psEllipseArc(ll,a2,r);
    psEllipseArc(a1,ul,r);
  }
  else
    psEllipseArc(a1,a2,r);
}				

void BaseEllipse::psEllipseArc(double a1, double a2, Vector& r)

{
  // bézier curve, valid for arcs of <M_PI/2
  double bcp = 4.0/3*(1-cos((a2-a1)/2))/sin((a2-a1)/2);
  Matrix mm = FlipY() * fwdCanvasMatrix();

  Vector t0 = Vector(r[0]*cos(a1),r[1]*sin(a1)) * mm;
  Vector x1 = Vector(r[0]*(cos(a1)-bcp*sin(a1)),r[1]*(sin(a1)+bcp*cos(a1)))*mm;
  Vector x2 = Vector(r[0]*(cos(a2)+bcp*sin(a2)),r[1]*(sin(a2)-bcp*cos(a2)))*mm;
  Vector t1 = Vector(r[0]*cos(a2),r[1]*sin(a2)) * mm;

  ostringstream str;
  str <<
    "newpath " <<
    t0.TkCanvasPs(parent->canvas) << "moveto " <<
    x1.TkCanvasPs(parent->canvas) << 
    x2.TkCanvasPs(parent->canvas) << 
    t1.TkCanvasPs(parent->canvas) << 
    " curveto stroke" << 
    endl;

  str << ends;

  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
}


