#  Copyright (C) 1999-2012
#  Smithsonian Astrophysical Observatory, Cambridge, MA, USA
#  For conditions of distribution and use, see copyright notice in "copyright"

package provide DS9 1.0

proc APDef {} {
    global iap
    global pap

    set iap(tt) {ap}
    set iap(windows) {}
    set iap(unique) 0

    set pap(grid) 1
    set pap(grid,log) linearlinear

    set pap(discrete) 0
    set pap(discrete,symbol) circle
    set pap(discrete,color) red

    set pap(linear) 1
    set pap(linear,width) 1
    set pap(linear,color) black
    set pap(linear,dash) no

    set pap(step) 0
    set pap(step,width) 1
    set pap(step,color) black
    set pap(step,dash) no

    set pap(quadratic) 0
    set pap(quadratic,width) 1
    set pap(quadratic,color) black
    set pap(quadratic,dash) no

    set pap(error) 1
    set pap(error,width) 1
    set pap(error,color) red
    set pap(error,style) 1

    set pap(titleFont) helvetica
    set pap(titleSize) 12
    set pap(titleWeight) normal
    set pap(titleSlant) roman

    set pap(textlabFont) helvetica
    set pap(textlabSize) 9
    set pap(textlabWeight) normal
    set pap(textlabSlant) roman

    set pap(numlabFont) helvetica
    set pap(numlabSize) 9
    set pap(numlabWeight) normal
    set pap(numlabSlant) roman
}

proc AnalysisPlot {tt wtt title xaxis yaxis dim data} {
    global iap

    # make the window name unique
    set ii [lsearch $iap(windows) $tt]
    if {$ii>=0} {
	append tt $iap(unique)
	incr iap(unique)
    }
    set varname $tt

    # add it to our xpa list
    lappend iap(windows) $varname

    # set the window title if none
    if {$wtt == {}} {
	set wtt $varname
    }

    APDialog $varname $wtt $title $xaxis $yaxis
    APDataSet $varname $dim $data
}

proc APDialog {varname wtt title xaxis yaxis} {
    upvar #0 $varname var
    global $varname

    set var(top) ".${varname}"
    set var(mb) ".${varname}mb"

    if {[APPing $varname]} {
	APRaise $varname
	return
    }

    global pap
    global ds9

    set var(stats) 0

    set var(data,total) 0
    set var(data,current) 0

    set var(xdata) {}
    set var(ydata) {}
    set var(xedata) {}
    set var(yedata) {}

    set var(graph,title) "$title"
    set var(graph,xaxis) "$xaxis"
    set var(graph,yaxis) "$yaxis"

    set var(graph,x,auto) 1
    set var(graph,x,min) {}
    set var(graph,x,max) {}
    set var(graph,y,auto) 1
    set var(graph,y,min) {}
    set var(graph,y,max) {}

    set var(format) 1
    set var(format,x) {}
    set var(format,y) {}

    set var(grid) $pap(grid)
    set var(grid,log) $pap(grid,log) 

    set var(discrete) $pap(discrete) 
    set var(discrete,symbol) $pap(discrete,symbol) 
    set var(discrete,color) $pap(discrete,color) 

    set var(linear) $pap(linear) 
    set var(linear,width) $pap(linear,width) 
    set var(linear,color) $pap(linear,color) 
    set var(linear,dash) $pap(linear,dash) 

    set var(step) $pap(step) 
    set var(step,width) $pap(step,width) 
    set var(step,color) $pap(step,color) 
    set var(step,dash) $pap(step,dash) 

    set var(quadratic) $pap(quadratic) 
    set var(quadratic,width) $pap(quadratic,width) 
    set var(quadratic,color) $pap(quadratic,color) 
    set var(quadratic,dash) $pap(quadratic,dash) 

    set var(error) $pap(error) 
    set var(error,width) $pap(error,width) 
    set var(error,color) $pap(error,color) 
    set var(error,style) $pap(error,style) 

    set var(titleFont) $pap(titleFont) 
    set var(titleSize) $pap(titleSize) 
    set var(titleWeight) $pap(titleWeight) 
    set var(titleSlant) $pap(titleSlant) 

    set var(textlabFont) $pap(textlabFont) 
    set var(textlabSize) $pap(textlabSize) 
    set var(textlabWeight) $pap(textlabWeight) 
    set var(textlabSlant) $pap(textlabSlant) 

    set var(numlabFont) $pap(numlabFont) 
    set var(numlabSize) $pap(numlabSize) 
    set var(numlabWeight) $pap(numlabWeight)
    set var(numlabSlant) $pap(numlabSlant)

    # create window
    Toplevel $var(top) $var(mb) 7 $wtt "APDestroy $varname"

    $var(mb) add cascade -label [msgcat::mc {File}] -menu $var(mb).file
    $var(mb) add cascade -label [msgcat::mc {Edit}] -menu $var(mb).edit
    $var(mb) add cascade -label [msgcat::mc {Graph}] -menu $var(mb).graph
    $var(mb) add cascade -label [msgcat::mc {Font}] -menu $var(mb).font
    $var(mb) add cascade -label [msgcat::mc {Dataset}] -menu $var(mb).dataset
    $var(mb) add cascade -label [msgcat::mc {View}] -menu $var(mb).view
    $var(mb) add cascade -label [msgcat::mc {Color}] -menu $var(mb).color
    $var(mb) add cascade -label [msgcat::mc {Line}] -menu $var(mb).line

    menu $var(mb).file
    $var(mb).file add command -label "[msgcat::mc {Load Data}]..." \
	-command "APLoadData $varname"
    $var(mb).file add command -label "[msgcat::mc {Save Data}]..." \
	-command "APSaveData $varname"
    $var(mb).file add command -label [msgcat::mc {Clear Data}] \
	-command "APClearData $varname"
    $var(mb).file add separator
    $var(mb).file add command -label "[msgcat::mc {Load Configuration}]..." \
	-command "APLoadConfig $varname"
    $var(mb).file add command -label "[msgcat::mc {Save Configuration}]..." \
	-command "APSaveConfig $varname"
    $var(mb).file add separator
    $var(mb).file add command -label [msgcat::mc {Statistics}] \
	-command "set ${varname}(stats) 1; APStats $varname"
    $var(mb).file add separator
    switch $ds9(wm) {
	x11 {
	    $var(mb).file add command \
		-label "[msgcat::mc {Page Setup}]..." \
		-command PSPageSetup
	    $var(mb).file add command -label "[msgcat::mc {Print}]..." \
		-command [list APPSPrint $varname]
	}
	win32 {
	    $var(mb).file add command \
		-label "[msgcat::mc {Postscript Page Setup}]..." \
		-command PSPageSetup
	    $var(mb).file add command \
		-label "[msgcat::mc {Postscript Print}]..." \
		-command [list APPSPrint $varname]
	    if {0} {
	    $var(mb).file add separator
	    $var(mb).file add command \
		-label "[msgcat::mc {Page Setup}]..." \
		-command Win32PageSetup
	    $var(mb).file add command \
		-label "[msgcat::mc {Print}]..." \
		-command [list APNativePrint $varname]
	    }
	}
	aqua {}
    }
    $var(mb).file add separator
    $var(mb).file add command -label [msgcat::mc {Close}] \
	-command "APDestroy $varname"

    menu $var(mb).edit
    $var(mb).edit add command -label [msgcat::mc {Cut}] \
	-state disabled -accelerator "${ds9(ctrl)}X"
    $var(mb).edit add command -label [msgcat::mc {Copy}] \
	-state disabled -accelerator "${ds9(ctrl)}C"
    $var(mb).edit add command -label [msgcat::mc {Paste}] \
	-state disabled -accelerator "${ds9(ctrl)}V"
    $var(mb).edit add command -label [msgcat::mc {Clear}] \
	-state disabled

    menu $var(mb).graph
    $var(mb).graph add checkbutton -label [msgcat::mc {Grid}] \
	-variable ${varname}(grid) \
	-command "APUpdateGraph $varname"
    $var(mb).graph add separator
    $var(mb).graph add radiobutton \
	-label "[msgcat::mc {Linear}]-[msgcat::mc {Linear}]" \
	-variable ${varname}(grid,log) -value linearlinear \
	-command "APUpdateGraph $varname"
    $var(mb).graph add radiobutton \
	-label "[msgcat::mc {Linear}]-[msgcat::mc {Log}]" \
	-variable ${varname}(grid,log) -value linearlog \
	-command "APUpdateGraph $varname"
    $var(mb).graph add radiobutton \
	-label "[msgcat::mc {Log}]-[msgcat::mc {Linear}]" \
	-variable ${varname}(grid,log) -value loglinear \
	-command "APUpdateGraph $varname"
    $var(mb).graph add radiobutton \
	-label "[msgcat::mc {Log}]-[msgcat::mc {Log}]" \
	-variable ${varname}(grid,log) -value loglog \
	-command "APUpdateGraph $varname"
    $var(mb).graph add separator
    $var(mb).graph add command -label "[msgcat::mc {Axes}]..." \
	-command "APAxes $varname"
    $var(mb).graph add command -label "[msgcat::mc {Labels}]..." \
	-command "APTitle $varname"

    menu $var(mb).font
    $var(mb).font add cascade -label [msgcat::mc {Axis Numbers}] \
	-menu $var(mb).font.numlab
    $var(mb).font add cascade -label [msgcat::mc {Axis Label}] \
	-menu $var(mb).font.textlab
    $var(mb).font add cascade -label [msgcat::mc {Title}] \
	-menu $var(mb).font.title

    FontMenu $var(mb).font.numlab $varname numlabFont numlabSize \
	numlabWeight numlabSlant [list APUpdateGraph $varname]
    FontMenu $var(mb).font.textlab $varname textlabFont textlabSize \
	textlabWeight textlabSlant [list APUpdateGraph $varname]
    FontMenu $var(mb).font.title $varname titleFont titleSize \
	titleWeight titleSlant [list APUpdateGraph $varname]

    menu $var(mb).dataset

    menu $var(mb).view
    $var(mb).view add checkbutton -label [msgcat::mc {Discrete}] \
	-variable ${varname}(discrete) -command "APUpdateState $varname"
    $var(mb).view add checkbutton -label [msgcat::mc {Linear}] \
	-variable ${varname}(linear) -command "APUpdateState $varname"
    $var(mb).view add checkbutton -label [msgcat::mc {Step}] \
	-variable ${varname}(step) -command "APUpdateState $varname"
    $var(mb).view add checkbutton -label [msgcat::mc {Quadratic}] \
	-variable ${varname}(quadratic) -command "APUpdateState $varname"
    $var(mb).view add checkbutton -label [msgcat::mc {Error}] \
	-variable ${varname}(error) -command "APUpdateState $varname"

    menu $var(mb).color
    $var(mb).color add cascade -label [msgcat::mc {Discrete}] \
	-menu $var(mb).color.discrete
    $var(mb).color add cascade -label [msgcat::mc {Linear}] \
	-menu $var(mb).color.linear
    $var(mb).color add cascade -label [msgcat::mc {Step}] \
	-menu $var(mb).color.step
    $var(mb).color add cascade -label [msgcat::mc {Quadratic}] \
	-menu $var(mb).color.quadratic
    $var(mb).color add cascade -label [msgcat::mc {Error}] \
	-menu $var(mb).color.error

    ColorMenu $var(mb).color.discrete $varname discrete,color \
	[list APUpdateState $varname]
    ColorMenu $var(mb).color.linear $varname linear,color \
	[list APUpdateState $varname]
    ColorMenu $var(mb).color.step $varname step,color \
	[list APUpdateState $varname]
    ColorMenu $var(mb).color.quadratic $varname quadratic,color \
	[list APUpdateState $varname]
    ColorMenu $var(mb).color.error $varname error,color \
	[list APUpdateState $varname]

    menu $var(mb).line
    $var(mb).line add cascade -label [msgcat::mc {Discrete}] \
	-menu $var(mb).line.discrete
    $var(mb).line add cascade -label [msgcat::mc {Linear}] \
	-menu $var(mb).line.linear
    $var(mb).line add cascade -label [msgcat::mc {Step}] \
	-menu $var(mb).line.step
    $var(mb).line add cascade -label [msgcat::mc {Quadratic}] \
	-menu $var(mb).line.quadratic
    $var(mb).line add cascade -label [msgcat::mc {Error}] \
	-menu $var(mb).line.error

    menu $var(mb).line.discrete
    $var(mb).line.discrete add radiobutton -label [msgcat::mc {Circle}] \
	-variable ${varname}(discrete,symbol) -value circle \
	-command "APUpdateState $varname"
    $var(mb).line.discrete add radiobutton -label [msgcat::mc {Diamond}] \
	-variable ${varname}(discrete,symbol) -value diamond \
	-command "APUpdateState $varname"
    $var(mb).line.discrete add radiobutton -label [msgcat::mc {Plus}] \
	-variable ${varname}(discrete,symbol) -value plus \
	-command "APUpdateState $varname"
    $var(mb).line.discrete add radiobutton -label [msgcat::mc {Cross}] \
	-variable ${varname}(discrete,symbol) -value cross \
	-command "APUpdateState $varname"

    APCreateLineMenu $var(mb).line.linear \
	${varname}(linear,width) ${varname}(linear,dash) \
	"APUpdateState $varname"
    APCreateLineMenu $var(mb).line.step \
	${varname}(step,width) ${varname}(step,dash) \
	"APUpdateState $varname"
    APCreateLineMenu $var(mb).line.quadratic \
	${varname}(quadratic,width) ${varname}(quadratic,dash) \
	"APUpdateState $varname"

    WidthDashMenu $var(mb).line.error $varname error,width {} \
	[list APUpdateState $varname] {}

    $var(mb).line.error add separator
    $var(mb).line.error add radiobutton -label "[msgcat::mc {Style}] 1" \
	-variable ${varname}(error,style) -value 1 \
	-command "APUpdateState $varname"
    $var(mb).line.error add radiobutton -label "[msgcat::mc {Style}] 2" \
	-variable ${varname}(error,style) -value 2 \
	-command "APUpdateState $varname"

    set var(graph) [blt::graph $var(top).g \
			-width 500 -height 400 \
			-plotrelief groove \
			-plotborderwidth 2 \
			]
    $var(graph) legend configure -hide yes

    switch $ds9(wm) {
	x11 -
	win32 {Blt_ZoomStack $var(graph)} 
	aqua {Blt_ZoomStack $var(graph) "ButtonPress-1" "ButtonPress-2"}
    }

    pack $var(graph) -expand yes -fill both

    APUpdateGraph $varname
}

proc APDestroy {varname} {
    upvar #0 $varname var
    global $varname

    global iap

    # see if it still is around
    if {![APPing $varname]} {
	return
    }

    set tt $varname

    # in the rare case where the user has start a zoom, then closes the dialog
    # we must clean up his mess
    global zoomInfo
    if [info exists zoomInfo($var(graph),corner)] {
	if {$zoomInfo($var(graph),corner) == {B}} {
	    blt::SetZoomPoint $var(graph) 0 0
	}
    }

    for {set nn 1} {$nn<=$var(data,total)} {incr nn} {
	if {$var($nn,manage)} {
	    blt::vector destroy \
		$var($nn,xdata) $var($nn,ydata) \
		$var($nn,xedata) $var($nn,yedata)
	}
    }

    destroy $var(top)
    destroy $var(mb)

    # stats window?
    if {$var(stats)} {
	SimpleTextDestroy "${varname}txt"
    }

    # delete it from the xpa list
    set ii [lsearch $iap(windows) $varname]
    if {$ii>=0} {
	set iap(windows) [lreplace $iap(windows) $ii $ii]
    }

    unset $varname
}

proc APAxisFormat {varname axis w nn} {
    upvar #0 $varname var
    global $varname

    return [format $var(format,$axis) $nn]
}

proc APCurrentData {varname} {
    upvar #0 $varname var
    global $varname

    if {$var(data,total) > 0} {
	set nn $var(data,current)

	set var(xdata) $var($nn,xdata)
	set var(ydata) $var($nn,ydata)
	set var(xedata) $var($nn,xedata)
	set var(yedata) $var($nn,yedata)

	set var(discrete) $var($nn,discrete) 
	set var(discrete,symbol) $var($nn,discrete,symbol) 
	set var(discrete,color) $var($nn,discrete,color) 

	set var(linear) $var($nn,linear) 
	set var(linear,width) $var($nn,linear,width) 
	set var(linear,color) $var($nn,linear,color) 
	set var(linear,dash) $var($nn,linear,dash) 

	set var(step) $var($nn,step) 
	set var(step,width) $var($nn,step,width) 
	set var(step,color) $var($nn,step,color) 
	set var(step,dash) $var($nn,step,dash) 

	set var(quadratic) $var($nn,quadratic) 
	set var(quadratic,width) $var($nn,quadratic,width) 
	set var(quadratic,color) $var($nn,quadratic,color) 
	set var(quadratic,dash) $var($nn,quadratic,dash) 

	set var(error) $var($nn,error) 
	set var(error,width) $var($nn,error,width) 
	set var(error,color) $var($nn,error,color) 
	set var(error,style) $var($nn,error,style) 
    }

    APStats $varname
}

proc APDataFormatDialog {xarname} {
    upvar $xarname xar
    global ed

    set w {.apdata}

    set ed(ok) 0
    set ed(dim) $xar

    DialogCreate $w [msgcat::mc {Data Format}] ed(ok)

    # Param
    set f [ttk::frame $w.param]

    ttk::label $f.title -text [msgcat::mc {Data Format}]
    ttk::radiobutton $f.xy -text {X Y} -variable ed(dim) -value xy
    ttk::radiobutton $f.xyex -text {X Y XErr} -variable ed(dim) -value xyex
    ttk::radiobutton $f.xyey -text {X Y YErr} -variable ed(dim) -value xyey
    ttk::radiobutton $f.xyexey -text {X Y XErr YErr} -variable ed(dim) \
	-value xyexey

    grid $f.title -padx 2 -pady 2 -sticky w
    grid $f.xy -padx 2 -pady 2 -sticky w
    grid $f.xyex -padx 2 -pady 2 -sticky w
    grid $f.xyey -padx 2 -pady 2 -sticky w
    grid $f.xyexey -padx 2 -pady 2 -sticky w

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed(ok) $w.param.xy
    DialogDismiss $w

    if {$ed(ok)} {
	set xar $ed(dim)
    }

    set rr $ed(ok)
    unset ed
    return $rr
}

proc APLoadData {varname} {
    upvar #0 $varname var
    global $varname

    global ds9
    global ed

    set filename [OpenFileDialog apdatafbox]
    if {$filename != {}} {
	set dim xy
	if {[APDataFormatDialog dim]} {
	    APPLoadData $varname $filename $dim
	}
    }
}

proc APPLoadData {varname filename dim} {
    upvar #0 $varname var
    global $varname

    set ch [open $filename]
    set data [read $ch]
    close $ch

    APRaise $varname
    APDataSet $varname $dim $data
}

proc APSaveData {varname} {
    upvar #0 $varname var
    global $varname

    APPSaveData $varname [SaveFileDialog apdatafbox]
}

proc APPSaveData {varname filename} {
    upvar #0 $varname var
    global $varname

    if {$filename != {}} {
	set ch [open $filename w]

	global $var(xdata) $var(ydata) $var(xedata) $var(yedata)
	set l [$var(xdata) length]

	if {([$var(xedata) length] == 0) && 
	    ([$var(yedata) length] == 0)} {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$var(xdata)\($i\)"]
		set y [expr "$$var(ydata)\($i\)"]
		puts $ch "$x $y"
	    }
	} elseif {([$var(xedata) length] == 0) &&
		  ([$var(yedata) length] != 0)} {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$var(xdata)\($i\)"]
		set y [expr "$$var(ydata)\($i\)"]
		set ye [expr "$$var(yedata)\($i\)"]
		puts $ch "$x $y $ye"
	    }
	} elseif {([$var(xedata) length] != 0) &&
		  ([$var(yedata) length] == 0)} {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$var(xdata)\($i\)"]
		set y [expr "$$var(ydata)\($i\)"]
		set xe [expr "$$var(xedata)\($i\)"]
		puts $ch "$x $y $xe"
	    }
	} else {
	    for {set i 0} {$i<$l} {incr i} {
		set x [expr "$$var(xdata)\($i\)"]
		set y [expr "$$var(ydata)\($i\)"]
		set xe [expr "$$var(xedata)\($i\)"]
		set ye [expr "$$var(yedata)\($i\)"]
		puts $ch "$x $y $xe $ye"
	    }
	}

	close $ch
    }
    APRaise $varname
}

proc APClearData {varname} {
    upvar #0 $varname var
    global $varname

    global ds9

    if {$var(data,total) == 0} {
	return
    }

    # this is hard coded to ignore an external first dataset
    if {$var(1,manage)} {
	set clear 1
	set nn 1
    } else {
	set clear 0
	set nn 2
    }

    for {} {$nn<=$var(data,total)} {incr nn} {
	# delete elements
	foreach el [$var(graph) element names] {
	    set f [split $el -]
	    if {[lindex $f 1] == $nn} {
		$var(graph) element delete $el
	    }
	}

	blt::vector destroy $var($nn,xdata) $var($nn,ydata) \
	    $var($nn,xedata) $var($nn,yedata)

	foreach x [array names $varname] {
	    set f [split $x ,]
	    if {([lindex $f 0] == $nn)} {
		unset ${varname}($x)
	    }
	}
    }

    if {$clear} {
	set var(data,total) 0
	set var(data,current) 0

	# reset other variables
	set var(xdata) {}
	set var(ydata) {}
	set var(xedata) {}
	set var(yedata) {}

	set var(graph,title) {}
	set var(graph,xaxis) {}
	set var(graph,yaxis) {}

	set var(graph,x,auto) 1
	set var(graph,x,min) {}
	set var(graph,x,max) {}
	set var(graph,y,auto) 1
	set var(graph,y,min) {}
	set var(graph,y,max) {}
	
	$var(mb).dataset delete $ds9(menu,start) end
    } else {
	if {$var(data,total) > 1} {
	    $var(mb).dataset delete [expr $ds9(menu,start)+1] end
	}
	set var(data,total) 1
	set var(data,current) 1

	APCurrentData $varname
    }

    APUpdateGraph $varname
    APStats $varname
}

proc APLoadConfig {varname} {
    upvar #0 $varname var
    global $varname

    APPLoadConfig $varname [OpenFileDialog apconfigfbox]
}

proc APPLoadConfig {varname filename} {
    upvar #0 $varname var
    global $varname

    if {$filename != {}} {
	source $filename

	array set $varname [array get analysisplot]
	APUpdateGraph $varname
    }
}

proc APSaveConfig {varname} {
    upvar #0 $varname var
    global $varname

    APPSaveConfig $varname [SaveFileDialog apconfigfbox]
}

proc APPSaveConfig {varname filename} {
    upvar #0 $varname var
    global $varname

    if {$filename != {}} {
	set ch [open $filename w]

	set analysisplot(grid) $var(grid)
	set analysisplot(grid,log) $var(grid,log) 

	set analysisplot(discrete) $var(discrete) 
	set analysisplot(discrete,symbol) $var(discrete,symbol) 
	set analysisplot(discrete,color) $var(discrete,color) 

	set analysisplot(linear) $var(linear) 
	set analysisplot(linear,width) $var(linear,width) 
	set analysisplot(linear,color) $var(linear,color) 
	set analysisplot(linear,dash) $var(linear,dash) 

	set analysisplot(step) $var(step) 
	set analysisplot(step,width) $var(step,width) 
	set analysisplot(step,color) $var(step,color) 
	set analysisplot(step,dash) $var(step,dash) 

	set analysisplot(quadratic) $var(quadratic) 
	set analysisplot(quadratic,width) $var(quadratic,width) 
	set analysisplot(quadratic,color) $var(quadratic,color) 
	set analysisplot(quadratic,dash) $var(quadratic,dash) 

	set analysisplot(error) $var(error) 
	set analysisplot(error,width) $var(error,width) 
	set analysisplot(error,color) $var(error,color) 
	set analysisplot(error,style) $var(error,style) 

	set analysisplot(titleFont) $var(titleFont) 
	set analysisplot(titleSize) $var(titleSize) 
	set analysisplot(titleWeight) $var(titleWeight) 
	set analysisplot(titleSlant) $var(titleSlant) 

	set analysisplot(textlabFont) $var(textlabFont) 
	set analysisplot(textlabSize) $var(textlabSize) 
	set analysisplot(textlabWeight) $var(textlabWeight) 
	set analysisplot(textlabSlant) $var(textlabSlant) 

	set analysisplot(numlabFont) $var(numlabFont) 
	set analysisplot(numlabSize) $var(numlabSize) 
	set analysisplot(numlabWeight) $var(numlabWeight)
	set analysisplot(numlabSlant) $var(numlabSlant)

	set analysisplot(format) 1
	set analysisplot(format,x) $var(format,x)
	set analysisplot(format,y) $var(format,y)

	puts $ch "array set analysisplot \{ [array get analysisplot] \}"
	close $ch
    }
}

proc APAxes {varname} {
    upvar #0 $varname var
    global $varname

    global ed

    set w {.aptitle}

    set ed(ok) 0
    set ed(xauto) $var(graph,x,auto)
    set ed(xmin) $var(graph,x,min)
    set ed(xmax) $var(graph,x,max)
    set ed(yauto) $var(graph,y,auto)
    set ed(ymin) $var(graph,y,min)
    set ed(ymax) $var(graph,y,max)
    set ed(format,x) $var(format,x)
    set ed(format,y) $var(format,y)

    DialogCreate $w [msgcat::mc {Axes}] ed(ok)

    # Param
    set f [ttk::frame $w.param]
    ttk::label $f.taxis -text [msgcat::mc {Axis}]
    ttk::label $f.tto -text [msgcat::mc {To}]
    ttk::label $f.tfrom -text [msgcat::mc {From}]
    ttk::label $f.tformat -text [msgcat::mc {Format}]
    ttk::label $f.tauto -text [msgcat::mc {Automatic}]

    ttk::label $f.tx -text [msgcat::mc {X}]
    ttk::entry $f.xmin -textvariable ed(xmin) -width 12
    ttk::entry $f.xmax -textvariable ed(xmax) -width 12
    ttk::entry $f.xformat -textvariable ed(format,x) -width 8
    ttk::checkbutton $f.xauto -variable ed(xauto)

    ttk::label $f.ty -text [msgcat::mc {Y}]
    ttk::entry $f.ymin -textvariable ed(ymin) -width 12
    ttk::entry $f.ymax -textvariable ed(ymax) -width 12
    ttk::entry $f.yformat -textvariable ed(format,y) -width 8
    ttk::checkbutton $f.yauto -variable ed(yauto)

    grid $f.taxis $f.tto $f.tfrom $f.tformat $f.tauto -padx 2 -pady 2 -sticky w
    grid $f.tx $f.xmin $f.xmax $f.xformat $f.xauto -padx 2 -pady 2 -sticky w
    grid $f.ty $f.ymin $f.ymax $f.yformat $f.yauto -padx 2 -pady 2 -sticky w

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed(ok) $w.param.xmin
    DialogDismiss $w

    if {$ed(ok)} {
	set var(graph,x,auto) $ed(xauto)
	set var(graph,x,min) $ed(xmin) 
	set var(graph,x,max) $ed(xmax) 
	set var(graph,y,auto) $ed(yauto) 
	set var(graph,y,min) $ed(ymin) 
	set var(graph,y,max) $ed(ymax) 
	set var(format,x) $ed(format,x)
	set var(format,y) $ed(format,y)

	APUpdateGraph $varname
    }
    
    set rr $ed(ok)
    unset ed
    return $rr
}

proc APTitle {varname} {
    upvar #0 $varname var
    global $varname

    global ed

    set w {.aptitle}

    set ed(ok) 0
    set ed(title) $var(graph,title)
    set ed(xaxis) $var(graph,xaxis)
    set ed(yaxis) $var(graph,yaxis)

    DialogCreate $w [msgcat::mc {Labels}] ed(ok)

    # Param
    set f [ttk::frame $w.param]
    ttk::label $f.label -text [msgcat::mc {Plot Title}]
    ttk::entry $f.title -textvariable ed(title) -width 23
    ttk::label $f.xlabel -text [msgcat::mc {X Axis Label}]
    ttk::entry $f.xtitle -textvariable ed(xaxis) -width 23
    ttk::label $f.ylabel -text [msgcat::mc {Y Axis Label}]
    ttk::entry $f.ytitle -textvariable ed(yaxis) -width 23

    grid $f.label $f.title -padx 2 -pady 2 -sticky ew
    grid $f.xlabel $f.xtitle -padx 2 -pady 2 -sticky ew
    grid $f.ylabel $f.ytitle -padx 2 -pady 2 -sticky ew
    grid columnconfigure $f 1 -weight 1

    # Buttons
    set f [ttk::frame $w.buttons]
    ttk::button $f.ok -text [msgcat::mc {OK}] -command {set ed(ok) 1} \
	-default active 
    ttk::button $f.cancel -text [msgcat::mc {Cancel}] -command {set ed(ok) 0}
    pack $f.ok $f.cancel -side left -expand true -padx 2 -pady 4

    bind $w <Return> {set ed(ok) 1}

    # Fini
    ttk::separator $w.sep -orient horizontal
    pack $w.buttons $w.sep -side bottom -fill x
    pack $w.param -side top -fill both -expand true

    DialogCenter $w 
    DialogWait $w ed(ok) $w.param.title
    DialogDismiss $w

    if {$ed(ok)} {
	set var(graph,title) $ed(title)
	set var(graph,xaxis) $ed(xaxis)
	set var(graph,yaxis) $ed(yaxis)
	APUpdateGraph $varname
    }
    
    set rr $ed(ok)
    unset ed
    return $rr
}

proc APCreateLineMenu {which var1 var2 cmd} {
    menu $which
    $which add radiobutton -label 1 -variable $var1 -value 1 -command $cmd
    $which add radiobutton -label 2 -variable $var1 -value 2 -command $cmd
    $which add radiobutton -label 3 -variable $var1 -value 3 -command $cmd
    $which add radiobutton -label 4 -variable $var1 -value 4 -command $cmd
    $which add separator
    $which add checkbutton -label [msgcat::mc {Dash}] -variable $var2 \
	-onvalue {yes} -offvalue {no} -command $cmd
}

proc APLineSymbolMenu {which var} {
    menu $which
    $which add radiobutton -label [msgcat::mc {Circle}] \
	-variable $var -value circle
    $which add radiobutton -label [msgcat::mc {Diamond}]  \
	-variable $var -value diamond
    $which add radiobutton -label [msgcat::mc {Plus}]  \
	-variable $var -value plus
    $which add radiobutton -label [msgcat::mc {Cross}]  \
	-variable $var -value cross
}

proc APLineWidthMenu {which var} {
    menu $which
    $which add radiobutton -label 0 -variable $var -value 0
    $which add radiobutton -label 1 -variable $var -value 1
    $which add radiobutton -label 2 -variable $var -value 2
    $which add radiobutton -label 3 -variable $var -value 3
}

proc APLineDashMenu {which var} {
    menu $which
    $which add radiobutton -label [msgcat::mc {No}] -variable $var \
	-value no
    $which add radiobutton -label [msgcat::mc {Yes}]  -variable $var \
	-value yes
}

# Print

proc APPSPrint {varname} {
    upvar #0 $varname var
    global $varname

    if {[APPrintDialog ap]} {
	if [catch {APPostScript $varname} printError] {
	    Error "[msgcat::mc {An error has occurred while printing}] $printError"
	}
    }
}

proc APPostScript {varname} {
    upvar #0 $varname var
    global $varname

    global ps
    global ds9

    global debug
    if {$debug(tcl,idletasks)} {
	puts stderr "APPostScript"
    }
    update idletasks

    # set postscript fonts
    $var(graph) configure \
	-font "$var(titleFont) $var(titleSize) $var(titleWeight) $var(titleSlant)"
    $var(graph) xaxis configure \
	-tickfont "$var(numlabFont) $var(numlabSize) $var(numlabWeight) $var(numlabSlant)" \
	-titlefont "$var(textlabFont) $var(textlabSize) $var(textlabWeight) $var(textlabSlant)"
    $var(graph) yaxis configure \
	-tickfont "$var(numlabFont) $var(numlabSize) $var(numlabWeight) $var(numlabSlant)" \
	-titlefont "$var(textlabFont) $var(textlabSize) $var(textlabWeight) $var(textlabSlant)"

    set options "-decorations false"

    # Color
    switch -- $ps(color) {
	rgb -
	cmyk {append options " -greyscale no"}
	gray {append options " -greyscale yes"}
    }

    # Size
    set ww [expr [winfo width $var(top)]*$ps(scale)/100./[tk scaling]]
    set hh [expr [winfo height $var(top)]*$ps(scale)/100./[tk scaling]]
    append options " -width $ww -height $hh"

    # Page size
    switch -- $ps(size) {
 	letter {append options " -paperwidth 8.5i -paperheight 11.i"}
 	legal {append options " -paperwidth 8.5i -paperheight 14.i"}
 	tabloid {append options " -paperwidth 11i -paperheight 17.i"}
 	poster {append options " -paperwidth 36.i -paperheight 48.i"}
 	a4 {append options " -paperwidth 195m -paperheight 282m"}
 	other {
 	    if {$ps(width) != {} && $ps(height) != {}} {
 		set w [expr $ps(width)]
 		set h [expr $ps(height)]
 		append options \
 		    " -paperwidth [append $w i] -paperheight [append $h i]"
 	    }
 	}
 	othermm {
 	    if {$ps(width) != {} && $ps(height) != {}} {
 		set w [expr $ps(width)]
 		set h [expr $ps(height)]
 		append options \
 		    " -paperwidth [append $w m] -paperheight [append $h m]"
 	    }
 	}
    }

    # Orientation
    switch -- $ps(orient) {
	portrait {append options " -landscape false"}
	landscape {append options " -landscape true"}
    }

    if {$ps(dest) == "file" && $ps(filename) != {}} {
	eval $var(graph) postscript output $ps(filename) $options
    } else {
	set ch [open "| $ps(cmd)" w]
	puts $ch [eval $var(graph) postscript output $options]
	close $ch
    }

    # reset fonts
    $var(graph) configure \
	-font "{$ds9($var(titleFont))} $var(titleSize) $var(titleWeight) $var(titleSlant)"
    $var(graph) xaxis configure \
	-tickfont "{$ds9($var(numlabFont))} $var(numlabSize) $var(numlabWeight) $var(numlabSlant)" \
	-titlefont "{$ds9($var(textlabFont))} $var(textlabSize) $var(textlabWeight) $var(textlabSlant)"
    $var(graph) yaxis configure \
	-tickfont "{$ds9($var(numlabFont))} $var(numlabSize) $var(numlabWeight) $var(numlabSlant)" \
	-titlefont "{$ds9($var(textlabFont))} $var(textlabSize) $var(textlabWeight) $var(textlabSlant)"

}

# Windows/Aqua
proc APNativePrint {varname} {
    upvar #0 $varname var
    global $varname

    global ps
    global ds9

    global debug
    if {$debug(tcl,idletasks)} {
	puts stderr "APNativePrint"
    }
    update idletasks

    # set background
    $var(graph) configure -plotbackground white -bg white -plotrelief flat
    $var(graph) print2
    # restore background
    $var(graph) configure -plotbackground $ds9(app,bg) -bg $ds9(app,bg) \
	-plotrelief groove
}

# Process Cmds

proc ProcessPlotCmd {xarname iname buf fn} {
    upvar $xarname xar
    upvar $iname i

    global iap
    set varname $iap(tt)
    set id 0

    # check for next command line option
    if {[string range [lindex $xar $i] 0 0] != {-}} {

	# determine which plot
	switch -- [string tolower [lindex $xar $i]] {
	    {} -
	    new {}

	    data -
	    load -
	    save -
	    clear -
	    loadconfig -
	    saveconfig -
	    print -
	    page -
	    pagesetup -
	    close -

	    graph -
	    font -
	    dataset -
	    view -
	    color -
	    line {
		set varname [lindex $iap(windows) end]
		set id [lsearch $iap(windows) $varname]
	    }

	    default {
		set varname [lindex $xar $i]
		set id [lsearch $iap(windows) $varname]
		incr i
	    }
	}
    }

    # we better have a tt by now
    if {$id == -1} {
	Error "[msgcat::mc {Unable to find plot window}] $varname"
	return
    }

    # check for next command line option
    if {[string range [lindex $xar $i] 0 0] != {-}} {

	# now, process plot command
	switch -- [string tolower [lindex $xar $i]] {
	    {} {
		if {$buf != {}} {
		    APProcessNew $varname $xarname $iname $buf
		} elseif {$fn != {}} {
		    if {[file exists $fn]} {
			set ch [open $fn r]
			set txt [read $ch]
			close $ch
			APProcessNew $varname $xarname $iname $txt
		    }
		} else {
		    APProcessNew $varname $xarname $iname {}
		}
	    }
	    new {
		incr i
		switch -- [lindex $xar $i] {
		    name {
			set varname [lindex $xar [expr $i+1]]
			incr i 2
		    }
		}
		if {$buf != {}} {
		    APProcessNew $varname $xarname $iname $buf
		} elseif {$fn != {}} {
		    if {[file exists $fn]} {
			set ch [open $fn r]
			set txt [read $ch]
			close $ch
			APProcessNew $varname $xarname $iname $txt
		    }
		} else {
		    APProcessNew $varname $xarname $iname {}
		}
	    }
	    data {
		incr i
		if {$buf != {}} {
		    APProcessData $varname $xarname $iname $buf
		} elseif {$fn != {}} {
		    if {[file exists $fn]} {
			set ch [open $fn r]
			set txt [read $ch]
			close $ch
			APProcessData $varname $xarname $iname $txt
		    }
		}
	    }
	    load {
		set ff [lindex $xar [expr $i+1]]
		set dim [lindex $xar [expr $i+2]]
		incr i 2
		APPLoadData $varname $ff $dim
		FileLast apdatafbox $ff
	    }
	    save {
		incr i
		set ff [lindex $xar $i]
		APPSaveData $varname $ff
		FileLast apdatafbox $ff
	    }
	    clear {APClearData $varname}
	    loadconfig {
		incr i
		set ff [lindex $xar $i]
		APPLoadConfig $varname $ff
		FileLast apconfigfbox $ff
	    }
	    saveconfig {
		incr i
		set ff [lindex $xar $i]
		APPSaveConfig $varname $ff
		FileLast apconfigfbox $ff
	    }
	    print {
		incr i
		APProcessPrint $varname $xarname $iname
	    }
	    page -
	    pagesetup {
		incr i
		APProcessPageSetup $varname $xarname $iname
	    }
	    close {APDestroy $varname}

	    graph {
		incr i
		APProcessGraph $varname $xarname $iname
	    }
	    font {
		incr i
		APProcessFont $varname $xarname $iname
	    }
	    dataset {
		incr i
		APProcessDataset $varname $xarname $iname
	    }
	    view {
		incr i
		APProcessView $varname $xarname $iname
	    }
	    color {
		incr i
		APProcessColor $varname $xarname $iname
	    }
	    line {
		incr i
		APProcessLine $varname $xarname $iname
	    }
	}
    } else {
	APProcessNew $varname $xarname $iname {}
    }
}

proc ProcessSendPlotCmd {proc id param} {
    global iap
    $proc $id "$iap(windows)\n"
}

proc APProcessNew {varname xarname iname buf} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    # check for next command line option
    if {[string range [lindex $xar $i] 0 0] != {-}} {

	switch -- [string tolower [lindex $xar $i]] {
	    stdin {AnalysisPlotStdin $varname {} $buf}
	    {} {AnalysisPlot $varname {} {} {} {} xy $buf; incr i -1}
	    default {
		AnalysisPlot $varname {} \
		    [lindex $xar $i] \
		    [lindex $xar [expr $i+1]] \
		    [lindex $xar [expr $i+2]] \
		    [lindex $xar [expr $i+3]] \
		    $buf
		incr i 3
	    }
	}
    } else {
	AnalysisPlot $varname {} {} {} {} xy $buf
	incr i -1
    }
}

proc APProcessData {varname xarname iname buf} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    APRaise $varname
    APDataSet $varname [lindex $xar $i] $buf
}

proc APProcessPrint {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    global ps
    global current

    switch -- [string tolower [lindex $xar $i]] {
	destination {incr i; set ps(dest) [lindex $xar $i]}
	command {incr i; set ps(cmd) [lindex $xar $i]}
	filename {incr i; set ps(filename) [lindex $xar $i] }
	palette -
	color {incr i; set ps(color) [lindex $xar $i] }

	{} {APPostScript $varname}
	default {incr i -1; APPostScript $varname}
    }
}

proc APProcessPageSetup {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    global ps

    switch -- [string tolower [lindex $xar $i]] {
	orientation -
	orient {incr i; set ps(orient) [lindex $xar $i]}
	pagesize -
	size {incr i; set ps(size) [lindex $xar $i] }
    }
}

proc APProcessGraph  {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    switch -- [string tolower [lindex $xar $i]] {
	grid {incr i; set var(grid) [FromYesNo [lindex $xar $i]]}
	scale {incr i; set var(grid,log) [lindex $xar $i]}
	format {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		x {incr i; set var(format,x) [lindex $xar $i]}
		y {incr i; set var(format,y) [lindex $xar $i]}
	    }
	}
	range {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		x {
		    incr i
		    switch -- [string tolower [lindex $xar $i]] {
			auto {incr i; set var(graph,x,auto) \
				  [FromYesNo [lindex $xar $i]]}
			min {incr i; set var(graph,x,min) [lindex $xar $i]}
			max {incr i; set var(graph,x,max) [lindex $xar $i]}
		    }
		}
		y {
		    incr i
		    switch -- [string tolower [lindex $xar $i]] {
			auto {incr i; set var(graph,y,auto) \
				  [FromYesNo [lindex $xar $i]]}
			min {incr i; set var(graph,y,min) [lindex $xar $i]}
			max {incr i; set var(graph,y,max) [lindex $xar $i]}
		    }
		}
	    }
	}
	labels {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		title {incr i; set var(graph,title) [lindex $xar $i]}
		xaxis {incr i; set var(graph,xaxis) [lindex $xar $i]}
		yaxis {incr i; set var(graph,yaxis) [lindex $xar $i]}
	    }
	}
    }
    APUpdateGraph $varname
}

proc APProcessFont  {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    switch -- [string tolower [lindex $xar $i]] {
	numbers {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		font {incr i; set var(numlabFont) [lindex $xar $i]}
		size {incr i; set var(numlabSize) [lindex $xar $i]}
		weight {incr i; set var(numlabWeight) [lindex $xar $i]}
		slant {incr i; set var(numlabSlant) [lindex $xar $i]}
		style {
		    incr i
		    switch [string tolower [lindex $xar $i]] {
			normal {
			    set var(numlabWeight) normal
			    set var(numlabSlant) roman
			}
			bold {
			    set var(numlabWeight) bold
			    set var(numlabSlant) roman
			}
			italic {
			    set var(numlabWeight) normal
			    set var(numlabSlant) italic
			}
		    }
		}
	    }
	}
	labels {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		font {incr i; set var(textlabFont) [lindex $xar $i]}
		size {incr i; set var(textlabSize) [lindex $xar $i]}
		weight {incr i; set var(textlabWeight) [lindex $xar $i]}
		slant {incr i; set var(textlabSlant) [lindex $xar $i]}
		style {
		    incr i
		    switch [string tolower [lindex $xar $i]] {
			normal {
			    set var(textlabWeight) normal
			    set var(textlabSlant) roman
			}
			bold {
			    set var(textlabWeight) bold
			    set var(textlabSlant) roman
			}
			italic {
			    set var(textlabWeight) normal
			    set var(textlabSlant) italic
			}
		    }
		}
	    }
	}
	title {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		font {incr i; set var(titleFont) [lindex $xar $i]}
		size {incr i; set var(titleSize) [lindex $xar $i]}
		weight {incr i; set var(titleWeight) [lindex $xar $i]}
		slant {incr i; set var(titleSlant) [lindex $xar $i]}
		style {
		    incr i
		    switch [string tolower [lindex $xar $i]] {
			normal {
			    set var(titleWeight) normal
			    set var(titleSlant) roman
			}
			bold {
			    set var(titleWeight) bold
			    set var(titleSlant) roman
			}
			italic {
			    set var(titleWeight) normal
			    set var(titleSlant) italic
			}
		    }
		}
	    }
	}
    }
    APUpdateGraph $varname
}

proc APProcessDataset  {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    set  var(data,current) [lindex $xar $i]
    APCurrentData $varname
}

proc APProcessView  {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    switch -- [string tolower [lindex $xar $i]] {
	discrete {incr i; set var(discrete) [FromYesNo [lindex $xar $i]]}
	line -
	linear {incr i; set var(linear) [FromYesNo [lindex $xar $i]]}
	step {incr i; set var(step) [FromYesNo [lindex $xar $i]]}
	quadratic {incr i; set var(quadratic) [FromYesNo [lindex $xar $i]]}
	errorbar {incr i; set var(error) [FromYesNo [lindex $xar $i]]}
    }

    APUpdateState $varname
}

proc APProcessColor  {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    switch -- [string tolower [lindex $xar $i]] {
	discrete {incr i; set var(discrete,color) [lindex $xar $i]}
	line -
	linear {incr i; set var(linear,color) [lindex $xar $i]}
	step {incr i; set var(step,color) [lindex $xar $i]}
	quadratic {incr i; set var(quadratic,color) [lindex $xar $i]}
	errorbar {incr i; set var(error,color) [lindex $xar $i]}
    }

    APUpdateState $varname
}

proc APProcessLine  {varname xarname iname} {
    upvar #0 $varname var
    global $varname

    upvar 2 $xarname xar
    upvar 2 $iname i

    switch -- [string tolower [lindex $xar $i]] {
	discrete {incr i; set var(discrete,symbol) [lindex $xar $i]}
	line -
	linear {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		width {incr i; set var(linear,width) [lindex $xar $i]}
		dash {
		    incr i
		    set var(linear,dash) [lindex $xar $i]}
	    }
	}
	step {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		width {incr i; set var(step,width) [lindex $xar $i]}
		dash {
		    incr i
		    set var(step,dash) [lindex $xar $i]}
	    }
	}
	quadratic {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		width {incr i; set var(quadratic,width) [lindex $xar $i]}
		dash {
		    incr i
		    set var(quadratic,dash) [lindex $xar $i]}
	    }
	}
	error -
	errorbar {
	    incr i
	    switch -- [string tolower [lindex $xar $i]] {
		width {incr i; set var(error,width) [lindex $xar $i]}
		style {
		    incr i;
		    set var(error,style) [lindex $xar $i]}
	    }
	}
    }

    APUpdateState $varname
}

proc APStats {varname} {
    upvar #0 $varname var
    global $varname

    if {!$var(stats)} {
	return
    }

    if {$var(ydata) != {}} {
	set min [format "%6.3f" [blt::vector expr min($var(ydata))]]
	set max [format "%6.3f" [blt::vector expr max($var(ydata))]]
	set mean [format "%6.3f" [blt::vector expr mean($var(ydata))]]
	set median [format "%6.3f" [blt::vector expr median($var(ydata))]]
	set varr [format "%6.3f" [expr [blt::vector expr var($var(ydata))]]]
	set sdev [format "%6.3f" [expr [blt::vector expr sdev($var(ydata))]]]
    } else {
	set min {}
	set max {}
	set mean {}
	set median {}
	set varr {}
	set sdev {}
    }
    
    set rr {}
    append rr "min $min\n"
    append rr "max $max\n"
    append rr "mean $mean\n"
    append rr "median $median\n"
    append rr "var $varr\n"
    append rr "sdev $sdev\n"

    SimpleTextDialog "${varname}txt" [msgcat::mc {Statistics}] 40 20 insert top $rr 0 APStatsDestroyCB $varname
}

proc APStatsDestroyCB {varname} {
    upvar #0 $varname var
    global $varname

    set var(stats) 0
}
