/* ---------------------------------------------------------------------
*
*  -- ScaLAPACK routine (version 1.0) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory,
*     and University of California, Berkeley.
*     November 17, 1996
*
*  ---------------------------------------------------------------------
*/
/*
* Include files
*/
#include "pblas.h"

void pchemv_( uplo, n, alpha, A, ia, ja, desc_A, X, ix, jx, desc_X, incx,
              beta, Y, iy, jy, desc_Y, incy )
/*
*  .. Scalar Arguments ..
*/
   F_CHAR      uplo;
   int         * ia, * incx, * incy, * ix, * iy, * ja, * jx, * jy, * n;
   complex     * alpha, * beta;
/* ..
*  .. Array Arguments ..
*/
   int         desc_A[], desc_X[], desc_Y[];
   complex     A[], X[], Y[];
{
/*
*  Purpose
*  =======
*
*  PCHEMV performs the distributed matrix-vector operation
*     sub( Y ) := alpha*sub( A ) * sub( X ) + beta*sub( Y ),
*
*  where sub( A ) denotes A(IA:IA+M-1,JA:JA+N-1),
*
*        sub( X ) denotes X(IX,JX:JX+N-1) if INCX = M_X,
*                         X(IX:IX+N-1,JX) if INCX = 1 and INCX <> M_X,
*
*        sub( Y ) denotes Y(IY,JY:JY+N-1) if INCY = M_Y,
*                         Y(IY:IY+N-1,JY) if INCY = 1 and INCY <> M_Y.
*
*  alpha and beta are scalars, sub( X ) and sub( Y ) are N element
*  distributed vectors and sub( A ) is an N-by-N hermitian distributed
*  matrix.
*
*  Notes
*  =====
*
*  Each global data object is described by an associated description
*  vector.  This vector stores the information required to establish
*  the mapping between an object element and its corresponding process
*  and memory location.
*
*  Let A be a generic term for any 2D block cyclicly distributed array.
*  Such a global array has an associated description vector descA.
*  In the following comments, the character _ should be read as
*  "of the global array".
*
*  NOTATION        STORED IN      EXPLANATION
*  --------------- -------------- --------------------------------------
*  DT_A   (global) descA[ DT_ ]   The descriptor type.  In this case,
*                                 DT_A = 1.
*  CTXT_A (global) descA[ CTXT_ ] The BLACS context handle, indicating
*                                 the BLACS process grid A is distribu-
*                                 ted over. The context itself is glo-
*                                 bal, but the handle (the integer
*                                 value) may vary.
*  M_A    (global) descA[ M_ ]    The number of rows in the global
*                                 array A.
*  N_A    (global) descA[ N_ ]    The number of columns in the global
*                                 array A.
*  MB_A   (global) descA[ MB_ ]   The blocking factor used to distribu-
*                                 te the rows of the array.
*  NB_A   (global) descA[ NB_ ]   The blocking factor used to distribu-
*                                 te the columns of the array.
*  RSRC_A (global) descA[ RSRC_ ] The process row over which the first
*                                 row of the array A is distributed.
*  CSRC_A (global) descA[ CSRC_ ] The process column over which the
*                                 first column of the array A is
*                                 distributed.
*  LLD_A  (local)  descA[ LLD_ ]  The leading dimension of the local
*                                 array.  LLD_A >= MAX(1,LOCr(M_A)).
*
*  Let K be the number of rows or columns of a distributed matrix,
*  and assume that its process grid has dimension p x q.
*  LOCr( K ) denotes the number of elements of K that a process
*  would receive if K were distributed over the p processes of its
*  process column.
*  Similarly, LOCc( K ) denotes the number of elements of K that a
*  process would receive if K were distributed over the q processes of
*  its process row.
*  The values of LOCr() and LOCc() may be determined via a call to the
*  ScaLAPACK tool function, NUMROC:
*          LOCr( M ) = NUMROC( M, MB_A, MYROW, RSRC_A, NPROW ),
*          LOCc( N ) = NUMROC( N, NB_A, MYCOL, CSRC_A, NPCOL ).
*  An upper bound for these quantities may be computed by:
*          LOCr( M ) <= ceil( ceil(M/MB_A)/NPROW )*MB_A
*          LOCc( N ) <= ceil( ceil(N/NB_A)/NPCOL )*NB_A
*
*  Because vectors may be seen as particular matrices, a distributed
*  vector is considered to be a distributed matrix.
*
*  The distributed vectors X and Y must be distributed along the same
*  axis, i.e if INCX = M_X then INCY must be equal to M_Y, similarly if
*  INCX = 1 and INCX <> M_X, then INCY must be equal to 1. Moreover,
*  MB_A = NB_A is required as well as MOD(IA-1,MB_A) = MOD(JA-1,NB_A).
*
*  If INCY = M_Y, the process column having the first entries of
*  sub( Y ) must also contain the first block of sub( A ), similarly,
*  if INCX = 1 and INCX <> M_X, the process row having the first
*  entries of sub( X ) must also contain the first block of sub( A ).
*
*  If INCX = M_X (resp. (INCX = 1 and INCX <> 1) ), the column (resp.
*  row) offset of sub( X ) must be equal to the row offset of sub( A ),
*  i.e MOD(JX-1,NB_X) (resp. MOD(IX-1,MB_X)) = MOD(JA-1,NB_A). Moreover,
*  the column (resp. row) blocksize of X must be equal to the column
*  blocksize of A, i.e NB_X = NB_A (resp. MB_X = NB_A).
*
*  If INCY = M_Y (resp. (INCY = 1 and INCY <> M_Y)), the column (resp.
*  row) offset of sub( Y ) must be equal to the row offset of sub( A ),
*  i.e MOD(JY-1,NB_Y) (resp. MOD(IY-1,MB_Y)) = MOD(IA-1,MB_A). Moreover,
*  the column (resp. row) blocksize of Y must be equal to the row
*  blocksize of A, i.e NB_Y = MB_A (resp. MB_Y = MB_A).
*
*  Parameters
*  ==========
*
*  UPLO    (global input) pointer to CHARACTER
*          On entry, UPLO specifies whether the upper or lower
*          triangular part of the distributed matrix sub( A ) is to be
*          referenced as follows:
*
*             UPLO = 'U' or 'u'   Only the upper triangular part of
*                                 sub( A ) is to be referenced.
*
*             UPLO = 'L' or 'l'   Only the lower triangular part of
*                                 sub( A ) is to be referenced.
*
*  N       (global input) pointer to INTEGER
*          The order of the distributed matrix sub( A ). N >= 0.
*
*  ALPHA   (global input) pointer to COMPLEX
*          On entry, ALPHA specifies the scalar alpha.
*
*  A       (local input) COMPLEX pointer into the local memory
*          to an array of dimension (LLD_A,LOCc(JA+N-1) containing the
*          local pieces of the distributed matrix sub( A ). Before
*          entry with  UPLO = 'U' or 'u', the leading N-by-N upper
*          triangular part of the distributed matrix sub( A ) must
*          and the strictly lower triangular part of sub( A ) is not
*          referenced.
*          Before entry with UPLO = 'L' or 'l', the leading N-by-N
*          lower triangular part of the distributed matrix sub( A )
*          must contain the lower triangular part of the hermitian
*          matrix and the strictly upper triangular part of sub( A )
*          is not referenced.
*
*  IA      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix A to operate on.
*
*  JA      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix A to operate on.
*
*  DESCA   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix A.
*
*  X       (local input) COMPLEX array containing the local
*          pieces of a distributed matrix of dimension of at least
*                  ( (JX-1)*M_X + IX + ( N - 1 )*abs( INCX ) )
*          This array contains the entries of the distributed vector
*          sub( X ).
*
*  IX      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix X to operate on.
*
*  JX      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix X to operate on.
*
*  DESCX   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix X.
*
*  INCX    (global input) pointer to INTEGER
*          The global increment for the elements of X. Only two values
*          of INCX are supported in this version, namely 1 and M_X.
*
*  BETA    (global input) pointer to COMPLEX
*          On entry,  BETA  specifies the scalar  beta.  When  BETA  is
*          supplied as zero then sub( Y ) need not be set on input.
*
*  Y       (local input/local output) COMPLEX array
*          containing the local pieces of a distributed matrix of
*          dimension of at least
*                ( (JY-1)*M_Y + IY + ( N - 1 )*abs( INCY ) )
*          This array contains the entries of the distributed vector
*          sub( Y ).  On exit, sub( Y ) is overwritten by the updated
*          distributed vector sub( Y ).
*
*  IY      (global input) pointer to INTEGER
*          The global row index of the submatrix of the distributed
*          matrix Y to operate on.
*
*  JY      (global input) pointer to INTEGER
*          The global column index of the submatrix of the distributed
*          matrix Y to operate on.
*
*  DESCY   (global and local input) INTEGER array of dimension 8.
*          The array descriptor of the distributed matrix Y.
*
*  INCY    (global input) pointer to INTEGER
*          The global increment for the elements of Y. Only two values
*          of INCY are supported in this version, namely 1 and M_Y.
*
*  =====================================================================
*
*  .. Local Scalars ..
*/
   char        UploA, xydist;
   int         iacol, iarow, ictxt, icoffa, iia, iix, iiy, info, iroffa,
               iszcmp, ixcol, ixpos, ixrow, iycol, iypos, iyrow, jja,
               jjx, jjy, lcm, lcmp, lcmq, locincx, locincy, mullen,
               mycol, myrow, nca, nn, np0, nprow, npcol, nra, nq0,
               szcmp, tmp1, tmp2, wksz;
/* ..
*  .. PBLAS Buffer ..
*/
   complex     * buff;
/* ..
*  .. External Functions ..
*/
   void        blacs_gridinfo_();
   void        pbchkmat();
   void        pbchkvect();
   void        pberror_();
   char        * getpbbuf();
   F_VOID_FCT  pbchemv_();
   F_INTG_FCT  ilcm_();
/* ..
*  .. Executable Statements ..
*
*  Get grid parameters
*/
   ictxt = desc_A[CTXT_];
   blacs_gridinfo_( &ictxt, &nprow, &npcol, &myrow, &mycol );
/*
*  Test the input parameters
*/
   info = 0;
   if( nprow == -1 )
      info = -(700+CTXT_+1);
   else
   {
      UploA = Mupcase( F2C_CHAR( uplo )[0] );
      iroffa = (*ia-1) % desc_A[MB_];
      icoffa = (*ja-1) % desc_A[NB_];
      pbchkmat( *n, 2, *n, 2, *ia, *ja, desc_A, 7, &iia, &jja,
                &iarow, &iacol, nprow, npcol, myrow, mycol,
                &nra, &nca, &info );
      pbchkvect( *n, 2, *ix, *jx, desc_X, *incx, 11, &iix, &jjx,
                 &ixrow, &ixcol, nprow, npcol, myrow, mycol, &info );
      pbchkvect( *n, 2, *iy, *jy, desc_Y, *incy, 17, &iiy, &jjy,
                 &iyrow, &iycol, nprow, npcol, myrow, mycol, &info );
      if( info == 0 )
      {
         if( ( UploA != 'U' ) && ( UploA != 'L' ) )
            info = -1;
         else if( iroffa != icoffa )
            info = -6;
         if( ( *incx == desc_X[M_] ) && ( *incy == desc_Y[M_] ) )
         {
            if( desc_A[NB_] != desc_A[MB_] )
               info = -(700+NB_+1);
            else if( ( ( (*jx-1) % desc_X[NB_]) != icoffa ) ||
                     ( ixcol != iacol ) )
               info = -10;
            else if( desc_A[NB_] != desc_X[NB_] )
               info = -(1100+NB_+1);
            else if( ( ( (*jy-1) % desc_Y[NB_]) != iroffa ) ||
                     ( iycol != iacol ) )
               info = -16;
            else if( desc_A[NB_] != desc_Y[NB_] )
               info = -(1700+NB_+1);
         }
         else if( ( *incx == 1 ) && ( *incx != desc_X[M_] ) &&
                  ( *incy == 1 ) && ( *incy != desc_Y[M_] ) )
         {
            if( desc_A[NB_] != desc_A[MB_] )
               info = -(700+NB_+1);
            else if( ( ( (*ix-1) % desc_X[MB_]) != icoffa ) ||
                     ( ixrow != iarow ) )
               info = -9;
            else if( desc_A[NB_] != desc_X[MB_] )
               info = -(1100+MB_+1);
            else if( ( ( (*iy-1) % desc_Y[MB_]) != iroffa ) ||
                     ( iyrow != iarow ) )
               info = -15;
            else if( desc_A[MB_] != desc_Y[MB_] )
               info = -(1700+MB_+1);
         }
         else
         {
            if( ( *incx == 1 ) && ( *incx != desc_X[M_] ) )
               info = -12;
            else if( ( *incy == 1 ) && ( *incy != desc_Y[M_] ) )
               info = -18;
         }
         if( ictxt != desc_X[CTXT_] )
            info = -(1100+CTXT_+1);
         if( ictxt != desc_Y[CTXT_] )
            info = -(1700+CTXT_+1);
      }
   }
   if( info )
   {
      pberror_( &ictxt, "PCHEMV", &info );
      return;
   }
/*
*  Quick return if possible.
*/
   if( ( *n == 0 ) ||
       ( ( alpha->re == ZERO && alpha->im == ZERO ) &&
         ( beta->re  == ONE  && beta->im  == ZERO ) ) )
      return;
/*
*  Figure out the arguments to be passed to pbchemv
*/
   nn = *n + iroffa;
   mullen = MULLENFAC * desc_A[MB_];
   lcm = ilcm_( &nprow, &npcol );
   if( ( *incx == desc_X[M_] ) && ( *incy == desc_Y[M_] ) )
   {
      lcmp = lcm / nprow;
      lcmq = lcm / npcol;
      xydist = 'R';
      locincx = desc_X[LLD_];
      locincy = desc_Y[LLD_];
      ixpos = ixrow;
      iypos = iyrow;
      tmp1 = nn / desc_A[MB_];
      np0 = MYROC0( tmp1, nn, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, nn, desc_A[NB_], npcol );

      tmp2 = lcmq * desc_A[MB_];
      iszcmp = CEIL( mullen, tmp2 );
      szcmp = iszcmp * iszcmp * lcmq*desc_A[MB_] * lcmp*desc_A[MB_];

      tmp2 = CEIL( nn, desc_A[MB_] );
      tmp2 = MIN( lcmp, tmp2 );
      tmp1 = np0 / desc_A[MB_];
      tmp1 = MYROC0( tmp1, np0, desc_A[MB_], lcmp ) * tmp2;
      wksz = 2*(np0 + nq0) + MAX( szcmp, tmp1 );
   }
   else
   {
      lcmp = lcm / nprow;
      lcmq = lcm / npcol;
      xydist = 'C';
      locincx = 1;
      locincy = 1;
      ixpos = ixcol;
      iypos = iycol;
      tmp1 = nn / desc_A[MB_];
      np0 = MYROC0( tmp1, nn, desc_A[MB_], nprow );
      tmp2 = nn / desc_A[NB_];
      nq0 = MYROC0( tmp2, nn, desc_A[NB_], npcol );

      tmp2 = lcmq * desc_A[MB_];
      iszcmp = CEIL( mullen, tmp2 );
      szcmp = iszcmp * iszcmp * lcmq*desc_A[MB_] * lcmp*desc_A[MB_];

      tmp2 = CEIL( nn, desc_A[MB_] );
      tmp2 = MIN( lcmq, tmp2 );
      tmp1 = nq0 / desc_A[NB_];
      tmp1 = MYROC0( tmp1, nq0, desc_A[NB_], lcmq ) * tmp2;
      wksz = 2*(np0 + nq0) + MAX( szcmp, tmp1 );
   }
   buff = (complex *)getpbbuf( "PCHEMV", wksz*sizeof(complex) );
/*
*  Call PB-BLAS routine
*/
   pbchemv_( &ictxt, uplo, C2F_CHAR( &xydist ), n, &desc_A[MB_], &iroffa,
             alpha, &A[iia-1+(jja-1)*desc_A[LLD_]], &desc_A[LLD_],
             &X[iix-1+(jjx-1)*desc_X[LLD_]], &locincx, beta,
             &Y[iiy-1+(jjy-1)*desc_Y[LLD_]], &locincy, &iarow,
             &iacol, &ixpos, &iypos, C2F_CHAR( NO ), C2F_CHAR( NO ),
             &mullen, buff );
}
