<?php
/**
 * Vacation_Driver_sql implements the Vacation_Driver API for SQL servers.
 *
 * $Horde: vacation/lib/Driver/customsql.php,v 1.2.2.4 2009/01/29 16:02:55 jan Exp $
 *
 * Copyright 2008-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Jan Schneider <jan@horde.org>
 * @since   Vacation 3.1
 * @package Vacation
 */
class Vacation_Driver_customsql extends Vacation_Driver {

    /**
     * SQL connection object.
     *
     * @var DB
     */
    var $_db;

    /**
     * State of SQL connection.
     *
     * @var boolean
     */
    var $_connected = false;

    /**
     * Constructor.
     *
     * @param string $user   A user name.
     * @param array $params  Configuration parameters for the backend.
     */
    function Vacation_Driver_customsql($user, $params = array())
    {
        $params = Horde::getDriverConfig('server', 'sql');
        parent::Vacation_Driver($user, $params);
    }

    /**
     * Sets up vacation notices for a user.
     *
     * @param string $password  The password for the user.
     * @param string $message   The text of the vacation notice.
     * @param string $subject   The subject of the vacation notice.
     * @param string $from      The From: address of the vacation notice.
     * @param string $alias     The alias email address.
     */
    function setVacation($password, $message, $subject, $from, $alias = '')
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build username.
        $myuser = $this->_buildUsername();

        // Encrypt password.
        $crypted_password = $this->_encryptPassword($password, $this->_getCurrentPassword($myuser));

        // Build the SQL query.
        $query = str_replace(
            array('\U', '\M', '\S', '\F', '\A', '\P'),
            array($this->_db->quote($myuser),
                  $this->_db->quote($message),
                  $this->_db->quote($subject),
                  $this->_db->quote($from),
                  $this->_db->quote($alias),
                  $this->_db->quote($crypted_password)),
            $this->_params[$this->_realm]['query_set']);

        // Execute the query.
        $result = $this->_db->query($query);
        if (is_a($result, 'PEAR_Error')) {
            Horde::logMessage($result, __FILE__, __LINE__, PEAR_LOG_ERR);
            return $result;
        }
        if ($this->_db->affectedRows() == 0) {
            return PEAR::raiseError(_("The vacation notice cannot be set. Check the password."));
        }
        if ($result !== DB_OK) {
            return PEAR::raiseError(_("An unknown error occured while enabling the vacation notice."));
        }
    }

    /**
     * Disables the vacation message for a user.
     *
     * @param string $password  The password of the user.
     */
    function unsetVacation($password)
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build username.
        $myuser = $this->_buildUsername();

        // Encrypt password.
        $crypted_password = $this->_encryptPassword($password, $this->_getCurrentPassword($myuser));

        // Build the SQL query.
        $query = str_replace(
            array('\U', '\P'),
            array($this->_db->quote($myuser),
                  $this->_db->quote($crypted_password)),
            $this->_params[$this->_realm]['query_unset']);

        // Execute the query.
        $result = $this->_db->query($query);
        if (is_a($result, 'PEAR_Error')) {
            Horde::logMessage($result, __FILE__, __LINE__, PEAR_LOG_ERR);
            return $result;
        }
        if ($this->_db->affectedRows() == 0) {
            return PEAR::raiseError(_("The vacation notice cannot be set. Check the password."));
        }
        if ($result !== DB_OK) {
            return PEAR::raiseError(_("An unknown error occured while enabling the vacation notice."));
        }
    }

    /**
     * Retrieves the current vacation details for the user.
     *
     * @param string $password  The password for user.
     *
     * @return array  Vacation details or PEAR_Error.
     */
    function _getUserDetails($password)
    {
        static $row;

        // If we already have the details, return now.
        if (isset($row)) {
            return $row;
        }

        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to database.
        if (is_a($connected = $this->_connect(), 'PEAR_Error')) {
            return $connected;
        }

        // Build username.
        $myuser = $this->_buildUsername();

        // Encrypt password.
        $crypted_password = $this->_encryptPassword($password, $this->_getCurrentPassword($myuser));

        // Build the SQL query.
        $query = str_replace(array('\U', '\P'),
                             array($this->_db->quote($myuser),
                                   $this->_db->quote($crypted_password)),
                             $this->_params[$this->_realm]['query_get']);

        // Execute the query.
        $result = $this->_db->query($query);

        if (is_a($result, 'PEAR_Error')) {
            Horde::logMessage($result, __FILE__, __LINE__, PEAR_LOG_ERR);
            return $result;
        }

        $row = $result->fetchRow(DB_FETCHMODE_ASSOC);
        if (is_a($row, 'PEAR_Error')) {
            Horde::logMessage($row, __FILE__, __LINE__, PEAR_LOG_ERR);
            return $row;
        }
        if (is_array($row)) {
            return $row;
        }

        return array('vacation' => false);
    }

    /**
     * Retrieves the current password details for a user.
     *
     * @return string  Current password.
     */
    function _getCurrentPassword()
    {
        if (empty($this->_params[$this->_realm]['query_password'])) {
            return '';
        }
            
        // Build the SQL query.
        $query = str_replace('\U',
                             $this->_db->quote($this->_buildUsername()),
                             $this->_params[$this->_realm]['query_password']);

        // Execute the query.
        $current_password = $this->_db->getOne($query);
        if (is_a($current_password, 'PEAR_Error')) {
            Horde::logMessage($current_password, __FILE__, __LINE__, PEAR_LOG_ERR);
            return '';
        }

        return $current_password;
    }

    /**
     * Builds a username based on presence of realm.
     *
     * @return string  Fully qualified username.
     */
    function _buildUsername()
    {
        if ($this->_realm === 'default' ||
            $this->_realm === '') {
            return $this->_user;
        } else {
            return $this->_user . '@' . $this->_realm;
        }
    }

    /**
     * Checks if the realm has a specific configuration. If not, tries to fall
     * back on the default configuration. If still not a valid configuration
     * then exits with an error.
     */
    function _checkConfig()
    {
        // If no realm passed in, or no table config for the realm passed in,
        // then we fall back to the default realm
        if (empty($this->_params[$this->_realm]['query_set'])) {
            $this->_realm = 'default';
        }

        if (empty($this->_params[$this->_realm]['query_get']) ||
            empty($this->_params[$this->_realm]['query_set']) ||
            empty($this->_params[$this->_realm]['query_unset'])) {
            return PEAR::raiseError(_("The vacation application is not properly configured."));
        }
    }

    /**
     * Does an SQL connect and logs in as user with privilege to change
     * vacation.
     *
     * @return boolean  True or PEAR_Error based on success of connect.
     */
    function _connect()
    {
        if ($this->_connected) {
            return;
        }

        // Build the params array to pass to DB
        $args = array_merge($this->_params, $this->_params[$this->_realm]);

        Horde::assertDriverConfig($args, 'server', array('phptype'),
                                  'Vacation SQL');

        if (!isset($args['database'])) {
            $args['database'] = '';
        }
        if (!isset($args['username'])) {
            $args['username'] = '';
        }
        if (!isset($args['hostspec'])) {
            $args['hostspec'] = '';
        }

        // Connect to the SQL server using the supplied parameters.
        require_once 'DB.php';
        $this->_db = &DB::connect($args,
                                  array('persistent' => !empty($args['persistent'])));
        if (is_a($this->_db, 'PEAR_Error')) {
            return $this->_db;
        }

        // Set DB portability options.
        switch ($this->_db->phptype) {
            case 'mssql':
                $this->_db->setOption(
                    'portability',
                    DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS | DB_PORTABILITY_RTRIM);
                break;
            default:
                $this->_db->setOption(
                    'portability',
                    DB_PORTABILITY_LOWERCASE | DB_PORTABILITY_ERRORS);
        }

        $this->_connected = true;
    }

}
