#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#if (QT_REL == 3)
#include <qlistview.h>
#include <qmessagebox.h>
#include <qinputdialog.h>
#include <qpopupmenu.h>
#include "../include/tulip/ClusterTree.h"
#else
#include <Qt3Support/q3listview.h>
#include <QtGui/qmessagebox.h>
#include <QtGui/qinputdialog.h>
#include <Qt3Support/q3popupmenu.h>
#include "tulip/Qt3ForTulip.h"
#include "../include/tulip/ClusterTreeQt3.h"
#endif

#include <tulip/SuperGraph.h>
#include <tulip/SelectionProxy.h>

using namespace std;

//=======================================================
struct ClusterListViewItem: public QListViewItem {
  SuperGraph *_superGraph;
  ClusterListViewItem(SuperGraph * superGraph, QListViewItem *item) : QListViewItem(item), _superGraph(superGraph) {}
  ClusterListViewItem(SuperGraph * superGraph, QListView *item) : QListViewItem(item), _superGraph(superGraph) {}
  SuperGraph * getSuperGraph() const {
    return _superGraph;
  }
};
//=======================================================
/* 
 *  Constructs a ClusterTree which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 */
ClusterTree::ClusterTree(SuperGraph *graph , QWidget* parent,  const char* name, Qt::WFlags fl) : 
  ClusterTreeData(parent, name, fl),
  _currentGraph(graph) {
  setCaption(trUtf8("Cluster Tree"));
  treeView->setColumnText(0,trUtf8("Subgraph Hierachy"));
#if (QT_REL == 3)
  connect(treeView, SIGNAL(currentChanged(QListViewItem *)), SLOT(changeSuperGraph(QListViewItem *)));
  connect(treeView, SIGNAL(contextMenuRequested ( QListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( QListViewItem *, const QPoint &, int )));
#else
  // because of moc control
  connect(treeView, SIGNAL(currentChanged(Q3ListViewItem *)), SLOT(changeSuperGraph(Q3ListViewItem *)));
  connect(treeView, SIGNAL(contextMenuRequested ( Q3ListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( Q3ListViewItem *, const QPoint &, int )));
#endif
  update();
}
//=======================================================
ClusterTree::ClusterTree(QWidget* parent, const char* name, Qt::WFlags fl) :
  ClusterTreeData(parent, name,fl),
  _currentGraph(0) {
  setCaption(trUtf8("Cluster Tree"));
  treeView->setColumnText(0,trUtf8("Subgraph Hierachy"));
#if (QT_REL == 3)
  connect(treeView, SIGNAL(currentChanged(QListViewItem *)), SLOT(changeSuperGraph(QListViewItem *)));
  connect(treeView, SIGNAL(contextMenuRequested ( QListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( QListViewItem *, const QPoint &, int )));
#else
  // because of moc control
  connect(treeView, SIGNAL(currentChanged(Q3ListViewItem *)), SLOT(changeSuperGraph(Q3ListViewItem *)));
  connect(treeView, SIGNAL(contextMenuRequested ( Q3ListViewItem *, const QPoint &, int )),
	  SLOT(rightButtonClusterTree( Q3ListViewItem *, const QPoint &, int )));
#endif
}
//=======================================================
SuperGraph* ClusterTree::getSuperGraph() const {
  return _currentGraph;
}
//=======================================================
void ClusterTree::setSuperGraph(SuperGraph *graph) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  if (graph == 0) {
    _currentGraph = 0;
    treeView->clear();
    //    QListViewItem *item = new QListViewItem( treeView, 0 );
    //    item->setText( 0, trUtf8( "No hierachy" ) );
    return;
  }
  if (_currentGraph==0 || (graphItems.get(graph->getId())==0)) {
    _currentGraph = graph;
    update();
  }
  currentGraphChanged(graph);
  emit supergraphChanged(_currentGraph);
}
//=======================================================
void ClusterTree::currentGraphChanged(const SuperGraph *graph) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  QListViewItem  *item = graphItems.get(graph->getId());
  if (item != 0) {
#if (QT_REL == 3)
    disconnect(treeView, SIGNAL(currentChanged(QListViewItem *)), this, SLOT(changeSuperGraph(QListViewItem *)));
#else
    disconnect(treeView, SIGNAL(currentChanged(Q3ListViewItem *)), this, SLOT(changeSuperGraph(Q3ListViewItem *)));
#endif
    treeView->setCurrentItem(item);
    treeView->ensureItemVisible(item);
#if (QT_REL == 3)
    connect(treeView, SIGNAL(currentChanged(QListViewItem *)), this, SLOT(changeSuperGraph(QListViewItem *)));
#else
    connect(treeView, SIGNAL(currentChanged(Q3ListViewItem *)), this, SLOT(changeSuperGraph(Q3ListViewItem *)));
#endif   
    _currentGraph = ((ClusterListViewItem *)item)->getSuperGraph();
  }
}
//=======================================================
//Building of the cluster tree view
void ClusterTree::buildTreeView(QListView *item, SuperGraph *p) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  QListViewItem *tmpItem = new ClusterListViewItem(p, item);
  tmpItem->setText(0, QString(p->getAttribute<string>("name").c_str()));
  graphItems.set(p->getId(), tmpItem);
  Iterator<SuperGraph *> *itS = p->getSubGraphs();
  while (itS->hasNext())
    buildTreeView(tmpItem, itS->next());
  delete itS;
  treeView->setOpen(tmpItem, true);
}
//=======================================================
void ClusterTree::buildTreeView(QListViewItem *item, SuperGraph *p) {
  QListViewItem *tmpItem = new ClusterListViewItem(p, item);
  tmpItem->setText(0, QString(p->getAttribute<string>("name").c_str()));
  graphItems.set(p->getId(), tmpItem);
  Iterator<SuperGraph *> *itS=p->getSubGraphs();
  while (itS->hasNext())
    buildTreeView(tmpItem, itS->next());
  delete itS;
}
//=======================================================
void ClusterTree::update() {  
  //cerr << __PRETTY_FUNCTION__ << endl;
  treeView->clear();
  graphItems.setAll(0);
  if (_currentGraph != 0) {
    buildTreeView(treeView, _currentGraph->getRoot());
    currentGraphChanged(_currentGraph);
  }
  show();
}
//=======================================================
//Cluster Tree Structure modification
void ClusterTree::removeSubgraph(SuperGraph *graph, bool recursive) {
  //cerr << __PRETTY_FUNCTION__ << endl;
  if (graph == 0) return;
  if (graph == graph->getRoot()) {
    QMessageBox::critical( 0, "Tulip Hierarchy Editor Remove Failed", QString("You cannot remove the root graph"));
    return;
  } 
  emit aboutToRemoveView(graph);
  _currentGraph = graph->getFather();
  if (!recursive)
    _currentGraph->delSubGraph(graph);
  else
    _currentGraph->delAllSubGraphs(graph);
  update();
  emit supergraphChanged(_currentGraph);
}
//=======================================================
//Cluster Tree Structure modification
void ClusterTree::contextRemoveCluster() {
  removeSubgraph(_currentGraph, false);
}
//=======================================================
void ClusterTree::contextRemoveAllCluster() {
  removeSubgraph(_currentGraph, true);
}
//=======================================================
void ClusterTree::contextCloneCluster() {  
  if (_currentGraph == _currentGraph->getRoot()) {
    QMessageBox::critical( 0, "Tulip Cluster Tree Editor Clone Failed",QString("You cannot clone the root cluster"));
    return;
  }
  bool ok;
  QString text = QInputDialog::getText( "Cluster name" , "Please enter the cluster name" ,
                                        QLineEdit::Normal,QString::null, &ok, this);
  if (ok) {
    SuperGraph *tmp=_currentGraph->getFather()->addSubGraph();
    tmp->setAttribute("name",string(text.latin1()));
    Iterator<node> *itN=_currentGraph->getNodes();
    while (itN->hasNext())
      tmp->addNode(itN->next());
    delete itN;
    Iterator<edge> *itE=_currentGraph->getEdges();
    while (itE->hasNext())
      tmp->addEdge(itE->next());
    delete itE;
    update();
  }
}
//=======================================================
void ClusterTree::contextCloneSubgraphCluster() {
  bool ok;
  QString text = QInputDialog::getText( "Cluster name" , "Please enter the cluster name" ,
                                        QLineEdit::Normal,QString::null, &ok, this);
  if (ok) {
    SelectionProxy sel1(_currentGraph);
    sel1.setAllNodeValue(true);
    sel1.setAllEdgeValue(true);
    _currentGraph = _currentGraph->addSubGraph(&sel1);
    _currentGraph->setAttribute("name",string(text.latin1()));
    update();
    emit supergraphChanged(_currentGraph);
  }
}
//=======================================================
void ClusterTree::contextMoveUpCluster() {
  std::cerr << "Not Implemented" << __PRETTY_FUNCTION__ << std::endl;
  //  _clusterTree->moveUp(_currentSubGraph);
  //  update();
}
//=======================================================
void ClusterTree::contextRenameCluster() {
  bool ok;
  QString text = QInputDialog::getText( trUtf8("Cluster Name") ,  trUtf8("Please enter the cluster name"),
					QLineEdit::Normal, _currentGraph->getAttribute<string>("name").c_str(), &ok, this);
  if (ok) {
    _currentGraph->setAttribute("name",string(text.latin1()));
    graphItems.get(_currentGraph->getId())->setText(0, text);
  }
}
//=======================================================
void ClusterTree::rightButtonClusterTree(QListViewItem *item, const QPoint &p, int c) {
  if (item == 0) return;
  QPopupMenu menu(this, "cluster_tree_context_menu");
  menu.insertItem(trUtf8("Remove"), this, SLOT(contextRemoveCluster()));
  menu.insertItem(trUtf8("Remove all"), this, SLOT(contextRemoveAllCluster()));
  menu.insertItem(trUtf8("Clone"), this, SLOT(contextCloneCluster()));
  menu.insertItem(trUtf8("SubGraph Clone"), this, SLOT(contextCloneSubgraphCluster()));
  menu.insertItem(trUtf8("Rename"), this, SLOT(contextRenameCluster()));
  menu.exec(p);
}
//=======================================================
void ClusterTree::changeSuperGraph(QListViewItem *item, const QPoint &p, int i) {
  changeSuperGraph(item);
}
//**********************************************************************
void ClusterTree::changeSuperGraph(QListViewItem *item) {
  //  cerr << __PRETTY_FUNCTION__ << endl;
  _currentGraph = ((ClusterListViewItem *)item)->getSuperGraph();
  emit supergraphChanged(_currentGraph);
}
//=======================================================
