#include <tulip/SuperGraph.h>
#include <tulip/GraphTools.h>
#include <tulip/DrawingTools.h>
#include <tulip/TlpTools.h>

#include "ConnnectedComponnentPacking.h"
#include "rectanglePackingFonctions.h"

using namespace std;

LAYOUTPLUGINOFGROUP(ConnectedComponnentPacking, "Connected Componnent Packing", "David Auber", "26/05/05", "beta", "0", "1", "Misc");

const float spacing = 8;
#define COMPLEXITY "auto;n5;n4logn;n4;n3logn;n3;n2logn;n2;nlogn;n;"
namespace {
const char* paramHelp[] = {
//Layout
HTML_HELP_OPEN() \
    HTML_HELP_DEF("Type", "LayoutProxy") \
    HTML_HELP_DEF("Values", "Any layout proxy") \
    HTML_HELP_DEF("Default", "viewLayout") \
    HTML_HELP_BODY() \
        "Choose the input coordinates of nodes and edges" \
HTML_HELP_CLOSE(),
//Sizes
HTML_HELP_OPEN() \
    HTML_HELP_DEF("Type", "SizesProxy") \
    HTML_HELP_DEF("Values", "Any sizes proxy") \
    HTML_HELP_DEF("Default", "viewSize") \
    HTML_HELP_BODY() \
        "Choose the input size of nodes" \
HTML_HELP_CLOSE(),
//Rotation
HTML_HELP_OPEN() \
    HTML_HELP_DEF("Type", "MetricProxy") \
    HTML_HELP_DEF("Values", "Any metric proxy used for rotation of nodes on z-axis") \
    HTML_HELP_DEF("Default", "viewRotation") \
    HTML_HELP_BODY() \
        "Choose the input rotation of nodes on z-axis" \
HTML_HELP_CLOSE(),
//Complexity
HTML_HELP_OPEN() \
    HTML_HELP_DEF("Type", "StringColection") \
    HTML_HELP_DEF("Default", "auto") \
    HTML_HELP_BODY() \
        "Choose the complexity of the algorithm.<br> n is the number of connected componnents in the graph" \
HTML_HELP_CLOSE()};
}                     

//====================================================================
ConnectedComponnentPacking::ConnectedComponnentPacking(const PropertyContext& context)
    : Layout(context) {
  addParameter<LayoutProxy> ("coordinates",paramHelp[0],"viewLayout");
  addParameter<SizesProxy> ("sizes",paramHelp[1],"viewSize");
  addParameter<MetricProxy> ("rotation",paramHelp[2],"viewRotation");
  addParameter<StringCollection> ("complexity", paramHelp[2], COMPLEXITY );
}
//====================================================================
bool ConnectedComponnentPacking::run() {
  
  LayoutProxy *layout = 0;
  SizesProxy *sizes = 0;
  MetricProxy *rotation = 0;
  string complexity("auto");

  workingGraph = tlp::newCloneSubGraph(superGraph, "workingGraph");

  if ( dataSet!=0 ) {
    dataSet->get("coordinates", layout);
    dataSet->get("sizes", sizes);
    dataSet->get("rotation", rotation);
    StringCollection complexityCol;
    if (dataSet->get("complexity", complexityCol))
      complexity = complexityCol.getCurrentString(); 
  }

  if (layout==0)
    layout = workingGraph->getProperty<LayoutProxy>("viewLayout");
  if (sizes==0)
    sizes = workingGraph->getProperty<SizesProxy>("viewSize");  
  if (rotation==0)
    rotation = workingGraph->getProperty<MetricProxy>("viewRotation");  
  if (complexity=="none")
    complexity = "auto";


  MetricProxy connectedComponnent(workingGraph);
  string err;
  workingGraph->computeProperty(string("Connected Component"), &connectedComponnent, err);
  DataSet tmp;
  tmp.set("Metric", &connectedComponnent);
  tlp::clusterizeGraph(workingGraph, err, &tmp, "Equal Value");

  vector<Rectangle<float> > rectangles;
  Iterator<SuperGraph *> *it = workingGraph->getSubGraphs();
  while(it->hasNext()) {
    SuperGraph *graph = it->next();
    pair<Coord, Coord> tmp = tlp::computeBoundingBox(graph, layout, sizes, rotation);
    Rectangle<float> tmpRec;
    tmpRec[1][0] = tmp.first[0] + spacing;
    tmpRec[1][1] = tmp.first[1] + spacing;
    tmpRec[0][0] = tmp.second[0] + spacing;
    tmpRec[0][1] = tmp.second[1] + spacing;
    rectangles.push_back(tmpRec);
  } delete it;

  if (complexity == "auto") {
    if (rectangles.size()<25) { 
      complexity="n5";
    } else
    if (rectangles.size()<50) { 
      complexity="n4logn";
    } else
    if (rectangles.size()<100) { 
      complexity="n4";
    } else
    if (rectangles.size()<150) { 
      complexity="n3logn";
    } else
    if (rectangles.size()<250) { 
      complexity="n3";
    } else
    if (rectangles.size()<500) { 
      complexity="n2logn";
    } else
    if (rectangles.size()<1000) { 
      complexity="n2";
    } else
    if (rectangles.size()<5000) { 
      complexity="nlogn";
    } else
    complexity="n";
  }

  vector<Rectangle<float> > rectanglesBackup(rectangles);
  RectanglePackingLimitRectangles(rectangles, complexity.c_str(), pluginProgress);
  Iterator<node> *itN = superGraph->getNodes();
  while(itN->hasNext()){
    node n = itN->next();
    layoutProxy->setNodeValue(n, layout->getNodeValue(n));
  } delete itN;
  Iterator<edge> *itE = superGraph->getEdges();
  while(itE->hasNext()){
    edge e = itE->next();
    layoutProxy->setEdgeValue(e, layout->getEdgeValue(e));
  } delete itE;
  unsigned int i = 0;
  it = workingGraph->getSubGraphs();
  while(it->hasNext()) {
    SuperGraph *graph = it->next();
    Coord move(rectangles[i][0][0]-rectanglesBackup[i][0][0], rectangles[i][0][1]-rectanglesBackup[i][0][1], 0);
    layoutProxy->translate(move, graph);
    ++i;
  } delete it;
  superGraph->delAllSubGraphs(workingGraph);
}
