/*
 * AnimationDialog.cpp
 *
 * Copyright (C) 2003 J. "MUFTI" Scheurich
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */
 
#include "stdafx.h"
#include <ctype.h>
#include "AnimationDialog.h"
#include "resource.h"
#include "MyString.h"
#include "newline.h"
#include "FieldValue.h"
#include "SFMFTypes.h"
#include "Field.h"
#include "EventIn.h"
#include "EventOut.h"
#include "ExposedField.h"
#include "Element.h"
#include "DuneApp.h"
#include "NodeTransform.h"

// for debugging
// #include "swt/motif/swt_structs.h"

#define X_SPACING 2
#define Y_SPACING 2

static SWND canvas;
static SWND scroll;
static SWND parent_window;

#define XPOS 20
#define YPOS 100
#ifdef WIN32
# define XMAX 300
# define YMAX 240
#else
# define XMAX 388
# define YMAX 300
#endif
#define SCROLL_SIZE 5

static int ymax;

static void
AnimationDialogRedraw(void* data)
{
    SDC	dc = swCreateDC(canvas);
    swSetFGColor(dc, swGetWindowColor(parent_window, SW_COLOR_WINDOW_BG));
    AnimationDialog* dialog = (AnimationDialog*) data;
    dialog->accountYmax();
    swFillRect(dc, 1, 1, XMAX-2, ymax-2);
    swSetFGColor(dc, 0);
    swDrawLine(dc,      0,      0, XMAX-1,   0);
    swDrawLine(dc,      0, ymax-1, XMAX-1, ymax-1);
    swDrawLine(dc,      0,      0,      0, ymax-1);
    swDrawLine(dc, XMAX-1,      0, XMAX-1, ymax-1);
    dialog->drawInterface(dc);
    swDestroyDC(dc);
}

static void
AnimationDialogExposeCallback(void *data, int x, int y, int width, int height)
{
    AnimationDialogRedraw(data);
}

#ifdef WIN32
static void
AnimationDialogEnterCallback(void *data, int command)
{
    AnimationDialogRedraw(data);
}
#endif

static int fontHeight;

AnimationDialog::AnimationDialog(SWND parent, Node* oldNode)
  : Dialog(parent, IDD_ANIMATION)
{
    parent_window = parent;
    _animationNode = oldNode;
    _newTimeSensorSeconds = 10;
    fontHeight = swGetFontHeight(swGetDefaultFont());
    ymax = YMAX;
    buildInterfaceData();
    LoadData();
    accountYmax();
    swInvalidateWindow(canvas);
}

AnimationDialog::~AnimationDialog()
{
    swDestroyWindow(canvas);
    swDestroyWindow(scroll); 
    for (int i = 0 ; i < numEventInNames() ; i++)
        swDestroyWindow(_buttons[i]);
}

void
AnimationDialog::buildInterfaceData(void)
{
    Proto* proto = _animationNode->getProto();
    if (proto == NULL)
        return;
    int index = 0;
    for (int i = 0; i < proto->getNumEventIns(); i++) {
        int type = proto->getEventIn(i)->getType();
        if (typeDefaultValue(type)->isAnimateable() &&
            typeDefaultValue(type)->hasAnimationSupport()) {
            _eventInTypes[index] = type;
            _eventInNames[index] = proto->getEventIn(i)->getName();
            _eventInFields[index] = i;
            index++; 
        }
    }
}

bool
AnimationDialog::Validate()
{
    for (int i = 0 ; i < numEventInNames() ; i++)
        if (_eventInIsAnimated[i] == true)
            return true;
    int commentID = _animationNode->getAnmationCommentID();
    if (commentID != -1) {
        char msg[256];
        swLoadString(commentID, msg, 255);
        swMessageBox(TheApp->mainWnd(), msg,
                     "animate what ?", SW_MB_OK, SW_MB_ERROR);
    }
    return false;
}

void
AnimationDialog::LoadData()
{
    int i;

    SWND comboTimeSensors = swGetDialogItem(_dlg, IDC_TIMESENSORS);
    swComboBoxDeleteAll(comboTimeSensors);

    Array<MyString> timeSensors;
    timeSensors[0] = "new TimeSensor"; 

    const NodeList *nodes = _animationNode->getScene()->getNodes();
    for (i = 0; i < nodes->size(); i++) {
        Node    *node = nodes->get(i);
        if (node->isInScene(_animationNode->getScene()))
            if (node->getType() == NODE_TIME_SENSOR)
                if (node->hasName())
                    timeSensors.append(node->getName()); 
    }
    
    for (i = 0;i < timeSensors.size(); i++)
        swComboBoxAppendItem(comboTimeSensors, timeSensors[i]);

    char        buf[128];

    snprintf(buf, 128, "%g", _newTimeSensorSeconds);
    swSetText(swGetDialogItem(_dlg, IDC_TIMESENSOR_SECONDS), buf);

    if (TheApp->is4Kids())
        if (_animationNode->getType() == NODE_TRANSFORM) {
            for (i = 0; i < _eventInNames.size(); i++) {
                _initIsAnimated.append(false);
                if (strcmp((const char *)_eventInNames[i], "set_rotation") == 0)
                    _initIsAnimated[i] = true;
                if (strcmp((const char *)_eventInNames[i], "set_translation") 
                    == 0)
                    _initIsAnimated[i] = true;
           }
       }            

    scroll = swCreateScrolledWindow(XPOS, YPOS, XMAX-XPOS, YMAX-YPOS, _dlg);
    canvas = swCreateCanvas("Animationdialog", 0, 0, XMAX, YMAX, scroll);
    swScrolledWindowSetChild(scroll, canvas);
    swSetScrollSizes(scroll, XMAX-XPOS-SCROLL_SIZE, YMAX-YPOS-SCROLL_SIZE);

    swSetClientData(canvas, this);
#ifdef WIN32
    swSetEnterCallback(canvas, AnimationDialogEnterCallback);
#else
    swSetExposeCallback(canvas, AnimationDialogExposeCallback);
#endif
    swInvalidateWindow(canvas);

    swShowWindow(canvas);
}

void 
AnimationDialog::SaveData()
{
    _timeSensor = NULL;

    int sensor = swComboBoxGetSelection(swGetDialogItem(_dlg, IDC_TIMESENSORS));

    if (sensor > 0) {
        int index = 1;
        const NodeList *nodes = _animationNode->getScene()->getNodes();
        for (int i = 0; i < nodes->size() && (_timeSensor == NULL); i++) {
            Node    *node = nodes->get(i);
            if (node->isInScene(_animationNode->getScene()))
                if (node->getType() == NODE_TIME_SENSOR)
                    if (node->hasName())
                        if (index++ == sensor) {
                            _timeSensor = (NodeTimeSensor *)node;
                            break;
                        }
        }
    }


    char        buf[128];

    swGetText(swGetDialogItem(_dlg, IDC_TIMESENSOR_SECONDS), buf, 128);
    _newTimeSensorSeconds = atof(buf);

    AnimationDialogRedraw(this);
}


void 
AnimationDialog::accountYmax()
{
    ymax = (numEventInNames() + 1) * (Y_SPACING + fontHeight) +
           2 * Y_SPACING; 
    if (ymax > YMAX-YPOS-SCROLL_SIZE) {
        swSetSize(canvas, XMAX-XPOS-SCROLL_SIZE, ymax);
        swSetScrollSizes(scroll, XMAX-XPOS-SCROLL_SIZE, ymax);
    } else {
        ymax = YMAX-YPOS-SCROLL_SIZE;
        swSetScrollSizes(scroll, XMAX-XPOS-SCROLL_SIZE, ymax);
    }
}   


void 
AnimationDialog::drawInterface(SDC dc)
{
    int y = 0;
    for (int i = 0 ; i < numEventInNames() ; i++) {
        y += Y_SPACING ;
  
        _eventInIsAnimated[i] = false;
        if (i < _buttons.size()) {
            if (swGetCheck(_buttons[i]) == 0)
                _eventInIsAnimated[i] = false;
            else
               _eventInIsAnimated[i] =  true;
            swDestroyWindow(_buttons[i]);
        }
        if (TheApp->is4Kids())
            if (_initIsAnimated[i]) {
                _eventInIsAnimated[i] = true;
                _initIsAnimated[i] = false;
            }
        _buttons[i]=swCreateCheckBox("", X_SPACING, y + Y_SPACING, 
                                     fontHeight, fontHeight, canvas);
        swSetCheck(_buttons[i], _eventInIsAnimated[i] ? 1 : 0);
        y += fontHeight;
        swDrawText(dc, 2 * X_SPACING + fontHeight, y, (const char*) 
                   _eventInNames[i]);
    }
}

