/*
 * Scene.h
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#ifndef _SCENE_H
#define _SCENE_H

#ifndef _STDIO_H
#include <stdio.h>
#endif

#ifndef _ARRAY_H
#include "Array.h"
#endif

#ifndef _MAP_H
#include "Map.h"
#endif

#ifndef _STACK_H
#include "Stack.h"
#endif

#ifndef _COMMAND_LIST_H
#include "CommandList.h"
#endif

#ifndef _NODE_LIST_H
#include "NodeList.h"
#endif

#ifndef _DUNE_STRING_H
#include "MyString.h"
#endif

#ifndef _URL_H
#include "URL.h"
#endif

#ifndef _PROTO_H
#include "Proto.h"
#endif

#ifndef _DUNE_APP_H
#include "DuneApp.h"
#endif

#include "TransformMode.h"

#define MAX_PATH_LEN 1024

class FieldValue;
class Node;
class Path;
class FontInfo;
class NodeViewpoint;
class SFVec3f;
class SceneView;
class Hint;
class NodeInline;

typedef Stack<Command *>	CommandStack;

typedef Map<MyString, Node *>	NodeMap;
typedef Map<MyString, int>	StringMap;
typedef Map<MyString, Proto *>	ProtoMap;

class Interpolator;

class Scene {
public:
			Scene();
		       ~Scene();
    void		def(const char *nodeName, Node *value);
    Node	       *use(const char *nodeName);
    void		undef(MyString nodeName);

    int			addSymbol(MyString s);
    const MyString     &getSymbol(int id) const;

    void		setRoot(Node *root)	{ _root = root; }
    Node	       *getRoot() const		{ return _root; }
    int	      	        getRootIndex() const	{ return _rootIndex; }

    void		setNodes(NodeList *nodes);
    const NodeList     *getNodes() const        { return &_nodes; }
    void       		addNodes(Node *target, NodeList *nodes);

    void 		scanForInlines(NodeList *nodes);
    void 		readInline(NodeInline *node);

    const char         *parse(FILE *f, bool protoLibrary, Node *target = NULL);
    Node	       *createNode(const char *nodeType);
    int			write(int filedes, const char *url, bool tempSave,
                              bool pureVRML97=false);

    void		addProto(MyString name, Proto *value);
    Proto	       *getProto(MyString name);

    static bool		validRoute(Node *src, int eventOut, Node *dst, int eventIn);
    bool		addRoute(Node *src, int eventOut, Node *dest, int eventIn);
    void		deleteRoute(Node *src, int eventOut, Node *dest, int eventIn);

    void		execute(Command *cmd);

    void		add(Command *change);

    void		undo();
    void		redo();

    int			canUndo() { return !_undoStack.empty(); }
    int			canRedo() { return !_redoStack.empty(); }

    int			getUndoStackTop() { return _undoStack.getTop(); }

    void                startMultipleUndo();
    void                optimizeMultipleUndo();

    void		errorf(const char *fmt, ...);

    void		invalidNode(const char *nodeName);
    void		invalidField(const char *node, const char *field);

    void		backupField(Node *node, int field);
    void                backupFieldsStart(void);
    void		backupFieldsAppend(Node *node, int field);
    void		backupFieldsDone(void);
    void		setField(Node *node, int field, FieldValue *value);
    Interpolator       *findUpstreamInterpolator(Node *node, int field) const;

    void		drawScene(bool pick = false, int x = 0, int y = 0);
    Path	       *pick(int x, int y);
    Path	       *searchTransform(void);
    void		drawHandles();
    void		drawHandlesRec(Node *node) const;

    void		setSelectedHandle(int handle)
			{ _selectedHandle = handle; }
    int			getSelectedHandle() const
			{ return _selectedHandle; }
    void		setOldSelectedHandle(int handle)
			{ _oldSelectedHandle = handle; }
    int			getOldSelectedHandle() const
			{ return _oldSelectedHandle; }

    void		setTransformMode(TMode tm)
			{ _transformMode->tmode = tm; }
    void		setTransformMode(TDimension td)
			{ _transformMode->tdimension = td; }
    void		setTransformMode(T2axes t2)
			{ _transformMode->t2axes = t2; }
    TransformMode      *getTransformMode()
			{ return _transformMode; }

    void		transform(const Path *path);
    void		projectPoint(float x, float y, float z,
				     float *wx, float *wy, float *wz);
    void		unProjectPoint(float wx, float wy, float wz,
				     float *x, float *y, float *z);
    int			allocateLight();
    int			freeLight();
    int			getNumLights() { return _numLights; }

    void		enableHeadlight();
    Path	       *processHits(unsigned hits, GLuint *pickBuffer);

    void		addViewpoint(Node *viewpoint);
    void		addFog(Node *fog);
    void		addBackground(Node *background);
    void		addTimeSensor(Node *timeSensor);

    void		setHeadlight(int headlight)
			{ _headlight = headlight; }

    void		moveCamera(float dx, float dy, float dz);
    void		turnCamera(float x, float y, float z, float ang);
    void		orbitCamera(float dtheta, float dphi);

    NodeViewpoint      *getCamera() const;

    void		applyCamera(void);

    void		addNode(Node *node);
    void		removeNode(Node *node);
    
    MyString		getUniqueNodeName(Node *node);
    MyString		generateUniqueNodeName(Node *node);

    const Path	       *getSelection() const { return _selection; }
    void		setSelection(const Path *selection);
    void		setSelection(Node *node);

    Path	       *newPath(Node *node);

    void		setURL(const char *url) { _URL = url; }
    const char	       *getURL() const { return _URL; }
    const char	       *getNewURL() const { return _newURL; }
    bool		isTempSave() const { return _tempSave; }

    void		setPath(const char *path) { _path = path; }
    const char	       *getPath() const { return _path; }

    void                setPathAllURL(const char *path);

    bool                isPureVRML97() { return _pureVRML97; }

    bool		isModified() const;
    bool		isRunning() const { return _running; }
    bool		isRecording() const { return _recording; }
    void		setRecording(bool recording) { _recording = recording; }

    void		start();
    void		stop();
    void		updateTime();

    void		OnFieldChange(Node *node, int field, int index = -1);
    void		OnAddNode(Node *node, Node *dest, int field);
    void		OnRemoveNode(Node *node, Node *src, int field);

    void		AddView(SceneView *view);
    void		RemoveView(SceneView *view);
    void		UpdateViews(SceneView *sender, int type,
				    Hint *hint = NULL);

    void		DeleteSelected();
    void		DeleteSelectedAppend(CommandList* list);

    int			OnDragOver(Node *src, Node *srcParent, int srcField, 
				   Node *dest, int destField, int modifiers);
    int			OnDrop(Node *src, Node *srcParent, int srcField, 
			       Node *dest, int destField, int effect);

    bool		Download(const URL &url, MyString *path);
    FontInfo           *LoadGLFont(const char *fontName, const char *style);
    int                 getNumProtoNames(void) { return _numProtoNames; }
    bool                addProtoName(MyString name);
    MyString            getProtoName(int i) { return _protoNames[i]; }
    void		addProtoDefinition(void);
    void                addToProtoDefinition(char* string);
    void		addRouteString(MyString string);
    int			writeRouteStrings(int f);
    int                 writeExternProto(int f, char* protoName);

    void                setHasFocus(void)    {_hasFocus=true;}
    void                deleteHasFocus(void) {_hasFocus=false;}
    bool                getHasFocus(void)    
                           {
                           if (TheApp->dontCareFocus())
                               return true;
                           else
                               return _hasFocus;
                           }  
    void                setNavigationMode(bool flag) {_navigationMode=flag;}
    bool                getNavigationMode() {return _navigationMode;}

    void                setViewOfLastSelection(SceneView* view);
    SceneView	       *getViewOfLastSelection(void);
    void	        deleteLastSelection(void);
    void 		DeleteLastSelection(void);

    bool		hasAlreadyName(MyString name);
    void 		updateURLs(Node* node);

    void		saveProtoStatus(void);
    void		restoreProtoStatus(void);

    void 		setErrorLineNumber(int lineno) 
                              { _errorLineNumber = lineno ;}
    int 		getErrorLineNumber(void) { return _errorLineNumber; }

    /* 
     * a extra modifiedflag is needed to signal possible changes 
     * (e.g. from the file -> Textedit menupoint)
     */
    void		setExtraModifiedFlag(void) { _extraModifiedFlag=true; }
    StringArray        *getAllNodeNames(void);
    void                draw3dCursor(int x, int y);
    void                use3dCursor(bool flag) { _use3dCursor = flag; }
    bool                use3dCursor(void);
    bool                setProtoPrefix(char* protoPrefix);
    char               *getProtoPrefix(void) { return _protoPrefix; }
    MyString	        getNodeWithPrefix(const MyString &nodeType);
    bool                hasExternProtoWarning(void) 
                           { return _externProtoWarning; }
    void                setExternProtoWarning(bool flag) 
                           { _externProtoWarning = flag; }
    Node               *convertConeToNurbs(Node* cone);                 
    bool                getXSymetricNurbsMode() 
                           { return TheApp->GetXSymetricMode(); }
    bool                isParsing(void) { return _isParsing; }
    int                 getNumberBuildinProtos(void) 
                           { return _numberBuildinProtos; }

private:
    NodeMap		_nodeMap;
    Node	       *_root;
    int	                _rootIndex;
    StringMap		_symbols;
    Array<MyString>	_symbolList;
    Array<MyString>	_protoNames;
    int                 _numProtoNames;
    int  		_statusNumProtoNames;
    Array<MyString>	_protoDefinitions;
    int                 _numProtoDefinitions;
    int  		_statusNumProtoDefinitions;
    ProtoMap		_protos;

    CommandStack	_undoStack;
    CommandStack	_redoStack;

    int                 _multipleUndoTop;
    CommandList        *_backupCommandList;
    int			_numLights;
    NodeList		_viewpoints;
    NodeList		_fogs;
    NodeList		_backgrounds;
    NodeList		_timeSensors;
    Stack<Node *>	_fogStack;
    int			_headlight;

    NodeList		_nodes;

    const Path	       *_selection;
    int			_selectedHandle;
    int			_oldSelectedHandle;
    TransformMode      *_transformMode;

    MyString		_URL;
    MyString		_newURL;
    bool		_tempSave;

    int		        _numSymbols;

    bool		_running;
    bool		_recording;

    Command	       *_unmodified;
    bool		_extraModifiedFlag;
    bool                _pureVRML97;

    NodeViewpoint      *_defaultViewpoint;
    NodeViewpoint      *_currentViewpoint;

    List<SceneView *>	_views;

    MyString		_compileErrors;
    Array<FontInfo *>	_fonts;

    MyString		_path;

    List<MyString>	_routeList;

    /* when picking several objects which one click, which one is selected ? */
    int                 _selectlevel;

    /* current scene is in mouse focus ? */
    bool                _hasFocus;

    bool		_navigationMode;

    SceneView 	       *_viewOfLastSelection;
    bool                _selection_is_in_scene;

    GLUquadricObj      *_obj3dCursor;
    bool                _use3dCursor;
    int 		_errorLineNumber;

    Array<char *>       _nodesWithExternProto;
    char               *_protoPrefix;
    bool                _externProtoWarning;
    bool                _isParsing;
    int                 _numberBuildinProtos;
};

enum {
    UPDATE_ALL = 0,
    UPDATE_SELECTION,
    UPDATE_FIELD,
    UPDATE_ADD_NODE,
    UPDATE_REMOVE_NODE,
    UPDATE_CHANGE_INTERFACE_NODE,
    UPDATE_ADD_ROUTE,
    UPDATE_DELETE_ROUTE,
    UPDATE_MODE,
    UPDATE_TIME,
    UPDATE_START_FIELD_EDIT,
    UPDATE_STOP_FIELD_EDIT,
    UPDATE_ENABLE_COLOR_CIRCLE,
    UPDATE_DISABLE_COLOR_CIRCLE,
    UPDATE_CLOSE_COLOR_CIRCLE,
    UPDATE_SELECTED_FIELD,
    UPDATE_SELECTION_NAME
};

class Hint {
};

class FieldUpdate : public Hint {
public:
                    FieldUpdate(Node *n, int f, int i = -1)
                    { node = n; field = f; index = i; }

    Node           *node;
    int             field;
    int             index;
};

class NodeUpdate : public Hint {
public:
                    NodeUpdate(Node *n, Node *p, int f)
                    { node = n; parent = p, field = f; }

    Node           *node;
    Node           *parent;
    int             field;
};

class RouteUpdate : public Hint {
public:
                    RouteUpdate(Node *s, int out, Node *d, int in)
                    { src = s; eventOut = out; dest = d; eventIn = in; }

    Node           *src;
    Node           *dest;
    int             eventOut;
    int             eventIn;
};

void BackupRoutesRec(Node *node, CommandList *list);
#endif // _SCENE_H
