/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * filename: m-init.c                                                      *
 *                                                                         *
 * UTIL C-source: Medical Image Conversion Utility                         *
 *                                                                         *
 * purpose      : initialize library                                       *
 *                                                                         *
 * project      : (X)MedCon by Erik Nolf                                   *
 *                                                                         *
 * Functions    : MdcIgnoreSIGFPE() - ignore floating point exception      *
 *                MdcAcceptSIGFPE() - accept floating point exception      *
 *                MdcSetLocale()    -   set POSIX locale and preserve      *
 *                MdcUnsetLocale()  - unset POSIX locale and retore        * 
 *                MdcInit()         - library usage initialized            *
 *                MdcFinish()       - library usage finished               *
 *                                                                         *
 * Notes        : no dynamic memory allocations here                       *
 *                                                                         * 
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
/* $Id: m-init.c,v 1.10 2009/05/14 20:12:17 enlf Exp $
 */

/*
   Copyright (C) 1997-2009 by Erik Nolf

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2, or (at your option) any later
   version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
   Public License for more details.

   You should have received a copy of the GNU General Public License along
   with this program; if not, write to the Free Software Foundation, Inc.,
   59 Place - Suite 330, Boston, MA 02111-1307, USA.  */

/****************************************************************************
                              H E A D E R S
****************************************************************************/

#include "m-depend.h"

#include <signal.h>
#include <locale.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include "m-init.h"

/****************************************************************************
                              D E F I N E S 
****************************************************************************/

static void (*mdc_old_handler)(int);

static char *mdc_old_locale = NULL;

/****************************************************************************
                            F U N C T I O N S
****************************************************************************/

void MdcIgnoreSIGFPE(void)                  /* before Accept! */
{
  mdc_old_handler = signal(SIGFPE, SIG_IGN);
}

void MdcAcceptSIGFPE(void)                  /* after  Ignore! */
{
  signal(SIGFPE, mdc_old_handler);
}

void MdcSetLocale(void)
{
  char *cur_locale;
  static char locale_string[30];

  /* preserve current locale */
  cur_locale = setlocale(LC_ALL,NULL);
  if (cur_locale == NULL) return;

  if (strlen(cur_locale) >= 30) return;

  strcpy(locale_string,cur_locale);
  mdc_old_locale = locale_string;

  /* set POSIX locale */
  setlocale(LC_ALL, "POSIX");

}

void MdcUnsetLocale(void)
{
  if (mdc_old_locale == NULL) return;

  /* restore previous locale */
  setlocale(LC_ALL,mdc_old_locale);

  /* clean up */
  mdc_old_locale = NULL;

}

void MdcInit(void)
{
  /* ignore floating point exception */
  MdcIgnoreSIGFPE();

  /* set POSIX locale */
  MdcSetLocale();

}

void MdcFinish(void)
{
  /* accept floating point exception */
  MdcAcceptSIGFPE();

  /* unset POSIX locale */
  MdcUnsetLocale();

}
