/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include <QFileDialog>
#include <QTextStream>
#include <QMessageBox>
#include <cmath>
#include <cassert>

#include "MeteoTable.h"
#include "GribPlot.h"
#include "Util.h"
#include "Settings.h"
#include "DataQString.h"


//===================================================================
// MeteoTable : dialog + MeteoTableWidget
//===================================================================
MeteoTableDialog::MeteoTableDialog (
					GriddedPlotter *plotter, 
					double lon, double lat, QString posName)
	: QDialog()
{
	this->plotter = plotter;
	this->lon = lon;
	this->lat = lat;
	this->posName = posName;
    
    if (!plotter->isReaderOk()) {
        QMessageBox::critical (this,
			tr("Error"),tr("Can't create Meteotable:\n\nNo GRIB file opened."));
		delete this;
		return;
    }
    GriddedReader *reader = plotter->getReader();
    GriddedRecord *record;
	if ((record=reader->getFirstRecord()) == NULL) {
        QMessageBox::critical (this,
            	tr("Error"),tr("Can't create Meteotable\n\nGRIB area undefined."));
		delete this;
		return;
    }
	if (!record->isPointInMap(lon, lat)) {
		QMessageBox::critical (this,
				tr("Error"),tr("Can't create Meteotable:\n\nPoint outside GRIB area."));
		delete this;
		return;
	}

	//----------------------------------------------
	setModal(false);
	
	meteoTableWidget = new MeteoTableWidget(plotter, lon,lat, this);
	assert(meteoTableWidget);
	optionsDialog = new DialogMeteotableOptions();
	assert(optionsDialog);
	
	scrollArea = new QScrollArea();
	assert(scrollArea);
	scrollArea->setWidget(meteoTableWidget);
	
	QVBoxLayout *layout = new QVBoxLayout(this);

	QString position = Util::formatPosition(lon, lat);
	QLabel *lbpos;
	if (posName == "") {
		setWindowTitle(position);
		lbpos = new QLabel(tr("Location: ") + position);
	}
	else {
		setWindowTitle(posName);
		lbpos = new QLabel(tr("Location: <b>") +posName + "</b> : " + position);
	}
	QLabel *lbdate = new QLabel(tr("Reference date: ")
							+ Util::formatDateTimeLong(reader->getRefDate()));
	//-------------------------------
	btClose   = new QPushButton(tr("Close"));
	btOptions = new QPushButton(tr("Options"));
	btExport  = new QPushButton(tr("Save"));
    assert(btClose);
    assert(btOptions);
    assert(btExport);
    QFrame *framebts = new QFrame(this);
    assert(framebts);
    QHBoxLayout *hlay = new QHBoxLayout();
    assert(hlay);
        hlay->addWidget(btClose);
        hlay->addWidget(btOptions);
        hlay->addWidget(btExport);
        framebts->setLayout(hlay);
        
	connect(btClose, SIGNAL(clicked()), this, SLOT(reject()));
	connect(btOptions, SIGNAL(clicked()), this, SLOT(slotBtOptions()));
	connect(btExport, SIGNAL(clicked()), this, SLOT(slotBtExport()));
	connect(optionsDialog, SIGNAL(accepted()), this, SLOT(slotOptionsChanged()));
							
	//-------------------------------
	layout->addWidget( lbpos);
	layout->addWidget( lbdate);
	layout->addWidget(scrollArea);	
	layout->addWidget(framebts, 0, Qt::AlignHCenter);
	
	// taille par défaut pour la 1ère ouverture
	adjustSize();
	int w = 800;
	int h = this->height()+80;
    resize( Util::getSetting("meteoTableDialogSize", QSize(w,h)).toSize() );
	show();	
}
//-----------------------------------------
MeteoTableDialog::~MeteoTableDialog()
{
	Util::setSetting("meteoTableDialogSize", size());
	delete optionsDialog;
}

//-----------------------------------------
void MeteoTableDialog::slotBtOptions()
{
	optionsDialog->show();
}

//-----------------------------------------
void MeteoTableDialog::slotOptionsChanged()
{
	delete meteoTableWidget;
	meteoTableWidget = new MeteoTableWidget(plotter, lon,lat, this);
	assert(meteoTableWidget);
	scrollArea->setWidget(meteoTableWidget);
}

//-----------------------------------------
void MeteoTableDialog::slotBtExport ()
{
	QString path = Util::getSetting("slkFilePath", "").toString();
	if (path == "")
		path = "./";
	else
		path += "/";
	QString fileName;
	fileName = QFileDialog::getSaveFileName (this,
					tr("Save SYLK file"), path, "*.slk");
    if (fileName != "")
    {
		if (! fileName.endsWith(".slk", Qt::CaseInsensitive))
			fileName += ".slk";
		
        QFile saveFile (fileName);
        bool ok;
        ok = saveFile.open(QIODevice::WriteOnly);
        if (ok) {
			ok = saveFileSYLK (saveFile);
		}
        if (ok) {
			QFileInfo info(saveFile);
			Util::setSetting("slkFilePath", info.absolutePath() );
        }
        else {
            QMessageBox::critical (this,
                    tr("Error"),  tr("Can't write file."));
        }
		if (saveFile.isOpen())
			saveFile.close();
	}
/*	std::set<DataCode> getAllDataCode (
	meteoTableWidget listVisibleData*/
	
}


//===============================================================
bool MeteoTableDialog::saveFileSYLK (QFile &file)
{
	QTextStream out(&file);
	int dl, dc, col;
	
	QList <MTGribData *> listData = meteoTableWidget->getListVisibleData();
	std::list<DataPointInfo *> lspinfos = meteoTableWidget->getListPinfos(); 
	std::list<DataPointInfo *>::iterator itp;
	DataPointInfo *pinfo;
	
	if (lspinfos.size() == 0) {
		return false;
	}
	
	//----------------------------------------
	// SYLK header
	out << "ID;PzyGrib\n";
	//----------------------------------------
	// Position
	pinfo = *(lspinfos.begin());
	out << "C;Y" << 2 << ";X" << 1
		<<";K\"" << Util::formatPosition (pinfo->x,pinfo->y) << "\"\n";
	if (posName != "")
		out << "C;Y" << 2 << ";X" << 2
			<<";K\"" << posName << "\"\n";
	out << "C;Y" << 1 << ";X" << 1
		<<";K\"" << "zyGrib" << "\"\n";
	//-----------------------------------------------------------------------
	// Dates
	dl = 2;
	dc = 4;
	for (col=0,itp=lspinfos.begin(); itp!=lspinfos.end(); col++,itp++) {
		DataPointInfo *pinfo = *itp;
		out << "C;Y" << dl << ";X" << col+dc
			<<";K\"" << Util::formatDateShort(pinfo->date) << "\"\n";
		out << "C;Y" << dl+1 << ";X" << col+dc
			<<";K\"" << Util::formatTime(pinfo->date) << "\"\n";
	}
	//-----------------------------------------------------------------------
	// All Data
	dl = 4;
	dc = 1;
	for (int j=0; j<listData.size(); j++) {
		MTGribData *data = listData.at(j);
		switch (data->dtc.dataType) {
			case GRB_WIND_XY2D    : 
				dl += SYLK_addData_wind (out, dl, dc, data->dtc);
				break;
			case GRB_WIND_VX      : 
			case GRB_WIND_VY      : 
			case GRB_WIND_DIR     : 
				break;
			default : 
				dl += SYLK_addData_gen (out, dl, dc, data->dtc);
		}
	}
	//----------------------------------------
	// SYLK end of file
	out << "E\n";
	return true;
}
//-----------------------------------------------------------------------------
int MeteoTableDialog::SYLK_addData_gen (QTextStream &out, int lig,int col, DataCode dtc)
{
	std::list<DataPointInfo *> lspinfos = meteoTableWidget->getListPinfos(); 
	std::list<DataPointInfo *>::iterator itp;
	int nblig;
	//-----------------------------------------------------------------------
	// Data Type ; Altitude ; Unit
	//-----------------------------------------------------------------------
	out << "C;Y" << lig << ";X" << col
		<<";K\"" << DataCodeStr::toString_name(dtc) << "\"\n";
	out << "C;Y" << lig << ";X" << col+1
		<<";K\"" << AltitudeStr::toString(dtc.getAltitude()) << "\"\n";
	out << "C;Y" << lig << ";X" << col+2
		<<";K\""+Util::getDataUnit(dtc)+"\"\n";

	int row = col+3;
	for (itp=lspinfos.begin(); itp!=lspinfos.end(); row++,itp++) 
	{
		DataPointInfo *pinfo = *itp;
		float val = pinfo->getDataValue (dtc);
		QString txt = "";
		if (val != GRIB_NOTDEF) {
			switch (dtc.dataType) {
				case GRB_SNOW_CATEG   : 
				case GRB_FRZRAIN_CATEG: 
					txt = Util::formatCategoricalData (val); 
					break;
				case GRB_TEMP_POT     : 
				case GRB_TMAX         : 
				case GRB_TMIN         : 
				case GRB_DEWPOINT     : 
				case GRB_TEMP         : 
				case GRB_THETA_E      : 
					txt = Util::formatTemperature (val, false); 
					break;
				case GRB_DIFF_TEMPDEW : 
					txt = Util::formatTemperature (val+273.15, false); 
					break;
				case GRB_SNOW_DEPTH   : 
					txt = Util::formatSnowDepth (val, false); 
					break;
				case GRB_CLOUD_TOT    : 
				case GRB_HUMID_SPEC   : 
				case GRB_HUMID_REL    : 
					txt = Util::formatPercentValue (val, false); 
					break;
				case GRB_GEOPOT_HGT   :
					txt = Util::formatGeopotAltitude (val, false); 
					break;
				case GRB_CAPE 		  : 
					txt = Util::formatCAPEsfc (val, false); 
					break;
				case GRB_PRESSURE     : 
					txt = Util::formatPressure (val, false); 
					break;
				case GRB_PRECIP_TOT   : 
				case GRB_PRECIP_RATE  : 
					txt = Util::formatRain (val, false); 
					break;
			}
			if (txt != "")
					out << "C;Y" << lig << ";X" << row
						<<";K"<< txt <<"\n";
		}
	}
	nblig = 1;
	return nblig;
}
//-----------------------------------------------------------------------------
int MeteoTableDialog::SYLK_addData_wind (QTextStream &out, int lig,int col, DataCode dtc)
{
	std::list<DataPointInfo *> lspinfos = meteoTableWidget->getListPinfos(); 
	std::list<DataPointInfo *>::iterator itp;
	int nblig = 2;
	//-----------------------------------------------------------------------
	// Line 1 header: Wind speed
	out << "C;Y" << lig << ";X" << col
		<<";K\""+ tr("Wind")+" ("+ tr("speed") +")\"\n";
	out << "C;Y" << lig << ";X" << col+1
		<<";K\"" << AltitudeStr::toString(dtc.getAltitude()) << "\"\n";
	out << "C;Y" << lig << ";X" << col+2
		<<";K\""+Util::getDataUnit(dtc)+"\"\n";
	//-----------------------------------------------------------------------
	// Line 2 header: Wind direction
	out << "C;Y" << lig+1 << ";X" << col
		<<";K\""+ tr("Wind")+" ("+ tr("direction") +")\"\n";
	out << "C;Y" << lig+1 << ";X" << col+1
		<<";K\"" << AltitudeStr::toString(dtc.getAltitude()) << "\"\n";
	out << "C;Y" << lig+1 << ";X" << col+2
		<<";K\""+ tr("°") +"\"\n";
	//-----------------------------------------------------------------------
	// Data
	int row = col+3;
	for (itp=lspinfos.begin(); itp!=lspinfos.end(); row++,itp++) 
	{
		DataPointInfo *pf = *itp;
		float v, dir;
		QString txt;
		if (pf->getWindValues (dtc.getAltitude(), &v, &dir)) {
			txt = Util::formatSpeed (v, false);
			if (txt != "")
					out << "C;Y" << lig << ";X" << row
						<<";K"<< txt <<"\n";
			txt = Util::formatDirection (dir, false);
			if (txt != "")
					out << "C;Y" << lig+1 << ";X" << row
						<<";K"<< txt <<"\n";
		}
	}

	return nblig;
}

    




