/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  double SPpcXcor (const float pc[], int Np, float cor[], int Ncof)

Purpose:
  Convert predictor coefficients to correlation values

Description:
  This routine finds the set of autocorrelation coefficients corresponding to
  a given set of predictor coefficients.  The procedure reverses the Durbin
  recursion.  The output autocorrelation coefficients are normalized such that
  the zero lag correlation is unity.

  This procedure returns Ncor correlations (including cor[0] = 1).  Consider
  a sequence generated by autoregressive process with exactly Np terms.  The
  first Np+1 terms of the autocorrelation function for that sequence match the
  autocorrelation values that have the Np predictor coefficients as the
  corresponding optimum predictor values.  The remaining autocorrelation values
  can be uniquely determined from the Np+1 values.  These extra values are the
  same as would be generated with zeros appended to the vector of predictor
  coefficients.

  Predictor coefficients are usually expressed algebraically as vectors with
  1-offset indexing.  The correspondence to the 0-offset C-arrays is as
  follows.
    p(1) <==> pc[0]       first predictor coefficient
    p(i) <==> pc[i-1]     1 <= i < Np

Parameters:
  <-  double SPpcXcor
      Normalized prediction error.  This is the energy of the prediction
      residual for a signal with the autocorrelation calculated by this routine
      applied to a predictor with the given predictor coefficients.  Note that
      this value may be negative if the given predictor coefficients do not
      correspond to a minimum phase prediction error filter.
   -> const float pc[]
      Vector of predictor coefficients (Np values).  These are the coefficients
      of the predictor filter, with pc[0] being the predictor coefficient
      corresponding to lag 1, and pc[Np-1] corresponding to lag Np.  The
      predictor coefficients must correspond to a minimum phase prediction
      error filter.
   -> int Np
      Number of predictor coefficients (at most 50)
  <-  float cor[]
      Autocorrelation coefficients (Np+1 values).  The first element
      corresponds to lag zero, while the last element corresponds to lag Np.
   -> int Ncof
      Number of correlation values to be returned (normally at least Np + 1)

Author / revision:
  P. Kabal  Copyright (C) 1997
  $Revision: 1.15 $  $Date: 1997/10/10 19:58:13 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: SPpcXcor.c 1.15 1997/10/10 libtsp-v3r0 $";

#include <libtsp.h>
#include <libtsp/SPmsg.h>

#define MINV(a, b)	(((a) < (b)) ? (a) : (b))
#define MAXNP		50


double
SPpcXcor(pc, Np, rxx, Ncor)

     const float pc[];
     int Np;
     float rxx[];
     int Ncor;

{
  int i, j, k;
  int Nv;
  double perr, t, sum;
  float rc[MAXNP];
  float Pc[MAXNP];

/*
   The process of converting to autocorrelation coefficients needs access to
   the predictor coefficients for predictors of all orders 1 to Np.  To do this
   we first convert to reflection coefficients and then build up the predictor
   coefficients of increasing order from the reflection coefficients
*/

  if (Np > MAXNP)
    UThalt ("SPpcXcor: %s", SPM_TooManyCoef);

/* Convert prediction coefficients to reflection coefficients
   SPpcXrc prints a message if the prediction error filter is non-minimum phase
*/
  SPpcXrc (pc, rc, Np);

/* Main loop: convert reflection coefficients to correlation coefficients */
  Nv = MINV (Np, Ncor - 1);
  rxx[0] = 1.0;
  perr = 1.0;
  for (k = 0; k < Nv; ++k) {

    sum = -rc[k] * perr;
    for (i = 0; i < k; ++i)
      sum = sum + rxx[k-i] * Pc[i];
    rxx[k+1] = sum;

    perr = perr * (1.0 - rc[k] * rc[k]);

/* Convert reflection coefficients to predictor coefficients (order k) */
    if (k >= Ncor - 2)
      break;
    Pc[k] = -rc[k];
    for (i = 0, j = k - 1; i < j; ++i, --j) {
      t = Pc[i] + rc[k] * Pc[j];
      Pc[j] = Pc[j] + rc[k] * Pc[i];
      Pc[i] = t;
    }
    if (i == j)
      Pc[i] = Pc[i] + rc[k] * Pc[i];
  }

/* Generate the remaining correlation values */
  for (k = Np; k < Ncor - 1; ++k) {
    sum = 0.0;
    for (i = 0; i < Np; ++i)
      sum = sum + rxx[k-i] * Pc[i];
    rxx[k+1] = sum;
  }

  return perr;
}
