/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


/**
 * @file diagnostics/unittest/test_system/current_test_logger.hpp
 *
 * $Id$
 *
 * @author Christian Schallhart
 *
 * @brief [LEVEL: beta] @ref diagnostics::unittest::internal::Current_Test_Logger
 *
 * @test none
 */


#ifndef DIAGNOSTICS__UNITTEST__TEST_SYSTEM__CURRENT_TEST_LOGGER__INCLUDE_GUARD
#define DIAGNOSTICS__UNITTEST__TEST_SYSTEM__CURRENT_TEST_LOGGER__INCLUDE_GUARD

#include <diagnostics/frame/namespace.hpp>
#include <diagnostics/unittest/namespace.hpp>

DIAGNOSTICS_NAMESPACE_BEGIN;

// used by reference in the interface
// diagnostics/frame/logger.hpp
class Logger;

UNITTEST_NAMESPACE_BEGIN;

// used as friend in the interface
// diagnostics/unittest/test_system/run_test_suite_traversal.hpp
class Run_Test_Suite_Traversal;


INTERNAL_NAMESPACE_BEGIN;

/**
 * @brief This class is for internal test purposes -- to switch on/off
 * the test logger.
 *
 * @attention To use this library, it is not necessary to understand
 * this clss. You only need this class if you intend to use
 * diagnostics to test diagnostics.
 *
 * @attention This class is only allowd to be used by @ref
 * Test_Run_Result and @ref No_Implicit_Test_Logging_Guard. If you use
 * this, make sure you really need it. The user interface is @ref
 * TEST_NO_IMPLICIT_LOGGING_ENTER and @ref
 * TEST_NO_IMPLICIT_LOGGING_EXIT.
 *
 * More precisely, when testing the testing facilities, it is
 * sometimes necessary to remove the @ref Test_Logger from the @ref
 * Record stream, since the occuring records are test-relevant -- but
 * their occurence is PART OF CURRENT TESTCASE and does NOT DESCRIBE 
 * THE TESTCASE at hand. 
 *
 * Therefore, we need a possibilty to remove the logger temporarily
 * from the stream; the records occuring in the meantime are not
 * included in the final assessment of the testcase (since they are
 * not logged by the correpsonding @ref Test_Logger).
 *
 * To make things worse, if we test a @ref Run_Test_Suite_Traversal
 * (directly or indirectly) with this framework, the following situation arises:
 *
 * @arg Run_Test_Suite_Traversal.traverse()  
 * @arg ... Testing Code
 * @arg ... ... Tested Code
 * @arg ... ... ... Run_Test_Suite_Traversal.traverse()
 * @arg ... ... ... ... Testing Code (for testing the testing)
 * @arg ... ... ... ... ... Tested Code (for testing the testing)
 *
 * In this case, both calls to @ref Run_Test_Suite_Traversal::traverse
 * will install their own @ref Test_Logger.
 *
 * But we would like to remove the first, while the second one stays
 * in place to be tested.
 * 
 * Therefore, Current_Test_Logger contains a stack of @ref
 * Test_Logger. The calls to @ref unregister_logger and @ref
 * register_logger disable/enable the last installed logger.
 *
 * The occuring calling sequence to resolve the above example:

 * @arg Run_Test_Suite_Traversal.traverse()  
 * @arg ... Run_Test_Suite_Traversal calls push_logger(logger1)
 * @arg ... Testing Code
 * @arg ... ... unregister_logger()
 * @arg ... ... Tested Code
 * @arg ... ... ... Run_Test_Suite_Traversal.traverse()
 * @arg ... ... ... ... Run_Test_Suite_Traversal calls push_logger(logger2)
 * @arg ... ... ... ... Testing Code (for testing the testing)
 * @arg ... ... ... ... ... Tested Code (for testing the testing)
 * @arg ... ... ... ... Run_Test_Suite_Traversal calls pop_logger()
 * @arg ... ... register_logger()
 * @arg ... Run_Test_Suite_Traversal calls pop_logger()
 *
 * @note long distance friend ship (to @ref Run_Test_Suite_Traversal)
 * is used, since this class should not be part of the interface
 * (Run_Test_Suite_Traversal), the common user relies on.
 *
 * @nosubgrouping
 */
struct Current_Test_Logger
{
    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Interface for Nested Testing
     * @{
     */
public:
    /**
     * @brief switches on the current top-level @ref Test_Logger. If
     * the current top-level @ref Test_Logger is already switched on,
     * this is a noop.
     *
     * @post the last pushed @ref Test_Logger is active
     *
     * @throw never
     */ 
    static void unregister_logger();

    /**
     * @brief switches off the current top-level @ref Test_Logger. If
     * the current top-level @ref Test_Logger is already switched off,
     * this is a noop.
     *
     * @post the last pushed @ref Test_Logger is passive
     *
     * @throw never
     */
    static void register_logger();
    // @}

    ////////////////////////////////////////////////////////////////////////////////
    /**
     * @name Interface for ::diagnostics::unittest::Run_Test_Suite_Traversal
     * @{
     */
private:
    friend class ::diagnostics::unittest::Run_Test_Suite_Traversal;
    /** 
     * @brief Set the current top-level @ref Test_Logger and activates
     * it.
     *
     * @throw never
     */
    static void push_logger(Logger * const logger);

    /**
     * @brief Deactivates the current top-level @ref Test_Logger and
     * removes it from the stack.
     *
     * @pre for each call to pop, there must be a call to push beforehand.
     *
     * @attention: not meeting the precondition means undefined behavior.
     *
     * throw never
     */ 
    static void pop_logger();
    // @}
};


/**
 * @class No_Implicit_Test_Logging_Guard
 *
 * @brief A guard for calling @ref
 * Current_Test_Logger::unregister_logger and later @ref
 * Current_Test_Logger::register_logger.
 *
 * @nosubgrouping
 */
struct No_Implicit_Test_Logging_Guard
{
    inline No_Implicit_Test_Logging_Guard()
    {
	Current_Test_Logger::unregister_logger();
    }
    
    inline ~No_Implicit_Test_Logging_Guard()
    {
	Current_Test_Logger::register_logger();
    }
};

/**
 * @brief A block with no implicit test logging. The current top-level
 * @ref ::diagnostics::unittest::Test_Logger is removed from the @ref
 * ::diagnostics::Logging_Config during the block.  Use with extreme
 * care. Only provided internally, to test some test facilities.
 */
#define TEST_NO_IMPLICIT_LOGGING_ENTER { ::diagnostics::unittest::internal::No_Implicit_Test_Logging_Guard guard; {
#define TEST_NO_IMPLICIT_LOGGING_EXIT  }}

INTERNAL_NAMESPACE_END;
UNITTEST_NAMESPACE_END;
DIAGNOSTICS_NAMESPACE_END;

#endif
// vim:ts=4:sw=4
