minetest.register_privilege("worldedit", "Can use WorldEdit commands")

worldedit.set_pos = {}

worldedit.pos1 = {}
worldedit.pos2 = {}

dofile(minetest.get_modpath("worldedit_commands") .. "/mark.lua")

--determines whether `nodename` is a valid node name, returning a boolean
worldedit.node_is_valid = function(nodename)
	return minetest.registered_nodes[nodename] ~= nil
	or minetest.registered_nodes["default:" .. nodename] ~= nil
end

--determines the axis in which a player is facing, returning an axis ("x", "y", or "z") and the sign (1 or -1)
worldedit.player_axis = function(name)
	local dir = minetest.env:get_player_by_name(name):get_look_dir()
	local x, y, z = math.abs(dir.x), math.abs(dir.y), math.abs(dir.z)
	if x > y then
		if x > z then
			return "x", dir.x > 0 and 1 or -1
		end
	elseif y > z then
		return "y", dir.y > 0 and 1 or -1
	end
	return "z", dir.z > 0 and 1 or -1
end

minetest.register_chatcommand("/reset", {
	params = "",
	description = "Reset the region so that it is empty",
	privs = {worldedit=true},
	func = function(name, param)
		worldedit.pos1[name] = nil
		worldedit.pos2[name] = nil
		worldedit.mark_pos1(name)
		worldedit.mark_pos2(name)
		minetest.chat_send_player(name, "WorldEdit region reset", false)
	end,
})

minetest.register_chatcommand("/mark", {
	params = "",
	description = "Show markers at the region positions",
	privs = {worldedit=true},
	func = function(name, param)
		worldedit.mark_pos1(name)
		worldedit.mark_pos2(name)
		minetest.chat_send_player(name, "WorldEdit region marked", false)
	end,
})

minetest.register_chatcommand("/pos1", {
	params = "",
	description = "Set WorldEdit region position 1 to the player's location",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = minetest.env:get_player_by_name(name):getpos()
		pos.x, pos.y, pos.z = math.floor(pos.x + 0.5), math.floor(pos.y + 0.5), math.floor(pos.z + 0.5)
		worldedit.pos1[name] = pos
		worldedit.mark_pos1(name)
		minetest.chat_send_player(name, "WorldEdit position 1 set to " .. minetest.pos_to_string(pos), false)
	end,
})

minetest.register_chatcommand("/pos2", {
	params = "",
	description = "Set WorldEdit region position 2 to the player's location",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = minetest.env:get_player_by_name(name):getpos()
		pos.x, pos.y, pos.z = math.floor(pos.x + 0.5), math.floor(pos.y + 0.5), math.floor(pos.z + 0.5)
		worldedit.pos2[name] = pos
		worldedit.mark_pos2(name)
		minetest.chat_send_player(name, "WorldEdit position 2 set to " .. minetest.pos_to_string(pos), false)
	end,
})

minetest.register_chatcommand("/p", {
	params = "set/set1/set2/get",
	description = "Set WorldEdit region, WorldEdit position 1, or WorldEdit position 2 by punching nodes, or display the current WorldEdit region",
	privs = {worldedit=true},
	func = function(name, param)
		if param == "set" then --set both WorldEdit positions
			worldedit.set_pos[name] = "pos1"
			minetest.chat_send_player(name, "Select positions by punching two nodes", false)
		elseif param == "set1" then --set WorldEdit position 1
			worldedit.set_pos[name] = "pos1only"
			minetest.chat_send_player(name, "Select position 1 by punching a node", false)
		elseif param == "set2" then --set WorldEdit position 2
			worldedit.set_pos[name] = "pos2"
			minetest.chat_send_player(name, "Select position 2 by punching a node", false)
		elseif param == "get" then --display current WorldEdit positions
			if worldedit.pos1[name] ~= nil then
				minetest.chat_send_player(name, "WorldEdit position 1: " .. minetest.pos_to_string(worldedit.pos1[name]), false)
			else
				minetest.chat_send_player(name, "WorldEdit position 1 not set", false)
			end
			if worldedit.pos2[name] ~= nil then
				minetest.chat_send_player(name, "WorldEdit position 2: " .. minetest.pos_to_string(worldedit.pos2[name]), false)
			else
				minetest.chat_send_player(name, "WorldEdit position 2 not set", false)
			end
		else
			minetest.chat_send_player(name, "Unknown subcommand: " .. param, false)
		end
	end,
})

minetest.register_on_punchnode(function(pos, node, puncher)
	local name = puncher:get_player_name()
	if name ~= "" and worldedit.set_pos[name] ~= nil then --currently setting position
		if worldedit.set_pos[name] == "pos1" then --setting position 1
			worldedit.pos1[name] = pos
			worldedit.mark_pos1(name)
			worldedit.set_pos[name] = "pos2" --set position 2 on the next invocation
			minetest.chat_send_player(name, "WorldEdit position 1 set to " .. minetest.pos_to_string(pos), false)
		elseif worldedit.set_pos[name] == "pos1only" then --setting position 1 only
			worldedit.pos1[name] = pos
			worldedit.mark_pos1(name)
			worldedit.set_pos[name] = nil --finished setting positions
			minetest.chat_send_player(name, "WorldEdit position 1 set to " .. minetest.pos_to_string(pos), false)
		elseif worldedit.set_pos[name] == "pos2" then --setting position 2
			worldedit.pos2[name] = pos
			worldedit.mark_pos2(name)
			worldedit.set_pos[name] = nil --finished setting positions
			minetest.chat_send_player(name, "WorldEdit position 2 set to " .. minetest.pos_to_string(pos), false)
		end
	end
end)

minetest.register_chatcommand("/volume", {
	params = "",
	description = "Display the volume of the current WorldEdit region",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local volume = worldedit.volume(pos1, pos2)
		minetest.chat_send_player(name, "Current WorldEdit region has a volume of " .. volume .. " nodes (" .. pos2.x - pos1.x .. "*" .. pos2.y - pos1.y .. "*" .. pos2.z - pos1.z .. ")", false)
	end,
})

minetest.register_chatcommand("/set", {
	params = "<node>",
	description = "Set the current WorldEdit region to <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		if param == "" or not worldedit.node_is_valid(param) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
        local count = worldedit.set(pos1, pos2, param, tenv)
		minetest.chat_send_player(name, count .. " nodes set", false)
	end,
})

minetest.register_chatcommand("/replace", {
	params = "<search node> <replace node>",
	description = "Replace all instances of <search node> with <replace node> in the current WorldEdit region",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, searchnode, replacenode = param:find("^([^%s]+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(searchnode) then
			minetest.chat_send_player(name, "Invalid search node name: " .. searchnode, false)
			return
		end
		if not worldedit.node_is_valid(replacenode) then
			minetest.chat_send_player(name, "Invalid replace node name: " .. replacenode, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.replace(pos1, pos2, searchnode, replacenode, tenv)
		minetest.chat_send_player(name, count .. " nodes replaced", false)
	end,
})

minetest.register_chatcommand("/replaceinverse", {
	params = "<search node> <replace node>",
	description = "Replace all nodes other than <search node> with <replace node> in the current WorldEdit region",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, searchnode, replacenode = param:find("^([^%s]+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(searchnode) then
			minetest.chat_send_player(name, "Invalid search node name: " .. searchnode, false)
			return
		end
		if not worldedit.node_is_valid(replacenode) then
			minetest.chat_send_player(name, "Invalid replace node name: " .. replacenode, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.replaceinverse(pos1, pos2, searchnode, replacenode, tenv)
		minetest.chat_send_player(name, count .. " nodes replaced", false)
	end,
})

minetest.register_chatcommand("/hollowsphere", {
	params = "<radius> <node>",
	description = "Add hollow sphere at WorldEdit position 1 with radius <radius>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, radius, nodename = param:find("^(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.hollow_sphere(pos, tonumber(radius), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes added", false)
	end,
})

minetest.register_chatcommand("/sphere", {
	params = "<radius> <node>",
	description = "Add sphere at WorldEdit position 1 with radius <radius>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, radius, nodename = param:find("^(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.sphere(pos, tonumber(radius), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes added", false)
	end,
})

minetest.register_chatcommand("/hollowdome", {
	params = "<radius> <node>",
	description = "Add hollow dome at WorldEdit position 1 with radius <radius>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, radius, nodename = param:find("^(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.hollow_dome(pos, tonumber(radius), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes added", false)
	end,
})

minetest.register_chatcommand("/dome", {
	params = "<radius> <node>",
	description = "Add dome at WorldEdit position 1 with radius <radius>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, radius, nodename = param:find("^(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.dome(pos, tonumber(radius), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes added", false)
	end,
})

minetest.register_chatcommand("/hollowcylinder", {
	params = "x/y/z/? <length> <radius> <node>",
	description = "Add hollow cylinder at WorldEdit position 1 along the x/y/z/? axis with length <length> and radius <radius>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, axis, length, radius, nodename = param:find("^([xyz%?])%s+([+-]?%d+)%s+(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if axis == "?" then
			axis, sign = worldedit.player_axis(name)
			length = length * sign
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.hollow_cylinder(pos, axis, tonumber(length), tonumber(radius), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes added", false)
	end,
})

minetest.register_chatcommand("/cylinder", {
	params = "x/y/z/? <length> <radius> <node>",
	description = "Add cylinder at WorldEdit position 1 along the x/y/z/? axis with length <length> and radius <radius>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, axis, length, radius, nodename = param:find("^([xyz%?])%s+([+-]?%d+)%s+(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if axis == "?" then
			axis, sign = worldedit.player_axis(name)
			length = length * sign
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.cylinder(pos, axis, tonumber(length), tonumber(radius), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes added", false)
	end,
})

minetest.register_chatcommand("/pyramid", {
	params = "<height> <node>",
	description = "Add pyramid at WorldEdit position 1 with height <height>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, size, nodename = param:find("(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.pyramid(pos, tonumber(size), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes added", false)
	end,
})

minetest.register_chatcommand("/spiral", {
	params = "<width> <height> <space> <node>",
	description = "Add spiral at WorldEdit position 1 with width <width>, height <height>, space between walls <space>, composed of <node>",
	privs = {worldedit=true},
	func = function(name, param)
		local pos = worldedit.pos1[name]
		if pos == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, width, height, space, nodename = param:find("(%d+)%s+(%d+)%s+(%d+)%s+([^%s]+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if not worldedit.node_is_valid(nodename) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.spiral(pos, tonumber(width), tonumber(height), tonumber(space), nodename, tenv)
		minetest.chat_send_player(name, count .. " nodes changed", false)
	end,
})

minetest.register_chatcommand("/copy", {
	params = "x/y/z/? <amount>",
	description = "Copy the current WorldEdit region along the x/y/z/? axis by <amount> nodes",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, axis, amount = param:find("^([xyz%?])%s+([+-]?%d+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if axis == "?" then
			axis, sign = worldedit.player_axis(name)
			amount = amount * sign
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.copy(pos1, pos2, axis, tonumber(amount), tenv)
		minetest.chat_send_player(name, count .. " nodes copied", false)
	end,
})

minetest.register_chatcommand("/move", {
	params = "x/y/z/? <amount>",
	description = "Move the current WorldEdit region along the x/y/z/? axis by <amount> nodes",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, axis, amount = param:find("^([xyz%?])%s+([+-]?%d+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if axis == "?" then
			axis, sign = worldedit.player_axis(name)
			amount = amount * sign
		end

		local count = worldedit.move(pos1, pos2, axis, tonumber(amount))

		pos1[axis] = pos1[axis] + amount
		pos2[axis] = pos2[axis] + amount
		worldedit.mark_pos1(name)
		worldedit.mark_pos2(name)

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.copy(pos1, pos2, axis, tonumber(amount), tenv)
		minetest.chat_send_player(name, count .. " nodes moved", false)
	end,
})

minetest.register_chatcommand("/stack", {
	params = "x/y/z/? <count>",
	description = "Stack the current WorldEdit region along the x/y/z/? axis <count> times",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, axis, count = param:find("^([xyz%?])%s+([+-]?%d+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if axis == "?" then
			axis, sign = worldedit.player_axis(name)
			count = count * sign
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.stack(pos1, pos2, axis, tonumber(count), tenv)
		minetest.chat_send_player(name, count .. " nodes stacked", false)
	end,
})

minetest.register_chatcommand("/transpose", {
	params = "x/y/z/? x/y/z/?",
	description = "Transpose the current WorldEdit region along the x/y/z/? and x/y/z/? axes",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, axis1, axis2 = param:find("^([xyz%?])%s+([xyz%?])$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if axis1 == "?" then
			axis1 = worldedit.player_axis(name)
		end
		if axis2 == "?" then
			axis2 = worldedit.player_axis(name)
		end
		if axis1 == axis2 then
			minetest.chat_send_player(name, "Invalid usage: axes are the same", false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count, pos1, pos2 = worldedit.transpose(pos1, pos2, axis1, axis2, tenv)

		--reset markers to transposed positions
		worldedit.pos1[name] = pos1
		worldedit.pos2[name] = pos2
		worldedit.mark_pos1(name)
		worldedit.mark_pos2(name)

		minetest.chat_send_player(name, count .. " nodes transposed", false)
	end,
})

minetest.register_chatcommand("/flip", {
	params = "x/y/z/?",
	description = "Flip the current WorldEdit region along the x/y/z/? axis",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		if param == "?" then
			param = worldedit.player_axis(name)
		end
		if param ~= "x" and param ~= "y" and param ~= "z" then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.flip(pos1, pos2, param, tenv)
		minetest.chat_send_player(name, count .. " nodes flipped", false)
	end,
})

minetest.register_chatcommand("/rotate", {
	params = "<axis> <angle>",
	description = "Rotate the current WorldEdit region around the axis <axis> by angle <angle> (90 degree increment)",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, axis, angle = param:find("^([xyz%?])%s+([+-]?%d+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if axis == "?" then
			axis = worldedit.player_axis(name)
		end
		if angle % 90 ~= 0 then
			minetest.chat_send_player(name, "Invalid usage: angle must be multiple of 90", false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count, pos1, pos2 = worldedit.rotate(pos1, pos2, axis, angle, tenv)

		--reset markers to rotated positions
		worldedit.pos1[name] = pos1
		worldedit.pos2[name] = pos2
		worldedit.mark_pos1(name)
		worldedit.mark_pos2(name)

		minetest.chat_send_player(name, count .. " nodes rotated", false)
	end,
})

minetest.register_chatcommand("/orient", {
	params = "<angle>",
	description = "Rotate oriented nodes in the current WorldEdit region around the Y axis by angle <angle> (90 degree increment)",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local found, _, angle = param:find("^([+-]?%d+)$")
		if found == nil then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end
		if angle % 90 ~= 0 then
			minetest.chat_send_player(name, "Invalid usage: angle must be multiple of 90", false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.orient(pos1, pos2, angle, tenv)

		minetest.chat_send_player(name, count .. " nodes oriented", false)
	end,
})

minetest.register_chatcommand("/fixlight", {
	params = "",
	description = "Fix the lighting in the current WorldEdit region",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.fixlight(pos1, pos2, tenv)
		minetest.chat_send_player(name, count .. " nodes updated", false)
	end,
})

minetest.register_chatcommand("/hide", {
	params = "",
	description = "Hide all nodes in the current WorldEdit region non-destructively",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.hide(pos1, pos2, tenv)
		minetest.chat_send_player(name, count .. " nodes hidden", false)
	end,
})

minetest.register_chatcommand("/suppress", {
	params = "<node>",
	description = "Suppress all <node> in the current WorldEdit region non-destructively",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		if param == "" or not worldedit.node_is_valid(param) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.suppress(pos1, pos2, param, tenv)
		minetest.chat_send_player(name, count .. " nodes suppressed", false)
	end,
})

minetest.register_chatcommand("/highlight", {
	params = "<node>",
	description = "Highlight <node> in the current WorldEdit region by hiding everything else non-destructively",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		if param == "" or not worldedit.node_is_valid(param) then
			minetest.chat_send_player(name, "Invalid node name: " .. param, false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.highlight(pos1, pos2, param, tenv)
		minetest.chat_send_player(name, count .. " nodes highlighted", false)
	end,
})

minetest.register_chatcommand("/restore", {
	params = "",
	description = "Restores nodes hidden with WorldEdit in the current WorldEdit region",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.restore(pos1, pos2, tenv)
		minetest.chat_send_player(name, count .. " nodes restored", false)
	end,
})

minetest.register_chatcommand("/save", {
	params = "<file>",
	description = "Save the current WorldEdit region to \"(world folder)/schems/<file>.we\"",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		if param == "" then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end

		local result, count = worldedit.serialize(pos1, pos2)

		local path = minetest.get_worldpath() .. "/schems"
		local filename = path .. "/" .. param .. ".we"
		os.execute("mkdir \"" .. path .. "\"") --create directory if it does not already exist
		local file, err = io.open(filename, "wb")
		if err ~= nil then
			minetest.chat_send_player(name, "Could not save file to \"" .. filename .. "\"", false)
			return
		end
		file:write(result)
		file:flush()
		file:close()

		minetest.chat_send_player(name, count .. " nodes saved", false)
	end,
})

minetest.register_chatcommand("/allocate", {
	params = "<file>",
	description = "Set the region defined by nodes from \"(world folder)/schems/<file>.we\" as the current WorldEdit region",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1 = worldedit.pos1[name]
		if pos1 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		if param == "" then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end

		local filename = minetest.get_worldpath() .. "/schems/" .. param .. ".we"
		local file, err = io.open(filename, "rb")
		if err ~= nil then
			minetest.chat_send_player(name, "Could not open file \"" .. filename .. "\"", false)
			return
		end
		local value = file:read("*a")
		file:close()

		if worldedit.valueversion(value) == 0 then --unknown version
			minetest.chat_send_player(name, "Invalid file: file is invalid or created with newer version of WorldEdit", false)
			return
		end
		local nodepos1, nodepos2, count = worldedit.allocate(pos1, value)

		worldedit.pos1[name] = nodepos1
		worldedit.mark_pos1(name)
		worldedit.pos2[name] = nodepos2
		worldedit.mark_pos2(name)

		minetest.chat_send_player(name, count .. " nodes allocated", false)
	end,
})

minetest.register_chatcommand("/load", {
	params = "<file>",
	description = "Load nodes from \"(world folder)/schems/<file>[.we[m]]\" with position 1 of the current WorldEdit region as the origin",
	privs = {worldedit=true},
	func = function(name, param)
		local pos1 = worldedit.pos1[name]
		if pos1 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		if param == "" then
			minetest.chat_send_player(name, "Invalid usage: " .. param, false)
			return
		end

		--find the file in the world path
		local testpaths = {
			minetest.get_worldpath() .. "/schems/" .. param,
			minetest.get_worldpath() .. "/schems/" .. param .. ".we",
			minetest.get_worldpath() .. "/schems/" .. param .. ".wem",
		}
		local file, err
		for index, path in ipairs(testpaths) do
			file, err = io.open(path, "rb")
			if not err then
				break
			end
		end
		if err then
			minetest.chat_send_player(name, "Could not open file \"" .. param .. "\"", false)
			return
		end
		local value = file:read("*a")
		file:close()

		if worldedit.valueversion(value) == 0 then --unknown version
			minetest.chat_send_player(name, "Invalid file: file is invalid or created with newer version of WorldEdit", false)
			return
		end

		local tenv = minetest.env
		if worldedit.ENABLE_QUEUE then
			tenv = worldedit.quene_aliasenv
		end
		local count = worldedit.deserialize(pos1, value, tenv)

		minetest.chat_send_player(name, count .. " nodes loaded", false)
	end,
})

minetest.register_chatcommand("/lua", {
	params = "<code>",
	description = "Executes <code> as a Lua chunk in the global namespace",
	privs = {worldedit=true, server=true},
	func = function(name, param)
		local err = worldedit.lua(param)
		if err then
			minetest.chat_send_player(name, "Code error: " .. err, false)
		else
			minetest.chat_send_player(name, "Code successfully executed", false)
		end
	end,
})

minetest.register_chatcommand("/luatransform", {
	params = "<code>",
	description = "Executes <code> as a Lua chunk in the global namespace with the variable pos available, for each node in the current WorldEdit region",
	privs = {worldedit=true, server=true},
	func = function(name, param)
		local pos1, pos2 = worldedit.pos1[name], worldedit.pos2[name]
		if pos1 == nil or pos2 == nil then
			minetest.chat_send_player(name, "No WorldEdit region selected", false)
			return
		end

		local err = worldedit.luatransform(pos1, pos2, param)
		if err then
			minetest.chat_send_player(name, "Code error: " .. err, false)
		else
			minetest.chat_send_player(name, "Code successfully executed", false)
		end
	end,
})
