/*
 * Copyright (C) 2016-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* System configuration controller (SYSCFG) */

/*
 * For most of the comments, infos, ... see:
 * ST - RM0383 - Reference Manual - STM32F411xC/E advanced ARM-based
 * 32-bit MCUs
 */

#define DEBUG_CONTROL_FLOW	1

#ifdef INCLUDE
#endif /* INCLUDE */
#ifdef STATE

struct {
	/* Input State */
	unsigned int state_in['h' - 'a' + 1][16];

	/* External interrupt configuration register 1-4 */
	uint8_t exti[16];

	uint32_t reg[0x1000 >> 2];
} NAME;

#endif /* STATE */
#ifdef EXPORT

/*forward*/ static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val);
/*forward*/ static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp);
/*forward*/ static void
NAME_(inNM_set)(struct cpssp *cpssp, int n, int m, unsigned int val);
/*forward*/ static void
NAME_(reset)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(create)(struct cpssp *cpssp);
/*forward*/ static void
NAME_(destroy)(struct cpssp *cpssp);

#endif /* EXPORT */
#ifdef BEHAVIOR

static void
NAME_(output_update)(struct cpssp *cpssp, int pin)
{
	unsigned int val;

	val = cpssp->NAME.state_in[cpssp->NAME.exti[pin] & 0x7][pin];

	NAME_(outN_set)(cpssp, pin, val);
}

static void
NAME_(st)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t val)
{
	unsigned int n;

	addr &= 0x3ff;

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, val=0x%08x\n",
				__FUNCTION__, addr, bs, val);
	}

	switch (addr) {
	case 0x008:
		/* External interrupt configuration register 1 */
	case 0x00c:
		/* External interrupt configuration register 2 */
	case 0x010:
		/* External interrupt configuration register 3 */
	case 0x014:
		/* External interrupt configuration register 4 */
		n = addr - 8;
		if ((bs >> 3) & 1) {
			/* Bit 31-24: Reserved */
		}
		if ((bs >> 2) & 1) {
			/* Bit 23-16: Reserved */
		}
		if ((bs >> 1) & 1) {
			cpssp->NAME.exti[n + 3] = (val >> 12) & 0xf;
			cpssp->NAME.exti[n + 2] = (val >> 8) & 0xf;
		}
		if ((bs >> 0) & 1) {
			cpssp->NAME.exti[n + 1] = (val >> 4) & 0xf;
			cpssp->NAME.exti[n + 0] = (val >> 0) & 0xf;
		}
		break;
	default:
		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);

		cpssp->NAME.reg[addr >> 2] = val;
		break;
	}
}

static void
NAME_(ld)(struct cpssp *cpssp, uint32_t addr, unsigned int bs, uint32_t *valp)
{
	unsigned int n;

	addr &= 0x3ff;

	switch (addr) {
	case 0x008:
		/* External interrupt configuration register 1 */
	case 0x00c:
		/* External interrupt configuration register 2 */
	case 0x010:
		/* External interrupt configuration register 3 */
	case 0x014:
		/* External interrupt configuration register 4 */
		n = addr - 8;

		*valp = 0;

		/* Bit 31-24: Reserved */

		/* Bit 23-16: Reserved */

		*valp |= cpssp->NAME.exti[n + 3] << 12;
		*valp |= cpssp->NAME.exti[n + 2] << 8;

		*valp |= cpssp->NAME.exti[n + 1] << 4;
		*valp |= cpssp->NAME.exti[n + 0] << 0;
		break;
	default:
		*valp = cpssp->NAME.reg[addr >> 2];

		fprintf(stderr, "WARNING: %s: addr=0x%08x\n",
				__FUNCTION__, addr);
		break;
	}

	if (DEBUG_CONTROL_FLOW) {
		fprintf(stderr, "%s: addr=0x%08x, bs=0x%x, *valp=0x%08x\n",
				__FUNCTION__, addr, bs, *valp);
	}
}

static void
NAME_(inNM_set)(struct cpssp *cpssp, int port, int pin, unsigned int val)
{
	assert('a' <= 'a' + port && port + 'a' <= 'h');
	assert(0 <= pin && pin < 16);

	cpssp->NAME.state_in[port][pin] = val;
	NAME_(output_update)(cpssp, pin);
}

static void
NAME_(reset)(struct cpssp *cpssp)
{
	unsigned int n;

	/* External interrupt configuration register 1-4 */
	for (n = 0; n < 16; n++) {
		cpssp->NAME.exti[n] = 0x0;
	}
}

static void
NAME_(create)(struct cpssp *cpssp)
{
	unsigned int port;
	unsigned int pin;

	for (port = 'a'; port <= 'h'; port++) {
		for (pin = 0; pin < 16; pin++) {
			cpssp->NAME.state_in[port - 'a'][pin] = SIG_STD_LOGIC_Z;
		}
	}
	for (pin = 0; pin < 16; pin++) {
		NAME_(output_update)(cpssp, pin);
	}
}

static void
NAME_(destroy)(struct cpssp *cpssp)
{
}

#endif /* BEHAVIOR */

#undef DEBUG_CONTROL_FLOW
